﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/edid.h>
#include <nn/vi/vi_Result.h>
#include <nn/vi/vi_ResultPrivate.h>
#include <nn/vi/vi_DisplayModeInfo.h>
#include "vi_AndroidDisplay.h"
#include "vi_IModeFilter.h"

nn::vi::detail::AndroidDisplay::AndroidDisplay(int id) NN_NOEXCEPT
    : m_Id(id)
{
    switch( id )
    {
    case android::ISurfaceComposer::eDisplayIdHdmi:
    case android::ISurfaceComposer::eDisplayIdMain:
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    };
}

nn::vi::detail::AndroidDisplay::~AndroidDisplay() NN_NOEXCEPT
{
    Close();
}

nn::Result nn::vi::detail::AndroidDisplay::Open() NN_NOEXCEPT
{
    // new is overloaded to use NvOsAlloc
    // this doesn't come from client heap
    m_Client = new android::SurfaceComposerClient();

    if( m_Client == nullptr )
    {
        return nn::vi::ResultOperationFailed();
    }

    m_Display = android::SurfaceComposerClient::getBuiltInDisplay(m_Id);

    if( m_Display == nullptr )
    {
        return nn::vi::ResultOperationFailed();
    }

    return nn::ResultSuccess();
}

void nn::vi::detail::AndroidDisplay::Close() NN_NOEXCEPT
{
    // necessary for reference counting
    m_Display = nullptr;
}

nn::Result nn::vi::detail::AndroidDisplay::GetHotplugState(HotplugState* pOutState) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_Display);

    DisplayModeInfo info;

    if( GetMode(&info).IsSuccess() )
    {
        *pOutState = HotplugState_Connected;
    }
    else
    {
        *pOutState = HotplugState_Disconnected;
    }

    return nn::ResultSuccess();
}

nn::Result nn::vi::detail::AndroidDisplay::GetMode(DisplayModeInfo* pOutMode) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_Display);

    android::DisplayInfo info;

    // TODO: double check error code
    if( android::SurfaceComposerClient::getDisplayInfo(m_Display, &info) != 0 )
    {
        return nn::vi::ResultDisplayDisconnected();
    }

    ConvertDisplayMode(pOutMode, info);

    return nn::ResultSuccess();
}

int nn::vi::detail::AndroidDisplay::ListModes(DisplayModeInfo* pOutModes, int modeCountMax, const IModeFilter* pFilter) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_Display);

    android::Vector<android::DisplayInfo> configs;

    if( android::SurfaceComposerClient::getDisplayConfigs(m_Display, &configs) != 0 )
    {
        return 0;
    }

    int writeIndex = 0;
    for( android::DisplayInfo info : configs )
    {
        if( writeIndex >= modeCountMax )
        {
            break;
        }

        DisplayModeInfo mode;
        ConvertDisplayMode(&mode, info);

        if( pFilter->IsValid(mode) )
        {
            bool isDuplicate = false;

            for( int i = 0; i < writeIndex; ++i )
            {
                if( pOutModes[i].width == mode.width &&
                    pOutModes[i].height == mode.height &&
                    pOutModes[i].refreshRate == mode.refreshRate &&
                    pOutModes[i].mode == mode.mode )
                {
                    isDuplicate = true;
                    break;
                }
            }

            if( !isDuplicate )
            {
                pOutModes[writeIndex] = mode;
                ++writeIndex;
            }
        }
    }

    return writeIndex;
}

nn::Result nn::vi::detail::AndroidDisplay::SetMode(const DisplayModeInfo* pMode, const IModeFilter* pFilter) NN_NOEXCEPT
{
    android::Vector<android::DisplayInfo> configs;

    if( !pFilter->IsValid(*pMode) )
    {
        return nn::vi::ResultNotSupported();
    }

    if( android::SurfaceComposerClient::getDisplayConfigs(m_Display, &configs) != 0 )
    {
        return nn::vi::ResultDisplayDisconnected();
    }

    int index = -1;

    for( int i = 0; i < configs.size(); ++i )
    {
        if( *pMode == configs[i] )
        {
            index = i;
            break;
        }
    }

    if( index == -1 )
    {
        return nn::vi::ResultNotSupported();
    }

    // TODO: double check error codes
    if( android::SurfaceComposerClient::setActiveConfig(m_Display, index) != 0 )
    {
        return nn::vi::ResultOperationFailed();
    }

    return nn::ResultSuccess();
}

int nn::vi::detail::AndroidDisplay::ListRgbRanges(RgbRange* pOutRanges, int rgbCountMax) const NN_NOEXCEPT
{
    android::DisplayEDID displayEdid;

    if( android::SurfaceComposerClient::getEDID(m_Display, displayEdid) != 0 )
    {
        return 0;
    }

    nn::edid::Edid edid;

    if( nn::edid::OpenEdid(&edid, displayEdid.getData(), displayEdid.getSize()) != nn::edid::Error_None )
    {
        return 0;
    }

    nn::edid::DisplayInfo info;
    nn::edid::GetDisplayInfo(&info, &edid);

    return WriteCeFormatRgbRange(pOutRanges, rgbCountMax, info.isRgbQuantizationSelectable);
}

nn::Result nn::vi::detail::AndroidDisplay::GetRgbRange(RgbRange* pOutRange) const NN_NOEXCEPT
{
    int range;
    if( android::SurfaceComposerClient::getRGBRange(m_Display, range) == 0 )
    {
        *pOutRange = ConvertRgbRange(range);

        return nn::ResultSuccess();
    }

    return nn::vi::ResultOperationFailed();
}

nn::Result nn::vi::detail::AndroidDisplay::SetRgbRange(RgbRange range) NN_NOEXCEPT
{
    if( range == RgbRange_Auto )
    {
        // Note: This is assuming CE video formats...
        //       Full is default for IT formats.
        range = RgbRange_Limited;
    }

    RgbRange supported[RgbRange_Max];
    int count = ListRgbRanges(supported, sizeof(supported) / sizeof(supported[0]));

    if( count > 0 )
    {
        bool found = false;

        for( int i = 0; i < count; ++i )
        {
            found = found || supported[i] == range;
        }

        if( !found )
        {
            return nn::vi::ResultNotSupported();
        }
    }
    else
    {
        // couldn't deduce what RGB ranges were supported from EDID
        return nn::vi::ResultOperationFailed();
    }

    int convertedRange = ConvertRgbRange(range);

    if( android::SurfaceComposerClient::setRGBRange(m_Display, convertedRange) == 0 )
    {
        return nn::ResultSuccess();
    }

    return nn::vi::ResultOperationFailed();
}

nn::Result nn::vi::detail::AndroidDisplay::SetUnderscan(int underscan) NN_NOEXCEPT
{
    if( underscan < 0 || underscan > 20 )
    {
        return nn::vi::ResultInvalidRange();
    }

    if( android::SurfaceComposerClient::setUnderscan(m_Display, underscan) == 0 )
    {
        return nn::ResultSuccess();
    }

    return nn::vi::ResultOperationFailed();
}

nn::Result nn::vi::detail::AndroidDisplay::GetUnderscan(int* pOutUnderscan) const NN_NOEXCEPT
{
    if( android::SurfaceComposerClient::getUnderscan(m_Display, *pOutUnderscan) == 0 )
    {
        return nn::ResultSuccess();
    }

    return nn::vi::ResultOperationFailed();
}

nn::Result nn::vi::detail::AndroidDisplay::SetAlpha(float alpha) NN_NOEXCEPT
{
    if( alpha < 0.f || alpha > 1.f )
    {
        return nn::vi::ResultInvalidRange();
    }

    android::SurfaceComposerClient::openGlobalTransaction();
    android::SurfaceComposerClient::setDisplayAlpha(m_Display, alpha);
    android::SurfaceComposerClient::closeGlobalTransaction();

    return nn::ResultSuccess();
}

nn::Result nn::vi::detail::AndroidDisplay::SetPowerState(PowerState state) NN_NOEXCEPT
{
    android::SurfaceComposerClient::openGlobalTransaction();
    android::status_t status = android::SurfaceComposerClient::setDisplayPowerMode(m_Display, ConvertPowerState(state));
    android::SurfaceComposerClient::closeGlobalTransaction();

    if( status == android::OK )
    {
        return nn::ResultSuccess();
    }

    return nn::vi::ResultOperationFailed();
}

nn::Result nn::vi::detail::AndroidDisplay::SetLayerStack(LayerStack id) NN_NOEXCEPT
{
    android::SurfaceComposerClient::openGlobalTransaction();
    android::SurfaceComposerClient::setDisplayLayerStack(m_Display, id);
    android::SurfaceComposerClient::closeGlobalTransaction();

    return nn::ResultSuccess();
}

nn::Result nn::vi::detail::AndroidDisplay::SetCmuLuma(float luma) NN_NOEXCEPT
{
    if( luma < -1.f || luma > 1.f )
    {
        return nn::vi::ResultInvalidRange();
    }

    if( android::SurfaceComposerClient::setBrightness(m_Display, luma) == android::OK )
    {
        return nn::ResultSuccess();
    }

    return nn::vi::ResultOperationFailed();
}

nn::Result nn::vi::detail::AndroidDisplay::GetCmuLuma(float* pOutLuma) const NN_NOEXCEPT
{
    if( android::SurfaceComposerClient::getBrightness(m_Display, *pOutLuma) == android::OK )
    {
        return nn::ResultSuccess();
    }

    return nn::vi::ResultOperationFailed();
}

nn::Result nn::vi::detail::AndroidDisplay::GetHotplugEvent(nn::os::SystemEventType* pOutEvent) NN_NOEXCEPT
{
    nn::os::NativeHandle handle;
    bool isManaged;

    if( m_Client->getHotplugEventHandle(handle, isManaged) == android::OK )
    {
        nn::os::AttachReadableHandleToSystemEvent(pOutEvent, handle, isManaged, nn::os::EventClearMode_AutoClear);

        return nn::ResultSuccess();
    }

    return nn::vi::ResultOperationFailed();
}
