﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vi/vi_DisplayEvents.h>
#include <nn/vi/vi_DisplayEventsDebug.h>
#include <nn/vi/vi_DisplayError.h>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_StringUtil.h>

#include <nn/vi/vi_Result.h>
#include "vi_Config.h"
#include "vi_MemoryManagement.h"
#include "vi_CommonUtility.h"
#include "vi_DisplayUtility.h"
#include "vi_LayerUtility.h"

namespace nn{ namespace vi{

//--------------------------
// Manager API
//--------------------------

    nn::Result GetDisplayHotplugEvent(nn::os::SystemEventType* pOutEvent, Display* pDisplay) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED_MANAGER();
        NN_SDK_REQUIRES_NOT_NULL(pOutEvent);

        nn::sf::NativeHandle handle;

        nn::Result result = GetManagerService()->GetDisplayHotplugEvent(&handle, GetDisplayIdWithValidation(pDisplay));

        if( result.IsSuccess() )
        {
            nn::os::AttachReadableHandleToSystemEvent(pOutEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
            handle.Detach();
        }

        return result;
    }

    Result GetDisplayHotplugState(HotplugState* pOutState, const Display* pDisplay) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED_MANAGER();
        NN_SDK_REQUIRES_NOT_NULL(pOutState);
        HotplugStateType value = {};
        NN_RESULT_DO(GetManagerService()->GetDisplayHotplugState(&value, GetDisplayIdWithValidation(pDisplay)));
        *pOutState = static_cast<HotplugState>(value);
        NN_RESULT_SUCCESS;
    }

    Result GetDisplayVsyncEventForDebug(nn::os::SystemEventType* pOutEvent, Display* pDisplay) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED();
        NN_SDK_REQUIRES_NOT_NULL(pOutEvent);

        nn::sf::NativeHandle handle;
        nn::Result result = GetService()->GetDisplayVsyncEventForDebug(&handle, GetDisplayIdWithValidation(pDisplay));

        if( result.IsSuccess() )
        {
            nn::os::AttachReadableHandleToSystemEvent(pOutEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
            handle.Detach();
        }

        return result;
    }

    nn::Result GetDisplayErrorEvent(nn::os::SystemEventType* pOutEvent, Display* pDisplay) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED_MANAGER();
        NN_SDK_REQUIRES_NOT_NULL(pOutEvent);

        nn::sf::NativeHandle handle;

        nn::Result result = GetManagerService()->GetDisplayErrorEvent(&handle, GetDisplayIdWithValidation(pDisplay));

        if( result.IsSuccess() )
        {
            nn::os::AttachReadableHandleToSystemEvent(pOutEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
            handle.Detach();
        }

        return result;
    }

    int GetCompositorErrorInfo(CompositorError* pOutErrorInfo, int errorID, Display* pDisplay) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED_MANAGER();
        NN_SDK_REQUIRES_NOT_NULL(pOutErrorInfo);
        int length;
        nn::Result result = GetManagerService()->GetCompositorErrorInfo(pOutErrorInfo, &length, errorID, GetDisplayIdWithValidation(pDisplay));

        if( result.IsSuccess() )
        {
            return static_cast<int>(length);
        }
        else
            return -1;

    }

    nn::Result GetDisplayModeChangedEvent(nn::os::SystemEventType* pOutEvent, Display* pDisplay) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED_MANAGER();
        NN_SDK_REQUIRES_NOT_NULL(pOutEvent);

        nn::sf::NativeHandle handle;

        NN_RESULT_DO(GetManagerService()->GetDisplayModeChangedEvent(&handle, GetDisplayIdWithValidation(pDisplay)));
        nn::os::AttachReadableHandleToSystemEvent(pOutEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
        handle.Detach();

        NN_RESULT_SUCCESS;
    }

}}
