﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_BitFlagSet.h>
#include <nn/os.h>

#include <nn/vi/vi_Lib.h>
#include <nn/vi/vi_ProxyName.private.h>
#include <nn/vi/vi_Result.h>

#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/applet/applet.h>

#include "vi_Config.h"
#include "vi_MemoryManagement.h"
#include "vi_CommonUtility.h"
#include "vi_DisplayUtility.h"
#include "vi_LayerUtility.h"

// HIPC
#include <nn/sm/sm_Result.h>
#include <nn/sf/sf_ShimLibraryUtility.h>
#include <nn/sf/sf_HipcClientProxyByName.h>
#include <nn/vi/sf/vi_DisplayService.sfdl.h>
#include <nn/vi/sf/vi_ServiceName.h>

// relay
#include <binder/HOSServiceManager.h>
#include <utils/String8.h>
#include <nvnflinger_service.h>

namespace nn {
    namespace vi {

        namespace {

            struct PortFlags
            {
                typedef nn::util::BitFlagSet<32, PortFlags>::Flag<0> Application;
                typedef nn::util::BitFlagSet<32, PortFlags>::Flag<1> System;
                typedef nn::util::BitFlagSet<32, PortFlags>::Flag<2> Manager;
            };

            typedef nn::util::BitFlagSet<32, PortFlags> Ports;

            typedef nn::visrv::sf::IApplicationRootService IApplicationRootServiceType;
            typedef nn::visrv::sf::ISystemRootService      ISystemRootServiceType;
            typedef nn::visrv::sf::IManagerRootService     IManagerRootServiceType;

            typedef nn::visrv::sf::IApplicationDisplayService IApplicationServiceType;
            typedef nn::visrv::sf::ISystemDisplayService      ISystemServiceType;
            typedef nn::visrv::sf::IManagerDisplayService     IManagerServiceType;

            Ports g_AvailablePorts = {};

            nn::sf::ShimLibraryObjectHolder<IApplicationRootServiceType> g_ApplicationRootServiceHolder = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;
            nn::sf::ShimLibraryObjectHolder<ISystemRootServiceType>      g_SystemRootServiceHolder = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;
            nn::sf::ShimLibraryObjectHolder<IManagerRootServiceType>     g_ManagerRootServiceHolder = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;
            nn::sf::HipcRef<IApplicationServiceType>  g_pApplicationService;
            nn::sf::HipcRef<IApplicationServiceType>  g_pApplicationBlockingService;
            nn::sf::SharedPointer<ISystemServiceType> g_pSystemService;
            nn::sf::SharedPointer<IManagerServiceType>g_pManagerService;

            nn::vi::ProxyName g_ProxyName = { NN_VI_DRIVER_SERVICE_NAME };

            size_t g_InitializeCount = 0;

            nn::os::Mutex g_lock(false);
        }

        nn::sf::SharedPointer<IApplicationServiceType> GetService() NN_NOEXCEPT
        {
            return g_pApplicationService;
        }

        nn::sf::SharedPointer<IApplicationServiceType> GetBlockingService() NN_NOEXCEPT
        {
            return g_pApplicationBlockingService;
        }

        nn::sf::SharedPointer<ISystemServiceType> GetSystemService() NN_NOEXCEPT
        {
            return g_pSystemService;
        }

        nn::sf::SharedPointer<IManagerServiceType> GetManagerService() NN_NOEXCEPT
        {
            return g_pManagerService;
        }

        // To reduce session count, blocking session will be created when requested
        nn::Result CreateBlockingService() NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(GetBlockingService() == nullptr, ResultAlreadyOpened());
            NN_RESULT_DO(g_pApplicationService.CloneSession<nn::sf::MemoryResourceAllocationPolicy>(
                &g_pApplicationBlockingService,
                ApplicationBlockingIpcSessionTag,
                GetClientManager()._allocator.GetMemoryResource()
            ));
            NN_RESULT_SUCCESS;
        }

        nn::sf::SimpleAllInOneHipcClientManager<ObjectCountMax>& GetClientManager() NN_NOEXCEPT
        {
            static nn::sf::SimpleAllInOneHipcClientManager<ObjectCountMax> s_AioManager = NN_SF_SIMPLE_ALL_IN_ONE_HIPC_CLIENT_MANAGER_INITIALIZER;
            return s_AioManager;
        }

        namespace {
            template<typename ShimLibraryHolder>
            nn::Result InitializeRootObject(ShimLibraryHolder* pRootHolder, const char* serviceName) NN_NOEXCEPT
            {
                auto viInitializationResult = GetClientManager().InitializeShimLibraryHolder(pRootHolder, serviceName);

                NN_RESULT_TRY(viInitializationResult)
                    NN_RESULT_CATCH(nn::sm::ResultNotPermitted)
                {
                    NN_RESULT_RETHROW;
                }
                NN_RESULT_CATCH_ALL
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(viInitializationResult);
                }
                NN_RESULT_END_TRY;
                NN_RESULT_SUCCESS;
            }

            void InitializeRootObjectSystemOrApplication() NN_NOEXCEPT
            {
                // try vi:m
                if (InitializeRootObject(&g_ManagerRootServiceHolder, nn::vi::sf::ManagerDisplayServiceName).IsSuccess())
                {
                    g_AvailablePorts.Set<PortFlags::Manager>();
                }

                // try vi:s
                if (InitializeRootObject(&g_SystemRootServiceHolder, nn::vi::sf::SystemDisplayServiceName).IsSuccess())
                {
                    g_AvailablePorts.Set<PortFlags::System>();
                }

                // try vi:u
                if (InitializeRootObject(&g_ApplicationRootServiceHolder, nn::vi::sf::ApplicationDisplayServiceName).IsSuccess())
                {
                    g_AvailablePorts.Set<PortFlags::Application>();
                }
            }

            void InitializeServiceObject(nn::vi::PolicyLevelType policyLevel) NN_NOEXCEPT
            {
                if (g_AvailablePorts.Test<PortFlags::Manager>())
                {
                    // Get ApplicationService
                    if (nn::util::Strncmp(g_ProxyName.value, NN_VI_DRIVER_SERVICE_NAME, sizeof(ProxyName)) == 0)
                    {
                        NN_ABORT_UNLESS_RESULT_SUCCESS(
                            g_ManagerRootServiceHolder.GetObject()->GetDisplayService(&g_pApplicationService, policyLevel)
                        );
                    }
                    else
                    {
                        NN_ABORT_UNLESS_RESULT_SUCCESS(
                            g_ManagerRootServiceHolder.GetObject()->GetDisplayServiceWithProxyNameExchange(&g_pApplicationService, policyLevel, g_ProxyName)
                        );
                    }

                    // Get ManagerService
                    NN_ABORT_UNLESS_RESULT_SUCCESS(
                        g_pApplicationService->GetManagerDisplayService(&g_pManagerService)
                    );
                }

                if (g_AvailablePorts.Test<PortFlags::System>())
                {
                    if (g_pApplicationService.Get() == nullptr)
                    {
                        // Get ApplicationService
                        if (nn::util::Strncmp(g_ProxyName.value, NN_VI_DRIVER_SERVICE_NAME, sizeof(ProxyName)) == 0)
                        {
                            NN_ABORT_UNLESS_RESULT_SUCCESS(
                                g_SystemRootServiceHolder.GetObject()->GetDisplayService(&g_pApplicationService, policyLevel)
                            );
                        }
                        else
                        {
                            NN_ABORT_UNLESS_RESULT_SUCCESS(
                                g_SystemRootServiceHolder.GetObject()->GetDisplayServiceWithProxyNameExchange(&g_pApplicationService, policyLevel, g_ProxyName)
                            );
                        }
                    }

                    // Get SystemService
                    NN_ABORT_UNLESS_RESULT_SUCCESS(
                        g_pApplicationService->GetSystemDisplayService(&g_pSystemService)
                    );
                }

                if (g_AvailablePorts.Test<PortFlags::Application>())
                {
                    if (g_pApplicationService.Get() == nullptr)
                    {
                        // Application can use only ApplicationService
                        NN_ABORT_UNLESS_RESULT_SUCCESS(
                            g_ApplicationRootServiceHolder.GetObject()->GetDisplayService(&g_pApplicationService, policyLevel)
                        );
                    }
                }

                // relay service
                {
                    nn::sf::SharedPointer<nns::hosbinder::IHOSBinderDriver> pRelayService;
                    NN_ABORT_UNLESS_RESULT_SUCCESS(
                        g_pApplicationService->GetRelayService(&pRelayService)
                    );
                    char proxyNameSafe[sizeof(ProxyName) + 1] = {};
                    nn::util::Strlcpy(proxyNameSafe, g_ProxyName.value, sizeof(proxyNameSafe));
                    // WAR:  If only vi::Initialize/vi::Finalize are called, a leak will occur after
                    //       nv::FinalizeGraphics.  Seems like the exit handler is pointing to
                    //       IPCThreadState::shutdown, but not sure why this isn't initialized
                    //       after setting up the service manager.
                    android::IPCThreadState::self();
                    NN_ABORT_UNLESS(
                        android::defaultHOSServiceManager()->addServiceProxy(android::String8(proxyNameSafe), pRelayService) == android::NO_ERROR
                    );
                }
            }
        }

        void Initialize() NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock(g_lock);

            if (g_InitializeCount == 0)
            {
                NN_SDK_REQUIRES(g_AvailablePorts.IsAllOff());

                InitializeMemory();

                // get root object
                InitializeRootObjectSystemOrApplication();
                NN_ABORT_UNLESS(g_AvailablePorts.IsAnyOn(), "failed to initialize nn::vi");

                // get per-client object
                nn::vi::PolicyLevel level;

                if (g_AvailablePorts.Test<PortFlags::System>() || g_AvailablePorts.Test<PortFlags::Manager>())
                {
                    level = nn::vi::PolicyLevel_Composition;
                }
                else
                {
                    level = nn::vi::PolicyLevel_Standard;
                }

                InitializeServiceObject(level);
            }

            // prevent overflow
            NN_SDK_REQUIRES_LESS(g_InitializeCount, std::numeric_limits<decltype(g_InitializeCount)>::max());
            ++g_InitializeCount;
        }

        void InitializeMinimum() NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock(g_lock);

            if(g_InitializeCount == 0)
            {
                NN_SDK_REQUIRES(g_AvailablePorts.IsAllOff());

                InitializeMemory();

                NN_ABORT_UNLESS_RESULT_SUCCESS(InitializeRootObject(&g_ManagerRootServiceHolder, "vi:m"));
                NN_ABORT_UNLESS_RESULT_SUCCESS(g_ManagerRootServiceHolder->GetDisplayService(&g_pApplicationService, nn::vi::PolicyLevel_Composition));
                NN_ABORT_UNLESS_RESULT_SUCCESS(g_pApplicationService->GetManagerDisplayService(&g_pManagerService));
                NN_ABORT_UNLESS_RESULT_SUCCESS(g_pApplicationService->GetSystemDisplayService(&g_pSystemService));
            }

            // prevent overflow
            NN_SDK_REQUIRES_LESS(g_InitializeCount, std::numeric_limits<decltype(g_InitializeCount)>::max());
            ++g_InitializeCount;
        }

        void Finalize() NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock(g_lock);

            if (g_InitializeCount == 0)
            {
                // not initialized
                return;
            }

            if (--g_InitializeCount == 0)
            {
                NN_SDK_REQUIRES(g_pApplicationService.Get() != nullptr);

                std::lock_guard<nn::os::Mutex> displayLock(ObjectManager::GetDisplayLock());
                std::lock_guard<nn::os::Mutex> layerLock(ObjectManager::GetLayerLock());

                // release all native windows
                for (auto& layerHolder : *ObjectManager::GetValidLayerHolderList())
                {
                    layerHolder.ReleaseNativeWindow();
                }

                // release relaying object
                {
                    char proxyNameSafe[sizeof(ProxyName) + 1] = {};
                    nn::util::Strlcpy(proxyNameSafe, g_ProxyName.value, sizeof(proxyNameSafe));
                    NN_ABORT_UNLESS(
                        android::defaultHOSServiceManager()->removeServiceProxy(android::String8(proxyNameSafe)) >= 0
                    );
                }

                // release per-client object
                g_pSystemService.Reset();
                g_pApplicationBlockingService.Reset();
                g_pApplicationService.Reset();
                g_pManagerService.Reset();

                // release root object
                if (g_AvailablePorts.Test<PortFlags::Application>())
                {
                    g_ApplicationRootServiceHolder.FinalizeHolder();
                }

                if (g_AvailablePorts.Test<PortFlags::System>())
                {
                    g_SystemRootServiceHolder.FinalizeHolder();
                }

                if (g_AvailablePorts.Test<PortFlags::Manager>())
                {
                    g_ManagerRootServiceHolder.FinalizeHolder();
                }

                g_AvailablePorts.Reset();

                // invalidate all displays
                auto pDisplayList = ObjectManager::GetValidDisplayHolderList();
                while (pDisplayList->size() > 0)
                {
                    ObjectManager::InvalidateDestroyDisplayHolder(pDisplayList->begin());
                }

                // all layers should also be removed
                NN_SDK_ASSERT_EQUAL(ObjectManager::GetValidLayerHolderList()->size(), 0);

                FinalizeMemory();
            }
        }

        void FinalizeMinimum() NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock(g_lock);

            if (g_InitializeCount == 0)
            {
                // not initialized
                return;
            }

            if (--g_InitializeCount == 0)
            {
                g_pSystemService.Reset();
                g_pManagerService.Reset();
                g_pApplicationService.Reset();
                g_ManagerRootServiceHolder.FinalizeHolder();

                {
                    std::lock_guard<nn::os::Mutex> displayLock(ObjectManager::GetDisplayLock());

                    // invalidate all displays
                    auto pDisplayList = ObjectManager::GetValidDisplayHolderList();
                    while (pDisplayList->size() > 0)
                    {
                        ObjectManager::InvalidateDestroyDisplayHolder(pDisplayList->begin());
                    }
                }

                FinalizeMemory();
            }
        }

        void SetRelayProxyName(const char* name) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(g_pApplicationService.Get() == nullptr);
            NN_SDK_REQUIRES_NOT_NULL(name);
            NN_SDK_REQUIRES_NOT_EQUAL('\0', name[0]);

            int i = 0;
            for (; i < sizeof(ProxyName); i++)
            {
                if (name[i] == '\0')
                {
                    break;
                }
                g_ProxyName.value[i] = name[i];
            }
            for (; i < sizeof(ProxyName); i++)
            {
                g_ProxyName.value[i] = '\0';
            }
        }

    }
}
