﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vi/manager/vi_ManagerSharedLowLevelLayer.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/vi/manager/vi_ManagerSharedLayer.h>

#define NN_VI_MNGR_CALLM(pService, Invoke)   \
    NN_ABORT_UNLESS_RESULT_SUCCESS(pService->GetManagerDisplayService()-> Invoke)

namespace nn{ namespace vi{ namespace manager{

    ManagerSharedLowLevelLayer::ManagerSharedLowLevelLayer() NN_NOEXCEPT
        : m_pService()
        , m_MyAruid()
        , m_DirectLayerId(0)
        , m_DirectDisplayName()
        , m_hIndirectProducer(0)
        , m_hBuffer()
        , m_pAttachedLayer()
        , m_SynchronizedEvent()
    {
    }

    bool ManagerSharedLowLevelLayer::IsInitialized() const NN_NOEXCEPT
    {
        return m_pService != nullptr;
    }

    bool ManagerSharedLowLevelLayer::IsAttached() const NN_NOEXCEPT
    {
        return m_pAttachedLayer != nullptr;
    }

    nn::vi::LayerId ManagerSharedLowLevelLayer::GetActiveLayerId() const NN_NOEXCEPT
    {
        if(m_DirectLayerId)
        {
            return m_DirectLayerId;
        }
        else
        {
            return static_cast<nn::vi::LayerId>(m_hIndirectProducer);
        }
    }

    //-----------------------------------------------------------------

    void ManagerSharedLowLevelLayer::InitializeForDirectLayer(
        DisplayManagerService* pService,
        nn::vi::LayerId layerId,
        const nn::vi::DisplayName& displayName,
        nn::vi::fbshare::SharedBufferHandle hBuffer,
        nn::applet::AppletResourceUserId myAruid
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!IsInitialized());
        NN_VI_MNGR_CALLM(pService, BindSharedLowLevelLayerToManagedLayer(layerId, displayName, myAruid));
        NN_ABORT_UNLESS_RESULT_SUCCESS(InitializeImpl(pService, layerId, hBuffer));
        m_pService          = pService;
        m_MyAruid           = myAruid;
        m_DirectLayerId     = layerId;
        m_DirectDisplayName = displayName;
        m_hBuffer           = hBuffer;
        m_pAttachedLayer    = nullptr;
    }

    void ManagerSharedLowLevelLayer::InitializeForIndirectLayer(
        DisplayManagerService* pService,
        nn::vi::IndirectProducerHandleType hProducer,
        nn::vi::fbshare::SharedBufferHandle hBuffer,
        nn::applet::AppletResourceUserId myAruid
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!IsInitialized());
        NN_VI_MNGR_CALLM(pService, BindSharedLowLevelLayerToIndirectLayer(hProducer, myAruid));
        NN_ABORT_UNLESS_RESULT_SUCCESS(InitializeImpl(pService, static_cast<nn::vi::LayerId>(hProducer), hBuffer));
        m_pService          = pService;
        m_MyAruid           = myAruid;
        m_hIndirectProducer = hProducer;
        m_hBuffer           = hBuffer;
        m_pAttachedLayer    = nullptr;
    }

    nn::Result ManagerSharedLowLevelLayer::InitializeImpl(
        DisplayManagerService* pService,
        nn::vi::LayerId layerId,
        nn::vi::fbshare::SharedBufferHandle hBuffer
    ) NN_NOEXCEPT
    {
        NN_VI_MNGR_CALLM(pService, ConnectSharedLowLevelLayerToSharedBuffer(layerId, hBuffer));

        {
            nn::sf::NativeHandle h = {};
            NN_VI_MNGR_CALLM(pService, GetSharedLowLevelLayerSynchronizedEvent(&h, layerId));
            nn::os::AttachReadableHandleToSystemEvent(&m_SynchronizedEvent, h.GetOsHandle(), h.IsManaged(), nn::os::EventClearMode_ManualClear);
            h.Detach();
        }

        NN_RESULT_SUCCESS;
    }

    void ManagerSharedLowLevelLayer::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        nn::os::DestroySystemEvent(&m_SynchronizedEvent);

        NN_VI_MNGR_CALLM(m_pService, UnbindSharedLowLevelLayer(GetActiveLayerId()));

        m_pService = nullptr;
        m_MyAruid = nn::applet::AppletResourceUserId::GetInvalidId();
        m_DirectLayerId = 0;
        m_DirectDisplayName = {};
        m_hIndirectProducer = 0;
        m_hBuffer = {0};
        m_pAttachedLayer = nullptr;
    }

    //-----------------------------------------------------------------

    bool ManagerSharedLowLevelLayer::Synchronize(int* pOutDisplayedIndex, nn::TimeSpan timeout) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());

        if(!nn::os::TimedWaitSystemEvent(&m_SynchronizedEvent, timeout))
        {
            return false;
        }

        int64_t index = -1;
        auto result = m_pService->GetManagerDisplayService()->CheckSharedLowLevelLayerSynchronized(&index, GetActiveLayerId());
        return result.IsSuccess();
    }

    void ManagerSharedLowLevelLayer::Blank() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());

        NN_VI_MNGR_CALLM(m_pService, DisconnectSharedLowLevelLayerFromSharedBuffer(GetActiveLayerId()));
        NN_VI_MNGR_CALLM(m_pService, ConnectSharedLowLevelLayerToSharedBuffer(GetActiveLayerId(), m_hBuffer));
    }

    //-----------------------------------------------------------------

    void ManagerSharedLowLevelLayer::Attach(
        ManagerSharedLayer* pLayer,
        const nn::vi::fbshare::SharedLayerTextureIndexList& bufferList
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES(!IsAttached());
        NN_SDK_REQUIRES_NOT_NULL(pLayer);
        NN_SDK_REQUIRES(pLayer->IsInitialized());
        NN_SDK_REQUIRES(!pLayer->IsAttached());

        auto hLayer = pLayer->GetHandle();
        NN_VI_MNGR_CALLM(m_pService, AttachSharedLayerToLowLevelLayer(hLayer, GetActiveLayerId(), bufferList));

        m_pAttachedLayer = pLayer;
        pLayer->m_pAttachedLowLevelLayer = this;
    }

    bool ManagerSharedLowLevelLayer::Detach(nn::TimeSpan timeout, bool force) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES(IsAttached());

        auto hLayer = m_pAttachedLayer->GetHandle();

        nn::os::SystemEventType detachReadyEvent = {};
        {
            nn::sf::NativeHandle h = {};
            NN_VI_MNGR_CALLM(m_pService, GetSharedLayerDetachReadyEvent(&h, hLayer));
            nn::os::AttachReadableHandleToSystemEvent(&detachReadyEvent, h.GetOsHandle(), h.IsManaged(), nn::os::EventClearMode_ManualClear);
            h.Detach();
        }
        NN_UTIL_SCOPE_EXIT{ nn::os::DestroySystemEvent(&detachReadyEvent); };

        NN_VI_MNGR_CALLM(m_pService, StartDetachSharedLayerFromLowLevelLayer(hLayer));
        if(nn::os::TimedWaitSystemEvent(&detachReadyEvent, timeout))
        {
            NN_VI_MNGR_CALLM(m_pService, FinishDetachSharedLayerFromLowLevelLayer(hLayer));
        }
        else if(force)
        {
            NN_VI_MNGR_CALLM(m_pService, ForceDetachSharedLayerFromLowLevelLayer(hLayer));
        }
        else
        {
            return false;
        }

        m_pAttachedLayer->m_pAttachedLowLevelLayer = nullptr;
        m_pAttachedLayer = nullptr;
        return true;
    }

    void ManagerSharedLowLevelLayer::ForceDetach() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES(IsAttached());

        auto hLayer = m_pAttachedLayer->GetHandle();
        NN_VI_MNGR_CALLM(m_pService, ForceDetachSharedLayerFromLowLevelLayer(hLayer));

        m_pAttachedLayer->m_pAttachedLowLevelLayer = nullptr;
        m_pAttachedLayer = nullptr;
    }


}}}

