﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/usb/usb_Pm.h>

#include "detail/usb_Util.h"
#include "pm/usb_PmServiceName.h"

using namespace nn::sf;

namespace nn {
namespace usb {

//--------------------------------------------------------------------------
//  PmClient class implementation
//--------------------------------------------------------------------------

Result PmClient::Initialize() NN_NOEXCEPT
{
    NativeHandle sfEventHandle;

    detail::UsbDiagInitialize();

    m_HeapHandle = nn::lmem::CreateExpHeap(&m_HeapBuffer,
                                           sizeof(m_HeapBuffer),
                                           nn::lmem::CreationOption_NoOption);
    NN_USB_ASSERT_IF_ZERO(m_HeapHandle);

    m_Allocator.Attach(m_HeapHandle);

    // Connect to the server
    NN_USB_ABORT_UNLESS_SUCCESS((
        CreateHipcProxyByName<pm::IPmService, MyAllocator::Policy>(
            &m_Handle, &m_Allocator, pm::ServiceName
        )
    ));

    // Setup the Data Role Change Event
    NN_USB_ABORT_UNLESS_SUCCESS(m_Handle->GetDataEvent(&sfEventHandle));
    nn::os::AttachReadableHandleToSystemEvent(
        &m_DataEvent,
        sfEventHandle.GetOsHandle(),
        sfEventHandle.IsManaged(),
        nn::os::EventClearMode_ManualClear
    );
    sfEventHandle.Detach();

    // Setup the Power State Change Event
    NN_USB_ABORT_UNLESS_SUCCESS(m_Handle->GetPowerEvent(&sfEventHandle));
    nn::os::AttachReadableHandleToSystemEvent(
        &m_PowerEvent,
        sfEventHandle.GetOsHandle(),
        sfEventHandle.IsManaged(),
        nn::os::EventClearMode_ManualClear
    );
    sfEventHandle.Detach();

    return ResultSuccess();
}

Result PmClient::Finalize() NN_NOEXCEPT
{
    nn::os::DestroySystemEvent(&m_DataEvent);
    nn::os::DestroySystemEvent(&m_PowerEvent);

    m_Handle = nullptr;

    nn::lmem::DestroyExpHeap(m_HeapHandle);
    m_HeapHandle = nullptr;

    detail::UsbDiagFinalize();

    return ResultSuccess();
}

nn::os::SystemEventType *PmClient::GetPowerEvent() NN_NOEXCEPT
{
    return &m_PowerEvent;
}

Result PmClient::GetPowerState(UsbPowerState *pOutState) NN_NOEXCEPT
{
    NN_USB_ABORT_IF_NULL(m_Handle);

    OutBuffer buffer(reinterpret_cast<char *>(pOutState), sizeof(UsbPowerState));

    Result result = m_Handle->GetPowerState(buffer);

    if (result.IsSuccess())
    {
        DumpPowerState(pOutState);
    }

    return result;
}

nn::os::SystemEventType *PmClient::GetDataEvent() NN_NOEXCEPT
{
    return &m_DataEvent;
}

Result PmClient::GetDataRole(UsbDataRole *pOutRole) NN_NOEXCEPT
{
    NN_USB_ABORT_IF_NULL(m_Handle);

    return m_Handle->GetDataRole(pOutRole);
}

Result PmClient::SetDiagData(DiagData data, uint32_t value) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    NN_USB_ABORT_IF_NULL(m_Handle);

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    result = m_Handle->SetDiagData(data, value);
#endif

    return result;
}

uint32_t PmClient::GetDiagData(DiagData data) NN_NOEXCEPT
{
    NN_USB_ABORT_IF_NULL(m_Handle);

    uint32_t value = 0;

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    m_Handle->GetDiagData(data, &value);
#endif

    return value;
}

void PmClient::DumpPowerState(UsbPowerState *pState) NN_NOEXCEPT
{
    static const char *PowerRole[] = {
        "Unknown", "Sink", "Source"
    };

    static const char *ChargerType[] = {
        "Unknown",
        "PD",
        "Type-C 1.5A", "Type-C 3.0A",
        "DCP", "CDP", "SDP",
        "Apple 500mA", "Apple 1000mA", "Apple 2000mA",
    };

    nn::usb::pd::Pdo pdo = pState->pdo;
    nn::usb::pd::Rdo rdo = pState->rdo;

    NN_USB_LOG_INFO("--------------= UsbPowerStatus =--------------\n");
    NN_USB_LOG_INFO("PowerRole                 : %s\n", PowerRole[pState->role]);
    NN_USB_LOG_INFO("ChargerType               : %s\n", ChargerType[pState->charger]);
    NN_USB_LOG_INFO("Voltage (non-PD)          : %dmV\n", pState->voltage);
    NN_USB_LOG_INFO("Current (non-PD)          : %dmA\n", pState->current);

    switch (pdo.Get<nn::usb::pd::Pdo::SupplyType>())
    {
    case 0:
        DumpPdoFixedSource(pdo);
        DumpRdoFixedOrVariable(rdo);
        break;

    case 1:
        DumpPdoBatterySource(pdo);
        DumpRdoBattery(rdo);
        break;

    case 2:
        DumpPdoVariableSource(pdo);
        DumpRdoFixedOrVariable(rdo);
        break;

    default:
        break;
    }

    NN_USB_LOG_INFO("--------------= UsbPowerStatus =--------------\n");
}

void PmClient::DumpPdoFixedSource(nn::usb::pd::Pdo &pdo) NN_NOEXCEPT
{
    static const char *Bool[] = {
        "False", "True"
    };

    NN_USB_LOG_INFO("PDO (PD)\n");
    NN_USB_LOG_INFO(" -SupplyType              : %s\n",    "Fixed");
    NN_USB_LOG_INFO(" -DualRolePower           : %s\n",    Bool[pdo.Get<nn::usb::pd::Pdo::DualRolePower>()]);
    NN_USB_LOG_INFO(" -UsbSuspendSupported     : %s\n",    Bool[pdo.Get<nn::usb::pd::Pdo::UsbSuspendSupported>()]);
    NN_USB_LOG_INFO(" -ExternallyPowered       : %s\n",    Bool[pdo.Get<nn::usb::pd::Pdo::ExternallyPowered>()]);
    NN_USB_LOG_INFO(" -UsbCommunicationsCapable: %s\n",    Bool[pdo.Get<nn::usb::pd::Pdo::USBCommunicationsCapable>()]);
    NN_USB_LOG_INFO(" -DataRoleSwap            : %s\n",    Bool[pdo.Get<nn::usb::pd::Pdo::DataRoleSwap>()]);
    NN_USB_LOG_INFO(" -PeakCurrent             : %d\n",    pdo.Get<nn::usb::pd::Pdo::PeakCurrent>());
    NN_USB_LOG_INFO(" -Voltage                 : %d mV\n", pdo.Get<nn::usb::pd::Pdo::Voltage>() * 50);
    NN_USB_LOG_INFO(" -MaximumCurrent          : %d mA\n", pdo.Get<nn::usb::pd::Pdo::MaximumCurrent>() * 10);
}

void PmClient::DumpPdoVariableSource(nn::usb::pd::Pdo &pdo) NN_NOEXCEPT
{
    NN_USB_LOG_INFO("PDO (PD)\n");
    NN_USB_LOG_INFO(" -SupplyType              : %s\n",    "Variable");
    NN_USB_LOG_INFO(" -MaximumVoltage          : %d mV\n", pdo.Get<nn::usb::pd::Pdo::MaximumVoltage>() * 50);
    NN_USB_LOG_INFO(" -MinimumVoltage          : %d mV\n", pdo.Get<nn::usb::pd::Pdo::MinimumVoltage>() * 50 );
    NN_USB_LOG_INFO(" -MaximumCurrent          : %d mA\n", pdo.Get<nn::usb::pd::Pdo::MaximumCurrent>() * 10);
}

void PmClient::DumpPdoBatterySource(nn::usb::pd::Pdo &pdo) NN_NOEXCEPT
{
    NN_USB_LOG_INFO("PDO (PD)\n");
    NN_USB_LOG_INFO(" -SupplyType              : %s\n",    "Battery");
    NN_USB_LOG_INFO(" -MaximumVoltage          : %d mV\n", pdo.Get<nn::usb::pd::Pdo::MaximumVoltage>() * 50);
    NN_USB_LOG_INFO(" -MinimumVoltage          : %d mV\n", pdo.Get<nn::usb::pd::Pdo::MinimumVoltage>() * 50 );
    NN_USB_LOG_INFO(" -MaximumAllowablePower   : %d mW\n", pdo.Get<nn::usb::pd::Pdo::MaximumAllowablePower>() * 250);
}

void PmClient::DumpPdoFixedSink(nn::usb::pd::Pdo &pdo) NN_NOEXCEPT
{
    static const char *Bool[] = {
        "False", "True"
    };

    NN_USB_LOG_INFO("PDO (PD)\n");
    NN_USB_LOG_INFO(" -SupplyType              : %s\n",    "Fixed");
    NN_USB_LOG_INFO(" -DualRolePower           : %s\n",    Bool[pdo.Get<nn::usb::pd::Pdo::DualRolePower>()]);
    NN_USB_LOG_INFO(" -HigherCapability        : %s\n",    Bool[pdo.Get<nn::usb::pd::Pdo::HigherCapability>()]);
    NN_USB_LOG_INFO(" -ExternallyPowered       : %s\n",    Bool[pdo.Get<nn::usb::pd::Pdo::ExternallyPowered>()]);
    NN_USB_LOG_INFO(" -UsbCommunicationsCapable: %s\n",    Bool[pdo.Get<nn::usb::pd::Pdo::USBCommunicationsCapable>()]);
    NN_USB_LOG_INFO(" -DataRoleSwap            : %s\n",    Bool[pdo.Get<nn::usb::pd::Pdo::DataRoleSwap>()]);
    NN_USB_LOG_INFO(" -Voltage                 : %d mV\n", pdo.Get<nn::usb::pd::Pdo::Voltage>() * 50);
    NN_USB_LOG_INFO(" -OperationalCurrent      : %d mA\n", pdo.Get<nn::usb::pd::Pdo::MaximumCurrent>() * 10);
}

void PmClient::DumpPdoVariableSink(nn::usb::pd::Pdo &pdo) NN_NOEXCEPT
{
    NN_USB_LOG_INFO("PDO (PD)\n");
    NN_USB_LOG_INFO(" -SupplyType              : %s\n",    "Variable");
    NN_USB_LOG_INFO(" -MaximumVoltage          : %d mV\n", pdo.Get<nn::usb::pd::Pdo::MaximumVoltage>() * 50);
    NN_USB_LOG_INFO(" -MinimumVoltage          : %d mV\n", pdo.Get<nn::usb::pd::Pdo::MinimumVoltage>() * 50 );
    NN_USB_LOG_INFO(" -OperationalCurrent      : %d mA\n", pdo.Get<nn::usb::pd::Pdo::MaximumCurrent>() * 10);
}

void PmClient::DumpPdoBatterySink(nn::usb::pd::Pdo &pdo) NN_NOEXCEPT
{
    NN_USB_LOG_INFO("PDO (PD)\n");
    NN_USB_LOG_INFO(" -SupplyType              : %s\n",    "Battery");
    NN_USB_LOG_INFO(" -MaximumVoltage          : %d mV\n", pdo.Get<nn::usb::pd::Pdo::MaximumVoltage>() * 50);
    NN_USB_LOG_INFO(" -MinimumVoltage          : %d mV\n", pdo.Get<nn::usb::pd::Pdo::MinimumVoltage>() * 50 );
    NN_USB_LOG_INFO(" -OperationalPower        : %d mW\n", pdo.Get<nn::usb::pd::Pdo::MaximumAllowablePower>() * 250);
}

void PmClient::DumpRdoFixedOrVariable(nn::usb::pd::Rdo &rdo) NN_NOEXCEPT
{
    static const char *Bool[] = {
        "False", "True"
    };

    NN_USB_LOG_INFO("RDO (PD)\n");
    NN_USB_LOG_INFO(" -ObjectPosition          : %d\n",    rdo.Get<nn::usb::pd::Rdo::ObjectPosition>());
    NN_USB_LOG_INFO(" -GiveBackFlag            : %s\n",    Bool[rdo.Get<nn::usb::pd::Rdo::GiveBackFlag>()]);
    NN_USB_LOG_INFO(" -CapabilityMismatch      : %s\n",    Bool[rdo.Get<nn::usb::pd::Rdo::CapabilityMismatch>()]);
    NN_USB_LOG_INFO(" -UsbCommunicationsCapable: %s\n",    Bool[rdo.Get<nn::usb::pd::Rdo::UsbCommunicationsCapable>()]);
    NN_USB_LOG_INFO(" -NoUsbSuspend            : %s\n",    Bool[rdo.Get<nn::usb::pd::Rdo::NoUsbSuspend>()]);
    NN_USB_LOG_INFO(" -OperatingCurrent        : %d mA\n", rdo.Get<nn::usb::pd::Rdo::OperatingCurrent>() * 10);

    if (rdo.Get<nn::usb::pd::Rdo::GiveBackFlag>())
    {
        NN_USB_LOG_INFO(" -MaximumOperatingCurrent : %d mA\n", rdo.Get<nn::usb::pd::Rdo::MaximumOperatingCurrent>() * 10);
    }
    else
    {
        NN_USB_LOG_INFO(" -MinimumOperatingCurrent : %d mA\n", rdo.Get<nn::usb::pd::Rdo::MinimumOperatingCurrent>() * 10);
    }
}

void PmClient::DumpRdoBattery(nn::usb::pd::Rdo &rdo) NN_NOEXCEPT
{
    static const char *Bool[] = {
        "False", "True"
    };

    NN_USB_LOG_INFO("RDO (PD)\n");
    NN_USB_LOG_INFO(" -ObjectPosition          : %d\n",    rdo.Get<nn::usb::pd::Rdo::ObjectPosition>());
    NN_USB_LOG_INFO(" -GiveBackFlag            : %s\n",    Bool[rdo.Get<nn::usb::pd::Rdo::GiveBackFlag>()]);
    NN_USB_LOG_INFO(" -CapabilityMismatch      : %s\n",    Bool[rdo.Get<nn::usb::pd::Rdo::CapabilityMismatch>()]);
    NN_USB_LOG_INFO(" -UsbCommunicationsCapable: %s\n",    Bool[rdo.Get<nn::usb::pd::Rdo::UsbCommunicationsCapable>()]);
    NN_USB_LOG_INFO(" -NoUsbSuspend            : %s\n",    Bool[rdo.Get<nn::usb::pd::Rdo::NoUsbSuspend>()]);
    NN_USB_LOG_INFO(" -OperatingPower          : %d mW\n", rdo.Get<nn::usb::pd::Rdo::OperatingPower>() * 250);

    if (rdo.Get<nn::usb::pd::Rdo::GiveBackFlag>())
    {
        NN_USB_LOG_INFO(" -MaximumOperatingPower   : %d mW\n", rdo.Get<nn::usb::pd::Rdo::MaximumOperatingPower>() * 250);
    }
    else
    {
        NN_USB_LOG_INFO(" -MinimumOperatingPower   : %d mW\n", rdo.Get<nn::usb::pd::Rdo::MinimumOperatingPower>() * 250);
    }
}

} // end of namespace usb
} // end of namespace nn
