﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os/os_SystemEvent.h>


#include <nn/psc.h>
#include <nn/usb/usb_PmTypes.h>
#include <nn/usb/pd/driver/usb_Pd.h>

#include "usb_PmPort.h"
#include "usb_PmServiceManager.h"

#include "../detail/usb_Util.h"
#include "../detail/usb_Memory.h"

#include "../detail/usb_Platform.h"
#include "../detail/usb_InputNotifier.h"
#include "../detail/usb_Complex.h"

namespace nn {
namespace usb {
namespace pm {

class Pm
{
public:
    Pm() NN_NOEXCEPT
        : m_pPlatform(nullptr)
        , m_pInputNotifier(nullptr)
        , m_pComplex(nullptr)
        , m_PortCount(0)
        , m_UpdMutex(false)
        , m_ListMutex(false)
        , m_IsAwake(true)
        , m_RepollEvent(nn::os::EventClearMode_ManualClear)
        , m_Charger(UsbChargerType_Sdp)
    {
        // do nothing
    }

    ~Pm() NN_NOEXCEPT
    {
        // do nothing
    }

    Result   Initialize(detail::UsbPlatform *pPlatform, detail::InputNotifier *pInputNotifier) NN_NOEXCEPT;
    Result   Finalize() NN_NOEXCEPT;
    Result   MainLoop() NN_NOEXCEPT;

    Result   BindPowerEvent(nn::os::SystemEvent *pEvent) NN_NOEXCEPT;
    Result   UnbindPowerEvent(nn::os::SystemEvent *pEvent) NN_NOEXCEPT;
    Result   GetPowerState(UsbPowerState *pState, nn::os::SystemEvent *pEvent) NN_NOEXCEPT;

    Result   BindDataEvent(nn::os::SystemEvent *pEvent) NN_NOEXCEPT;
    Result   UnbindDataEvent(nn::os::SystemEvent *pEvent) NN_NOEXCEPT;
    Result   GetDataRole(UsbDataRole *pRole, nn::os::SystemEvent *pEvent) NN_NOEXCEPT;

private:
    detail::UsbPlatform                      *m_pPlatform;
    detail::InputNotifier                    *m_pInputNotifier;
    detail::UsbComplex                       *m_pComplex;

    uint32_t                                  m_PortCount;
    PmPort                                    m_Port[HwLimitMaxPortCount];

    // UPD: USB Power Delivery
    nn::usb::pd::driver::Session              m_UpdSession;
    nn::usb::pd::Status                       m_UpdStatus;
    nn::usb::pd::Notice                       m_UpdNotice;
    nn::os::SystemEvent                       m_UpdEvent;
    nn::os::Mutex                             m_UpdMutex;

    detail::StdList<nn::os::SystemEvent*>     m_PowerEventList;
    detail::StdList<nn::os::SystemEvent*>     m_DataEventList;
    nn::os::Mutex                             m_ListMutex;

    bool                                      m_IsAwake;
    nn::os::Event                             m_RepollEvent;

    // Charger detection
    UsbChargerType                            m_Charger;

    // PSC: Power State Coordinator
    nn::psc::PmModule                         m_PmModule;

    // SF: Service Framework
    PmServiceManager                          m_ServiceManager;

private:
    void     DumpUpdStatus() NN_NOEXCEPT;
    Result   PollUpdStatus() NN_NOEXCEPT;
    Result   SignalPowerEvent() NN_NOEXCEPT;
    Result   SignalDataEvent() NN_NOEXCEPT;
    Result   SetDataRole(PmPort& port, nn::usb::pd::StatusDataRole role) NN_NOEXCEPT;
    void     HandlePowerEvent() NN_NOEXCEPT;
    void     OnFullAwake() NN_NOEXCEPT;
    void     OnMinimumAwake() NN_NOEXCEPT;
    void     OnSleepReady() NN_NOEXCEPT;
    void     OnShutdownReady() NN_NOEXCEPT;
    uint32_t GetChargerVoltage(UsbChargerType chargerType) NN_NOEXCEPT;
    uint32_t GetChargerCurrent(UsbChargerType chargerType) NN_NOEXCEPT;
};

} // end of namespace pm
} // end of namespace usb
} // end of namespace nn

