﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include "../detail/usb_Util.h"

#include "usb_Pm.h"

namespace nn {
namespace usb {
namespace pm {

//----------------------------------------------------------------------------
// Public Methods
//----------------------------------------------------------------------------

Result Pm::Initialize(detail::UsbPlatform *pPlatform, detail::InputNotifier *pInputNotifier) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    m_pPlatform       = pPlatform;
    m_pInputNotifier  = pInputNotifier;

    // If there are more than one complex on the platform, it's PD driver's
    // responsibility to tell PM the source of DataRoleSwap event.
    //
    // For now, hard code to Tegra21x.
    m_pComplex = m_pPlatform->GetComplex(ComplexId_Tegra21x);
    NN_USB_ABORT_IF_NULL(m_pComplex);

    NN_USB_ABORT_UNLESS_SUCCESS(m_pComplex->Enable());

    // Populate the ports
    m_PortCount = m_pPlatform->GetConfig().portCount;
    for (uint32_t i = 0; i < m_PortCount; i++)
    {
        NN_USB_ABORT_UNLESS_SUCCESS(m_Port[i].Initialize(this, i, m_pComplex));
        NN_USB_ABORT_UNLESS_SUCCESS(m_Port[i].Enable());
    }

    // Register PSC dependencies
    const nn::psc::PmModuleId usbDependencies[] = {
        nn::psc::PmModuleId_PcvClock,
        nn::psc::PmModuleId_PcvVoltage,
        nn::psc::PmModuleId_Gpio,
        nn::psc::PmModuleId_I2c,
        nn::psc::PmModuleId_Pcie,
        nn::psc::PmModuleId_PsmLow,
    };
    NN_USB_ABORT_UPON_ERROR(
        m_PmModule.Initialize(nn::psc::PmModuleId_Usb,
                              usbDependencies,
                              NN_USB_ARRAY_SIZE(usbDependencies),
                              nn::os::EventClearMode_ManualClear)
    );

    // Prepare PD
    nn::usb::pd::driver::OpenSession(&m_UpdSession);
    nn::usb::pd::driver::BindNoticeEvent(m_UpdEvent.GetBase(), m_UpdSession);

    // Initial Poll
    PollUpdStatus();

    // start the service after initial poll, so GetPowerState() always
    // returns meaningful data
    m_ServiceManager.Initialize(this);

    return result;
}

// TODO: This is never called actually. Probably let OnShutDownReady() call it
//       and adjust the logic acoordingly
Result Pm::Finalize() NN_NOEXCEPT
{
    Result result = ResultSuccess();

    m_ServiceManager.Finalize();

    nn::usb::pd::driver::UnbindNoticeEvent(m_UpdSession);
    nn::usb::pd::driver::CloseSession(&m_UpdSession);

    m_PmModule.Finalize();

    for (uint32_t i = 0; i < m_PortCount; i++)
    {
        NN_USB_ABORT_UPON_ERROR(m_Port[i].Disable());
        NN_USB_ABORT_UPON_ERROR(m_Port[i].Finalize());
    }

    NN_USB_ABORT_UNLESS_SUCCESS(m_pComplex->Disable());

    return result;
}

Result Pm::BindPowerEvent(nn::os::SystemEvent *pEvent) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_ListMutex);

    m_PowerEventList.push_back(pEvent);

    return ResultSuccess();
}

Result Pm::UnbindPowerEvent(nn::os::SystemEvent *pEvent) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_ListMutex);

    m_PowerEventList.remove(pEvent);

    return ResultSuccess();
}

Result Pm::GetPowerState(UsbPowerState *pState, nn::os::SystemEvent *pEvent) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_UpdMutex);

    pEvent->Clear();

    memset(pState, 0, sizeof(UsbPowerState));

    if (!m_UpdStatus.IsActive())
    {
        pState->role = UsbPowerRole_Unknown;
    }
    else
    {
        switch (m_UpdStatus.GetPowerRole())
        {
        case nn::usb::pd::StatusPowerRole_Sink:
            pState->role = UsbPowerRole_Sink;
            break;

        case nn::usb::pd::StatusPowerRole_Source:
            pState->role = UsbPowerRole_Source;
            break;

        default:
            NN_UNEXPECTED_DEFAULT;
            break;
        }

        pState->charger = m_Charger;
        pState->pdo     = m_UpdStatus.m_CurrentPdo;
        pState->rdo     = m_UpdStatus.m_CurrentRdo;
        pState->voltage = GetChargerVoltage(pState->charger);
        pState->current = GetChargerCurrent(pState->charger);
    }

    return ResultSuccess();
}

Result Pm::BindDataEvent(nn::os::SystemEvent *pEvent) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_ListMutex);

    m_DataEventList.push_back(pEvent);

    return ResultSuccess();
}

Result Pm::UnbindDataEvent(nn::os::SystemEvent *pEvent) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_ListMutex);

    m_DataEventList.remove(pEvent);

    return ResultSuccess();
}

Result Pm::GetDataRole(UsbDataRole *pRole, nn::os::SystemEvent *pEvent) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_UpdMutex);

    pEvent->Clear();

    switch (m_UpdStatus.GetDataRole())
    {
    case nn::usb::pd::StatusDataRole_Unknown:
        *pRole = UsbDataRole_Unknown;
        break;

    case nn::usb::pd::StatusDataRole_Ufp:
        *pRole = UsbDataRole_Device;
        break;

    case nn::usb::pd::StatusDataRole_Dfp:
        *pRole = UsbDataRole_Host;
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

    return ResultSuccess();
}

Result Pm::MainLoop() NN_NOEXCEPT
{
    Result result = ResultSuccess();

    nn::os::SystemEvent *pPowerEvent = m_PmModule.GetEventPointer();
    nn::os::Event       *pResetEvent = m_pComplex->GetResetEvent();

    NN_USB_LOG_INFO("Start USB Port Manager MainLoop\n");

    while (true)
    {
        auto index = nn::os::WaitAny(m_UpdEvent.GetBase(),
                                     m_RepollEvent.GetBase(),
                                     pPowerEvent->GetBase(),
                                     pResetEvent->GetBase());
        switch (index)
        {
        case 0:
            m_UpdEvent.Clear();
            PollUpdStatus();
            break;

        case 1:
            m_RepollEvent.Clear();
            PollUpdStatus();
            break;

        case 2:
            pPowerEvent->Clear();
            HandlePowerEvent();
            break;

        case 3:
            pResetEvent->Clear();

            // Don't try to re-enable the controller if we are in the sleep sequence.
            //   - it's logically incorrect as we want the controller to be disabled
            //   - it might halt the processor if the clock is stopped when we access
            //     the registers (SIGLO-47698)
            if (!m_IsAwake)
            {
                break;
            }

            for (uint32_t i = 0; i < m_PortCount; i++)
            {
                m_Port[i].Disable();
            }
            m_pComplex->Disable();
            m_pComplex->Enable();
            for (uint32_t i = 0; i < m_PortCount; i++)
            {
                m_Port[i].Enable();
            }
            break;

        default:
            NN_UNEXPECTED_DEFAULT;
            break;
        }
    }

    return result;
}

//----------------------------------------------------------------------------
// Private Methods
//----------------------------------------------------------------------------

void Pm::DumpUpdStatus() NN_NOEXCEPT
{
    const char *PlugOrientation[] = {
        "CC1", "CC2",
    };
    const char *PowerRole[] = {
        "Sink", "Source",
    };
    const char *DataRole[] = {
        "Unknown", "UFP", "DFP", "AccessoryMode",
    };
    const char *AccessoryMode[] = {
        "None", "Audio", "Debug", "VConnPowered",
    };
    const char *DeviceType[] = {
        "Unknown", "Cradle", "RelayBox", "AcAdaptor", "TableDock",
    };

    NN_USB_LOG_INFO("-------------- UPD Status --------------\n");
    NN_USB_LOG_INFO("Active       : %s\n", m_UpdStatus.IsActive() ? "Yes": "No");
    NN_USB_LOG_INFO("Orientation  : %s\n", PlugOrientation[m_UpdStatus.GetPlugOrientation()]);
    NN_USB_LOG_INFO("DataRole     : %s\n", DataRole[m_UpdStatus.GetDataRole()]);
    NN_USB_LOG_INFO("PowerRole    : %s\n", PowerRole[m_UpdStatus.GetPowerRole()]);
    NN_USB_LOG_INFO("ElecMark     : %s\n", m_UpdStatus.IsElectronicallyMarkedCable() ? "Yes" : "No");
    NN_USB_LOG_INFO("AccessoryMode: %s\n", AccessoryMode[m_UpdStatus.GetAccessoryMode()]);
    NN_USB_LOG_INFO("DeviceType   : %s\n", DeviceType[m_UpdStatus.GetDeviceType()]);
    NN_USB_LOG_INFO("Notice       :");

    if (m_UpdNotice.IsActiveNotice())
    {
        NN_SDK_LOG(" Active");
    }
    if (m_UpdNotice.IsErrorNotice())
    {
        NN_SDK_LOG(" Error");
    }
    if (m_UpdNotice.IsDataRoleNotice())
    {
        NN_SDK_LOG(" DataRole");
    }
    if (m_UpdNotice.IsPowerRoleNotice())
    {
        NN_SDK_LOG(" PowerRole");
    }
    if (m_UpdNotice.IsDeviceNotice())
    {
        NN_SDK_LOG(" Device");
    }
    NN_SDK_LOG("\n");

    NN_USB_LOG_INFO("-------------- UPD Status --------------\n");
}

Result Pm::SignalPowerEvent() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_ListMutex);

    for (nn::os::SystemEvent *pEvent : m_PowerEventList)
    {
        pEvent->Signal();
    }

    return ResultSuccess();
}

Result Pm::SignalDataEvent() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_ListMutex);

    for (nn::os::SystemEvent *pEvent : m_DataEventList)
    {
        pEvent->Signal();
    }

    return ResultSuccess();
}

Result Pm::PollUpdStatus() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_UpdMutex);

    /*
     * SIGLO-47698: In the sleep sequence, there is a short period that the XUSB
     * clock is stopped, but the CPU is still running (i.e. the code below is
     * being executed). System hangs if we access registers without clock.
     *
     * We "force repoll" in the wake sequence so we won't miss any PD events.
     */
    if (!m_IsAwake)
    {
        // Postpone until awake
        return ResultSuccess();
    }

    nn::usb::pd::driver::GetNotice(&m_UpdNotice, m_UpdSession);
    nn::usb::pd::driver::GetStatus(&m_UpdStatus, m_UpdSession);

    DumpUpdStatus();

    // FIXME: this really should be PowerRoleNotice or PowerContractNotice
    if (m_UpdNotice.IsActiveNotice() || m_UpdNotice.IsErrorNotice())
    {
        m_Charger = UsbChargerType_Unknown;

        // charger detction only when we are connected and sinking power
        if (m_UpdStatus.IsActive() &&
            m_UpdStatus.GetPowerRole() == nn::usb::pd::StatusPowerRole_Sink)
        {
            if (m_UpdStatus.m_CurrentRdo.storage != 0)  // with PD contract
            {
                m_Charger = UsbChargerType_Pd;
            }
            else  // no PD contract
            {
                auto pdoVoltage    = m_UpdStatus.m_CurrentPdo.Get<pd::Pdo::Voltage>();
                auto pdoMaxCurrent = m_UpdStatus.m_CurrentPdo.Get<pd::Pdo::MaximumCurrent>();

                pdoVoltage    *= pd::PdoMilliVoltUnit;
                pdoMaxCurrent *= pd::PdoMilliAmpereUnit;

                if (pdoVoltage == 5000 && pdoMaxCurrent == 3000)      // Type-C Current 3.0A
                {
                    m_Charger = UsbChargerType_TypeC30;
                }
                else if (pdoVoltage == 5000 && pdoMaxCurrent == 1500) // Type-C Current 1.5A
                {
                    m_Charger = UsbChargerType_TypeC15;
                }
                else if (pdoVoltage == 5000 && pdoMaxCurrent == 0)    // USB Default
                {
                    // BC1.2 charger detection only in Device Mode
                    if (m_UpdStatus.GetDataRole() == nn::usb::pd::StatusDataRole_Ufp)
                    {
                        m_Charger = m_pComplex->ChargerDetection();
                    }
                }
            }
        }

        SignalPowerEvent();
        m_pInputNotifier->Notify();
    }

    // FIXME: this really should be DataRoleNotice
    if (m_UpdNotice.IsActiveNotice())
    {
        // It's unlikely that we will have multiple dual role ports. PD shall
        // be revised if someday that comes true.
        for (uint32_t i = 0; i < m_PortCount; i++)
        {
            if (m_Port[i].IsDualRole())
            {
                SetDataRole(m_Port[i], m_UpdStatus.GetDataRole());
                break;
            }
        }
    }

    return ResultSuccess();
}

Result Pm::SetDataRole(PmPort& port, nn::usb::pd::StatusDataRole role) NN_NOEXCEPT
{
    Result result = ResultSuccess();

#if 0  // This allows Copper to enumerate a Cradle as a USB device (hub)
    if(m_UpdStatus.GetDeviceType() == nn::usb::pd::StatusDeviceType_Cradle)
    {
        role = nn::usb::pd::StatusDataRole_Dfp;
    }
#endif

    switch (role)
    {
    case nn::usb::pd::StatusDataRole_Ufp:
        result = port.EnableAsDevice();
        break;

    case nn::usb::pd::StatusDataRole_Dfp:
        result = port.EnableAsHost();
        break;

    case nn::usb::pd::StatusDataRole_Unknown:
        result = port.Disable();
        if (result.IsSuccess())
        {
            // Don't preserve the data role
            port.ResetRole();
        }
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

    if (result.IsSuccess())
    {
        SignalDataEvent();
    }

    return result;
}

void Pm::HandlePowerEvent() NN_NOEXCEPT
{
    Result result;
    nn::psc::PmState    state;
    nn::psc::PmFlagSet  flags;

    NN_USB_ABORT_UPON_ERROR(
        m_PmModule.GetRequest(&state, &flags)
    );

    switch (state)
    {
    case nn::psc::PmState_FullAwake:
        OnFullAwake();
        break;

    case nn::psc::PmState_MinimumAwake:
        OnMinimumAwake();
        break;

    case nn::psc::PmState_SleepReady:
        OnSleepReady();
        break;

    case nn::psc::PmState_ShutdownReady:
        OnShutdownReady();
        break;

    default:
        NN_USB_LOG_INFO("PmState %d (unhandled)\n", state);
        break;
    }

    NN_USB_ABORT_UPON_ERROR(
        m_PmModule.Acknowledge(state, result)
    );

    NN_USB_LOG_INFO("PmState Acknowledged\n");
}

void Pm::OnFullAwake() NN_NOEXCEPT
{
    NN_USB_LOG_INFO("PmState FullAwake\n");

    nn::usb::pd::driver::FullAwake();

    // USB is already enabled in "EssentialServiceAwake -> MinimumAwake"

    m_IsAwake = true;
}

void Pm::OnMinimumAwake() NN_NOEXCEPT
{
    NN_USB_LOG_INFO("PmState MinimumAwake\n");

    nn::usb::pd::driver::MinimumAwake();

    if (m_IsAwake == true)
    {
        // Sleep sequence: FullAwake -> MinimumAwake
        // Do nothing here. USB is disabled later in SleepReady
    }
    else
    {
        // Wake sequence: EssentialServiceAwake -> MinimumAwake
        // Enable the USB here
        m_pComplex->Enable();

        for (uint32_t i = 0; i < m_PortCount; i++)
        {
            m_Port[i].Enable();
        }

        // SIGLO-47698: force re-poll
        m_RepollEvent.Signal();
    }

    m_IsAwake = true;
}

void Pm::OnSleepReady() NN_NOEXCEPT
{
    NN_USB_LOG_INFO("PmState SleepReady\n");

    // Sleep sequence: MinimumAwake -> SleepReady
    for (uint32_t i = 0; i < m_PortCount; i++)
    {
        m_Port[i].Disable();
    }

    m_pComplex->Disable();

    nn::usb::pd::driver::Sleep();

    m_IsAwake = false;
}

void Pm::OnShutdownReady() NN_NOEXCEPT
{
    NN_USB_LOG_INFO("PmState ShutdownReady\n");

    // Shutdown sequence: MinimumAwake -> ShutdownReady
    for (uint32_t i = 0; i < m_PortCount; i++)
    {
        m_Port[i].Disable();
    }

    m_pComplex->Disable();

    nn::usb::pd::driver::Sleep();

    m_IsAwake = false;
}

uint32_t Pm::GetChargerVoltage(UsbChargerType chargerType) NN_NOEXCEPT
{
    uint32_t voltage;

    switch (chargerType)
    {
    case UsbChargerType_TypeC15:
    case UsbChargerType_TypeC30:
    case UsbChargerType_Dcp:
    case UsbChargerType_Cdp:
    case UsbChargerType_Sdp:
    case UsbChargerType_Apple500ma:
    case UsbChargerType_Apple1000ma:
    case UsbChargerType_Apple2000ma:
        voltage = 5000;
        break;

    default:
        voltage = 5000;
        break;
    }

    return voltage;
}

uint32_t Pm::GetChargerCurrent(UsbChargerType chargerType) NN_NOEXCEPT
{
    uint32_t current;

    switch (chargerType)
    {
    case UsbChargerType_TypeC15:
        current = 1500;
        break;

    case UsbChargerType_TypeC30:
        current = 3000;
        break;

    case UsbChargerType_Dcp:
    case UsbChargerType_Cdp:
        current = 1500;
        break;

    case UsbChargerType_Sdp:
        // FIXME: No enumeration, default to 500mA. This is not spec compliant.
        current = 500;
        break;

    case UsbChargerType_Apple500ma:
        current = 500;
        break;

    case UsbChargerType_Apple1000ma:
        current = 900;
        break;

    case UsbChargerType_Apple2000ma:
        current = 2000;
        break;

    default:
        current = 500;
        break;
    }

    return current;
}

} // end of namespace pm
} // end of namespace usb
} // end of namespace nn
