﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <cstring>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/os.h>
#include <nn/util/util_BitPack.h>

#include "usb_PdDriver.h"

namespace nn{
namespace usb{
namespace pd{
namespace driver {
namespace detail{

bool Driver::EnableCradle() NN_NOEXCEPT
{
    Result result;

    // Insert 検出後にクレードル USB HUB 制御タイマースタート
    StartCradleUsbHubTimer();

    // 電力変更
    bool isNonPdDevice = false;
    if ( !ChangePower( &isNonPdDevice ) )
    {
        StopCradleUsbHubTimer();
        // PD コン非搭載デバイスでも IsActive にしなければならない
        if ( isNonPdDevice )
        {
            USBPD_DBG_LOG("PDC unmounted device inserted.\n");
            return true;
        }
        return false;
    }
    // UFP → DFP 遷移
    result = ChangeUfpIntoDfp();
    if ( !result.IsSuccess() )
    {
        StopCradleUsbHubTimer();
        // DataRoleSwap 非対応デバイスでも IsActive にしなければならない
        if ( result <= ResultPdcCommandRejected() )
        {
            USBPD_DBG_LOG("DataRoleSwap unsupported device inserted.\n");
            return true;
        }
        // Split PDO 受信時は DFP 遷移済み
        if ( result <= ResultDataRoleAlreadySwapped() )
        {
            USBPD_WARNING("Data role is already swapped!\n");
            return true;
        }
        return false;
    }
    // クレードル識別
    result = IdentifyCradle();
    if ( !result.IsSuccess() )
    {
        StopCradleUsbHubTimer();
        // クレードル以外でも IsActive にしなければならない
        if ( result <= ResultVdmUnknownDevice() )
        {
            return true;
        }
        return false;
    }
    // DisplayPort 有効化
    result = EnableDisplayPort();
    if ( result.IsSuccess() )
    {
        m_IsDisplayPortAlternateMode = true;
    }
    else if ( !(result <= ResultDisplayPortUnsupported()) && !(result <= ResultPowerShortage()) )
    {
        StopCradleUsbHubTimer();
        // クレードル VDM を送信できない状態は UnknownDevice にする
        std::memset( &m_IdentityVdm, 0, sizeof(m_IdentityVdm) );
        return true;
    }
    // クレードル VDM ステータス取得
    if ( !GetCradleVdmStatus() )
    {
        StopCradleUsbHubTimer();
        return false;
    }
    return true;
}

bool Driver::ChangePower( bool* pIsNonPdDevice ) NN_NOEXCEPT
{
    Result result;
    Status1 status1;

    *pIsNonPdDevice = false;

    os::TimerEvent timer( os::EventClearMode_AutoClear );

    int retryCount = 0;
    m_NewContractCancelEvent.Clear();
    while (1)
    {
        // Power Contract 待ち
        if ( !WaitForPowerContract() )
        {
            return false;
        }
        bool isNewPdos = false;
        // PD コン非搭載デバイス時の Power Contract が長い
        // WaitForPowerContract() 中の Remove はエラーになり、
        // 再 Insert は継続可能なので PlugInsertRemove イベントをクリア
        m_pAlertEvent[AlertStatus::PlugInsertRemove::Pos]->Clear();
        {
            std::lock_guard<nn::os::Mutex> lock(m_AlertMutex);

            // NewContractConsumer アラートと NewContractEvent クリアの同期を取る必要あり
            m_pAlertEvent[AlertStatus::NewContractConsumer::Pos]->Clear();
            m_NewContractEvent.Clear();
            // NewPdos アラートも同期を取ってより安全に
            isNewPdos = m_pAlertEvent[AlertStatus::NewPdos::Pos]->TryWait();
            m_pAlertEvent[AlertStatus::NewPdos::Pos]->Clear();
        }

        // Power Contract → SetSysRdy コマンド間のタイマースタート
        //（実際は EP2-2 から不要だが過渡期のため DP1 からウェイトを削除）
        if ( m_HostPdcFwRevision < HostPdcFwRevisionDp1 )
        {
            timer.Stop();
            timer.Clear();
            timer.StartOneShot( WaitForSetSysRdyTimeSpan );
        }

        // 電力取得
        int voltage = 0;
        int maxAmperePdo = 0;
        int maxAmpereRdo = 0;
        GetCurrentPower( &voltage, &maxAmperePdo, &maxAmpereRdo );
        if ( retryCount )
        {
            // プラグ接続して ChangePowerCore() 呼び出し後から通る
            int v = m_SelectedPdo.Get<Pdo::Voltage>() * PdoMilliVoltUnit;
            int a = m_SetRdo.Get<Rdo::MaximumOperatingCurrent>() * RdoMilliAmpereUnit;
            UpdateAlert();
            isNewPdos |= m_pAlertEvent[AlertStatus::NewPdos::Pos]->TryWait();
            if ( !isNewPdos && m_IsSendRdoCompleted &&
                 voltage == v && maxAmperePdo == a && maxAmpereRdo == a
               )
            {
                // 電力リストの更新が無く、指定電力が反映されていれば正常終了
                break;
            }
            if ( retryCount < PowerContractRetryMax )
            {
                USBPD_WARNING("Retry power contract!\n");
            }
        }

        // Power Contract リトライ上限
        if ( ++retryCount > PowerContractRetryMax )
        {
            USBPD_WARNING("Give up to retry power contract!\n");
            return false;
        }

        // 電力変更（→ Console モード時： 15V * 2.6A = 39W）
        if ( !ChangePowerCore( pIsNonPdDevice, voltage ) )
        {
            return false;
        }
    }

    // Power Contract → SetSysRdy コマンド間のウェイト
    //（実際は EP2-2 から不要だが過渡期のため DP1 からウェイトを削除）
    if ( m_HostPdcFwRevision < HostPdcFwRevisionDp1 )
    {
        timer.Wait();
        timer.Clear();
    }
    USBPD_DBG_LOG("AlertGpio = %d\n", gpio::GetValue( &m_AlertSession ));

    // 充電開始（EP2 以降はコマンド不要）
    if ( m_HostPdcFwRevision < HostPdcFwRevisionEp2 )
    {
        USBPD_DBG_LOG("SetSysRdy\n");
        result = SendAndWaitCommand2( &status1, AlertStatus::CommandCompleteOrAbort::Mask, L2Command_SetSysRdy );
        if ( result <= ResultPdcCommandGiveUp() )
        {
            return false;
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        if ( status1.Get<Status1::LastCommand>() != Status1LastCommand_Completed )
        {
            return false;
        }
    }

    return true;
}

bool Driver::WaitForPowerContract() NN_NOEXCEPT
{
    // Power Contract 待ち
    //   AlertStatus::Error（OVP）発生時はタイムアウト前に AlertStatus::PlugRemove も発生
    os::MultiWaitHolderType* pEventHolder = os::TimedWaitAny( &m_NewContractWaiter, NewContractConsumerTimeoutSpan );
    if ( !pEventHolder )
    {
        USBPD_WARNING("Power contract is timeout!\n");
        return false;
    }
    else if ( pEventHolder == &m_NewContractCancelEventHolder )
    {
        USBPD_WARNING("Power contract is canceled!\n");
        return false;
    }

    return true;
}

void Driver::GetCurrentPower( int* pVoltage, int* pMaxAmperePdo, int* pMaxAmpereRdo ) NN_NOEXCEPT
{
    Result result;

    result = ReceiveCommand1( &m_CurrentPdo, L1CommandSize_CurrentPdo, L1Command_CurrentPdo );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    int voltage = m_CurrentPdo.Get<Pdo::Voltage>() * PdoMilliVoltUnit;
    int maxAmpere = m_CurrentPdo.Get<Pdo::MaximumCurrent>() * PdoMilliAmpereUnit;
    USBPD_DBG_LOG("CurrentPDO(%08x)::Voltage = %d mV, MaximumCurrent = %d mA\n", m_CurrentPdo, voltage, maxAmpere );
    result = ReceiveCommand1( &m_CurrentRdo, L1CommandSize_CurrentRdo, L1Command_CurrentRdo );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    int a = m_CurrentRdo.Get<Rdo::OperatingCurrent>() * RdoMilliAmpereUnit;
    NN_UNUSED(a);
    int ma = m_CurrentRdo.Get<Rdo::MaximumOperatingCurrent>() * RdoMilliAmpereUnit;
    USBPD_DBG_LOG("CurrentRDO(%08x)::OperatingCurrent = %d mA, MaximumCurrent = %d mA\n", m_CurrentRdo, a, ma );

    if ( pVoltage )
    {
        *pVoltage = voltage;
    }
    if ( pMaxAmperePdo )
    {
        *pMaxAmperePdo = maxAmpere;
    }
    if ( pMaxAmpereRdo )
    {
        *pMaxAmpereRdo = ma;
    }
}

bool Driver::ChangePowerCore( bool* pIsNonPdDevice, int voltage ) NN_NOEXCEPT
{
    *pIsNonPdDevice = false;

    // 更新電力選択
    int pdoPosition = 0;
    int volt = 0;
    int amp = 0;
    if ( !SelectNewPower( pIsNonPdDevice, &pdoPosition, &volt, &amp, voltage ) )
    {
        return false;
    }

    // 電力リスト追加更新時はシーケンスの最初からやり直し
    UpdateAlert();
    bool isNewPdos = m_pAlertEvent[AlertStatus::NewPdos::Pos]->TryWait();
    if ( isNewPdos )
    {
        m_SelectedPdo.Clear();
        m_SetRdo.Clear();
        m_IsSendRdoCompleted = false;
        return true;
    }

    // 電力更新
    if ( !RequestNewPower( pdoPosition, volt, amp ) )
    {
        return false;
    }

    return true;
}

bool Driver::SelectNewPower( bool* pIsNonPdDevice, int* pPdoPosition, int* pVolt, int* pAmp, int voltage ) NN_NOEXCEPT
{
    Result result;

    if ( pIsNonPdDevice )
    {
        *pIsNonPdDevice = false;
    }

    // 電力リスト取得
    m_SelectedPdo.Clear();
    Pdo pdos[PdosSrcConsumerSnkProviderNum];
    result = ReceiveCommand1( &pdos, sizeof(pdos), L1Command_PdosSrcConsumerSnkProvider );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("PdosSrcConsumerSnkProvider = %08x %08x %08x %08x %08x %08x %08x\n", pdos[0], pdos[1], pdos[2], pdos[3], pdos[4], pdos[5], pdos[6]); // 最大７つ
    // PD コン非搭載デバイスでは電力変更は行わない
    if ( voltage == MilliVoltOnNonPd && pdos[0].storage == 0 )
    {
        if ( pIsNonPdDevice )
        {
            *pIsNonPdDevice = true;
        }
        return false;
    }
    int pdoPosition = 0;
    int volt = 0;
    int amp = 0;
    for ( int i = 0; i < PdosSrcConsumerSnkProviderNum; i++ )
    {
        int v = pdos[i].Get<Pdo::Voltage>() * PdoMilliVoltUnit;
        int a = pdos[i].Get<Pdo::MaximumCurrent>() * PdoMilliAmpereUnit;
        int oa = a;
        if ( pdos[i].storage )
        {
            USBPD_DBG_LOG("PDO[%d]::Voltage = %d mV, MaximumCurrent = %d mA\n", i + 1, v, a );
        }
        if ( v < MilliVoltMin || MilliVoltMax < v )
        {
            v = 0;
        }
        if ( a < MilliAmpereMin || MilliAmpereMax < a )
        {
            a = 0;
        }
        if ( v * a > volt * amp ||
             ( volt * amp > 0 && v * a == volt * amp && v > volt ) ||
             ( pdoPosition == 0 && v == MilliVoltOnNonPd && oa == 0 ) // クレードルへ PC から電力供給時
           )
        {
            m_SelectedPdo = pdos[i];
            volt = v;
            amp = a;
            pdoPosition = i + 1;
        }
    }
    if ( pPdoPosition )
    {
        *pPdoPosition = pdoPosition;
    }
    if ( pVolt )
    {
        *pVolt = volt;
    }
    if ( pAmp )
    {
        *pAmp = amp;
    }

    return true;
}

bool Driver::RequestNewPower( int pdoPosition, int volt, int amp ) NN_NOEXCEPT
{
    Result result;
    Status1 status1;

    m_IsSendRdoCompleted = false;

    // 最大電力要求
    m_SetRdo.Set<Rdo::ObjectPosition>( pdoPosition );
    m_SetRdo.Set<Rdo::OperatingCurrent>( amp / RdoMilliAmpereUnit );
    m_SetRdo.Set<Rdo::MaximumOperatingCurrent>( amp / RdoMilliAmpereUnit );
    result = SendCommand1( &m_SetRdo, L1CommandSize_SetRdo, L1Command_SetRdo );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("SetRDO = %08x\n", m_SetRdo);
    if ( m_HostPdcFwRevision < HostPdcFwRevisionEp2 )
    {
        // EP1 以前は PS_RDY 待ちが必要
        os::SleepThread( WaitForPsRdyTimeSpan );
    }
    result = SendAndWaitCommand2( &status1, AlertStatus::CommandCompleteOrAbort::Mask, L2Command_SendRdo );
    if ( result <= ResultPdcCommandGiveUp() )
    {
        return false;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    int lastCommand = status1.Get<Status1::LastCommand>();
    if ( lastCommand != Status1LastCommand_Completed )
    {
        USBPD_WARNING("SendRDO is failure (Status1::LastCommand = %d)!\n", lastCommand);
    }
    else
    {
        m_IsSendRdoCompleted = true;
    }

    return true;
}

Result Driver::ChangeUfpIntoDfp() NN_NOEXCEPT
{
    Result result;
    Status1 status1;

    // Split PDO 受信時は DFP 遷移済み
    if ( m_Status1.Get<Status1::DataRole>() == Status1DataRole_Dfp )
    {
        return ResultDataRoleAlreadySwapped();
    }

    // UFP → DFP 要求
    USBPD_DBG_LOG("SendDataRoleSwap\n");
    if ( m_HostPdcFwRevision < HostPdcFwRevisionEp2 )
    {
        // EP1 以前は PS_RDY 待ちが必要
        os::SleepThread( WaitForPsRdyTimeSpan );
    }
    m_pAlertEvent[AlertStatus::DataRoleSwapComplete::Pos]->Clear();
    result = SendAndWaitCommand2( &status1, AlertStatus::CommandCompleteOrAbort::Mask, L2Command_SendDataRoleSwap );
    if ( result <= ResultPdcCommandGiveUp() )
    {
        return result;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    bool isDataRoleSwap = m_pAlertEvent[AlertStatus::DataRoleSwapComplete::Pos]->TryWait();
    m_pAlertEvent[AlertStatus::DataRoleSwapComplete::Pos]->Clear();
    USBPD_DBG_LOG("Status1::DataRole: %s\n", g_pDataRoleName[status1.Get<Status1::DataRole>()]);
    if ( !isDataRoleSwap ||
         status1.Get<Status1::DataRole>() != Status1DataRole_Dfp ||
         status1.Get<Status1::SpdSnk>() != Status1SpdSnk_On ||
         status1.Get<Status1::LastCommand>() != Status1LastCommand_Completed
       )
    {
        USBPD_WARNING("DataRoleSwap is failure!\n");
        if ( status1.Get<Status1::LastCommand>() != Status1LastCommand_Completed )
        {
            return GetStatus1LastCommandResult( status1 );
        }
        else
        {
            return ResultDataRoleSwapFailure();
        }
    }

    return ResultSuccess();
}

Result Driver::IdentifyCradle() NN_NOEXCEPT
{
    Result result;
    Status1 status1;
    Status2 status2;
    VdmHeader sendVdm;

    if ( m_HostPdcFwRevision < HostPdcFwRevisionEp2 )
    {
        // EP1 以前は非対応
        return ResultSuccess();
    }

    USBPD_DBG_LOG("DiscoverIdentity\n");
    result = SendStandardVdm( &status1, &sendVdm, VdmHeaderCommand_DiscoverIdentity );
    if ( result <= ResultPdcCommandGiveUp() )
    {
        // DiscoverIdentity 非対応デバイスの可能性あり
        return ResultVdmUnknownDevice();
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    if ( status1.Get<Status1::LastCommand>() != Status1LastCommand_Completed )
    {
        USBPD_WARNING("DiscoverIdentity is failure!\n");
        if ( status1.Get<Status1::LastCommand>() == Status1LastCommand_Rejected )
        {
            // DiscoverIdentity 非対応デバイスの可能性あり
            return ResultVdmUnknownDevice();
        }
        return GetStatus1LastCommandResult( status1 );
    }
    result = ReceiveStadardVdm( &status2, &m_IdentityVdm, sizeof(m_IdentityVdm) );
    if ( result <= ResultVdmReplyGiveUp() )
    {
        // DiscoverIdentity 非対応デバイスの可能性あり
        return ResultVdmUnknownDevice();
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    VdmUnit* pVdm = &m_IdentityVdm.m_Header;
    NN_UNUSED(pVdm);
    VdmVendorId vid = m_IdentityVdm.m_IdHeader.Get<VdmIdHeader::VendorId>();
    VdmProductId pid = m_IdentityVdm.m_ProductVdo.Get<ProductVdo::ProductId>();
    USBPD_DBG_LOG("DiscoverIdentityVDM = %08x %08x %08x %08x\n", pVdm[0], pVdm[1], pVdm[2], pVdm[3]);
    sendVdm.Set<VdmHeader::CommandType>( VdmCommandType_Ack );
    if ( m_IdentityVdm.m_Header.storage != sendVdm.storage )
    {
        USBPD_WARNING("DiscoverIdentity::VdmHeader is invalid(%04x)!\n", m_IdentityVdm.m_Header);
        return ResultVdmUnknownDevice();
    }
    if ( vid != VdmNintendoId )
    {
        USBPD_WARNING("DiscoverIdentity::VdmNintendoID(%04x) is invalid(%04x)!\n", VdmNintendoId, vid);
        return ResultVdmUnknownDevice();
    }
    if ( pid != VdmCradleId && pid != VdmTableDockId )
    {
        if ( !IdentifyAcAdaptor( pid ) )
        {
            USBPD_WARNING("DiscoverIdentity::VdmCradleID(%04x) is invalid(%04x)!\n", VdmCradleId, pid);
        }
        return ResultVdmUnknownDevice();
    }
    if ( status2.Get<Status2::IncomingVdmType>() != Status2IncomingVdmType_Sop )
    {
        USBPD_WARNING("DiscoverIdentity::IncomingVdmType is invalid!\n");
        return ResultVdmUnknownDevice();
    }
    USBPD_DBG_LOG("EnterMode\n");
    result = SendStandardVdm( &status1, &sendVdm, VdmHeaderCommand_EnterMode );
    if ( result <= ResultPdcCommandGiveUp() )
    {
        return ResultVdmUnknownDevice();
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    if ( status1.Get<Status1::LastCommand>() != Status1LastCommand_Completed )
    {
        USBPD_WARNING("EnterMode is failure!\n");
        if ( status1.Get<Status1::LastCommand>() == Status1LastCommand_Rejected )
        {
            return ResultVdmUnknownDevice();
        }
        return GetStatus1LastCommandResult( status1 );
    }
    EnterModeVdm enterModeVdm;
    result = ReceiveStadardVdm( &status2, &enterModeVdm, sizeof(enterModeVdm) );
    if ( result <= ResultVdmReplyGiveUp() )
    {
        // Prooduct ID がクレードルなら EnterMode に対応しているはず
        return result;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("EnterModeVDM = %08x\n", enterModeVdm.m_Header);
    sendVdm.Set<VdmHeader::CommandType>( VdmCommandType_Ack );
    if ( enterModeVdm.m_Header.storage != sendVdm.storage )
    {
        USBPD_WARNING("EnterMode::VdmHeader is invalid(%04x)!\n", enterModeVdm.m_Header);
        return ResultVdmUnknownDevice();
    }
    if ( status2.Get<Status2::IncomingVdmType>() != Status2IncomingVdmType_Sop )
    {
        USBPD_WARNING("EnterMode::IncomingVdmType is invalid!\n");
        return ResultVdmUnknownDevice();
    }

    return ResultSuccess();
}

bool Driver::IsDisplayPortPowerEnough() NN_NOEXCEPT
{
    // 39W 未満では DisplayPort 無効化
    int voltage = m_CurrentPdo.Get<Pdo::Voltage>() * PdoMilliVoltUnit;
    int ampere = m_CurrentRdo.Get<Rdo::OperatingCurrent>() * RdoMilliAmpereUnit;
    int watt = voltage * ampere / 1000;
    if ( watt < MilliWattOnConsole )
    {
        USBPD_WARNING("AC adaptor is power shortage!\n");
        m_IsPowerShortage = true;
        return false;
    }

    return true;
}

Result Driver::EnableDisplayPort() NN_NOEXCEPT
{
    Status1 status1;

    VdmVendorId vid = m_IdentityVdm.m_IdHeader.Get<VdmIdHeader::VendorId>();
    VdmProductId pid = m_IdentityVdm.m_ProductVdo.Get<ProductVdo::ProductId>();
    if ( vid == VdmNintendoId && pid == VdmTableDockId )
    {
        return ResultDisplayPortUnsupported();
    }

    if ( !IsDisplayPortPowerEnough() )
    {
        return ResultPowerShortage();
    }

    // DisplayPort 有効化
    USBPD_DBG_LOG("StartDisplayPortEnterMode\n");
    auto result = SendAndWaitCommand2( &status1, AlertStatus::CommandCompleteOrAbort::Mask, L2Command_StartDisplayPortEnterMode );
    if ( result <= ResultPdcCommandGiveUp() )
    {
        return result;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    if ( status1.Get<Status1::LastCommand>() != Status1LastCommand_Completed )
    {
        return GetStatus1LastCommandResult( status1 );
    }
    VdmUnit dpVdm;
    // 特別に VDM ヘッダが付かない
    result = ReceiveCommand1( &dpVdm, sizeof(dpVdm), L1Command_IncomingVdm );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("Recv DpVDM = %08x\n", dpVdm);
    if ( !dpVdm.Get<VdmDisplayPortCapabilities::UfpdPinAssignmentSupportedD>() )
    {
        return ResultDisplayPortError();
    }

    // DisplayPort 挿抜検出有効化
    dpVdm.Clear();
    dpVdm.Set<VdmDisplayPortCapabilities::PortCapability>( VdmPortCapability_DfpD );
    dpVdm.Set<VdmDisplayPortCapabilities::SignalingForTransport>( VdmSignalingForTransport_Dp13 );
    dpVdm.Set<VdmDisplayPortCapabilities::UfpdPinAssignmentsSupported>( VdmPinAssignment_D );
    result = SendCommand1( &dpVdm, sizeof(dpVdm), L1Command_OutgoingVdm );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("Send DpVDM = %08x\n", dpVdm);

    USBPD_DBG_LOG("StartDisplayPortConfigureAndStartHandlingHpd\n");
    result = SendAndWaitCommand2( &status1, AlertStatus::CommandCompleteOrAbort::Mask,
                                  L2Command_StartDisplayPortConfigureAndStartHandlingHpd );
    if ( result <= ResultPdcCommandGiveUp() )
    {
        return result;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    if ( status1.Get<Status1::LastCommand>() != Status1LastCommand_Completed )
    {
        return GetStatus1LastCommandResult( status1 );
    }

    // DisplayPortStatus チェック
    result = ReceiveCommand1( &m_DisplayPortStatus, L1CommandSize_DisplayPortStatus, L1Command_DisplayPortStatus );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("DisplayPortStatus = %04x\n", m_DisplayPortStatus);

    return ResultSuccess();
}

bool Driver::GetCradleVdmStatus() NN_NOEXCEPT
{
    Result result;

    result = SendAndReceiveNxVdm( &m_CradleDeviceStatus, sizeof(m_CradleDeviceStatus), nullptr, VdmCommand_DeviceState );
    if ( result <= ResultVdmReplyGiveUp() )
    {
        // DP1 から VDM タイムアウトは Inactive 扱い
        return m_HostPdcFwRevision < HostPdcFwRevisionDp1 ? true : false;
    }
    else if ( result <= ResultVdmError() )
    {
        return false;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("CradleDeviceStateVDM = %08x\n", m_CradleDeviceStatus);
    result = SendAndReceiveNxVdm( &m_CradlePdcHFwVersion, sizeof(m_CradlePdcHFwVersion), nullptr, VdmCommand_PdcHFwVersion );
    if ( result <= ResultVdmReplyGiveUp() )
    {
        return m_HostPdcFwRevision < HostPdcFwRevisionDp1 ? true : false;
    }
    else if ( result <= ResultVdmError() )
    {
        return false;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("CradlePdcHFwVersionVDM = %08x\n", m_CradlePdcHFwVersion);
    result = SendAndReceiveNxVdm( &m_CradleMcuFwVersion, sizeof(m_CradleMcuFwVersion), nullptr, VdmCommand_McuFwVersion );
    if ( result <= ResultVdmReplyGiveUp() )
    {
        return m_HostPdcFwRevision < HostPdcFwRevisionDp1 ? true : false;
    }
    else if ( result <= ResultVdmError() )
    {
        return false;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("CradleMcuFwVersionVDM = %08x\n", m_CradleMcuFwVersion);

    return true;
}

void Driver::NotifyCradleError() NN_NOEXCEPT
{
    Result result;

    m_pNoticeVdmEvent[VdmNoticeType_DeviceError]->Clear();

    USBPD_WARNING("Notify Cradle Error!\n");
    VdmHeader header;
    header.Clear();
    header.Set<VdmHeader::SvId>( VdmNintendoId );
    header.Set<VdmHeader::VdmType>( VdmType_UnstructuredVdm );
    header.Set<VdmHeader::VdmVersion>( VdmVersion_1_0 );
    header.Set<VdmHeader::ObjectPosition>( 0 );
    header.Set<VdmHeader::CommandType>( VdmCommandType_Initiator );
    header.Set<VdmHeader::HeaderCommand>( VdmHeaderCommand_None );
    result = ReceiveNxVdm( &m_CradleDeviceStatus, sizeof(m_CradleDeviceStatus), &header, VdmCommand_DeviceErrorNotice );
    if ( result <= ResultVdmError() )
    {
        return;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("CradleDeviceState = %08x\n", m_CradleDeviceStatus);

    // クレードルエラー通知
    UpdateStatus( false, Notice::DeviceNotice::Mask | Notice::ErrorNotice::Mask );
}

void Driver::StartCradleUsbHubTimer() NN_NOEXCEPT
{
    if ( m_HostPdcFwRevision < HostPdcFwRevisionDp1 )
    {
        // EP2 以前は非対応
        return;
    }

    StopCradleUsbHubTimer();
    m_CradleUsbHubTimerEvent.StartOneShot( CradleUsbHubStartTimeSpan );
}

void Driver::StopCradleUsbHubTimer() NN_NOEXCEPT
{
    m_CradleUsbHubTimerEvent.Stop();
    m_CradleUsbHubTimerEvent.Clear();
    m_CradleUsbHubRepeatCount = 0;
}

void Driver::DetectUsbHubInCradle() NN_NOEXCEPT
{
    Result result;

    USBPD_DBG_LOG("Detecting cradle USB HUB...\n");
    m_CradleUsbHubTimerEvent.Stop();
    m_CradleUsbHubTimerEvent.Clear();
    if ( m_CradlePdcHFwVersion == 0 )
    {
        USBPD_WARNING("Cradle has been removed!\n");
        return;
    }
    else if ( m_CradlePdcHFwVersion < CradlePdcHFwRevisionDp1 )
    {
        // クレードル EP2 以前は非対応
        USBPD_WARNING("This cradle doesn't support USB HUB control!\n");
        return;
    }
    // クレードル USB HUB VBusDetect を Enable
    result = SendAndReceiveNxVdmWithRequest( &m_CradleUsbHubState, sizeof(m_CradleUsbHubState), nullptr, VdmSet_UsbHubEnableRequest, VdmCommand_UsbHubVBusDetect );
    if ( result <= ResultVdmError() )
    {
        USBPD_WARNING("Detecting cradle USB HUB is ResultVdmError(%d-%04d)!\n", result.GetModule(), result.GetDescription());
        return;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    if ( !m_CradleUsbHubState.Get<VdmCommonVdo1::VBusDetectError>() )
    {
        USBPD_DBG_LOG("Cradle USB HUB is detected.\n");
    }
    else if ( m_CradleUsbHubRepeatCount < CradleUsbHubRetryMax )
    {
        m_CradleUsbHubRepeatCount++;
        if ( m_CradleUsbHubRepeatCount < CradleUsbHubRetryMax )
        {
            m_CradleUsbHubTimerEvent.StartOneShot( CradleUsbHubRepeatTimeSpan );
            USBPD_WARNING("Retry to detect cradle USB HUB!\n");
        }
        else
        {
            USBPD_WARNING("Cradle USB HUB VBUS undetected error!\n");
            // クレードル USB HUB 検出エラー通知
            //   クレードルの USB Type-A コネクタが使えない状態であることをユーザへ通知したり
            //   上位層からクレードルのリセット API を呼んで復旧させる可能性を想定
            m_IpcError = StatusError_CradleUsbHubUndetected;
            UpdateStatus( false, Notice::ErrorNotice::Mask );
        }
    }
}

Result Driver::ResetCradleUsbHub() NN_NOEXCEPT
{
    Result result;

    if ( m_HostPdcFwRevision < HostPdcFwRevisionDp1 )
    {
        // EP2 以前は非対応
        return ResultSuccess();
    }

    USBPD_DBG_LOG("Resetting cradle USB HUB...\n");

    // クレードル USB HUB VBusDetect を Disable
    result = SendAndReceiveNxVdmWithRequest( &m_CradleUsbHubState, sizeof(m_CradleUsbHubState), nullptr, VdmSet_UsbHubDisableRequest, VdmCommand_UsbHubVBusDetect );
    if ( result <= ResultVdmError() )
    {
        return result;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("Cradle USB HUB changed into semihost mode.\n");

    // クレードル USB HUB をリセット
    result = SendAndReceiveNxVdmWithRequest( nullptr, 0, nullptr, VdmSet_NoSubRequest, VdmCommand_UsbHubReset );
    if ( result <= ResultVdmError() )
    {
        return result;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("Cradle USB HUB was reset.\n");

    // クレードル USB HUB 制御タイマースタート
    StartCradleUsbHubTimer();

    return ResultSuccess();
}

bool Driver::IsCradleFamily( StatusDeviceType deviceType ) NN_NOEXCEPT
{
    return deviceType == StatusDeviceType_Cradle ||
           deviceType == StatusDeviceType_RelayBox ||
           deviceType == StatusDeviceType_TableDock;
}

} // detail
} // driver
} // pd
} // usb
} // nn
