﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief Usb Request Block (URB)
 *
 * @details
 *
 *
 */

#pragma once

namespace nn { namespace usb { namespace hs {

class UrbPool;

class UsbRequestBlock
{
public:
    enum State
    {
        State_Invalid = 0,
        State_Free,
        State_Allocated,
        State_Pending,
    };
    typedef void (*Callback)(Hs* pHs, UsbRequestBlock* pUrb);

    /*
     * Ref [xHCI r1.1] 6.4.1
     *   "Data buffers referenced by Transfer TRBs shall not span 64KB
     *    boundaries. If a physical data buffer spans a 64KB boundary,
     *    software shall chain multiple TRBs to describe the buffer."
     *
     * Since we have a pretty big IoVa heap, we get the luxury to ask for
     * 64KB IoVa alignment for every data buffer. However, remember that we
     * could still be in trouble if a batch xfer uses a buffer bigger than
     * 64KB. For now, we leave this problem unsolved until a class driver
     * actually do so.
     */
    const size_t BufferAlignment = 64 * 1024;

    // URB management
    nn::util::IntrusiveListNode   m_ListNode;
    State                         m_State;

    // Data source / sink
    HostControllerDriverEndpoint* m_pHcEp;

    // CtrlRequest data
    UsbCtrlRequest                m_CtrlRequest;

    nn::dd::ProcessHandle         m_ProcHandle;
    detail::SmmuSpace            *m_pSmmuSpace;
    uint64_t                      m_ProcVa;
    nn::dd::DeviceVirtualAddress  m_IoVa;
    uint32_t                      m_BufferSize;
    bool                          m_IsSmmuMapDynamic;
    UsbBusToken                   m_Token;

    struct {
        uint32_t size;
        uint32_t xferredSize;
        Result   result;
    } m_Xfer[HsLimitMaxXferPerUrbCount];
    uint32_t                      m_XferCount;
    uint32_t                      m_CompletionCount;

    ClientEpSession::UrbDeferredRequest *m_pDeferredRequest;
    void*                         m_Context;
    Callback                      m_Callback;

    SchedulePolicy                m_Policy;
    uint32_t                      m_FrameId;

    uint32_t                      m_TimeoutInMs;
    LocalEventType                m_TimeoutEvent;

public:
    UsbRequestBlock(UrbPool& urbPool, HostControllerDriverEndpoint *pHcEp);
    ~UsbRequestBlock();

    Result Submit();
    void   Complete();

    void * operator new(size_t size) NN_NOEXCEPT
    {
        return detail::UsbMemoryAllocAligned(
            size, ObjectMemAlignmentSize, "UsbRequestBlock"
        );
    }

    void operator delete(void *p, size_t size) NN_NOEXCEPT
    {
        NN_UNUSED(size);
        detail::UsbMemoryFree(p, "UsbRequestBlock");
    }

private:
    static const uint32_t ObjectMemAlignmentSize = HwLimitDataCacheLineSize;

    UrbPool&            m_Pool;
};

class UrbPool
{
public:
    UrbPool(Hs *pHs, HostControllerDriverEndpoint *pHcEp);
    ~UrbPool();

    Result Initialize(uint32_t count);
    Result Finalize();

    // Free -> Allocated
    UsbRequestBlock* AllocateUrb();

    // used by class driver
    Result AllocateUrb(UsbRequestBlock                    **ppCreatedUrb,
                       ClientEpSession::UrbDeferredRequest *pRequest);

    // internal use, e.g. hub intr xfer
    Result AllocateUrb(UsbRequestBlock         **ppCreatedUrb,
                       nn::dd::ProcessHandle     procHandle,
                       uint64_t                  procVa,
                       uint32_t                  bufferSize,
                       bool                      isSmmuMapDynamic,
                       UsbBusToken               token,
                       UsbRequestBlock::Callback callback,
                       void                     *context,
                       uint32_t                  timeoutInMs);

    // internal use, ctrl xfer
    Result AllocateUrb(UsbRequestBlock         **ppCreatedUrb,
                       nn::dd::ProcessHandle     procHandle,
                       uint64_t                  procVa,
                       UsbCtrlRequest&           ctrlRequest,
                       bool                      isSmmuMapDynamic,
                       UsbRequestBlock::Callback callback,
                       void                     *context,
                       uint32_t                  timeoutInMs);

    void FreeUrb(UsbRequestBlock *pUrb);

    void OnUrbSubmission(UsbRequestBlock *pUrb);
    void OnUrbCompletion(UsbRequestBlock *pUrb);

    void CompleteAll();

    void StartTimedEvent(LocalEventType *pEvent, uint32_t timeoutInMs);
    void StopTimedEvent(LocalEventType *pEvent);

private:
    typedef detail::NnList<UsbRequestBlock> UrbListType;

    Hs                            *m_pHs;
    LocalEventPoolType             m_LocalEventPool;
    LocalEventType                 m_LocalEventStorage[1];

    HostControllerDriverEndpoint  *m_pHcEp;
    PlatformController            *m_pPlatform;

    UrbListType                    m_UrbFreeList;
    UrbListType                    m_UrbAllocatedList;
    UrbListType                    m_UrbPendingList;

private:
    void DeleteAllUrbs();

    static void TimerCallbackStatic(void *pContext, LocalEventType *pEvent);
};

} // end of namespace hs
} // end of namespace usb
} // end of namespace nn

