﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "usb_HsPrivateIncludes.h"

namespace nn { namespace usb { namespace hs {


//--------------------------------------------------------------------------
//  ControlTransferManager::Transaction implementation
//--------------------------------------------------------------------------
void* ControlTransferManager::Transaction::operator new(size_t size) NN_NOEXCEPT
{
    return detail::UsbMemoryAllocAligned(size, Transaction::ObjectMemAlignmentSize,
                                         "ControlTransferManager::Transaction");
}
void ControlTransferManager::Transaction::operator delete(void *p, size_t size) NN_NOEXCEPT
{
    detail::UsbMemoryFree(p, "ControlTransferManager::Transaction");
}


//--------------------------------------------------------------------------
//  ControlTransferManager implementation
//--------------------------------------------------------------------------
void* ControlTransferManager::operator new(size_t size) NN_NOEXCEPT
{
    return detail::UsbMemoryAllocAligned(size, ControlTransferManager::ObjectMemAlignmentSize,
                                         "ControlTransferManager");
}
void ControlTransferManager::operator delete(void *p, size_t size) NN_NOEXCEPT
{
    detail::UsbMemoryFree(p, "ControlTransferManager");
}

const char *ControlTransferManager::StateNames[ControlTransferManager::State_Maximum] =
{
    "Null", "Idle", "Processing", "StallRecovery"
};
const char *ControlTransferManager::EventNames[ControlTransferManager::Event_Maximum] =
{
    "Request", "UrbComplete", "StallRecoveryComplete", "Error"
};


void ControlTransferManager::Initialize(HostControllerDriverEndpoint *pHcEp,
                                        const char *objectName)
{
    char fsmName[HsLimitMaxDebugNameSize] = { 0 };

    m_pHcEp = pHcEp;
    nn::util::SNPrintf(fsmName, sizeof(fsmName), "CtrlXfer(%s)", objectName);

    NN_USB_ABORT_UNLESS_SUCCESS(
        Fsm::Initialize(fsmName,
                        EventNames, Event_Maximum,
                        StateNames, State_Maximum,
                        State_Idle)
    );
}

/*
 * This finalize cancels anything pending, does not do callback completions.
 *
 * FIXME: Careful, this is flawed. Bottom half of async operations (URB, Reset
 *        EP) in this FSM could still queue events even after this call being
 *        made. For now, this call must be placed very carefully...
 */
void ControlTransferManager::Finalize()
{
    NN_USB_ABORT_UNLESS_SUCCESS(Fsm::Finalize());

    CompleteCurrentTransaction(ResultEndpointClosed());

    while (!m_TransactionList.empty())
    {
        m_pCurrentTxn = &m_TransactionList.front();
        m_TransactionList.erase(m_TransactionList.iterator_to(*m_pCurrentTxn));
        CompleteCurrentTransaction(ResultEndpointClosed());
    }

    m_pHcEp = nullptr;
}

Result ControlTransferManager::SubmitAsync(uint8_t bRequest, uint8_t bmRequestType, uint16_t wValue,
                                           uint16_t wIndex, uint16_t wLength,
                                           void *pData, bool isSmmuMapDynamic,
                                           CompletionCallback callback,
                                           void *context, uint32_t timeoutInMs)
{
    Result result = ResultSuccess();

    Transaction *pTxn =
        new Transaction(bmRequestType, bRequest, wValue, wIndex, wLength,
                        nn::dd::GetCurrentProcessHandle(), (uint64_t)pData, isSmmuMapDynamic,
                        callback, context, timeoutInMs);
    if (pTxn != nullptr)
    {
        m_TransactionList.push_back(*pTxn);
        Fsm::QueueFsmEvent(Event_Request, pTxn);
    }
    else
    {
        result = ResultMemAllocFailure();
    }

    return result;
}

Result ControlTransferManager::SubmitAsync(ClientIfSession::CtrlXferDeferredRequest *pDefReq)
{
    Result result = ResultSuccess();
    ClientIfSession::CtrlXferArgDataType& data = pDefReq->m_Data;
    Transaction *pTxn = new Transaction(data.bmRequestType,
                                        data.bRequest,
                                        data.wValue,
                                        data.wIndex,
                                        data.wLength,
                                        data.procHandle,
                                        data.procVa,
                                        true,
                                        DeferredRequestCompletionCallback,
                                        pDefReq,
                                        data.timeoutInMs);
    if (pTxn != nullptr)
    {
        m_TransactionList.push_back(*pTxn);
        Fsm::QueueFsmEvent(Event_Request, pTxn);
    }
    else
    {
        result = ResultMemAllocFailure();
    }

    return result;
}

void ControlTransferManager::CompleteCurrentTransaction(Result status)
{
    if (m_pCurrentTxn != nullptr)
    {
        m_pCurrentTxn->m_Status = status;
        (*m_pCurrentTxn->m_Callback)(m_pHs, m_pCurrentTxn->m_Context,
                                     m_pCurrentTxn->m_Status, m_pCurrentTxn->m_TransferredSize);
        delete m_pCurrentTxn;
        m_pCurrentTxn = nullptr;
    }
}

void ControlTransferManager::DeferredRequestCompletionCallback(Hs *pHs, void *context, Result status, uint32_t transferredSize)
{
    Result result = status;
    ClientIfSession::CtrlXferDeferredRequest *pDefReq =
        reinterpret_cast<ClientIfSession::CtrlXferDeferredRequest *>(context);

    /*
     * HACK: SF won't pass back <OUT> parameters if returned Result is not
     *       successful. This is a problem with the short transfer. So let's
     *       pretend to be success for short packet error, so transferredSize
     *       can be returned.
     *
     *       Now it's caller's responsibility to check if it's a short or not.
     */
    if (ResultShortPacket::Includes(result))
    {
        result = ResultSuccess();
    }

    pDefReq->CompleteRequest(result, transferredSize);
}

void ControlTransferManager::ResetEndpointCompletionCallback(Hs* pHs,  Result status, HostControllerDriverEndpoint* pHcEp, void *context)
{
    ControlTransferManager *pThis = static_cast<ControlTransferManager *>(context);
    pThis->Fsm::QueueFsmEvent(Event_StallRecoveryComplete, status);
}

Result ControlTransferManager::FsmHandler(int32_t state, LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();

    switch (state)
    {
    case State_Null:
        break;
    case State_Idle:
        result = IdleStateHandler(pEvent);
        break;
    case State_Processing:
        result = ProcessingStateHandler(pEvent);
        break;
    case State_StallRecovery:
        result = StallRecoveryStateHandler(pEvent);
        break;
    default:
        NN_USB_ABORT("ControlTransferManager::state %d invalid", state);
        break;
    }

    return result;
}

Result ControlTransferManager::IdleStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();

    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        if (!m_TransactionList.empty())
        {
            Fsm::QueueFsmEvent(Event_Request, result);
        }
        break;

    case Event_Request:
        Fsm::SetState(State_Processing);
        break;

    case Fsm::Event_Exit:
        break;

    default:
        result = ResultUnexpectedEvent();
        break;
    }

    return result;
}

Result ControlTransferManager::ProcessingStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    UsbRequestBlock *pUrb = nullptr;

    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        // grab a ctrl transaction request
        if (m_TransactionList.empty())
        {
            result = ResultInternalStateError();
            Fsm::QueueFsmEvent(Event_Error, result);
            break;
        }
        m_pCurrentTxn = &m_TransactionList.front();
        m_TransactionList.erase(m_TransactionList.iterator_to(*m_pCurrentTxn));

        // allocate an URB
        result = m_pHcEp->CreateCtrlUrb(
            m_pCurrentTxn->m_ProcHandle,
            m_pCurrentTxn->m_ProcVa,
            m_pCurrentTxn->m_CtrlRequest,
            m_pCurrentTxn->m_IsSmmuMapDynamic,
            [](Hs *pHs, UsbRequestBlock *pUrb) {
                auto *self = static_cast<ControlTransferManager *>(pUrb->m_Context);
                auto& xfer = pUrb->m_Xfer[0];
                self->QueueFsmEvent(Event_UrbComplete, xfer.result, xfer.xferredSize);
            },
            this,
            m_pCurrentTxn->m_TimeoutInMs, &pUrb
        );
        if (result.IsFailure())
        {
            Fsm::QueueFsmEvent(Event_Error, result);
            break;
        }

        // submit the URB
        result = pUrb->Submit();
        if (result.IsFailure())
        {
            m_pHcEp->DestroyUrb(pUrb);
            Fsm::QueueFsmEvent(Event_Error, result);
            break;
        }

        // now wait for the URB completion
        break;

    case Event_UrbComplete:
        m_pCurrentTxn->m_TransferredSize = pEvent->args[0].dataUint32;

        if (pEvent->status.IsFailure())
        {
            Fsm::QueueFsmEvent(Event_Error, pEvent->status);
        }
        else
        {
            CompleteCurrentTransaction(pEvent->status);
            Fsm::SetState(State_Idle);
        }

        break;

    case Event_Request:
        // ignore the new ctrl transaction request, it will be
        // handled when we go back to idle state
        break;

    case Event_Error:
        if (ResultStalled::Includes(pEvent->status))
        {
            // Don't complete the transaction until stall recovery is done
            Fsm::SetState(State_StallRecovery);
        }
        else
        {
            CompleteCurrentTransaction(pEvent->status);
            Fsm::SetState(State_Idle);
        }

        break;

    case Fsm::Event_Exit:
        break;

    default:
        result = ResultUnexpectedEvent();
        break;
    }

    return result;
}

Result ControlTransferManager::StallRecoveryStateHandler(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    switch (pEvent->eventId)
    {
    case Fsm::Event_Entry:
        m_pHcEp->ResetAsync(ResetEndpointCompletionCallback, this);
        break;

    case Event_Request:
        // deferred
        break;

    case Event_StallRecoveryComplete:
        CompleteCurrentTransaction(ResultStalled());
        Fsm::SetState(State_Idle);
        break;

    case Fsm::Event_Exit:
        break;

    default:
        result = ResultUnexpectedEvent();
        break;
    }
    return result;
}





} // end of namespace hs
} // end of namespace usb
} // end of namespace nn
