﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//#define NN_USB_ENABLE_TRACE

#include "usb_DsProtocol.h"
#include "usb_DsController.h"
#include "usb_DsDummyController.h"
#include <nn/nn_SystemThreadDefinition.h>

#include "usb_DsController30-soc.tegra.h"

#undef  NN_USB_TRACE_CLASS_NAME
#define NN_USB_TRACE_CLASS_NAME "DsProtocol"

using namespace nn::dd;

namespace nn {
namespace usb {
namespace ds {

DsDummyController DsProtocol::s_DummyController;

//----------------------------------------------------------------------------
//  Control Request Dispatch Table
//----------------------------------------------------------------------------

const DsProtocol::CtrlHandlerRecord DsProtocol::CtrlHandlerTable[] = {
    // standard request
    {0xe0, 0x80, 0xff, UsbCtrlXferReq_GetStatus,        &DsProtocol::GetStatus       },
    {0xff, 0x00, 0xff, UsbCtrlXferReq_ClearFeature,     &DsProtocol::ClearFeature    },
    {0xff, 0x00, 0xff, UsbCtrlXferReq_SetFeature,       &DsProtocol::SetFeature      },
    {0xff, 0x80, 0xff, UsbCtrlXferReq_GetDescriptor,    &DsProtocol::GetDescriptor   },
    {0xff, 0x00, 0xff, UsbCtrlXferReq_SetAddress,       &DsProtocol::SetAddress      },
    {0xff, 0x80, 0xff, UsbCtrlXferReq_GetConfiguration, &DsProtocol::GetConfiguration},
    {0xff, 0x00, 0xff, UsbCtrlXferReq_SetConfiguration, &DsProtocol::SetConfiguration},

    // Endpoint request
    {0xff, 0x02, 0xff, UsbCtrlXferReq_ClearFeature,     &DsProtocol::ClearFeature},
    {0xff, 0x02, 0xff, UsbCtrlXferReq_SetFeature,       &DsProtocol::SetFeature},

    // Hid request, this sort of thing should really be exported to the client
    // however at this time we don't want to fire up a thread to handle these
    // notifications. These are for the default charging device.
    {0xff, 0x81, 0xff, UsbCtrlXferReq_GetDescriptor,    &DsProtocol::GetReportDescriptor},
    {0xff, 0x21, 0xff, 0x0a,                            &DsProtocol::SetIdle},

    // Required for USB 3.0
    {0xff, 0x00, 0xff, UsbCtrlXferReq_SetSel,           &DsProtocol::SetSel          },
    {0xff, 0x00, 0xff, UsbCtrlXferReq_SetIsocDelay,     &DsProtocol::SetIsocDelay    },


    // stall for all other requests
};


//----------------------------------------------------------------------------
//  Public Methods
//----------------------------------------------------------------------------
DsProtocol::DsProtocol() NN_NOEXCEPT
    : m_DsDeviceRef(0)
    , m_DeviceIsEnabled(false)
    , m_ProtocolMutex(true)
    , m_StateChangeMutex(false)
    , m_EndPointMutex(true)
    , m_SetupMutex(false)
{
    NN_USB_TRACE;
    for (int i = 0; i < DsLimitMaxInterfacesPerConfigurationCount; i++)
    {
        m_pSetupEvent[i]        = nullptr;
        m_IsInterfaceEnabled[i] = false;
    }
}


//////////////////////////////////////////////////////////////////////////////
DsProtocol::~DsProtocol() NN_NOEXCEPT
{
    NN_USB_TRACE;
    // do nothing
}


//////////////////////////////////////////////////////////////////////////////
Result DsProtocol::Initialize() NN_NOEXCEPT
{
    NN_USB_TRACE;
    m_CtrlFlow      = CtrlFlow_Invalid;
    m_State         = UsbState_Detached;
    m_Configuration = 0;

    // All hardware operations are routed to dummy controller for now.
    // Attach() must be called in order to route to real controller.
    m_pController           = &s_DummyController;
    m_pPhysicalController   = nullptr;

    // descriptor container for this protocol instance
    NN_USB_ABORT_UNLESS_SUCCESS(m_Descriptor.Initialize(this));

    // multi wait
    nn::os::InitializeMultiWait(&m_MultiWait);

    nn::os::InitializeEvent(&m_CtrlRequestEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeMultiWaitHolder(&m_CtrlRequestHolder, &m_CtrlRequestEvent);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_CtrlRequestHolder);

    nn::os::InitializeEvent(&m_CtrlIOEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeMultiWaitHolder(&m_CtrlIOHolder, &m_CtrlIOEvent);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_CtrlIOHolder);

    nn::os::InitializeEvent(&m_CtrlFinalizeEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeMultiWaitHolder(&m_CtrlFinalizeHolder, &m_CtrlFinalizeEvent);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_CtrlFinalizeHolder);

    // start the dedicated ctrl pipe thread
    NN_USB_ABORT_UNLESS_SUCCESS(
        nn::os::CreateThread(&m_CtrlPipeThread,
                             CtrlPipeThread,
                             this,
                             &m_ThreadStack,
                             sizeof(m_ThreadStack),
                             NN_SYSTEM_THREAD_PRIORITY(usb,DsProtocol))
    );
    nn::os::SetThreadNamePointer(&m_CtrlPipeThread, NN_SYSTEM_THREAD_NAME(usb, DsProtocol));
    nn::os::StartThread(&m_CtrlPipeThread);

    // Create EP0 urbs for completion events.
    GetEndPointFromAddress(UsbEndpointAddressMask_DirDeviceToHost).AllocUrbRing(m_pController, UsbEndpointAddressMask_DirDeviceToHost);
    GetEndPointFromAddress(UsbEndpointAddressMask_DirHostToDevice).AllocUrbRing(m_pController, UsbEndpointAddressMask_DirHostToDevice);

    GetEndPointFromAddress(UsbEndpointAddressMask_DirDeviceToHost).Enable();
    GetEndPointFromAddress(UsbEndpointAddressMask_DirHostToDevice).Enable();

    // Start the charging device here, if another client joins it will be stopped.
    m_DsChargingDevice.Initialize(this);

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result DsProtocol::Finalize() NN_NOEXCEPT
{
    NN_USB_TRACE;
    // stop the ctrl loop thread
    nn::os::SignalEvent(&m_CtrlFinalizeEvent);
    nn::os::WaitThread(&m_CtrlPipeThread);
    nn::os::DestroyThread(&m_CtrlPipeThread);

    nn::os::UnlinkMultiWaitHolder(&m_CtrlRequestHolder);
    nn::os::FinalizeMultiWaitHolder(&m_CtrlRequestHolder);
    nn::os::FinalizeEvent(&m_CtrlRequestEvent);

    nn::os::UnlinkMultiWaitHolder(&m_CtrlIOHolder);
    nn::os::FinalizeMultiWaitHolder(&m_CtrlIOHolder);
    nn::os::FinalizeEvent(&m_CtrlIOEvent);

    nn::os::UnlinkMultiWaitHolder(&m_CtrlFinalizeHolder);
    nn::os::FinalizeMultiWaitHolder(&m_CtrlFinalizeHolder);
    nn::os::FinalizeEvent(&m_CtrlFinalizeEvent);

    nn::os::FinalizeMultiWait(&m_MultiWait);

    m_Descriptor.Finalize();

    GetEndPointFromAddress(UsbEndpointAddressMask_DirDeviceToHost).FreeUrbRing();
    GetEndPointFromAddress(UsbEndpointAddressMask_DirHostToDevice).FreeUrbRing();

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result DsProtocol::ClearDeviceData() NN_NOEXCEPT
{
    NN_USB_TRACE;
    return m_Descriptor.ClearDeviceData();
}


//////////////////////////////////////////////////////////////////////////////
Result DsProtocol::AddUsbStringDescriptor(UsbStringDescriptor *pUsbStringDescriptor, uint8_t *pIndex) NN_NOEXCEPT
{
    NN_USB_TRACE;
    return m_Descriptor.AddUsbStringDescriptor(pUsbStringDescriptor, pIndex);
}


//////////////////////////////////////////////////////////////////////////////
Result DsProtocol::DeleteUsbStringDescriptor(uint8_t index) NN_NOEXCEPT
{
    NN_USB_TRACE;
    return m_Descriptor.DeleteUsbStringDescriptor(index);
}


//////////////////////////////////////////////////////////////////////////////
Result DsProtocol::SetUsbDeviceDescriptor(UsbDeviceDescriptor *pUsbDeviceDescriptor, UsbDeviceSpeed usbDeviceSpeed) NN_NOEXCEPT
{
    NN_USB_TRACE;
    return m_Descriptor.SetUsbDeviceDescriptor(pUsbDeviceDescriptor, usbDeviceSpeed);
}


//////////////////////////////////////////////////////////////////////////////
Result DsProtocol::SetBinaryObjectStore(uint8_t *pData, int size) NN_NOEXCEPT
{
    NN_USB_TRACE;
    return m_Descriptor.SetBinaryObjectStore(pData, size);
}


//////////////////////////////////////////////////////////////////////////////
Result DsProtocol::AppendConfigurationData(uint8_t bInterfaceNumber, UsbDeviceSpeed usbDeviceSpeed, uint8_t *pData, int bytes) NN_NOEXCEPT
{
    NN_USB_TRACE;
    return m_Descriptor.AppendConfigurationData(bInterfaceNumber, usbDeviceSpeed, pData, bytes);
}


//////////////////////////////////////////////////////////////////////////////
Result DsProtocol::SetReportData(uint8_t bInterfaceNumber, uint8_t *pData, int bytes) NN_NOEXCEPT
{
    NN_USB_TRACE;
    return m_Descriptor.SetReportData(bInterfaceNumber, pData, bytes);
}


//////////////////////////////////////////////////////////////////////////////
Result DsProtocol::AttachDsController(DsController *pController) NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_ProtocolMutex);

    m_pController           = pController;
    m_pPhysicalController   = pController;

    if (m_DeviceIsEnabled)
    {
        AttachDevice();
    }

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result DsProtocol::DetachDsController() NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_ProtocolMutex);

    if (m_DeviceIsEnabled)
    {
        DetachDevice();
    }

    m_pPhysicalController   = nullptr;
    m_pController           = &s_DummyController;

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result DsProtocol::EnableDevice() NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_ProtocolMutex);

    m_DeviceIsEnabled = true;
    AttachDevice();

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result DsProtocol::DisableDevice() NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_ProtocolMutex);

    m_DeviceIsEnabled = false;
    DetachDevice();

    return ResultSuccess();
}


//----------------------------------------------------------------------------
// Interface
//----------------------------------------------------------------------------
Result DsProtocol::RegisterInterface(uint8_t bInterfaceNumber) NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_ProtocolMutex);

    return m_Descriptor.AddInterface(bInterfaceNumber);
}


//////////////////////////////////////////////////////////////////////////////
Result DsProtocol::UnRegisterInterface(uint8_t ifNum) NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_ProtocolMutex);

    return m_Descriptor.DelInterface(ifNum);
}


//////////////////////////////////////////////////////////////////////////////
Result DsProtocol::EnableInterface(uint8_t ifNum) NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_ProtocolMutex);

    m_IsInterfaceEnabled[ifNum] = true;

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result DsProtocol::DisableInterface(uint8_t ifNum) NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_ProtocolMutex);

    m_IsInterfaceEnabled[ifNum] = false;

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
bool DsProtocol::IsInterfaceEnabled(uint8_t ifNum) NN_NOEXCEPT
{
    NN_USB_TRACE;
    return (ifNum < DsLimitMaxInterfacesPerConfigurationCount) &&
           (m_IsInterfaceEnabled[ifNum]);
}


//----------------------------------------------------------------------------
// Endpoint
//----------------------------------------------------------------------------
Result DsProtocol::RegisterEndpoint(uint8_t epAddr, uint8_t ifNum) NN_NOEXCEPT
{
    NN_USB_TRACE;
    Result result = ResultSuccess();
    std::lock_guard<nn::os::Mutex> lock(m_ProtocolMutex);

    result = m_Descriptor.AddEndpoint(epAddr, ifNum);

    if (result.IsSuccess())
    {
        std::lock_guard<nn::os::Mutex> lock(m_EndPointMutex);

        if (GetEndPointFromAddress(epAddr).GetUrbRingPtr() != nullptr)
        {
            result = ResultResourceBusy();
        }

        if (result.IsSuccess())
        {
            if (GetEndPointFromAddress(epAddr).AllocUrbRing(m_pController, epAddr) == nullptr)
            {
                result = ResultMemAllocFailure();
            }
        }

        if (!result.IsSuccess())
        {
            m_Descriptor.DelEndpoint(epAddr);
            NN_USB_LOG_INFO("%s Fail! epAddr 0x%02x\n", __FUNCTION__, epAddr);
        }
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result DsProtocol::UnRegisterEndpoint(uint8_t epAddr) NN_NOEXCEPT
{
    NN_USB_TRACE;
    Result result = ResultSuccess();

    std::lock_guard<nn::os::Mutex> protocoLock(m_ProtocolMutex);
    std::lock_guard<nn::os::Mutex> endpointLock(m_EndPointMutex);

    NN_USB_RETURN_UPON_ERROR(
        m_Descriptor.DelEndpoint(epAddr)
    );

    m_DsEndPoint[detail::GetEndpointIndex(epAddr)].FreeUrbRing();

    return result;
}


//----------------------------------------------------------------------------
// USB State
//----------------------------------------------------------------------------
Result DsProtocol::SetState(UsbState state) NN_NOEXCEPT
{
    NN_USB_TRACE;

    // State change happens in two scenarios:
    //   - interrupt handler: e.g. attach / detach event
    //   - protocol operation: e.g. SetAddress(), SetConfiguration()
    std::lock_guard<nn::os::Mutex> lock(m_StateChangeMutex);

    Result result = ResultSuccess();

    if (state == m_State)
    {
        return result;
    }

    switch (state)
    {
    case UsbState_Detached:

        NN_USB_LOG_INFO("UsbState_Detached\n");
        // disable all the endpoints
        DisableAllEndpoints();
        break;

    case UsbState_Attached:

        NN_USB_LOG_INFO("UsbState_Attached\n");
        // do nothing
        break;

    case UsbState_Powered:

        NN_USB_LOG_INFO("UsbState_Powered\n");
        // do nothing
        break;

    case UsbState_Default:

        NN_USB_LOG_INFO("UsbState_Default\n");
        {
            std::lock_guard<nn::os::Mutex> lock(m_EndPointMutex);

            // enable Ep0 Out/In
            UsbEndpointDescriptor usbEndpointDescriptor[2] =
            {
                {
                    UsbDescriptorSize_Endpoint,
                    UsbDescriptorType_Endpoint,
                    0x00,
                    UsbEndpointAttributeMask_XferTypeControl,
                    64,
                    0
                },
                {
                    UsbDescriptorSize_Endpoint,
                    UsbDescriptorType_Endpoint,
                    0x80,
                    UsbEndpointAttributeMask_XferTypeControl,
                    64,
                    0
                }
            };

            m_pController->EnableEndpoint(&usbEndpointDescriptor[0], NULL);
            m_pController->EnableEndpoint(&usbEndpointDescriptor[1], NULL);

            GetEndPointFromAddress(UsbEndpointAddressMask_DirDeviceToHost).Enable();
            GetEndPointFromAddress(UsbEndpointAddressMask_DirHostToDevice).Enable();
        }

        break;

    case UsbState_Address:
        NN_USB_LOG_INFO("UsbState_Address\n");

        // make sure endpoints other than Ep0 are disabled
        EnableDisableEndpoints(false, m_pController->GetConnectSpeed());

        break;

    case UsbState_Configured:

        NN_USB_LOG_INFO("UsbState_Configured\n");
        // bringup registerred endpoints
        // Per X1 TRM 23.16.1.40 USB2_CONTROLLER_USB2D_ENDPTCTRLn_0:
        //   "An Endpoint should be enabled only after it has been configured"
        EnableDisableEndpoints(true, m_pController->GetConnectSpeed());

        break;

    case UsbState_Suspended:
        NN_USB_LOG_INFO("UsbState_Suspended\n");
        // do nothing, for now
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

    m_State = state;

    // inform every client listening on state change event
    for (nn::os::SystemEvent *pEvent : m_StateChangeEventList)
    {
        pEvent->Signal();
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
UsbState DsProtocol::GetState(nn::os::SystemEvent *pEvent) NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_StateChangeMutex);

    if (pEvent != nullptr)
    {
        pEvent->Clear();
    }
    NN_USB_TRACE_END;
    return m_State;
}


//////////////////////////////////////////////////////////////////////////////
void DsProtocol::BindStateChangeEvent(nn::os::SystemEvent *pEvent) NN_NOEXCEPT
{
    NN_USB_TRACE;

    // reuse the mutex to protect the list
    {
        std::lock_guard<nn::os::Mutex> lock(m_StateChangeMutex);
        m_StateChangeEventList.push_back(pEvent);
        m_DsDeviceRef++;
    }

    if (m_DsDeviceRef == 1)
    {
        m_DsChargingDevice.Finalize();
    }
}


//////////////////////////////////////////////////////////////////////////////
void DsProtocol::UnbindStateChangeEvent(nn::os::SystemEvent *pEvent) NN_NOEXCEPT
{
    NN_USB_TRACE;

    // reuse the mutex to protect the list
    {
        std::lock_guard<nn::os::Mutex> lock(m_StateChangeMutex);
        m_StateChangeEventList.remove(pEvent);
        m_DsDeviceRef--;
    }

    if (m_DsDeviceRef == 0)
    {
        m_DsChargingDevice.Initialize(this);
    }
}

//----------------------------------------------------------------------------
// SETUP Packet Handling
//----------------------------------------------------------------------------
Result DsProtocol::SetSetupPacket(uint8_t         epAddr,
                                  UsbCtrlRequest *pSetup) NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_SetupMutex);

    // TODO: support SETUP on other than default pipe
    if (epAddr != 0x00)
    {
        return ResultNotImplemented();
    }

    memcpy(&m_SetupPacket, pSetup, sizeof(UsbCtrlRequest));

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result DsProtocol::GetSetupPacket(uint8_t epAddr, char *buffer, size_t size,
                                  nn::os::SystemEvent *pEvent) NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_SetupMutex);

    // TODO: support SETUP on other than default pipe
    if (epAddr != 0x00)
    {
        return ResultNotImplemented();
    }

    memcpy(buffer, &m_SetupPacket, NN_USB_MIN(sizeof(UsbCtrlRequest), size));

    if (pEvent != nullptr)
    {
        pEvent->Clear();
    }

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result DsProtocol::BindSetupEvent(uint8_t epAddr, uint8_t ifNum,
                                  nn::os::SystemEvent *pEvent) NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_SetupMutex);

    // TODO: support SETUP on other than default pipe
    if (epAddr != 0x00)
    {
        return ResultNotImplemented();
    }

    if (m_pSetupEvent[ifNum] != nullptr)
    {
        return ResultResourceBusy();
    }
    else
    {
        m_pSetupEvent[ifNum] = pEvent;
        return ResultSuccess();
    }
}


//////////////////////////////////////////////////////////////////////////////
Result DsProtocol::UnbindSetupEvent(uint8_t epAddr, uint8_t ifNum) NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_SetupMutex);

    // TODO: support SETUP on other than default pipe
    if (epAddr != 0x00)
    {
        return ResultNotImplemented();
    }

    m_pSetupEvent[ifNum] = nullptr;

    return ResultSuccess();
}


// this is supposed to be called in interrupt context, so let's
// return as soon as we can
Result DsProtocol::DoCtrlRequest() NN_NOEXCEPT
{
    NN_USB_TRACE;
    nn::os::SignalEvent(&m_CtrlRequestEvent);

    return ResultSuccess();
}

//----------------------------------------------------------------------------
// I/O
//----------------------------------------------------------------------------
Result DsProtocol::PostBufferAsync(uint8_t        epAddr,
                                   ProcessHandle  processHandle,
                                   uint64_t       procVa,
                                   uint32_t       bytes,
                                   uint32_t      *pUrbId) NN_NOEXCEPT
{
    NN_USB_TRACE;
    uint32_t urbId;
    Result result = Enqueue(&urbId, epAddr, processHandle, procVa, bytes);

    if(result.IsSuccess())
    {
        *pUrbId = urbId;
    }

    //NN_USB_TRACE_END;
    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result DsProtocol::PostBuffer(uint8_t        epAddr,
                              ProcessHandle  processHandle,
                              uint64_t       procVa,
                              uint32_t       bytes,
                              uint32_t      *pBytesTransferred) NN_NOEXCEPT
{
    NN_USB_TRACE;
    NN_ABORT_UNLESS_ALIGNED(procVa, HwLimitDmaBufferAlignmentSize);

    uint32_t urbId;
    Result result = EnqueueAndBlock(&urbId, epAddr, processHandle, procVa, bytes);

    // query for status
    UrbReport report;
    GetUrbReport(epAddr, &report);

    if (report.count == 1 && report.report[0].id == urbId)
    {
        switch (report.report[0].status)
        {
        case UrbStatus_Cancelled:
            result = ResultInterrupted();
            break;

        case UrbStatus_Failed:
            result = ResultTransactionError();
            break;

        case UrbStatus_Finished:
            result = ResultSuccess();
            break;

        default:
            result = ResultInternalStateError();
            break;
        }

        if (pBytesTransferred != nullptr)
        {
            *pBytesTransferred = report.report[0].transferredSize;
        }
    }
    else
    {
        result = ResultInternalStateError();
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result DsProtocol::PostBuffer(uint8_t   epAddr,
                              void     *buffer,
                              uint32_t  bytes,
                              uint32_t *pBytesTransferred) NN_NOEXCEPT
{
    NN_USB_TRACE;
    Result result;

    if (bytes)
    {
        if (epAddr & UsbEndpointAddressMask_DirDeviceToHost)  // device -> host
        {
            FlushDataCache(buffer, bytes);
        }
        else
        {
            InvalidateDataCache(buffer, bytes);
        }
    }

    result = PostBuffer(epAddr,
                        GetCurrentProcessHandle(),
                        reinterpret_cast<uint64_t>(buffer),
                        bytes,
                        pBytesTransferred);

    if (bytes)
    {
        if ((epAddr & UsbEndpointAddressMask_DirDeviceToHost) == 0) // device <- host
        {
            InvalidateDataCache(buffer, bytes);
        }
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
nn::os::SystemEventType* DsProtocol::GetCompletionEvent(uint8_t epAddr) NN_NOEXCEPT
{
    NN_USB_TRACE;
    nn::os::SystemEventType *pEvent = NULL;

    std::lock_guard<nn::os::Mutex> lock(m_EndPointMutex);

    if(GetEndPointFromAddress(epAddr).GetUrbRingPtr() != nullptr)
    {
        pEvent = GetEndPointFromAddress(epAddr).GetUrbRingPtr()->GetCompletionEvent();
    }

    return pEvent;
}


//////////////////////////////////////////////////////////////////////////////
Result DsProtocol::GetUrbReport(uint8_t epAddr, UrbReport *pReport) NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_EndPointMutex);
    DsUrbRing *pUrbRing  = m_DsEndPoint[detail::GetEndpointIndex(epAddr)].GetUrbRingPtr();

    if(pUrbRing == nullptr)
    {
        return ResultInvalidEndpoint();
    }

    return pUrbRing->GetReport(pReport);
}


//////////////////////////////////////////////////////////////////////////////
Result DsProtocol::Stall(uint8_t epAddr) NN_NOEXCEPT
{
    NN_USB_TRACE;
    return m_pController->StallEndpoint(epAddr);
}


//////////////////////////////////////////////////////////////////////////////
Result DsProtocol::Cancel(uint8_t epAddr) NN_NOEXCEPT
{
    NN_USB_TRACE;

    std::lock_guard<nn::os::Mutex> lock(m_EndPointMutex);

    Result result = m_pController->FlushEndpoint(epAddr);

    if(result.IsSuccess())
    {
        if(!GetEndPointFromAddress(epAddr).GetUrbRingPtr())
        {
            return ResultInvalidEndpoint();
        }

        result = GetEndPointFromAddress(epAddr).GetUrbRingPtr()->Flush();
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result DsProtocol::SetZlt(uint8_t epAddr, bool zlt) NN_NOEXCEPT
{
    NN_USB_TRACE;
    return m_pController->SetZlt(epAddr, zlt);
}


//////////////////////////////////////////////////////////////////////////////
void DsProtocol::CompleteBuffer(uint8_t epAddr, Result result, uint32_t bytesRemaining, bool lockMutex) NN_NOEXCEPT
{
    NN_USB_TRACE;

    std::lock_guard<nn::os::Mutex> lock(m_EndPointMutex);

    if(GetEndPointFromAddress(epAddr).GetUrbRingPtr() != nullptr)
    {
        GetEndPointFromAddress(epAddr).GetUrbRingPtr()->Dequeue(result, bytesRemaining, lockMutex);
    }

    NN_USB_TRACE_END;
}


//----------------------------------------------------------------------------
//  Private Methods
//----------------------------------------------------------------------------
void DsProtocol::AttachDevice() NN_NOEXCEPT
{
    NN_USB_TRACE;

    // caller holds mutex lock
    SetControllerAllEndPoints();
    m_pController->Attach();
}


//////////////////////////////////////////////////////////////////////////////
void DsProtocol::DetachDevice() NN_NOEXCEPT
{
    NN_USB_TRACE;

    // caller holds mutex lock
    m_State = UsbState_Default;
    SetState(UsbState_Detached);
    m_pController->Detach();
}


//////////////////////////////////////////////////////////////////////////////
void DsProtocol::CtrlPipeLoop() NN_NOEXCEPT
{
    NN_USB_TRACE;
    while (true)
    {
        nn::os::MultiWaitHolderType* holder = nn::os::WaitAny(&m_MultiWait);

        // new control request arrival
        if (holder == &m_CtrlRequestHolder)
        {
            nn::os::ClearEvent(&m_CtrlRequestEvent);
            HandleCtrlRequest();
        }

        if (holder == &m_CtrlIOHolder)
        {
            nn::os::ClearEvent(&m_CtrlIOEvent);
            // TODO: handle ctrl-in/-out request
        }

        if (holder == &m_CtrlFinalizeHolder)
        {
            nn::os::ClearEvent(&m_CtrlFinalizeEvent);
            break;
        }
    }
}


//////////////////////////////////////////////////////////////////////////////
void DsProtocol::HandleCtrlRequest() NN_NOEXCEPT
{
    NN_USB_TRACE;
    UsbCtrlRequest setup;

    {
        // make a local copy, so we won't block ISR
        std::lock_guard<nn::os::Mutex> lock(m_SetupMutex);
        memcpy(&setup, &m_SetupPacket, sizeof(UsbCtrlRequest));
    }

    for (unsigned int i = 0; i < NN_USB_ARRAY_SIZE(CtrlHandlerTable); i++)
    {
        if ((setup.bmRequestType & CtrlHandlerTable[i].tmask) == CtrlHandlerTable[i].type &&
            (setup.bRequest      & CtrlHandlerTable[i].rmask) == CtrlHandlerTable[i].request)
        {
            (this->*(DsProtocol::CtrlHandlerTable[i].handler))(&setup);
            NN_USB_TRACE_END;
            return;
        }
    }
/*
    NN_USB_LOG_INFO("#### unhandled request bmRequestType 0x%x bRequest 0x%x wValue %d wIndex %d wLength %d\n",
        setup.bmRequestType,
        setup.bRequest,
        setup.wValue,
        setup.wIndex,
        setup.wLength);
 */
    // unsupported request
    m_pController->StallEndpoint(0x80);
}


//////////////////////////////////////////////////////////////////////////////
void DsProtocol::GetConfiguration(UsbCtrlRequest *pCtrl) NN_NOEXCEPT
{
    NN_UNUSED(pCtrl);

    NN_USB_TRACE;
    m_TxBuffer[0] = m_Configuration;

    // data stage
    NN_USB_VOID_RETURN_UPON_ERROR(
        PostBuffer(UsbEndpointAddressMask_DirDeviceToHost, m_TxBuffer, 1, NULL)
    );

    GetStatusFromHost();
}


//////////////////////////////////////////////////////////////////////////////
void DsProtocol::GetDescriptor(UsbCtrlRequest *pCtrl) NN_NOEXCEPT
{
    NN_USB_TRACE;

    int      bytes   = 0;
    uint8_t  type    = pCtrl->wValue >> 8;
    uint8_t  index   = pCtrl->wValue & 0xff;
    uint16_t length  = pCtrl->wLength;

    switch (type)
    {
    case UsbDescriptorType_Device:
        //NN_USB_LOG_INFO("### UsbDescriptorType_Device post\n");
        bytes = m_Descriptor.GetDeviceDescriptor(m_TxBuffer, length, m_pController->GetConnectSpeed());
        break;

    case UsbDescriptorType_Config:
        //NN_USB_LOG_INFO("### UsbDescriptorType_Config post\n");
        bytes = m_Descriptor.GetConfigurationDescriptor(m_TxBuffer, index, length);
        break;

    case UsbDescriptorType_String:

        //NN_USB_LOG_INFO("### UsbDescriptorType_String post\n");
        bytes = m_Descriptor.GetStringDescriptor(m_TxBuffer, index, length);
        break;

    case UsbDescriptorType_Bos:
        //NN_USB_LOG_INFO("### UsbDescriptorType_Bos post\n");
        bytes = m_Descriptor.GetBinaryObjectStore(m_TxBuffer, length);
        break;

    default:
//        NN_USB_LOG_INFO("unknown supported get descriptor request for type %d\n", type);
        bytes = -1;  // so we can stall
        break;
    }

    // If m_pController is changed, it will go to a version that supports this call.
    if (bytes < 0)
    {
        //NN_USB_LOG_INFO("### StallEndpoint \n");
        m_pController->StallEndpoint(UsbEndpointAddressMask_DirDeviceToHost);
    }
    else
    {
        // data stage
        NN_USB_VOID_RETURN_UPON_ERROR(PostBuffer(UsbEndpointAddressMask_DirDeviceToHost, m_TxBuffer, bytes, NULL));
        GetStatusFromHost();
    }
}


//////////////////////////////////////////////////////////////////////////////
void DsProtocol::GetStatus(UsbCtrlRequest *pCtrl) NN_NOEXCEPT
{
    NN_UNUSED(pCtrl);

    NN_USB_TRACE;
    m_TxBuffer[0] = 0x01;    // self powered
    m_TxBuffer[1] = 0;

    //NN_USB_LOG_INFO("### GET_STATUS\n");
    // data stage
    NN_USB_VOID_RETURN_UPON_ERROR(
        PostBuffer(UsbEndpointAddressMask_DirDeviceToHost, m_TxBuffer, 2, NULL)
    );

    GetStatusFromHost();
}


//////////////////////////////////////////////////////////////////////////////
void DsProtocol::SetAddress(UsbCtrlRequest *pCtrl) NN_NOEXCEPT
{
    NN_USB_TRACE;

    m_pController->Ctrl(pCtrl);
    SetState(UsbState_Address);

    //NN_USB_LOG_INFO("### SET_ADDRESS\n");

    SetStatusToHost();

    // Update the Configuration depending on connect speed
    m_Descriptor.UpdateConfigurationDescriptor(m_pController->GetConnectSpeed());
}


//////////////////////////////////////////////////////////////////////////////
void DsProtocol::SetConfiguration(UsbCtrlRequest *pCtrl) NN_NOEXCEPT
{
    NN_USB_TRACE;
    switch (pCtrl->wValue & 0xff)
    {
    case 0: // moves to Address state

        m_Configuration = 0;
        m_pController->Ctrl(pCtrl);
        SetState(UsbState_Address);

        //NN_USB_LOG_INFO("### SetConfiguration 0\n");

        SetStatusToHost();

        break;

    case 1: // move to Configured state

        m_Configuration = 1;
        SetState(UsbState_Configured);
        m_pController->Ctrl(pCtrl);

        //NN_USB_LOG_INFO("### SetConfiguration 1\n");

        SetStatusToHost();

        break;

    default: // invalid config value

        NN_USB_LOG_INFO("### SetConfiguration %d unsupported, stall\n", pCtrl->wValue & 0xff);
        m_pController->StallEndpoint(UsbEndpointAddressMask_DirDeviceToHost);

        break;
    }
}


//////////////////////////////////////////////////////////////////////////////
void DsProtocol::SetSel(UsbCtrlRequest *pCtrl) NN_NOEXCEPT
{
    NN_USB_TRACE;
    NN_UNUSED(pCtrl);

    //NN_USB_LOG_INFO("### SetSel post\n");

    m_pController->Ctrl(pCtrl);

    // data stage
    NN_USB_VOID_RETURN_UPON_ERROR(PostBuffer(UsbEndpointAddressMask_DirHostToDevice, m_RxBuffer, pCtrl->wLength, NULL));
}


//////////////////////////////////////////////////////////////////////////////
void DsProtocol::SetIsocDelay(UsbCtrlRequest *pCtrl) NN_NOEXCEPT
{
    NN_USB_TRACE;
    NN_UNUSED(pCtrl);

    //NN_USB_LOG_INFO("### SetIsocDelay post\n");

    m_pController->Ctrl(pCtrl);

    SetStatusToHost();
}


//////////////////////////////////////////////////////////////////////////////
void DsProtocol::GetReportDescriptor(UsbCtrlRequest *pCtrl) NN_NOEXCEPT
{
    NN_USB_TRACE;

    int bytes = m_Descriptor.GetReportData(pCtrl->wIndex, m_TxBuffer, pCtrl->wLength);

    if (bytes <= pCtrl->wLength)
    {
        //NN_USB_LOG_INFO("### UsbDescriptorType_Report post\n");

        // data stage
        PostBuffer(UsbEndpointAddressMask_DirDeviceToHost, m_TxBuffer, bytes, NULL);
        GetStatusFromHost();
    }
    else
    {
//        NN_USB_LOG_INFO("### GetReportDescriptor StallEndpoint \n");
        m_pController->StallEndpoint(UsbEndpointAddressMask_DirDeviceToHost);
    }

    NN_USB_TRACE_END;
}


//////////////////////////////////////////////////////////////////////////////
void DsProtocol::SetFeature(UsbCtrlRequest *pCtrl) NN_NOEXCEPT
{
    NN_USB_TRACE;

    //NN_USB_LOG_INFO("### SET_FEATURE\n");

    m_pController->Ctrl(pCtrl);

    SetStatusToHost();
}


//////////////////////////////////////////////////////////////////////////////
void DsProtocol::ClearFeature(UsbCtrlRequest *pCtrl) NN_NOEXCEPT
{
    NN_USB_TRACE;
    NN_UNUSED(pCtrl);

    //NN_USB_LOG_INFO("### CLEAR_FEATURE\n");

    m_pController->Ctrl(pCtrl);

    SetStatusToHost();
}


//////////////////////////////////////////////////////////////////////////////
void DsProtocol::SetIdle(UsbCtrlRequest *pCtrl) NN_NOEXCEPT
{
    NN_USB_TRACE;
    NN_UNUSED(pCtrl);

    //NN_USB_LOG_INFO("### SetIdle post\n");

    // This is here for the default charging device to handle HID SET_IDLE
    // command

    SetStatusToHost();
}


//////////////////////////////////////////////////////////////////////////////
void DsProtocol::GetStatusFromHost() NN_NOEXCEPT
{
    NN_USB_TRACE;

    // status stage to ack request
    NN_USB_VOID_RETURN_UPON_ERROR(PostBuffer(UsbEndpointAddressMask_DirHostToDevice, m_RxBuffer, 0, NULL));
}


//////////////////////////////////////////////////////////////////////////////
void DsProtocol::SetStatusToHost() NN_NOEXCEPT
{
    NN_USB_TRACE;

    // status stage to ack request
    NN_USB_VOID_RETURN_UPON_ERROR(PostBuffer(UsbEndpointAddressMask_DirDeviceToHost, m_TxBuffer, 0, NULL));
}


//////////////////////////////////////////////////////////////////////////////
void DsProtocol::EnableDisableEndpoints(bool enable, UsbDeviceSpeed usbDeviceSpeed) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_EndPointMutex);

    NN_USB_TRACE;

    // For each interface present, enable endpoints based on alternate setting
    for (uint8_t bInterfaceNumber = 0; bInterfaceNumber < DsLimitMaxInterfacesPerConfigurationCount; bInterfaceNumber++)
    {
        int     bytes;
        uint8_t interfaceData[MaxInterfaceDataSize];
        Result  result = m_Descriptor.GetConfigurationData(bInterfaceNumber, usbDeviceSpeed, interfaceData, &bytes);

        if (result.IsSuccess())
        {
            uint8_t bAlternateSetting = 0;
            uint8_t *pData = interfaceData;

            // Find alternate setting
            while (bytes > 0)
            {
                UsbDescriptorHeader *pUsbDescriptorHeader = reinterpret_cast<UsbDescriptorHeader*>(pData);

                if (pUsbDescriptorHeader->bDescriptorType == UsbDescriptorType_Interface)
                {
                    UsbInterfaceDescriptor *pUsbInterfaceDescriptor = reinterpret_cast<UsbInterfaceDescriptor*>(pData);

                    bAlternateSetting = pUsbInterfaceDescriptor->bAlternateSetting;
                }

                // TODO: add alternate setting param to DsProtocol, support alt setting
                // for now just use alt setting 0
                if (bAlternateSetting == 0)
                {
                    // Find endpoints
                    pUsbDescriptorHeader = reinterpret_cast<UsbDescriptorHeader*>(pData);

                    if (pUsbDescriptorHeader->bDescriptorType == UsbDescriptorType_Endpoint)
                    {
                        UsbEndpointDescriptor *pUsbEndpointDescriptor = reinterpret_cast<UsbEndpointDescriptor*>(pData);

                        if (enable)
                        {
                            // For USB 3.0 also use attached companion descriptor
                            if(usbDeviceSpeed == UsbDeviceSpeed_Super)
                            {
                                if (bytes >= static_cast<int>(pUsbEndpointDescriptor->bLength + sizeof(UsbEndpointCompanionDescriptor)))
                                {
                                    UsbEndpointCompanionDescriptor *pUsbEndpointCompanionDescriptor = reinterpret_cast<UsbEndpointCompanionDescriptor*>(pData + sizeof(UsbEndpointDescriptor));

                                    if (pUsbEndpointCompanionDescriptor->bDescriptorType == UsbDescriptorType_EndpointCompanion)
                                    {
                                        m_pController->EnableEndpoint(pUsbEndpointDescriptor, pUsbEndpointCompanionDescriptor);
                                        GetEndPointFromAddress(pUsbEndpointDescriptor->bEndpointAddress).Enable();
                                    }
                                }
                            }
                            else
                            {
                                m_pController->EnableEndpoint(pUsbEndpointDescriptor, NULL);
                                GetEndPointFromAddress(pUsbEndpointDescriptor->bEndpointAddress).Enable();
                            }
                        }
                        else // disable
                        {
                            if (GetEndPointFromAddress(pUsbEndpointDescriptor->bEndpointAddress).IsEnabled())
                            {
                                GetEndPointFromAddress(pUsbEndpointDescriptor->bEndpointAddress).Disable();
                                m_pController->DisableEndpoint(pUsbEndpointDescriptor->bEndpointAddress);
                            }
                        }
                    }
                }

                pData += pUsbDescriptorHeader->bLength;
                bytes -= pUsbDescriptorHeader->bLength;
            }
        }
    }
}


//////////////////////////////////////////////////////////////////////////////
void DsProtocol::DisableAllEndpoints() NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_EndPointMutex);

    for (uint8_t i = 0; i < 16; i++)
    {
        if(m_DsEndPoint[i].IsEnabled())
        {
            m_pController->DisableEndpoint(i);
            m_DsEndPoint[i].Disable();
        }

        if(m_DsEndPoint[i + 16].IsEnabled())
        {
            m_pController->DisableEndpoint(i | 0x80);
            m_DsEndPoint[i + 16].Disable();
        }
    }
}


//////////////////////////////////////////////////////////////////////////////
void DsProtocol::SetControllerAllEndPoints() NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_EndPointMutex);

    for(int i=0; i<UsbLimitMaxEndpointsCount; i++)
    {
        if(m_DsEndPoint[i].GetUrbRingPtr())
        {
            m_DsEndPoint[i].GetUrbRingPtr()->SetController(m_pController);
        }
    }
}


//////////////////////////////////////////////////////////////////////////////
Result DsProtocol::Enqueue(uint32_t *urbId, uint8_t epAddr, ProcessHandle processHandle, uint64_t procVa, uint32_t bytes) NN_NOEXCEPT
{
    NN_USB_TRACE;
    std::lock_guard<nn::os::Mutex> lock(m_EndPointMutex);

    if(!GetEndPointFromAddress(epAddr).GetUrbRingPtr())
    {
        return ResultInvalidEndpoint();
    }

    if(!GetEndPointFromAddress(epAddr).IsEnabled())
    {
//        NN_USB_LOG_INFO("### Endpoint 0x%x Not Enabled -bail! \n", epAddr);
        return ResultInvalidEndpoint();
    }

    if (procVa != 0 && bytes > 0)
    {
        if (epAddr & UsbEndpointAddressMask_DirDeviceToHost)
        {
            nn::dd::FlushProcessDataCache(processHandle, procVa, bytes);
        }
        else
        {
            nn::dd::InvalidateProcessDataCache(processHandle, procVa, bytes);
        }
    }

    Result result =  GetEndPointFromAddress(epAddr).GetUrbRingPtr()->Enqueue(urbId, processHandle, procVa, bytes);

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result DsProtocol::EnqueueAndBlock(uint32_t *urbId, uint8_t epAddr, ProcessHandle processHandle, uint64_t procVa, uint32_t bytes) NN_NOEXCEPT
{
    NN_USB_TRACE;

    Result result  = Enqueue(urbId, epAddr, processHandle, procVa, bytes);

    if(result.IsSuccess())
    {
        GetEndPointFromAddress(epAddr).GetUrbRingPtr()->BlockForCompletion();
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
void DsProtocol::LockEndPointMutex() NN_NOEXCEPT
{
    m_EndPointMutex.lock();
}


//////////////////////////////////////////////////////////////////////////////
void DsProtocol::UnlockEndPointMutex() NN_NOEXCEPT
{
    m_EndPointMutex.unlock();
}


// -------------------------------------------------------------------------
// -------------------------------------------------------------------------
// DsProtocolEndPoint Class
// -------------------------------------------------------------------------
// -------------------------------------------------------------------------

#undef NN_USB_TRACE_CLASS_NAME
#define NN_USB_TRACE_CLASS_NAME "DsProtocolEndPoint"

// Note: All locking for this class is handled by its parent.

DsProtocolEndPoint &DsProtocol::GetEndPointFromAddress(uint8_t epAddr) NN_NOEXCEPT
{
    int epIndex = detail::GetEndpointIndex(epAddr);
    return m_DsEndPoint[epIndex];
}
DsProtocolEndPoint::DsProtocolEndPoint() NN_NOEXCEPT
{
    m_IsEnabled = false;
    m_UrbRingPtr = nullptr;
}

DsProtocolEndPoint::~DsProtocolEndPoint() NN_NOEXCEPT
{
    // Add assertion for m_UrbRingPtr exists and such
}

void DsProtocolEndPoint::Enable() NN_NOEXCEPT
{
    NN_USB_TRACE;
    m_IsEnabled = true;
}

void DsProtocolEndPoint::Disable() NN_NOEXCEPT
{
    NN_USB_TRACE;

    m_IsEnabled = false;

    if(m_UrbRingPtr)
    {
        m_UrbRingPtr->Flush();
    }
}

bool DsProtocolEndPoint::IsEnabled() NN_NOEXCEPT
{
    return m_IsEnabled;
}

DsUrbRing*DsProtocolEndPoint::GetUrbRingPtr() NN_NOEXCEPT
{
    return m_UrbRingPtr;
}

DsUrbRing* DsProtocolEndPoint::AllocUrbRing(DsController *pController, uint8_t address) NN_NOEXCEPT
{
    NN_USB_TRACE;

    m_UrbRingPtr = new DsUrbRing(pController, address);
    return m_UrbRingPtr;
}

void DsProtocolEndPoint::FreeUrbRing() NN_NOEXCEPT
{
    NN_USB_TRACE;

    delete m_UrbRingPtr;
    m_UrbRingPtr = nullptr;
}

} // end of namespace ds
} // end of namespace usb
} // end of namespace nn
