﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//#define NN_USB_ENABLE_TRACE 1
#include <nn/os/os_MemoryAttribute.h>
#include <nn/pcv/pcv.h>
#include <cstring>
#include <nn/os/os_SdkThreadCommon.h>
#include <nn/nn_SystemThreadDefinition.h>

#include "usb_DsController30-soc.tegra.h"
#include "../detail/usb_ComplexTegra21x.h"

using namespace nn::dd;
using namespace nn::usb::detail;

#undef  NN_USB_TRACE_CLASS_NAME
#define NN_USB_TRACE_CLASS_NAME "DsControllerTegra30"

namespace {

const uint32_t PORTSC_MASK = 0xFF15FFFF;


} // anonymous namespace

namespace nn {
namespace usb {
namespace ds {

//////////////////////////////////////////////////////////////////////////////
//  public functions
//////////////////////////////////////////////////////////////////////////////
Result DsControllerTegra30::Initialize(UsbComplex *pComplex,detail::UsbController::Config* pConfig) NN_NOEXCEPT
{
    NN_USB_TRACE;
    Result result = DsController::Initialize(pComplex, pConfig);

    if (result.IsSuccess())
    {
        m_pComplex = static_cast<detail::UsbComplexTegra21x*>(pComplex);
        m_pPlatform = m_pComplex->GetPlatform();

        for(int i=0; i<UsbXusbDevNumEndpointContext; i++)
        {
            m_EndpointArray[i].pTransferRingBuffer = nullptr;
        }
    }

    nn::os::InitializeInterruptEvent(&m_XusbDevIrqEvent, Usb3DevInterruptName,nn::os::EventClearMode_ManualClear);
    nn::os::InitializeMultiWaitHolder(&m_XusbDevIrqHolder, &m_XusbDevIrqEvent);

    nn::os::InitializeTimerEvent(&m_PortResetWarEvent, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeMultiWaitHolder(&m_PortResetWarEventHolder, &m_PortResetWarEvent);

    nn::os::InitializeTimerEvent(&m_PlcResetWarEvent, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeMultiWaitHolder(&m_PlcResetWarEventHolder, &m_PlcResetWarEvent);

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result DsControllerTegra30::Finalize() NN_NOEXCEPT
{
    NN_USB_TRACE;

    nn::os::FinalizeMultiWaitHolder(&m_PlcResetWarEventHolder);
    nn::os::FinalizeTimerEvent(&m_PlcResetWarEvent);

    nn::os::FinalizeMultiWaitHolder(&m_PortResetWarEventHolder);
    nn::os::FinalizeTimerEvent(&m_PortResetWarEvent);

    nn::os::FinalizeMultiWaitHolder(&m_XusbDevIrqHolder);
    nn::os::FinalizeInterruptEvent(&m_XusbDevIrqEvent);

    Result result = DsController::Finalize();

    return result;
}


//////////////////////////////////////////////////////////////////////////////
// Pd calls Enable to enable driver
//////////////////////////////////////////////////////////////////////////////
Result DsControllerTegra30::Enable() NN_NOEXCEPT
{
    NN_USB_TRACE;
    Result result;

    if (m_ControllerRefCount > 0)
    {
        m_ControllerRefCount++;
        result = ResultSuccess();
    }
    else
    {
        result = InitializeXusbDevice();
        InitializeDriverMemory(this);
        InitializeEndpoint(&m_EndpointArray[0]);

        if(result.IsSuccess())
        {
            result = DsController::Enable();

            if(result.IsSuccess())
            {
                protocol.AttachDsController(this);
                m_ControllerRefCount++;
            }
        }
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
// Pd calls Disable to disable driver
//////////////////////////////////////////////////////////////////////////////
Result DsControllerTegra30::Disable() NN_NOEXCEPT
{
    NN_USB_TRACE;
    Result result;

    if (m_ControllerRefCount > 1)
    {
        m_ControllerRefCount--;
        result = ResultSuccess();
    }
    else
    {
        protocol.DetachDsController();

        result = DsController::Disable();

        if(result.IsSuccess())
        {
            result = FinalizeXusbDevice();

            if (result.IsSuccess())
            {
                m_ControllerRefCount--;
            }
        }
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
// DsProtocol calls Attach to enable controller
//////////////////////////////////////////////////////////////////////////////
Result DsControllerTegra30::Attach() NN_NOEXCEPT
{
    NN_USB_TRACE;

    protocol.SetState(UsbState_Default);

    SetInterruptModeration(100);

    EnableRxDetect();
    XusbDeviceBaseRegisters.SetField32(XusbDevControlOffset, XusbDevControlLseMask | XusbDevControlIeMask);
    XusbDeviceBaseRegisters.SetField32(XusbDevCtrlPortHaltOffset,XusbDevCtrlPortHaltStChgIntrEnabledMask);

    XusbDeviceBaseRegisters.SetField32(XusbDevControlOffset, XusbDevControlEnableMask);

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
// DsProtocol calls Detach to disable controller
//////////////////////////////////////////////////////////////////////////////
Result DsControllerTegra30::Detach() NN_NOEXCEPT
{
    NN_USB_TRACE;

    Reset();

    // Make sure deferred workarounds have been completed
    while ((m_WaitForSecPrcRefCount > 0) || (m_WaitForCscRefCount > 0))
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
    }

    XusbDeviceBaseRegisters.ClearField32(XusbDevControlOffset, XusbDevControlIeMask | XusbDevControlEnableMask);

    m_DeviceState       = UsbState_Default;
    m_ConnectedToHost   = false;

    // SS connction needs a sleep here
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
// DsProtocol calls this for Ctrl transactions, HW has to handle some
// device requests
//////////////////////////////////////////////////////////////////////////////
Result DsControllerTegra30::Ctrl(UsbCtrlRequest *pCtrl) NN_NOEXCEPT
{
    NN_USB_TRACE;

    if (pCtrl->bmRequestType == 0) // Handle only host to device requests
    {
        switch (pCtrl->bRequest)
        {
        case UsbCtrlXferReq_SetAddress:

            {
                m_DevAddr = pCtrl->wValue;

                uint32_t reg = XusbDeviceBaseRegisters.Read32(XusbDevControlOffset);
                reg &= ~NN_USB_DEFINE_DEV_ADDR(-1);
                reg |= NN_USB_DEFINE_DEV_ADDR(m_DevAddr);
                XusbDeviceBaseRegisters.Write32(XusbDevControlOffset, reg);

                EpContextWriteDevAddr(m_pEpContext, m_DevAddr);

                m_DeviceState = UsbState_Address;
            }

            break;

        case UsbCtrlXferReq_SetConfiguration:

            if (pCtrl->wValue == 0)
            {
                XusbDeviceBaseRegisters.ClearField32(XusbDevControlOffset, XusbDevControlRunMask);
                m_DeviceState = UsbState_Address;
            }
            else
            {
                XusbDeviceBaseRegisters.SetField32(XusbDevControlOffset, XusbDevControlRunMask);
                m_DeviceState = UsbState_Configured;
            }

            break;

        case UsbCtrlXferReq_SetFeature:

            if (pCtrl->wValue == 0x30)
            {
                uint32_t portPm = XusbDeviceBaseRegisters.Read32(XusbDevCtrlPortPmOffset);

                portPm &= ~(1 << 28);
                portPm |= XusbDevCtrlPortPmU1eMask;
                XusbDeviceBaseRegisters.Write32(XusbDevCtrlPortPmOffset, portPm);
            }

            if (pCtrl->wValue == 0x31)
            {
                uint32_t portPm = XusbDeviceBaseRegisters.Read32(XusbDevCtrlPortPmOffset);

                portPm &= ~(1 << 29);
                portPm |= XusbDevCtrlPortPmU2eMask;
                XusbDeviceBaseRegisters.Write32(XusbDevCtrlPortPmOffset, portPm);
            }

            break;

        default:

            break;
        }
    }

    if (pCtrl->bmRequestType == 2) // Handle endpoint requests
    {
        uint8_t epIndex = UsbAddressToEndPointIndex(pCtrl->wIndex & 0xff);

        switch (pCtrl->bRequest)
        {
            case UsbCtrlXferReq_SetFeature:     // set ep halt

                EpHalt(epIndex);

                break;

            case UsbCtrlXferReq_ClearFeature:   // clear ep halt

                {
                    EpContext *pEpContext = m_pEpContext + m_EndpointArray[epIndex].dci;

                    EpContextWriteState(pEpContext, EndpointState_Disabled);

                    EpReload(epIndex);

                    EpContextWriteState(pEpContext, EndpointState_Running);
                    EpContextWriteSeqNum(pEpContext, 0);

                    EpReload(epIndex);
                    EpUnpause(epIndex);
                    EpUnhalt(epIndex);
                    Ringdoorbell(&m_EndpointArray[epIndex]);
                }

                break;

            default:

                break;
        }
    }

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
// DsProtocol calls GetConnectSpeed to determine speed (for descriptors)
//////////////////////////////////////////////////////////////////////////////
UsbDeviceSpeed DsControllerTegra30::GetConnectSpeed() NN_NOEXCEPT
{
    return m_Speed;
}


//////////////////////////////////////////////////////////////////////////////
Result DsControllerTegra30::EnableEndpoint(UsbEndpointDescriptor *pUsbEndpointDescriptor, UsbEndpointCompanionDescriptor *pUsbEndpointCompanionDescriptor) NN_NOEXCEPT
{
    NN_USB_TRACE;

    if (pUsbEndpointDescriptor->bmAttributes != UsbEndpointAttributeMask_XferTypeControl)
    {
        Endpoint *pEndpoint = UsbEpStructSetup(pUsbEndpointDescriptor, pUsbEndpointCompanionDescriptor);

        if(pEndpoint)
        {
            InitializeEndpoint(pEndpoint);
        }
        else
        {
            NN_USB_LOG_ERROR("%s(@%d) No valid Endpoint and/or EpDescriptor! ep=0x%x, bInterval=%d wMaxPacketSize=%d bEndpointAddress= 0x%x\n",
                        __FUNCTION__,__LINE__,
                        pEndpoint,
                        pUsbEndpointDescriptor->bInterval,
                        pUsbEndpointDescriptor->wMaxPacketSize,
                        pUsbEndpointDescriptor->bEndpointAddress);
            return ResultMemAllocFailure();
        }
    }
    else
    {
         return ResultNotImplemented();
    }

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result DsControllerTegra30::DisableEndpoint(uint8_t bEndpointAddress) NN_NOEXCEPT
{
    NN_USB_TRACE;

    int epIndex = UsbAddressToEndPointIndex(bEndpointAddress);
    Endpoint *pEndpoint = &m_EndpointArray[epIndex];

    if(pEndpoint == nullptr)
    {
        return ResultEndpointStateInvalid();
    }

    // Non-control
    if (epIndex != 0)
    {
        if (pEndpoint->pTransferRingBuffer == nullptr)
        {
            return ResultSuccess();
        }

        FinalizeEpContext(pEndpoint);
    }

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result DsControllerTegra30::SetZlt(uint8_t address, bool zlt) NN_NOEXCEPT
{
    NN_USB_TRACE;
    NN_UNUSED(address);
    NN_UNUSED(zlt);
    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result DsControllerTegra30::PostBufferAsync(uint8_t        epAddr,
                                            ProcessHandle  processHandle,
                                            uint64_t       procVa,
                                            uint32_t       bytes) NN_NOEXCEPT
{
    NN_UNUSED(processHandle);
    NN_USB_TRACE;

    Result result = HandleDataPacket(epAddr,procVa,bytes);

    if (result.IsFailure())
    {
        protocol.CompleteBuffer(epAddr, result, 0, false);
        NN_USB_LOG_ERROR("%s(@%d)HandleDataPacket() error..epAddr=0x%x\n",__FUNCTION__,__LINE__,epAddr);
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result DsControllerTegra30::FlushEndpoint(uint8_t address) NN_NOEXCEPT
{
    NN_USB_TRACE;

    int epIndex             = UsbAddressToEndPointIndex(address);
    Endpoint *pEndpoint     = &m_EndpointArray[epIndex];
    EpContext *pEpContext   = (EpContext*)m_pEpContext + pEndpoint->dci;

    // Stop DMA on the endpoint
    if (EpContextReadState(pEpContext) == EndpointState_Running)
    {
        EpPause(epIndex);
        EpWaitForInactive(epIndex);
    }

    // Initializing the TRB ring will get rid of pending TRBs at XUSB, the
    // DsProtocol will then complete pending URBs.
    EndpointInitializeTransferRing(pEndpoint);

    // Resume
    EpUnpause(epIndex);

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result DsControllerTegra30::StallEndpoint(uint8_t address) NN_NOEXCEPT
{
    NN_USB_TRACE;
    EpHalt(UsbAddressToEndPointIndex(address));
    m_SetupStatus = WaitForSetup;

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
//  Private functions
//////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::XusbDeviceIrqHandler()
{
    volatile uint32_t*CtrlStatusReg = reinterpret_cast<volatile uint32_t*>(XusbDeviceBaseRegisters.GetBaseVa() + XusbDevCtrlStatusOffset);
    volatile uint64_t*CtrlEventRingReg = reinterpret_cast<volatile uint64_t*>(XusbDeviceBaseRegisters.GetBaseVa() + XusbDevCtrlEvtRingDpLoOffset);

    while (true)
    {
        uint32_t opStatus = *CtrlStatusReg;

        if(!(opStatus & XusbDevCtrlStatusIpMask))
        {
            break;
        }

        *CtrlStatusReg = opStatus | XusbDevCtrlStatusIpMask | XusbDevCtrlStatusDseMask;

        while (NN_USB_DEFINE_GET_FIELD_VALUE(EventTrbDw3_CycleBit, m_pEvtDqPtr->eventTrbDw3) == m_Ccs)
        {
            HandleTrbEvent(m_pEvtDqPtr);
            m_pEvtDqPtr++;
            m_pEvtDqIoVaPtr += sizeof(EventTrb);

            if (m_pEvtDqPtr == m_pEvtSeg1LastTrb)
            {
                m_Ccs ^= 1;
                m_pEvtDqPtr = m_pEvtRingSeg0;
                m_pEvtDqIoVaPtr = m_pEvtDqIoVaBasePtr;
            }
        }

        *CtrlEventRingReg = m_pEvtDqIoVaPtr;
    }
}


//////////////////////////////////////////////////////////////////////////////
Result DsControllerTegra30::InitializeXusbDevice() NN_NOEXCEPT
{
    NN_USB_TRACE;

    XusbDeviceBaseRegisters.Initialize(&Register, DeviceControllerBaseOffset, 0x8000);
    XusbDevFpciConfigRegisters.Initialize(&Register, FpciConfigRegistersOffset, 0x1000);
    XusbDeviceIpfsRegisters.Initialize(&Register, IpfsDeviceRegistersOffset, 0x1000);

    // result and do{} while(false) for NN_USB_BREAK_UPON_ERROR macro
    Result result = ResultSuccess();

    do
    {
        NN_USB_BREAK_UPON_ERROR(MapDriverMemory(this));

        NN_USB_BREAK_UPON_ERROR(nn::pcv::SetReset(nn::pcv::Module_XusbSs, false)); // partition_id_xusba
        NN_USB_BREAK_UPON_ERROR(nn::pcv::SetReset(nn::pcv::Module_XusbDevice, false));// partition_id_xusbb for DS
        NN_USB_BREAK_UPON_ERROR(nn::pcv::SetPowerEnabled(nn::pcv::Module_XusbSs, true));
        NN_USB_BREAK_UPON_ERROR(nn::pcv::SetPowerEnabled(nn::pcv::Module_XusbDevice, true)); // partition_id_xusbb for DS
        NN_USB_BREAK_UPON_ERROR(nn::pcv::SetClockEnabled(nn::pcv::Module_XusbDevice, true));
        NN_USB_BREAK_UPON_ERROR(nn::pcv::SetClockEnabled(nn::pcv::Module_XusbSs, true));

        XusbDeviceIpfsRegisters.SetField32(XusbDevIpfsConfigOffset, XusbDevIpfsConfigEnFpciMask);
        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(15));

        // Check the Vendor ID and the Device ID for sanity
        size_t VendorId = NN_USB_GET_FIELD32(XusbDevFpciCfg0VendorId, XusbDevFpciConfigRegisters.Read32(XusbDevFpciCfg0Offset));
        NN_USB_ABORT_UNLESS(XusbDevFpciCfg0Tegra21xVenId == VendorId);
        size_t DeviceId = NN_USB_GET_FIELD32(XusbDevFpciCfg0DeviceId, XusbDevFpciConfigRegisters.Read32(XusbDevFpciCfg0Offset));
        NN_USB_ABORT_UNLESS(XusbDevFpciCfg0Tegra21xDevId == DeviceId);

        // Program the following XUSB Device IPFS registers to allow SW accesses to XUSB Device’s MMIO registers
        // i.e. Device Controller registers
        XusbDeviceIpfsRegisters.SetField32(XusbDevIpfsAxiBar0StartOffset,XusbDevIpfsAxiBar0StartAxiBar0StartMask,static_cast<uint32_t>(UsbXusbDevBase));
        XusbDeviceIpfsRegisters.SetField32(XusbDevIpfsAxiBar0SzOffset, XusbDevIpfsAxiBar0SzBar0SizeMask,0x8);

        // NOTE: TRM values are incorrect. We need to use 0x00700d00 not 0x10000 otherwise controller
        // registers cannot be accessed. Reset value is 0x00700d01. Bit 1 indicates IO config access.
        XusbDeviceIpfsRegisters.SetField32(XusbDevIpfsFpciBar0Offset, XusbDevIpfsFpciBar0FpciBar0StartMask,0x700d01);

        XusbDevFpciConfigRegisters.SetField32(XusbDevFpciCfg1Offset, XusbDevFpciCfg1MemorySpaceEnabledMask | XusbDevFpciCfg1BusMasterEnabledMask | XusbDevFpciCfg1IoSpaceEnabledMask);

        // NOTE: TRM values are incorrect. Use base address not 0x2000
        XusbDevFpciConfigRegisters.SetField32(XusbDevFpciCfg4Offset,XusbDevFpciCfg4BaseAddressMask,static_cast<uint32_t>(UsbXusbDevBase));

        // NOTE: Configuration for the registers below do not appear in the TRM

        // NOTE: Configuration of FpciCfg5 is not in TRM but it's part of the 'gadget'
        XusbDevFpciConfigRegisters.Write32(XusbDevFpciCfg5Offset,static_cast<uint32_t>(UsbXusbDevBase >> 32));
        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(150));

        // Enable interrupt Assertion
        XusbDeviceIpfsRegisters.SetField32(XusbDevIpfsIntrMasksOffset, XusbDevIpfsIntrMasksIpIntMask);

        // NOTE: These registers are currently not defined in the TRM
        XusbDeviceBaseRegisters.SetField32(XusbDevCtrlSspxCorePadctl4Offset, XusbDevCtrlSspxCorePadctl4RxDatVldTimeoutU3Mask,0x5dc0);
        // Ping LFPS TBURST is greater than spec defined value of 200ns
        // Reduce ping tburst to 0x6 in order to make link layer tests pass
        XusbDeviceBaseRegisters.SetField32(XusbDevCtrlSspxCoreCnt0Offset, XusbDevCtrlSspxCoreCnt0PingTbrstMask,0x6);
        // Increase tPortConfiguration timeout to 0x978
        XusbDeviceBaseRegisters.SetField32(XusbDevCtrlSspxCoreCnt30Offset, XusbDevCtrlSspxCoreCnt30LmpitpTimerMask,0x978);
        // WAR for TD 6.5 Polling LFPS Duration Test.
        // Test suite is violating polling lfps burst max of 1.4us
        // Sending 1.45us instead
        // program POLL_TBRST_MAX to 0xB0 in order to make the test pass
        XusbDeviceBaseRegisters.SetField32(XusbDevCtrlSspxCoreCnt32Offset, XusbDevCtrlSspxCoreCnt32PollTbrstMaxMask,0xF0);

        // Register ISRs
        NN_USB_BREAK_UPON_ERROR(m_pPlatform->RequestIrq(&m_XusbDevIrqHolder, &m_XusbDevIrqHandler));

        m_SetupStatus = WaitForSetup;
    } while(false);

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result DsControllerTegra30::FinalizeXusbDevice() NN_NOEXCEPT
{
    NN_USB_TRACE;

    // result and do{} while(false) for NN_USB_BREAK_UPON_ERROR macro
    Result result = ResultSuccess();

    do
    {
        // Unregister ISRs
        NN_USB_BREAK_UPON_ERROR(m_pPlatform->FreeIrq(&m_XusbDevIrqHolder));

        // Disable interrupt Assertion
        XusbDeviceIpfsRegisters.ClearField32(XusbDevIpfsIntrMasksOffset, XusbDevIpfsIntrMasksIpIntMask);

        FinalizeEpContext(&m_EndpointArray[0]);

        NN_USB_BREAK_UPON_ERROR(FinalizeDriverMemory(this));

        memset(m_EvtRingSeg0, 0,sizeof(m_EvtRingSeg0));
        memset(m_EpContext,   0,sizeof(m_EpContext));

        // Disable PCI related
        XusbDevFpciConfigRegisters.ClearField32(XusbDevFpciCfg1Offset, XusbDevFpciCfg1MemorySpaceEnabledMask | XusbDevFpciCfg1BusMasterEnabledMask | XusbDevFpciCfg1IoSpaceEnabledMask);
        XusbDevFpciConfigRegisters.ClearField32(XusbDevFpciCfg4Offset,XusbDevFpciCfg4BaseAddressMask);
        XusbDevFpciConfigRegisters.ClearField32(XusbDevFpciCfg5Offset,XusbDevFpciCfg5BaseAddressMask);

        // Disallow SW accesses to XUSB Device’s MMIO registers
        XusbDeviceIpfsRegisters.ClearField32(XusbDevIpfsAxiBar0StartOffset,XusbDevIpfsAxiBar0StartAxiBar0StartMask);
        XusbDeviceIpfsRegisters.ClearField32(XusbDevIpfsAxiBar0SzOffset, XusbDevIpfsAxiBar0SzBar0SizeMask);
        XusbDeviceIpfsRegisters.ClearField32(XusbDevIpfsFpciBar0Offset, XusbDevIpfsFpciBar0FpciBar0StartMask);

        // Disable the IPFS device block so it no longer visible on the IPFS bus
        XusbDeviceIpfsRegisters.ClearField32(XusbDevIpfsConfigOffset, XusbDevIpfsConfigEnFpciMask);

        // Put IP into reset
        NN_USB_BREAK_UPON_ERROR(nn::pcv::SetReset(nn::pcv::Module_XusbDevice, true));
        NN_USB_BREAK_UPON_ERROR(nn::pcv::SetReset(nn::pcv::Module_XusbSs, true));

        // Stop IP clocks
        NN_USB_BREAK_UPON_ERROR(nn::pcv::SetClockEnabled(nn::pcv::Module_XusbDevice, false));
        NN_USB_BREAK_UPON_ERROR(nn::pcv::SetClockEnabled(nn::pcv::Module_XusbSs, false));

        // Disable IP power
        NN_USB_BREAK_UPON_ERROR(nn::pcv::SetPowerEnabled(nn::pcv::Module_XusbDevice, false));
        NN_USB_BREAK_UPON_ERROR(nn::pcv::SetPowerEnabled(nn::pcv::Module_XusbSs, false));
    } while(false);

    if(result.IsSuccess())
    {
        XusbDeviceBaseRegisters.Finalize();
        XusbDevFpciConfigRegisters.Finalize();
        XusbDeviceIpfsRegisters.Finalize();
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result DsControllerTegra30::MapDriverMemory(DsControllerTegra30 *pController) NN_NOEXCEPT
{
    NN_USB_TRACE;

    m_SizeOfControlTransferRing = UsbXusbDevControlTransferRingSize;
    m_SizeOfBulkTransferRing    = UsbXusbDevBulkTransferRingSize;
    m_pEvtRingSeg0              = reinterpret_cast<EventTrb*>(m_EvtRingSeg0);
    m_pEpContext                = reinterpret_cast<EpContext*>(m_EpContext);

    nn::os::SetMemoryAttribute(reinterpret_cast<uintptr_t>(m_pEvtRingSeg0), sizeof(m_EvtRingSeg0), nn::os::MemoryAttribute_Uncached);
    nn::os::SetMemoryAttribute(reinterpret_cast<uintptr_t>(m_pEpContext), sizeof(m_EpContext), nn::os::MemoryAttribute_Uncached);

    Result result = pController->DoSmmu().Map(&m_pEvtDqIoVaPtr, m_EvtRingSeg0, sizeof(m_EvtRingSeg0));

    if (result.IsSuccess())
    {
        // The only time this is used, it is always or'd with this value, so do it now to save time.
        m_pEvtDqIoVaPtr |= XusbDevCtrlEvtRingDpLoEhbMask;
        m_pEvtDqIoVaBasePtr = m_pEvtDqIoVaPtr;

        result = pController->DoSmmu().Map(m_EpContext, sizeof(m_EpContext));

        if (result.IsSuccess())
        {
            // Reserve memory and SMMU mappings for Ep0
            UsbEndpointDescriptor usbEndpointDescriptor =
            {
                UsbDescriptorSize_Endpoint,
                UsbDescriptorType_Endpoint,
                UsbDirAddrCtrl,
                UsbEndpointAttributeMask_XferTypeControl,
                64,
                0
            };

            if (UsbEpStructSetup(&usbEndpointDescriptor, NULL))
            {
                // Dequeue Pointer starts with the base of Ring 0
                m_pEvtDqPtr         = m_pEvtRingSeg0; // va
                m_pEvtSeg1LastTrb   = m_pEvtRingSeg0 + UsbXusbDevMaxRingSize * 2;
            }
            else
            {
                result = ResultMemAllocFailure();
            }
        }
        else
        {
            pController->DoSmmu().Unmap(m_EvtRingSeg0);
            pController->DoSmmu().Unmap(m_EpContext);
        }
    }

    if (!result.IsSuccess())
    {
        result = ResultMemAllocFailure();
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::UnMapDriverMemory() NN_NOEXCEPT
{
    NN_USB_TRACE;

    m_SizeOfControlTransferRing = 0;
    m_SizeOfBulkTransferRing    = 0;

    m_pXusbDeviceController->DoSmmu().Unmap(m_EvtRingSeg0);
    m_pXusbDeviceController->DoSmmu().Unmap(m_EpContext);

    m_pEvtRingSeg0      = NULL;
    m_pEvtDqPtr         = m_pEvtRingSeg0;
    m_pEvtSeg1LastTrb   = NULL;

    // free memory for all transfer rings
    Endpoint *pEndpoint;

    for (int i = 0; i < UsbXusbDevNumEndpointContext; i++)
    {
        pEndpoint = &m_EndpointArray[i];

        if (pEndpoint->pTransferRingBuffer)
        {
            if(!(this->DoSmmu().Unmap(pEndpoint->pTransferRingBuffer)).IsSuccess())
            {
                NN_USB_LOG_ERROR("unmap failed for transferRing=0x%x\n",__FUNCTION__,__LINE__,pEndpoint->pTransferRingBuffer);
            }

            nn::os::SetMemoryAttribute(reinterpret_cast<uintptr_t>(pEndpoint->pTransferRingBuffer), UsbXusbDevMaxRingSize, nn::os::MemoryAttribute_Normal);

            detail::UsbMemoryFree(pEndpoint->pTransferRingBuffer,"TransferRingBuffer");
            pEndpoint->pTransferRingBuffer = nullptr;
            pEndpoint->transferRingIoVa = 0;
            memset(pEndpoint, 0, sizeof(Endpoint));
        }
    }

    nn::os::SetMemoryAttribute(reinterpret_cast<uintptr_t>(m_pEpContext), sizeof(m_EpContext), nn::os::MemoryAttribute_Normal);
    nn::os::SetMemoryAttribute(reinterpret_cast<uintptr_t>(m_pEvtRingSeg0), sizeof(m_EvtRingSeg0), nn::os::MemoryAttribute_Normal);
}


//////////////////////////////////////////////////////////////////////////////
Result DsControllerTegra30::InitializeDriverMemory(DsControllerTegra30 *pController) NN_NOEXCEPT
{
    NN_USB_TRACE;
    m_pXusbDeviceController = pController;
    Result result = ResultSuccess();

    // Event Ring Segment Table Base Size, where the maximum size of the Event Ring Segment Table is 2 ERST Max
    // One event ring segment won't work on T210 HW. TRM 22.9.8
    // NOTE: The TRM specifies this register as read only, however in 'gadget' this register is written to.
    XusbDeviceBaseRegisters.Write32(XusbDevCtrlIdBaseOffset, UsbXusbDevErstSize << XusbDevCtrlIdBaseErstMaxShift);
    XusbDeviceBaseRegisters.Write32(XusbDevCtrlEvtSgmtRingSzOffset, UsbXusbDevMaxRingSize | (UsbXusbDevMaxRingSize << XusbDevCtrlEvtSgmtRingSzErst1szShift));

    // Initialize event ring0
    nn::dd::DeviceVirtualAddress p_EvtRingSeg0 = pController->DoSmmu().GetIoVa(m_EvtRingSeg0);
    XusbDeviceBaseRegisters.Write64(XusbDevCtrlEvtRingSgmt0AddrLoOffset, p_EvtRingSeg0);

    // Initialize event ring1
    nn::dd::DeviceVirtualAddress p_EvtRingSeg1 = pController->DoSmmu().GetIoVa(m_EvtRingSeg0) + (sizeof(EventTrb) * UsbXusbDevMaxRingSize);
    XusbDeviceBaseRegisters.Write64(XusbDevCtrlEvtRingSgmt1AddrLoOffset, p_EvtRingSeg1);

    // The producer (HW) places items in a Transfer Ring at the Enqueue Pointer,
    // and the consumer (driver) removes items from the Transfer Ring at the Dequeue Pointer.

    ////////////////////// Initialize DP //////////////////////
    // Initialize the event ring segment table dequeue pointer (DP) register with the
    // physical memory address of  the event ring segment pointed by event ring segment table entry 0 (base)
    // SW also updates this EvtRingDpLo/Hi after it has popped events.
    XusbDeviceBaseRegisters.Write64(XusbDevCtrlEvtRingDpLoOffset, p_EvtRingSeg0);

    ////////////////////// Initialize EQ //////////////////////
    // Initial value of enqueue pointer (EQ) is base of event ring 0
    XusbDeviceBaseRegisters.Write64(XusbDevCtrlEvtRingEqLoOffset, p_EvtRingSeg0);

    // Set control bit
    // SW can write to the register when Control Enable bit is cleared (reset default value of 0)
    // SW specifies the segment index to be used and HW samples the value written by SW on a posedge of control enable.
    // This set tells hardware to update EREP after posting an event
    XusbDeviceBaseRegisters.Write32(XusbDevCtrlEvtRingEqLoOffset, (p_EvtRingSeg0 & 0xFFFFFFFF) | XusbDevCtrlEvtRingEqLoEreploEcsMask);

    //// Initialize Control endpoint contexts (one IN and one OUT) and the transfer rings that they point to  ////
    ////  with the physical memory address of the first entry in the array of endpoint contexts (control EPs) ////

    ////////////////////// Initialize EC //////////////////////
    XusbDeviceBaseRegisters.Write64(XusbDevCtrlEcpLoOffset, pController->DoSmmu().GetIoVa(m_EpContext));
    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result DsControllerTegra30::FinalizeDriverMemory(DsControllerTegra30 *pController) NN_NOEXCEPT
{
    NN_USB_TRACE;
    NN_UNUSED(pController);

    Result result = ResultSuccess();

    // Clear Event Ring Segment Table Base Size
    XusbDeviceBaseRegisters.ClearField32(XusbDevCtrlIdBaseOffset, XusbDevCtrlIdBaseErstMaxMask);
    XusbDeviceBaseRegisters.ClearField32(XusbDevCtrlEvtSgmtRingSzOffset, XusbDevCtrlEvtSgmtRingSzErst0szMask);
    XusbDeviceBaseRegisters.ClearField32(XusbDevCtrlEvtSgmtRingSzOffset, XusbDevCtrlEvtSgmtRingSzErst1szMask);

    UnMapDriverMemory();

    // clear event ring0
    XusbDeviceBaseRegisters.ClearField32(XusbDevCtrlEvtRingSgmt0AddrLoOffset,XusbDevCtrlEvtRingSgmt0AddrLoErst0BalLOAddrLOMask);
    XusbDeviceBaseRegisters.ClearField32(XusbDevCtrlEvtRingSgmt0AddrHiOffset,XusbDevCtrlEvtRingSgmt0AddrHiErst0BalHIAddrHIMask);

    // clear event ring1
    XusbDeviceBaseRegisters.ClearField32(XusbDevCtrlEvtRingSgmt1AddrLoOffset,XusbDevCtrlEvtRingSgmt1AddrLoErst1BalLOAddrLOMask);
    XusbDeviceBaseRegisters.ClearField32(XusbDevCtrlEvtRingSgmt1AddrHiOffset,XusbDevCtrlEvtRingSgmt1AddrHiErst1BalHIAddrHIMask);

    // Clear value of dequeue pointer (dP)
    XusbDeviceBaseRegisters.ClearField32(XusbDevCtrlEvtRingDpLoOffset,XusbDevCtrlEvtRingDpLoErdpLOAddrLOMask); // base
    XusbDeviceBaseRegisters.ClearField32(XusbDevCtrlEvtRingDpHiOffset,XusbDevCtrlEvtRingDpHiErdpHIAddrHIMask); // base
    XusbDeviceBaseRegisters.ClearField32(XusbDevCtrlEvtRingEqLoOffset,XusbDevCtrlEvtRingDpLoErdpLOAddrLOMask); // base

    // Clear value of enqueue pointer (EP)
    XusbDeviceBaseRegisters.ClearField32(XusbDevCtrlEvtRingEqLoOffset,XusbDevCtrlEvtRingEqLoEreploEcsMask);
    XusbDeviceBaseRegisters.ClearField32(XusbDevCtrlEvtRingEqHiOffset,XusbDevCtrlEvtRingEqHiErepHIAddrHIMask); // base

    // Clear Control endpoint contexts
    XusbDeviceBaseRegisters.ClearField32(XusbDevCtrlEcpLoOffset,XusbDevCtrlEcpLoEcpLOAddrLOMask);
    XusbDeviceBaseRegisters.ClearField32(XusbDevCtrlEcpHiOffset,XusbDevCtrlEcpHiEcpHIAddrHIMask);

    return result;
}


//////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::InitializeEpContext(Endpoint *pEndpoint) NN_NOEXCEPT
{
    uint16_t maxpacket  = EndpointMaxPacketSize(pEndpoint) & 0x07ff;
    uint16_t maxburst   = 0;
    uint16_t esit       = 0;

    if (m_Speed == UsbDeviceSpeed_Super)
    {
        switch (EndpointType(pEndpoint))
        {
        case UsbEndpointAttributeMask::UsbEndpointAttributeMask_XferTypeIsoc:
        case UsbEndpointAttributeMask::UsbEndpointAttributeMask_XferTypeInt:

            esit = EndpointCompBytesPerInterval(pEndpoint);

            break;

        case UsbEndpointAttributeMask::UsbEndpointAttributeMask_XferTypeBulk:

            maxburst = EndpointCompMaxBurst(pEndpoint);

            break;

        default:

            break;
        }
    }
    else if (m_Speed < UsbDeviceSpeed_Super)
    {
        switch (EndpointType(pEndpoint))
        {
        case UsbEndpointAttributeMask::UsbEndpointAttributeMask_XferTypeIsoc:
        case UsbEndpointAttributeMask::UsbEndpointAttributeMask_XferTypeInt:

            if (m_Speed == UsbDeviceSpeed_High)
            {
                maxburst = (EndpointMaxPacketSize(pEndpoint) >> 11) & 3;

                if (maxburst == 3)
                {
                    maxburst = 2;
                }
            }

            esit = maxpacket * (maxburst + 1);

            break;

        default:

            break;
        }
    }

    EpContext *pEpContext = m_pEpContext + pEndpoint->dci;

    memset(pEpContext, 0, sizeof(EpContext));

    EpContextWriteState(pEpContext, EndpointState_Running);
    EpContextWriteInterval(pEpContext, EndpointInterval(pEndpoint));

    if (m_Speed == UsbDeviceSpeed_Super)
    {
        switch (EndpointType(pEndpoint))
        {
        case UsbEndpointAttributeMask::UsbEndpointAttributeMask_XferTypeIsoc:

            EpContextWriteMult(pEpContext, EndpointCompAttributes(pEndpoint) & 3);

            break;

        case UsbEndpointAttributeMask::UsbEndpointAttributeMask_XferTypeBulk:

            EpContextWriteMaxPstreams(pEpContext, EndpointCompAttributes(pEndpoint) & 0x1f);
            EpContextWriteLsa(pEpContext, 1);

            break;

        default:

            break;
        }
    }

    if (
        (EndpointType(pEndpoint) != UsbEndpointAttributeMask::UsbEndpointAttributeMask_XferTypeControl) &&
        (EndpointDirection(pEndpoint) == UsbEndpointAddressMask::UsbEndpointAddressMask_DirHostToDevice)
        )
    {
        EpContextWriteType(pEpContext, EndpointType(pEndpoint));
    }
    else
    {
        EpContextWriteType(pEpContext, EndpointType(pEndpoint) + 4);
    }

    EpContextWriteCerr(pEpContext, 3);
    EpContextWriteMaxPacketSize(pEpContext, maxpacket);
    EpContextWriteMaxBurstSize(pEpContext, maxburst);
    EpContextWriteDeqPtr(pEpContext, m_pXusbDeviceController->DoSmmu().GetIoVa(m_EndpointArray[pEndpoint->dci].pTransferRingBuffer));
    EpContextWriteDcs(pEpContext, pEndpoint->pcs);

    switch (EndpointType(pEndpoint))
    {
    case UsbEndpointAttributeMask::UsbEndpointAttributeMask_XferTypeControl:

        EpContextWriteAvgTrbLen(pEpContext, 8);

        break;

    case UsbEndpointAttributeMask::UsbEndpointAttributeMask_XferTypeInt:

        EpContextWriteAvgTrbLen(pEpContext, 1024);

        break;

    case UsbEndpointAttributeMask::UsbEndpointAttributeMask_XferTypeBulk:
    case UsbEndpointAttributeMask::UsbEndpointAttributeMask_XferTypeIsoc:

        EpContextWriteAvgTrbLen(pEpContext, 3072);

        break;

    default:

        break;
    }

    EpContextWriteMaxEsitPayload(pEpContext, esit);
    EpContextWriteCerrCnt(pEpContext, 3);
}


//////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::EnableRxDetect() NN_NOEXCEPT
{
    NN_USB_TRACE;

    uint32_t tempReg;

    XusbDeviceBaseRegisters.SetField32(XusbDevCtrlCfgDevFeOffset, XusbDevCtrlCfgDevFeRegSelectMask, UsbXusbDevFeHsFsPiSelect);

    tempReg = XusbDeviceBaseRegisters.Read32(XusbDevCtrlPortScOffset);
    tempReg &= PORTSC_MASK;
    tempReg &= ~NN_USB_PORTSC_PLS(~0);
    tempReg |= Pls_RxDetect;
    tempReg |= XusbDevCtrlPortScLwsMask;
    XusbDeviceBaseRegisters.Write32(XusbDevCtrlPortScOffset, tempReg);

    XusbDeviceBaseRegisters.SetField32(XusbDevCtrlCfgDevFeOffset, XusbDevCtrlCfgDevFeRegSelectMask, UsbXusbDevFeSsPiSelect);

    tempReg = XusbDeviceBaseRegisters.Read32(XusbDevCtrlPortScOffset);
    tempReg &= PORTSC_MASK;
    tempReg &= ~NN_USB_PORTSC_PLS(~0);
    tempReg |= Pls_RxDetect;
    tempReg |= XusbDevCtrlPortScLwsMask;
    XusbDeviceBaseRegisters.Write32(XusbDevCtrlPortScOffset, tempReg);

    XusbDeviceBaseRegisters.SetField32(XusbDevCtrlCfgDevFeOffset, XusbDevCtrlCfgDevFeRegSelectMask, UsbXusbDevFeRestorePortReg);

    XusbDeviceBaseRegisters.SetField32(XusbDevCtrlCfgDevFeOffset, XusbDevCtrlCfgDevFeSsRetryMask);
    XusbDeviceBaseRegisters.Write32(XusbDevCtrlSspiTransferOffset, 0xf000);

    m_Ccs = 1;

    return;
}


//////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::SetInterruptModeration(uint32_t us)
{
    NN_USB_TRACE;

    uint32_t regData;
    uint32_t imodi = us * 4; // 1 us = 4 x 250 ns
    const uint32_t defaultMinIrqInterval = 1000; // 1 ms

    if (imodi > 0xFFFF)
    {
        NN_USB_LOG_WARN("%s(@%d)invalid interrupt interval %u us\n", us);
        imodi = defaultMinIrqInterval * 4;
    }

    regData =XusbDeviceBaseRegisters.Read32(XusbDevCtrlRtImodOffset);
    regData &= ~NN_USB_DEFINE_RT_IMOD_IMODI(-1);
    regData |= NN_USB_DEFINE_RT_IMOD_IMODI(imodi);
    // HW won't overwrite IMODC. Set IMODC to the new value also
    regData &= ~NN_USB_DEFINE_RT_IMOD_IMODC(-1);
    regData |= NN_USB_DEFINE_RT_IMOD_IMODC(imodi);
    XusbDeviceBaseRegisters.Write32(XusbDevCtrlRtImodOffset, regData);
}


//////////////////////////////////////////////////////////////////////////////
int DsControllerTegra30::UsbAddressToEndPointIndex(uint8_t address) NN_NOEXCEPT
{
    NN_USB_TRACE;

    //even is out and top bit clear
    int baseAddress = address & 0x0f;
    int inOutBitAddress = address >> 7;

    // In/Out control Endpoint both mapped to 0 (bi-directional)
    if(baseAddress == 0)
    {
        return 0;
    }

    return (baseAddress << 1) | inOutBitAddress;
}


//////////////////////////////////////////////////////////////////////////////
DsControllerTegra30::Endpoint *DsControllerTegra30::UsbEpStructSetup(UsbEndpointDescriptor *pUsbEndpointDescriptor, UsbEndpointCompanionDescriptor *pUsbEndpointCompanionDescriptor) NN_NOEXCEPT
{
    NN_USB_TRACE;

    Endpoint *pEndpoint = &m_EndpointArray[UsbAddressToEndPointIndex(pUsbEndpointDescriptor->bEndpointAddress)];

    memcpy(&pEndpoint->usbEndpointDescriptor, pUsbEndpointDescriptor, sizeof(UsbEndpointDescriptor));

    if (pUsbEndpointCompanionDescriptor)
    {
        memcpy(&pEndpoint->usbEndpointCompanionDescriptor, pUsbEndpointCompanionDescriptor, sizeof(UsbEndpointCompanionDescriptor));
    }

    // maybe we should do the following for all enpoints at MapDriverMemory()
    pEndpoint->dci = UsbAddressToEndPointIndex(pUsbEndpointDescriptor->bEndpointAddress);

    if(!pEndpoint->pTransferRingBuffer)
    {
        pEndpoint->pTransferRingBuffer = (uint8_t *) detail::UsbMemoryAllocAligned(UsbXusbDevMaxRingSize, nn::dd::DeviceAddressSpaceMemoryRegionAlignment, "UsbTRB");

        if (pEndpoint->pTransferRingBuffer)
        {
            nn::os::SetMemoryAttribute(reinterpret_cast<uintptr_t>(pEndpoint->pTransferRingBuffer), UsbXusbDevMaxRingSize, nn::os::MemoryAttribute_Uncached);

            if((this->DoSmmu().Map(&pEndpoint->transferRingIoVa, pEndpoint->pTransferRingBuffer, UsbXusbDevMaxRingSize)).IsSuccess())
            {
                memset(pEndpoint->pTransferRingBuffer, 0, UsbXusbDevMaxRingSize);
            }
            else
            {
                detail::UsbMemoryFree(pEndpoint->pTransferRingBuffer,"TransferRingBuffer");
                NN_USB_LOG_ERROR("%s(@%d) Could not map memory address 0x%p\n",__FUNCTION__,__LINE__, pEndpoint->pTransferRingBuffer);
                return nullptr;
            }
        }
        else
        {
            NN_USB_LOG_ERROR("%s(@%d) Could not allocate memry for transfer ring\n",__FUNCTION__,__LINE__);
            return nullptr;
        }
    }

    return pEndpoint;
}


//////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::HandleTrbEvent(EventTrb *pEvent) NN_NOEXCEPT
{
    switch (NN_USB_DEFINE_GET_FIELD_VALUE(EventTrbDw3_Type, pEvent->eventTrbDw3))
    {
        case TrbType_Evt_PortStatusChange:
            HandlePortStatusChange();
            break;

        case TrbType_Evt_Transfer:
            HandleDataPacketCmpl(pEvent);
            break;

        case TrbType_Evt_SetupPkt:
            CheckSetupPkt(pEvent);
            break;

        default:
            NN_USB_LOG_ERROR("%s(@%d)TRB_TYPE = 0x%x\n",__FUNCTION__,__LINE__,NN_USB_DEFINE_GET_FIELD_VALUE(EventTrbDw3_Type, pEvent->eventTrbDw3));
            break;
    }

    return;
}

/*
//////////////////////////////////////////////////////////////////////////////
void PrintPortStatus(uint32_t portStatusChange)
{

    NN_USB_LOG_INFO("portStatusChange %08x\n", portStatusChange);
    NN_USB_LOG_INFO("   31:     Reserved                    %d\n", (portStatusChange & (1 << 31)) >> 31);
    NN_USB_LOG_INFO("   30:     WPR                         %d\n", (portStatusChange & (1 << 30)) >> 30);
    NN_USB_LOG_INFO("   29:28   RSVD4                       %d\n", (portStatusChange & (3 << 28)) >> 28);
    NN_USB_LOG_INFO("   27:24   Reserved                    %d\n", (portStatusChange & (0xf << 24)) >> 24);
    NN_USB_LOG_INFO("   23      CEC                         %d\n", (portStatusChange & (1 << 23)) >> 23);
    NN_USB_LOG_INFO("   22      PLC                         %d\n", (portStatusChange & (1 << 22)) >> 22);
    NN_USB_LOG_INFO("   21      PRC                         %d\n", (portStatusChange & (1 << 21)) >> 21);
    NN_USB_LOG_INFO("   20      Reserved                    %d\n", (portStatusChange & (1 << 20)) >> 20);
    NN_USB_LOG_INFO("   19      WRC                         %d\n", (portStatusChange & (1 << 19)) >> 19);
    NN_USB_LOG_INFO("   18      RSVD3                       %d\n", (portStatusChange & (1 << 18)) >> 18);
    NN_USB_LOG_INFO("   17      CSC                         %d\n", (portStatusChange & (1 << 17)) >> 17);;
    NN_USB_LOG_INFO("   16      LWS                         %d\n", (portStatusChange & (1 << 16)) >> 16);
    NN_USB_LOG_INFO("   15      Reserved                    %d\n", (portStatusChange & (1 << 15)) >> 15);
    NN_USB_LOG_INFO("   14      RSVD2                       %d\n", (portStatusChange & (1 << 14)) >> 14);
    NN_USB_LOG_INFO("   13:10   PS                          %d\n", (portStatusChange & (0xf << 10)) >> 10);
    NN_USB_LOG_INFO("   9       LANE_POLARITY_VALUE         %d\n", (portStatusChange & (1 << 9)) >> 9);
    NN_USB_LOG_INFO("   8:5     PLS                         %d\n", (portStatusChange & (0xf << 5)) >> 5);
    NN_USB_LOG_INFO("   4       PR                          %d\n", (portStatusChange & (1 << 4)) >> 4);
    NN_USB_LOG_INFO("   3       LANE_POLARITY_OVRD_VALUE    %d\n", (portStatusChange & (1 << 3)) >> 3);
    NN_USB_LOG_INFO("   2       LANE_POLARITY_OVRD          %d\n", (portStatusChange & (1 << 2)) >> 2);
    NN_USB_LOG_INFO("   1       PED                         %d\n", (portStatusChange & (1 << 1)) >> 1);
    NN_USB_LOG_INFO("   0       CCS                         %d\n", (portStatusChange & (1 << 0)) >> 0);
}
*/

//////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::SetDeviceSpeed(uint32_t portStatusChange) NN_NOEXCEPT
{
    uint32_t speed = (portStatusChange >> PortScPsShift) & PortScPsMask;

    switch (speed)
    {
    case 1:

        NN_USB_LOG_INFO("Full Speed connection\n");
        m_Speed = UsbDeviceSpeed_Full;

        break;

    case 2:

        NN_USB_LOG_INFO("Low Speed connection\n");
        m_Speed = UsbDeviceSpeed_Low;

        break;

    case 3:

        NN_USB_LOG_INFO("High Speed connection\n");
        m_Speed = UsbDeviceSpeed_High;

        break;

    case 4:

        NN_USB_LOG_INFO("Super Speed connection\n");
        m_Speed = UsbDeviceSpeed_Super;

        break;

    default:

        NN_USB_LOG_ERROR("Undefined port speed %d\n", speed);

        break;
    }
}


//////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::EnableDoorBell() NN_NOEXCEPT
{
    // clear run change bit to enable door bell
    uint32_t StatusData = XusbDeviceBaseRegisters.Read32(XusbDevCtrlStatusOffset);

    if (XusbDevCtrlStatusRcMask & StatusData)
    {
        XusbDeviceBaseRegisters.Write32(XusbDevCtrlStatusOffset, XusbDevCtrlStatusRcMask);
    }
}


//////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::HandleConnectionStatusChange(uint32_t portStatusChange) NN_NOEXCEPT
{
    if (XusbDevCtrlPortScCcsMask & portStatusChange)
    {
        SetDeviceSpeed(portStatusChange);

        protocol.SetState(UsbState_Default);
        UpdateEp0MaxPacketSize();

        // TODO: complete any requests on ep0

        m_SetupStatus = WaitForSetup;
        PortpmConfigWar();
        m_ConnectedToHost = true;
    }
    else
    {
        Reset();
        protocol.SetState(UsbState_Detached);
        EnableDoorBell();
        m_ConnectedToHost = false;
    }

    m_WaitForCsc = false;
}


//////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::HandlePortLinkStateChange(uint32_t portStatusChange) NN_NOEXCEPT
{
    uint32_t pls = portStatusChange & XusbDevCtrlPortScPlsMask;

    switch (pls)
    {
    case Pls_U0:

        if (m_Speed < UsbDeviceSpeed_Super)
        {
            ResumeState(false);
            protocol.SetState(UsbState_Default);
        }

        break;

    case Pls_U3:

        m_ResumeState = m_DeviceState;
        m_DeviceState = UsbState_Suspended;
        protocol.SetState(UsbState_Suspended);

        break;

    case Pls_Disabled:

        break;

    case Pls_RxDetect:

        break;

    case Pls_Inactive:

        if ((m_WaitForCsc == false) && (m_WaitForCscRefCount == 0))
        {
            m_WaitForCsc = true;
            m_WaitForCscRefCount++;

            NN_USB_ABORT_UNLESS_SUCCESS(
                m_pPlatform->RequestIrq(&m_PlcResetWarEventHolder, &m_PlcResetWarHandler)
            );

            nn::os::StartOneShotTimerEvent(
                &m_PlcResetWarEvent,
                nn::TimeSpan::FromMilliSeconds(100)
            );
        }

        break;

    case Pls_Resume:

        if (m_Speed == UsbDeviceSpeed_Super)
        {
            ResumeState(false);
            protocol.SetState(UsbState_Default);
        }

        break;

    default:

        NN_USB_LOG_ERROR("Unhandled PLS state %d\n", pls >> 5);

        break;
    }
}


//////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::HandlePortStatusChange() NN_NOEXCEPT
{
    uint32_t portStatusChange = XusbDeviceBaseRegisters.Read32(XusbDevCtrlPortScOffset);

//PrintPortStatus(portStatusChange);

    if (portStatusChange & (XusbDevCtrlPortScPrcMask | XusbDevCtrlPortScPrMask | XusbDevCtrlPortScWrcMask))
    {
        portStatusChange |= XusbDevCtrlPortScPedMask;
    }

    XusbDeviceBaseRegisters.Write32(XusbDevCtrlPortScOffset, portStatusChange);

    if ((XusbDevCtrlPortScPrcMask & portStatusChange) && (XusbDevCtrlPortScPrMask & portStatusChange))
    {
        if ((m_WaitForSecPrc == false) && (m_WaitForSecPrcRefCount == 0))
        {
            m_WaitForSecPrc = true;
            m_WaitForSecPrcRefCount++;

            NN_USB_ABORT_UNLESS_SUCCESS(
                m_pPlatform->RequestIrq(&m_PortResetWarEventHolder, &m_PortResetWarHandler)
            );

            nn::os::StartOneShotTimerEvent(
                &m_PortResetWarEvent,
                nn::TimeSpan::FromMilliSeconds(100)
            );
        }
    }

    if ((XusbDevCtrlPortScPrcMask & portStatusChange) && (!(XusbDevCtrlPortScPrMask & portStatusChange)))
    {
        Reset();

        PortpmConfigWar();

        m_WaitForSecPrc = false;

        // Reset processing has been moved to CSC handling. This is only here
        // as workaround for Relay Box.
        if (m_ConnectedToHost)
        {
            // When using the "Relay Box" we don't actually get a detach so
            // lets set detached state then ASSERT the CSC as if there was an
            // detach then attach interrupt.
            protocol.SetState(UsbState_Detached);
            portStatusChange |= XusbDevCtrlPortScCscMask;
        }
    }

    // check status change request in Port HALT
    uint32_t portHaltRegister = XusbDeviceBaseRegisters.Read32(XusbDevCtrlPortHaltOffset);

    if (XusbDevCtrlPortHaltStChgReqMask & portHaltRegister)
    {
        // Clear HALT_LTSSM otherwise USB3 will not initiate Link Training
        // If this is not cleared then the device will enumerate as a High Speed Device
        if (XusbDevCtrlPortHaltHltLtssmMask & portHaltRegister)
        {
            uint32_t tempReg = portHaltRegister & 0x071F0003;
            tempReg |= XusbDevCtrlPortHaltStChgReqMask;

            if (XusbDevCtrlPortHaltHltLtssmMask & tempReg)
            {
                tempReg &= 0xfffffffe;//~XusbDevCtrlPortHaltHltLtssmMask;
                XusbDeviceBaseRegisters.Write32(XusbDevCtrlPortHaltOffset, tempReg);
            }
        }
    }

    if ((XusbDevCtrlPortScWrcMask & portStatusChange) && (!(XusbDevCtrlPortScWprMask & portStatusChange)))
    {
        Reset();
        PortpmConfigWar();
        EnableDoorBell();
    }

    if (XusbDevCtrlPortScCscMask & portStatusChange)
    {
        HandleConnectionStatusChange(portStatusChange);
    }

    if (XusbDevCtrlPortScPlcMask & portStatusChange)
    {
        HandlePortLinkStateChange(portStatusChange);
    }
}


/////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::UpdateEp0MaxPacketSize() NN_NOEXCEPT
{
    EpContext *pEpContext = m_pEpContext;

    uint16_t maxPacketSize;

    switch (m_Speed)
    {
    case UsbDeviceSpeed_Full:
    case UsbDeviceSpeed_Low:
    case UsbDeviceSpeed_High:

        maxPacketSize = 64;

        break;

    case UsbDeviceSpeed_Super:

        maxPacketSize = 512;

        break;

    default:

        maxPacketSize = 0;

        break;
    }

    EpContextWriteMaxPacketSize(pEpContext, maxPacketSize);
}


/////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::PollEpStopped(const char *_fmt, size_t _val) NN_NOEXCEPT
{
    uint32_t _i, _reg;

    for (_i = 0; _i < (StateChgRegTimeoutInuSecs * 20); _i++)
    {
        _reg = XusbDeviceBaseRegisters.Read32(XusbDevCtrlDevEpStoppedOffset);

        if ((_reg & _val) == _val)
        {
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromNanoSeconds(50));
    }

    if (_i == (EpStoppedRegTimeoutInuSecs * 20))
    {
        NN_USB_LOG_ERROR("EP_STOPPED Timeout: %s\n", _fmt);
        NN_USB_LOG_ERROR("EP_STOPPED=0x%.8x, EpMask=0x%.8x\n",_reg, _val);
    }
}


/////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::PollStchg(const char *_fmt, size_t _val) NN_NOEXCEPT
{
    do {
        uint32_t _i, _reg;
        for (_i = 0; _i < (StateChgRegTimeoutInuSecs * 20); _i++)
        {
            _reg = XusbDeviceBaseRegisters.Read32(XusbDevCtrlEpStchgOffset);

            if ((_reg & _val) == _val)
                break;
            nn::os::SleepThread(nn::TimeSpan::FromNanoSeconds(50));
        }

        if (_i == (StateChgRegTimeoutInuSecs * 20))
        {
            NN_USB_LOG_ERROR("STCHG Timeout: %s\n", _fmt);
            NN_USB_LOG_ERROR("STCHG=0x%.8x, EpMask=0x%.8x\n",_reg, _val);
        }
    } while (0);
}


/////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::ReadPoll32(size_t offset, uint32_t mask, uint32_t value) NN_NOEXCEPT
{
    uint32_t i, reg;

    for (i = 0; i < (ReloadRegTimeoutInuSecs * 20); i++)
    {
        reg = XusbDeviceBaseRegisters.Read32(offset);

        if ((reg & mask) == value)
        {
            return;
        }

        nn::os::SleepThread(nn::TimeSpan::FromNanoSeconds(50));
    }

    NN_USB_LOG_ERROR("ReadPoll32 Timeout\n");
}


/////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::Reset() NN_NOEXCEPT
{
    uint32_t regData = 0;
    Endpoint *pEndpoint;
    EpContext *pEpContext;
    m_SetupStatus = WaitForSetup;

    m_DeviceState = UsbState_Attached;

    // clear PAUSE for all the endpoints
    regData = XusbDeviceBaseRegisters.Read32(XusbDevCtrlEpPauseOffset);
    XusbDeviceBaseRegisters.Write32(XusbDevCtrlEpPauseOffset, 0);

    PollStchg(__FUNCTION__, regData);
    XusbDeviceBaseRegisters.Write32(XusbDevCtrlEpStchgOffset, regData);

    for (size_t i = 2; i < 32; i++)
    {
        pEndpoint = &m_EndpointArray[i];
        //pEndpoint->transferRingFull = false;
EndpointInitializeTransferRing(pEndpoint);
    }

    pEndpoint  = &m_EndpointArray[UsbEndpointControlIndex];

    pEpContext = m_pEpContext;

    EpContextWriteSeqNum(pEpContext, 0);
    EpContextWriteDeqPtr(pEpContext, m_pXusbDeviceController->DoSmmu().GetIoVa(pEndpoint->pEnqPtrTransferTrb));
    EpContextWriteDcs(pEpContext, pEndpoint->pcs);

    pEndpoint->pDeqPtrTransferTrb = pEndpoint->pEnqPtrTransferTrb;

    EpReload(0);
    EpUnpause(0);
}


/////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::PlcResetWarWork() NN_NOEXCEPT
{
    // This function is called from Irq thread, no need to lock against ISR

    if (m_WaitForCsc)
    {
        NN_USB_LOG_INFO("Executing PlcResetWarWork() %08x\n", XusbDeviceBaseRegisters.Read32(XusbDevCtrlPortScOffset) & XusbDevCtrlPortScPlsMask);

        if ((XusbDeviceBaseRegisters.Read32(XusbDevCtrlPortScOffset) & XusbDevCtrlPortScPlsMask) == Pls_Inactive)
        {
            NN_USB_LOG_INFO("Toggle VBUS\n");
            // This WAR does not affect USB2, however it affects USB3.
            // Without this WAR setup packets are sent from the host (no interrupt generated though)
            m_pComplex->XusbPadctlToggleVbusId0(false);
            m_pComplex->XusbPadctlToggleVbusId0(true);
        }

        m_WaitForCsc = false;
    }

    m_pPlatform->FreeIrq(&m_PlcResetWarEventHolder);
    m_WaitForCscRefCount--;
}


/////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::PortResetWarWork() NN_NOEXCEPT
{
    // This function is called from Irq thread, no need to lock against ISR
    if (m_WaitForSecPrc) // vbus assumed to be set
    {
        NN_USB_LOG_INFO("Executing PortResetWarWork()\n");

        uint32_t pls = XusbDeviceBaseRegisters.Read32(XusbDevCtrlPortScOffset) & XusbDevCtrlPortScPlsMask;
        NN_USB_LOG_INFO("%s(@%d)pls = 0x%x \n",__FUNCTION__,__LINE__,XusbDevCtrlPortScPlsMask);

        uint32_t val = m_pComplex->XusbPadctlReadBatteryChrgOtgPadCtl0Hs();
        NN_USB_LOG_INFO("%s(@%d)battery_chrg_otgpad_ctl0 = 0x%x\n",__FUNCTION__,__LINE__,val);

        uint32_t zip = NN_USB_DEFINE_SET_BIT(18);
        uint32_t zin = NN_USB_DEFINE_SET_BIT(22);

        if ((pls == Pls_Disabled) && ((val & zin) || (val & zip)))
        {
            NN_USB_LOG_INFO("Toggle VBUS\n");

            // PRC doesn't complete in 100ms, toggle the vbus
            m_pComplex->XusbPadctlToggleVbusId0(false);
            m_pComplex->XusbPadctlToggleVbusId0(true);
        }

        m_WaitForSecPrc = false;
    }

    m_pPlatform->FreeIrq(&m_PortResetWarEventHolder);
    m_WaitForSecPrcRefCount--;
}


/////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::RetryStreamRejectedWork(Endpoint *pEndpoint) NN_NOEXCEPT
{
    // When the stream is rejected by the Host, we wait for this much
    // time(in milliseconds) before we retry sending an ERDY. so in total
    // we retry for 400ms( 20 count x 20ms sleep each time) for the Host to
    // respond and then stop retrying and wait for the PRIME from Host
    static uint32_t streamRejectedSleepmmSecs = 20;

    uint32_t regData;
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(streamRejectedSleepmmSecs));

    EpWaitForStopped(pEndpoint->dci);

    // missing queue EP code.

    regData = NN_USB_DEFINE_DB_TARGET(pEndpoint->dci);

    XusbDeviceBaseRegisters.Write32(XusbDevCtrlDoorbellBaseOffset, regData);
}


/////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::PortpmConfigWar() NN_NOEXCEPT
{
    // WAR: Disable u1 to improve link stability
    uint32_t portPm = XusbDeviceBaseRegisters.Read32(XusbDevCtrlPortPmOffset);
    portPm &= ~XusbDevCtrlPortPmU1timeoutMask;

    XusbDeviceBaseRegisters.Write32(XusbDevCtrlPortPmOffset, portPm);

    // Work around (WAR) for the following controller issue
    // No support of U1 or U2 timers
    // One of the most common problems with LPM implementation is the failure to initiate U1 or U2 transition
    // when the timer expires. Even after the software has programmed U1 or U2 timeout values for DS ports,
    // some hubs or controllers do not initiate a transition to U1 or U2 on the expiration of the timer.
    // This behavior prevents power savings through LPM.
    if (m_Speed <= UsbDeviceSpeed_High)
    {
        // Set L1 State to NYET. See TRM 23.16.15.20 T_ XUSB_DEV_XHCI_PORTPM for actual value definition
        XusbDeviceBaseRegisters.SetField32(XusbDevCtrlPortPmOffset, XusbDevCtrlPortPmL1stateMask, 2);
    }
}


/////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::ResumeState(bool deviceInit) NN_NOEXCEPT
{
    uint32_t epPaused, scRegData;

    // Unpausing the Endpoints (by writing to itself)
    epPaused = XusbDeviceBaseRegisters.Read32(XusbDevCtrlEpPauseOffset);
    XusbDeviceBaseRegisters.Write32(XusbDevCtrlEpPauseOffset, 0);
    PollStchg(__FUNCTION__, epPaused);
    XusbDeviceBaseRegisters.Write32(XusbDevCtrlEpStchgOffset, epPaused);

    scRegData = XusbDeviceBaseRegisters.Read32(XusbDevCtrlPortScOffset);

    if (Pls_U0 != (scRegData & XusbDevCtrlPortScPlsMask))
    {
        uint32_t tempReg = scRegData & PORTSC_MASK;
        tempReg &= ~XusbDevCtrlPortScPlsMask;
        tempReg |= XusbDevCtrlPortScLwsMask; // Plus an or of 0, which is a nop

        XusbDeviceBaseRegisters.Write32(XusbDevCtrlPortScOffset, tempReg);
    }

    if (m_DeviceState == UsbState_Suspended)
    {
        m_DeviceState = m_ResumeState;
        m_ResumeState = 0;
    }

    if (deviceInit)
    {
        NN_ABORT("Device Init resume not implemented\n");
    }

    // If the time difference between SW clearing the Pause bit
    // and SW ringing doorbell was very little (less than 200ns),
    // this could lead to the doorbell getting dropped and the
    // corresponding transfer not completing.
    // WAR: add 500ns delay before ringing the door bell
    nn::os::SleepThread(nn::TimeSpan::FromNanoSeconds(500));

    // ring door bell for the paused endpoints. There are no EP paused during initial enumeration
    DoorbellForUnpause(epPaused);
}


/////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::DoorbellForUnpause(uint32_t pausedBits) NN_NOEXCEPT
{
    int i;
    uint32_t dci;
    Endpoint *pEndpoint;

    for (i = 0; i < UsbXusbDevNumEndpointContext; i++)
    {
        if (!pausedBits)
        {
            break;
        }

        if (pausedBits & 1)
        {
            pEndpoint = &m_EndpointArray[i];

            dci = pEndpoint->dci;
            dci = NN_USB_DEFINE_DB_TARGET(dci);

            XusbDeviceBaseRegisters.Write32(XusbDevCtrlDoorbellBaseOffset, dci);
        }

        pausedBits = pausedBits >> 1;
    }
}


/////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::CheckSetupPkt(EventTrb *pEvent) NN_NOEXCEPT
{
    uint16_t seqNum = 0;
    UsbCtrlRequest *pUsbCtrlReq = NULL;
    uint64_t trb = (uint64_t)pEvent->trbPointerLo + ((uint64_t)(pEvent->trbPointerHi) << 32);

    pUsbCtrlReq = (UsbCtrlRequest *)&trb;

    seqNum = NN_USB_DEFINE_GET_FIELD_VALUE(EventTrbDw2_SeqNum,pEvent->eventTrbDw2);

    if (m_SetupStatus != WaitForSetup)
    {
        NN_USB_LOG_ERROR("%s(@%d) status not WaitForSetup = %d. Now sending setup pkt to Protocol\n", __FUNCTION__, __LINE__, m_SetupStatus);
    }

    HandleSetupPacket(pUsbCtrlReq, seqNum);

    return;
}


/////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::HandleSetupPacket(UsbCtrlRequest *pUsbCtrlReq, uint16_t seqNum) NN_NOEXCEPT
{
    m_SetupStatus   = SetupPktProcessInProgress;
    m_CtrlSeqNum    = seqNum;

    EpUnhalt(0);

    // war for ctrl request with seqNum = 0xfffe or 0xffff
    if (seqNum == 0xfffe || seqNum == 0xffff)
    {
        NN_USB_LOG_WARN("%s(@%d)Applied WAR for ctrl request with seqNum = 0xfffe or 0xffff\n",__FUNCTION__,__LINE__);
        EpHalt(0);
        m_SetupStatus = WaitForSetup;
        return;
    }

    m_SetupStatus =  (pUsbCtrlReq->bmRequestType & UsbDirAddrCtrl) ? DataStageTransferToHost :  DataStageReceivedFromHost;
    protocol.SetSetupPacket(0x00, pUsbCtrlReq);
    protocol.DoCtrlRequest();
}


/////////////////////////////////////////////////////////////////////////////
Result DsControllerTegra30::HandleDataPacket(uint8_t epAddr, uint64_t transferData,uint32_t transferLength) NN_NOEXCEPT
{
    NN_USB_TRACE;
    Result result = ResultSuccess();
    int epIndex = UsbAddressToEndPointIndex(epAddr);
    Endpoint *pEndpoint = &m_EndpointArray[epIndex];

    // status is being sent from the upper layer
    if (transferLength == 0 && UsbEndpointIsControl(&pEndpoint->usbEndpointDescriptor))
    {
        if (!m_DataTransferComplete)
        {
            m_SetupStatus = StatusStageTransferToHost;
        }
        else
        {
            m_DataTransferComplete = false;
            m_SetupStatus = StatusStageReceivedFromHost;
        }

        Ep0Status();

        return result;
    }

    // request length is possible to be 0. Like SCSI blank command

    // If the transfer ring for this particular end point is full,
    // then simply queue the request and return
    if (pEndpoint->transferRingFull == true)
    {
        NN_USB_LOG_ERROR("%s(@%d)Transfer Ring Full\n",__FUNCTION__,__LINE__);
        result = ResultSuccess();
    }
    else
    {
        // build td
        uint32_t numTrbsNeeded = (uint32_t)(transferLength / TrbMaxBufferSize);

        // Zero length packets
        if ((transferLength == 0) || (transferLength % TrbMaxBufferSize))
        {
            numTrbsNeeded += 1;
        }

        // push the request to the transfer ring if possible.
        if (UsbEndpointIsControl(&pEndpoint->usbEndpointDescriptor))
        {
            result = BuildEp0DataIn(pEndpoint,numTrbsNeeded,transferData,(uint64_t)transferLength);
        }
        else
        {
            size_t transferRingSize = 0;
            if (UsbEndpointIsBulk(&pEndpoint->usbEndpointDescriptor))
            {
                transferRingSize = UsbXusbDevBulkTransferRingSize;
            }
            else if (UsbEndpointIsInt(&pEndpoint->usbEndpointDescriptor))
            {
                transferRingSize = UsbXusbDevIntTransferRingSize;
            }
            else
            {
                // ISOC is not implemented yet
                return ResultNotImplemented();
            }

            uint32_t numTrbsAva = RoomOnRing(transferRingSize, pEndpoint->pFirstTransferTrb, pEndpoint->pEnqPtrTransferTrb, pEndpoint->pDeqPtrTransferTrb);
            if (numTrbsAva >= numTrbsNeeded)
            {
                result = QueueTrbs(pEndpoint,transferRingSize,numTrbsNeeded,transferData,(uint64_t)transferLength);
                pEndpoint->bulkTransferComplete = true;
            }
            else
            {
                // starting count of TRBs
                pEndpoint->numTrbsStillNeeded = numTrbsNeeded;
                pEndpoint->numTrbsAva         = numTrbsAva;
                pEndpoint->bytesSent          = numTrbsAva * TrbMaxBufferSize;
                pEndpoint->addressOfTransferDataSent = transferData;

                if (pEndpoint->numTrbsStillNeeded > pEndpoint->numTrbsAva)
                {
                    pEndpoint->numTrbsStillNeeded -= pEndpoint->numTrbsAva; // Use the TRBS available and update the TRBs need to account for those used

                    // make sure we have enought TRBs in the ring
                    pEndpoint->numTrbsAva  = RoomOnRing(transferRingSize, pEndpoint->pFirstTransferTrb, pEndpoint->pEnqPtrTransferTrb, pEndpoint->pDeqPtrTransferTrb);
                    pEndpoint->bytesSent   = pEndpoint->numTrbsAva * TrbMaxBufferSize;
                    result = QueueTrbs(pEndpoint,transferRingSize,pEndpoint->numTrbsAva,pEndpoint->addressOfTransferDataSent,(uint64_t)pEndpoint->bytesSent);
                    pEndpoint->addressOfTransferDataSent += pEndpoint->bytesSent;

                    pEndpoint->bulkTransferComplete = false;
                }
            }
        }
    }

    return result;
}


/////////////////////////////////////////////////////////////////////////////
Result DsControllerTegra30::BuildEp0DataIn(Endpoint *pEp0, uint32_t numTrbsNeeded, uint64_t transferData, uint64_t transferLength) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    EpContext *pEpContext = m_pEpContext + UsbEndpointControlIndex;

    uint8_t EpState = EpContextReadState(pEpContext);
    Trb *pEnqPtrTransferTrb = pEp0->pEnqPtrTransferTrb;
    Trb *pDeqPtrTransferTrb = pEp0->pDeqPtrTransferTrb;
    uint32_t length;
    uint8_t  Ioc;
    uint32_t trbsUsed = 0;
    uint8_t  dir = 0;

    // Need to process the request even ep is paused or halted
    if (EpState != EndpointState_Running)
    {
        NN_USB_LOG_ERROR("%s(@%d)invalid state EP State = 0x%x\n",__FUNCTION__,__LINE__, EpState);
        return ResultInterfaceInvalidState();
    }

    // For control endpoint, we handle one setup request at a time. TDs are NOT queued for EP0
    if (pEnqPtrTransferTrb == pDeqPtrTransferTrb)
    {
        uint32_t tempLen = 0;
        uint32_t numTrb  = 0;

        // Transfer ring is empty, therefore setup data stage TRBs
        if (m_SetupStatus ==  DataStageTransferToHost)
        {
            dir = 1;
        }
        else if (m_SetupStatus == DataStageReceivedFromHost)
        {
            dir = 0;
        }
        else
        {
            NN_USB_LOG_ERROR("%s(@%d)unexpected m_SetupStatus=%s\n", __FUNCTION__, __LINE__, m_SetupStatus);
            return ResultInterfaceInvalidState();
        }

        for (numTrb = 0; numTrb < numTrbsNeeded; numTrb++)
        {
            if (numTrb < (numTrbsNeeded - 1))
            {
                length = TrbMaxBufferSize;
                Ioc = 0; // No Interrupt on Completion
            }
            else
            {
                tempLen = TrbMaxBufferSize * numTrb;
                length = (uint32_t) transferLength - tempLen;  // transferLength provided by the driver
                Ioc = 1; // Interrupt on completion
            }

            // Setup TRB to transfer data
            memset(pEnqPtrTransferTrb, 0, sizeof(Trb));

            TrbWriteDataPtr(pEnqPtrTransferTrb, transferData);
            TrbWriteTransferLen(pEnqPtrTransferTrb, length);
            TrbWriteTdSize(pEnqPtrTransferTrb, trbsUsed);
            TrbWriteCycle(pEnqPtrTransferTrb, pEp0->pcs);
            TrbWriteIsp(pEnqPtrTransferTrb, 1);
            TrbWriteIoc(pEnqPtrTransferTrb, Ioc);
            TrbWriteType(pEnqPtrTransferTrb, TrbType_Transfer_Data_Stage);
            TrbWriteDataStageDir(pEnqPtrTransferTrb, dir);

            // udpate enqueue pointer
            pEnqPtrTransferTrb++;
            pEnqPtrTransferTrb = CheckEndofSegment(pEp0,pEnqPtrTransferTrb);
        }

        pEp0->pEnqPtrTransferTrb = pEnqPtrTransferTrb;

        uint32_t regData = 0;
        regData = NN_USB_DEFINE_DB_TARGET(0);
        regData |= NN_USB_DEFINE_DB_STREAMID(m_CtrlSeqNum);
        XusbDeviceBaseRegisters.Write32(XusbDevCtrlDoorbellBaseOffset, regData);
    }
    else
    {
        // we process one setup request at a time, so ring should already be empty.
        // If there are TD pending in the transfer ring, wait for the sequence number error event,
        // then put the new request to tranfer ring
        NN_USB_LOG_ERROR("%s(@%d)Eq = 0x%x != Dq = 0x%x\n",__FUNCTION__,__LINE__,pEnqPtrTransferTrb, pDeqPtrTransferTrb);
    }

    return result;
}


/////////////////////////////////////////////////////////////////////////////
Trb* DsControllerTegra30::TranTrbDmaToVirt(Endpoint *pEndpoint, uint64_t address) NN_NOEXCEPT
{
    if (address & 0xf)
    {
        NN_USB_LOG_ERROR("%s(@%d)transfer ring dma address incorrect\n",__FUNCTION__,__LINE__);
        return NULL;
    }

    uint64_t offset = address - pEndpoint->transferRingIoVa;

    if (offset > UsbXusbDevMaxRingSize)
    {
        NN_USB_LOG_ERROR("%s(@%d)dma address isn't within transfer ring\n",__FUNCTION__,__LINE__);
        NN_USB_LOG_ERROR("%s(@%d)dma address 0x%llx\n",__FUNCTION__,__LINE__,address);
        NN_USB_LOG_ERROR("%s(@%d)ring is 0x%llx ~ 0x%llx\n",__FUNCTION__,__LINE__,
                        pEndpoint->transferRingIoVa,(uint64_t) (pEndpoint->transferRingIoVa + UsbXusbDevMaxRingSize - 1));
        return NULL;
    }

    offset /= sizeof(Trb);

    // This is the CPU va
    return pEndpoint->pFirstTransferTrb + offset;
}


/////////////////////////////////////////////////////////////////////////////
Result DsControllerTegra30::HandleDataPacketCmpl(EventTrb *pEvent) NN_NOEXCEPT
{
    Result result           = ResultSuccess();
    uint8_t epIndex         = NN_USB_DEFINE_GET_FIELD_VALUE(EventTrbDw3_EndpointId, pEvent->eventTrbDw3);
    Endpoint  *pEndpoint    = &m_EndpointArray[epIndex];
    EpContext *pEpContext   = m_pEpContext + epIndex; // FIXME: Verify epIndex range
    uint16_t compCode;

    // handle exfer event
    if (!pEndpoint->pFirstTransferTrb || EpContextReadState(pEpContext) == EndpointState_Disabled)
    {
        NN_USB_LOG_ERROR("%s(@%d)WARNING: Invalid State/TransferRing epIndex=%d\n",__FUNCTION__,__LINE__,epIndex);
        return ResultInterfaceInvalidState();
    }

    UpdateEpDequeuePtr(pEvent, pEndpoint);

    compCode = NN_USB_DEFINE_GET_FIELD_VALUE(EventTrbDw2_ComplCode, pEvent->eventTrbDw2);

    switch (compCode)
    {
        case TrbCmplCode_Success:

            if (PrimeNotRcvdWar)
            {
                uint32_t temp;
                temp = NN_USB_DEFINE_SET_BIT(epIndex);

                // When the stream is being rejected by the Host and then
                // a valid packet is exchanged on this pipe, we clear the
                // stream rejected condition and cancel any pending work that
                // was scheduled to retry the stream
                if (m_StreamRejected & temp)
                {
                    NN_USB_LOG_INFO("%s(@%d)m_StreamRejected\n",__FUNCTION__,__LINE__);
                    m_StreamRejected &= ~temp;
                    // TODO: cancel stream rejected work?
                    pEndpoint->streamRejectedRetryCount = 0;
                }
            }

            HandleCmplCodeSuccess(pEvent, pEndpoint);

            return result;

        case TrbCmplCode_Short_Pkt:

            HandleCmplCodeShortPkt(pEvent, pEndpoint);

            return result;

            // Remaining cases will exit via reporting an error condition to protocol.

        case TrbCmplCode_HostRejected:

            HandleCmplCodeHostRejected(pEvent, pEndpoint);

            break;

        case TrbCmplCode_PrimePipeReceived:

            HandleCmplCodePrimePipeReceived(pEvent, pEndpoint);

            break;

        case TrbCmplCode_CtrlSeqnumErr:

            HandleCmplCodeCtrlSeqnumErr(pEvent, pEndpoint);

            break;

        // All these cases handled as one
        case TrbCmplCode_BabbleDetectedErr:

            EpWaitForStopped(pEndpoint->dci);

        case TrbCmplCode_StreamNumpError:
        case TrbCmplCode_CtrlDirErr:
        case TrbCmplCode_InvalidStreamTypeErr:
        case TrbCmplCode_RingUnderrun:
        case TrbCmplCode_RingOverrun:
        case TrbCmplCode_IsochBufferOverrun:
        case TrbCmplCode_UsbTransErr:
        case TrbCmplCode_TrbErr:

            NN_USB_LOG_ERROR("%s(@%d)compCode = 0x%x EP DCI = 0x%x\n",__FUNCTION__,__LINE__,compCode,pEndpoint->dci);
            EpHalt(pEndpoint->dci);

            break;

        case TrbCmplCode_Stopped:

            EndpointInitializeTransferRing(pEndpoint);

            break;

        default:

            NN_USB_LOG_ERROR("%s(@%d)compCode = 0x%x EP DCI = 0x%x\n",__FUNCTION__,__LINE__,compCode,pEndpoint->dci);

            EpHalt(epIndex);

            break;
    }

    protocol.CompleteBuffer(pEndpoint->usbEndpointDescriptor.bEndpointAddress, ResultTransactionError(), 0, true);

    return result;
}


/////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::UpdateEpDequeuePtr(EventTrb *pEvent, Endpoint *pEndpoint) NN_NOEXCEPT
{
    uint32_t deqPtLo    = pEvent->trbPointerLo;
    uint32_t deqPtHi    = pEvent->trbPointerHi;
    uint64_t dqPtAddr   = (uint64_t)deqPtLo + ((uint64_t)deqPtHi << 32);
    Trb *pDeqPt;

    pDeqPt = TranTrbDmaToVirt(pEndpoint, dqPtAddr);

    if (!pDeqPt)
    {
        NN_USB_LOG_ERROR("%s(@%d) invalid dqPtAddr 0x%llx\n",__FUNCTION__,__LINE__,(uint64_t) dqPtAddr);
        return;
    }

    pDeqPt++;

    if (TrbReadType(pDeqPt) == TrbType_Link)
    {
        pDeqPt = pEndpoint->pFirstTransferTrb;
    }

    pEndpoint->pDeqPtrTransferTrb = pDeqPt;
}


/////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::HandleCmplCodeSuccess(EventTrb *pEvent,Endpoint *pEndpoint) NN_NOEXCEPT
{
    uint64_t trb;
    Trb *pTrb=NULL;
    uint32_t bytesRemaining;

    trb = (uint64_t)pEvent->trbPointerLo + ((uint64_t)(pEvent->trbPointerHi) << 32);

    pTrb = TranTrbDmaToVirt(pEndpoint, trb);
    if (!pTrb)
    {
        NN_USB_LOG_ERROR("%s(@%d)invalid trb\n",__FUNCTION__,__LINE__);
        return;
    }

    if (!TrbReadChain(pTrb))
    {
        bytesRemaining = NN_USB_DEFINE_GET_FIELD_VALUE(EventTrbDw2_TranLen,pEvent->eventTrbDw2);

        ///ZLP code goes here

        //@IMPLEMENT complete req (upper layer notification)
        if (UsbEndpointIsControl(&pEndpoint->usbEndpointDescriptor))
        {
            //ep0 req complete
            HandleEp0StatusComplete(pEndpoint);
        }
        else if (UsbEndpointIsBulk(&pEndpoint->usbEndpointDescriptor))
        {
            // Bulk transfer is done
            if (pEndpoint->bulkTransferComplete)
            {
                protocol.CompleteBuffer(pEndpoint->usbEndpointDescriptor.bEndpointAddress, ResultSuccess(), bytesRemaining, true);
            }
            else
            {
                Result result = ResultSuccess();
                if (pEndpoint->numTrbsStillNeeded > pEndpoint->numTrbsAva)
                {
                    pEndpoint->numTrbsStillNeeded -= pEndpoint->numTrbsAva; // update TRB count
                    // make sure we have enought TRBs in the ring
                    pEndpoint->numTrbsAva  = RoomOnRing(m_SizeOfBulkTransferRing, pEndpoint->pFirstTransferTrb, pEndpoint->pEnqPtrTransferTrb, pEndpoint->pDeqPtrTransferTrb);
                    pEndpoint->bytesSent = pEndpoint->numTrbsAva * TrbMaxBufferSize;
                    result = QueueTrbs(pEndpoint,m_SizeOfBulkTransferRing,pEndpoint->numTrbsAva,pEndpoint->addressOfTransferDataSent,(uint64_t)pEndpoint->bytesSent);
                    pEndpoint->addressOfTransferDataSent += pEndpoint->bytesSent;
                    pEndpoint->bulkTransferComplete = false;
                }
                else
                {
                    pEndpoint->bytesSent = pEndpoint->numTrbsStillNeeded * TrbMaxBufferSize;
                    result = QueueTrbs(pEndpoint,m_SizeOfBulkTransferRing,pEndpoint->numTrbsStillNeeded,pEndpoint->addressOfTransferDataSent,(uint64_t)pEndpoint->bytesSent);
                    // Now we can call back into the protocol
                    pEndpoint->bulkTransferComplete = true;
                    pEndpoint->numTrbsAva = 0;
                    pEndpoint->numTrbsStillNeeded = 0;
                    pEndpoint->bytesSent = 0;
                    pEndpoint->addressOfTransferDataSent = 0;
                }
            }
        }
        else if (UsbEndpointIsInt(&pEndpoint->usbEndpointDescriptor))
        {
            protocol.CompleteBuffer(pEndpoint->usbEndpointDescriptor.bEndpointAddress, ResultSuccess(), bytesRemaining, true);
        }
    }
    else
    {
        NN_USB_LOG_ERROR("%s(@%d)chain bit set\n",__FUNCTION__,__LINE__);
    }
}


/////////////////////////////////////////////////////////////////////////////
bool DsControllerTegra30::HandleCmplCodeShortPkt(EventTrb *pEvent,Endpoint *pEndpoint) NN_NOEXCEPT
{
    uint8_t epIndex = NN_USB_DEFINE_GET_FIELD_VALUE(EventTrbDw3_EndpointId, pEvent->eventTrbDw3);

    if (PrimeNotRcvdWar)
    {
        uint32_t temp;
        temp = NN_USB_DEFINE_SET_BIT(epIndex);

        // When the stream is being rejected by the Host and then
        // a valid packet is exchanged on this pipe, we clear the
        // stream rejected condition and cancel any pending work that
        // was scheduled to retry the stream
        if (m_StreamRejected & temp)
        {
            m_StreamRejected &= ~temp;

            // TODO: cancel stream rejected work?

            pEndpoint->streamRejectedRetryCount = 0;
        }
    }

    if (UsbEndpointIsHostToDevice(&pEndpoint->usbEndpointDescriptor))
    {
        Trb *pTrb = TranTrbDmaToVirt(pEndpoint, (uint64_t)pEvent->trbPointerLo + ((uint64_t)(pEvent->trbPointerHi) << 32));

        if (pTrb)
        {
            if (TrbReadIoc(pTrb))
            {
                uint32_t bytesRemaining = 0;

                // This means the bytesRemaining was computerd on TRB that shorted
                // prior to TRB carrying IOC
                if (pEndpoint->trbCountForShortTransfer)
                {
                    bytesRemaining = pEndpoint->bytesRemainingForShortTransfer;
                }
                else // Same TRB shorted and carry IOC
                {
                    // The EvetTrb has remainder for TRB
                    bytesRemaining = TrbReadTransferLen(reinterpret_cast<Trb*>(pEvent)) & 0x1ffff;
                }

                protocol.CompleteBuffer(pEndpoint->usbEndpointDescriptor.bEndpointAddress, ResultSuccess(), bytesRemaining, true);
                pEndpoint->trbCountForShortTransfer = 0;
            }
            else
            {
                // The EvetTrb has remainder for TRB
                pEndpoint->bytesRemainingForShortTransfer = TrbReadTransferLen(reinterpret_cast<Trb*>(pEvent)) & 0x1ffff;

                // Trace the rest of TRB chain to gather bytesRemaining
                while (TrbReadIoc(pTrb) == 0)
                {
                    pEndpoint->trbCountForShortTransfer++;
                    pTrb++;

                    if (TrbReadType(pTrb) == TrbType_Link)
                    {
                        pTrb = pEndpoint->pFirstTransferTrb;
                    }

                    pEndpoint->bytesRemainingForShortTransfer += TrbReadTransferLen(pTrb) & 0x1ffff;
                }
            }
        }
    }
    else
    {
        protocol.CompleteBuffer(pEndpoint->usbEndpointDescriptor.bEndpointAddress, ResultSuccess(), 0, true);
    }

    return true;
}


/////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::HandleCmplCodeHostRejected(EventTrb *pEvent,Endpoint *pEndpoint) NN_NOEXCEPT
{
    uint8_t epIndex = NN_USB_DEFINE_GET_FIELD_VALUE(EventTrbDw3_EndpointId, pEvent->eventTrbDw3);
    uint32_t temp;
    NN_USB_LOG_ERROR("%s(@%d)TrbCmplCode_HostRejected\n",__FUNCTION__,__LINE__);

    // bug 1285731, 1376479
    // PCD driver needs to block the ERDY until Prime Pipe Received. But Since currently the Host is not sending
    // a PRIME packet when the stream is rejected, we are retrying the ERDY till some time before stopping and waiting
    // for the PRIME
    temp = NN_USB_DEFINE_SET_BIT(epIndex);

    if (PrimeNotRcvdWar)
    {
        if ((m_StreamRejected & temp) && (pEndpoint->streamRejectedRetryCount >= MaxStreamRejectedRetryCount))
        {
            // We have tried retrying the stream for some time
            // but the Host is not responding. So we stop
            // retrying now and wait for the PRIME
            pEndpoint->streamRejectedRetryCount = 0;
            NN_USB_LOG_ERROR("%s(@%d)Stream retry Limit reached\n",__FUNCTION__,__LINE__);
        }
        else
        {
            m_StreamRejected |= temp;
            pEndpoint->streamRejectedRetryCount++;
            NN_USB_LOG_ERROR("%s(@%d)Applied WAR for stream rejected work\n",__FUNCTION__,__LINE__);
            RetryStreamRejectedWork(pEndpoint); // TODO: create a separate thread
        }
    }
    else
    {
        m_StreamRejected |= temp;
    }
}


/////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::HandleCmplCodePrimePipeReceived(EventTrb *pEvent,Endpoint *pEndpoint) NN_NOEXCEPT
{
    uint8_t epIndex = NN_USB_DEFINE_GET_FIELD_VALUE(EventTrbDw3_EndpointId, pEvent->eventTrbDw3);
    uint32_t temp;
    NN_USB_LOG_ERROR("%s(@%d)HandleCmplCodePrimePipeReceived\n",__FUNCTION__,__LINE__);

    temp = NN_USB_DEFINE_SET_BIT(epIndex);

    if (m_StreamRejected & temp)
    {
        NN_USB_LOG_ERROR("%s(@%d)m_StreamRejected\n",__FUNCTION__,__LINE__);
        m_StreamRejected &= ~temp;
        if (PrimeNotRcvdWar)
        {
            pEndpoint->streamRejectedRetryCount = 0;
            // TODO: cancel stream rejected work
        }
        //
        // When a stream is rejected for particular ep, its
        // corresponding bit is set in the EP_STOPPED register
        // The bit has to be cleared by writing 1 to it before
        // sending an ERDY on the endpoint.
        //
        PollEpStopped("stream_rejected", temp);
        XusbDeviceBaseRegisters.Write32(XusbDevCtrlDevEpStoppedOffset, temp);
    }

    // re-ring the door bell if ERDY is pending
    // @IMPLEMENT Do we need to do this if nothing is queued?
    temp = pEndpoint->dci;
    temp = NN_USB_DEFINE_DB_TARGET(temp);
    XusbDeviceBaseRegisters.Write32(XusbDevCtrlDoorbellBaseOffset, temp);
    NN_USB_LOG_ERROR("%s(@%d)DOORBELL STREAM = 0x%x\n",__FUNCTION__,__LINE__,temp);
}


/////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::HandleCmplCodeCtrlSeqnumErr(EventTrb *pEvent,Endpoint *pEndpoint) NN_NOEXCEPT
{
    NN_UNUSED(pEvent);

    NN_USB_LOG_ERROR("%s(@%d)TrbCmplCode_CtrlSeqnumErr\n",__FUNCTION__,__LINE__);

    // skip seqnum err event until last one arrives.
    if (pEndpoint->pDeqPtrTransferTrb == pEndpoint->pEnqPtrTransferTrb)
    {
        m_SetupStatus = WaitForSetup;
        NN_USB_LOG_ERROR("%s(@%d)m_SetupStatus --> WaitForSetup \n",__FUNCTION__,__LINE__);
    }
    else
    {
        NN_USB_LOG_ERROR("%s(@%d)seqNum err skipped: pDeqPtrTransferTrb != pEnqPtrTransferTrb: 0x%x, 0x%x\n"
                        ,__FUNCTION__,__LINE__,pEndpoint->pDeqPtrTransferTrb,pEndpoint->pEnqPtrTransferTrb);
    }
}


/////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::HandleEp0StatusComplete(Endpoint *pEndpoint) NN_NOEXCEPT
{
    NN_UNUSED(pEndpoint);

    switch (m_SetupStatus)
    {
        case DataStageTransferToHost:

            m_SetupStatus = StatusStageReceivedFromHost;
            protocol.CompleteBuffer(UsbEndpointAddressMask_DirDeviceToHost,ResultSuccess(), 0, true);
            m_DataTransferComplete = true;

            break;

        case DataStageReceivedFromHost:

            m_SetupStatus = StatusStageTransferToHost;
            Ep0Status();
            protocol.CompleteBuffer(UsbEndpointAddressMask_DirHostToDevice,ResultSuccess(), 0, true);

            break;

        case StatusStageTransferToHost:

            protocol.CompleteBuffer(UsbEndpointAddressMask_DirDeviceToHost,ResultSuccess(), 0, true);
            m_SetupStatus = WaitForSetup;

            break;

        case StatusStageReceivedFromHost:

            protocol.CompleteBuffer(UsbEndpointAddressMask_DirHostToDevice,ResultSuccess(), 0, true);
            m_SetupStatus = WaitForSetup;

            break;

        case WaitForSetup:

            protocol.CompleteBuffer(UsbEndpointAddressMask_DirHostToDevice,ResultSuccess(), 0, true);
            m_SetupStatus = WaitForSetup;

            break;

        default:

            NN_USB_LOG_ERROR("%s(@%d)unknown state\n", __FUNCTION__, __LINE__);

            break;
    }
}


/////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::Ep0Status() NN_NOEXCEPT
{
    Endpoint *pEndpoint = &m_EndpointArray[UsbEndpointControlIndex];
    Trb *pEnqPtrTransferTrb = pEndpoint->pEnqPtrTransferTrb;
    uint32_t temp;

    // ACK
    memset(pEnqPtrTransferTrb, 0, sizeof(Trb));

    TrbWriteDataPtr(pEnqPtrTransferTrb, 0LL);
    TrbWriteCycle(pEnqPtrTransferTrb, pEndpoint->pcs);
    TrbWriteIoc(pEnqPtrTransferTrb, 1);
    TrbWriteType(pEnqPtrTransferTrb, TrbType_Transfer_Status_Stage);
    TrbWriteDataStageDir(pEnqPtrTransferTrb, m_SetupStatus == StatusStageTransferToHost ? 1 : 0);

    // Inc Trb
    pEnqPtrTransferTrb++;

    pEndpoint->pEnqPtrTransferTrb = CheckEndofSegment(pEndpoint,pEnqPtrTransferTrb);

    // Note: for ep0, streamid field is also used for seqnum.
    temp = 0;
    temp |= NN_USB_DEFINE_DB_STREAMID(m_CtrlSeqNum);

    // DB should be zero
    XusbDeviceBaseRegisters.Write32(XusbDevCtrlDoorbellBaseOffset, temp);
}


//////////////////////////////////////////////////////////////////////////////
Result DsControllerTegra30::InitializeEndpoint(Endpoint *pEndpoint) NN_NOEXCEPT
{
    NN_USB_TRACE;

    UsbEndpointDescriptor *pUsbEndpointDescriptor = &pEndpoint->usbEndpointDescriptor;

    if  (pUsbEndpointDescriptor->bDescriptorType != UsbDescriptorType_Endpoint)
    {
        NN_USB_LOG_ERROR("%s(@%d)Invalid descriptor\n",__FUNCTION__,__LINE__);
        return ResultInvalidDescriptor();
    }

    EpContext *pEpContext = (EpContext *) m_pEpContext + pEndpoint->dci;

    if (EpContextReadState(pEpContext) != EndpointState_Disabled)
    {
        EpContextWriteState(pEpContext, EndpointState_Disabled);
    }

    EndpointInitializeTransferRing(pEndpoint);

    pEndpoint->trbCountForShortTransfer         = 0;
    pEndpoint->bytesRemainingForShortTransfer   = 0;

    // Setup EP state
    EpReload(pEndpoint->dci);
    EpUnpause(pEndpoint->dci);
    EpUnhalt(pEndpoint->dci);

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::FinalizeEpContext(Endpoint *pEndpoint) NN_NOEXCEPT
{
    NN_USB_TRACE;

    EpContext *pEpContext = (EpContext *)m_pEpContext + pEndpoint->dci;

    EpContextWriteState(pEpContext, EndpointState_Disabled);
    EpReload(pEndpoint->dci);

    memset(&pEndpoint->usbEndpointDescriptor, 0, sizeof(UsbEndpointDescriptor));
    memset(&pEndpoint->usbEndpointCompanionDescriptor, 0, sizeof(UsbEndpointCompanionDescriptor));
    memset(pEpContext, 0, sizeof(EpContext));

    EpUnpause(pEndpoint->dci);
    EpUnhalt(pEndpoint->dci);
    EpClearStop(pEndpoint->dci);
}


//////////////////////////////////////////////////////////////////////////////
Result DsControllerTegra30::Ringdoorbell(Endpoint *pEndpoint) NN_NOEXCEPT
{
    // TODO check to see if queue is empty. if so, then return

    Result result = ResultSuccess();

    uint32_t regData = NN_USB_DEFINE_DB_TARGET(pEndpoint->dci);

    if (pEndpoint->usbEndpointCompanionDescriptor.bLength && UsbSsMaxStreams(&pEndpoint->usbEndpointCompanionDescriptor))
    {
        if (m_StreamRejected & NN_USB_DEFINE_SET_BIT(pEndpoint->dci))
        {
            return ResultOperationDenied();
        }

        // TODO: Provide STREAM ID. This must come from layer above
        //regData |= DB_STREAMID(udc_req_ptr->usb_req.stream_id);
    }

    XusbDeviceBaseRegisters.Write32(XusbDevCtrlDoorbellBaseOffset, regData);

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result DsControllerTegra30::QueueTrbs(Endpoint *pEndpoint, uint32_t transferRingSize, uint32_t numTrbsNeeded, uint64_t transferData, uint64_t BufferLength) NN_NOEXCEPT
{
    NN_USB_TRACE;

    Trb *pTransferRing = pEndpoint->pFirstTransferTrb, *pEnqPtrTransferTrb = pEndpoint->pEnqPtrTransferTrb;
    uint32_t numTrbsAva = 0, i, j = 1, totalTrbsToUse, intrRate = 5; // BULK_EP_INTERRUPT_RATE;
    uint64_t bufLenTemp = 0;
    uint8_t trbsUsed, ChainBit = 1, shortPkt = 0, intrOnCompl = 0;
    const uint8_t trbType = TrbType_Transfer_Normal;
    bool fullTd = true;
    const bool needZlp = false;
    Result result = ResultSuccess();

    trbsUsed = numTrbsNeeded;
    numTrbsAva = RoomOnRing(transferRingSize, pTransferRing, pEndpoint->pEnqPtrTransferTrb, pEndpoint->pDeqPtrTransferTrb);

    // DMA is the starting address of the USB request
    // points to the addr of the buffer that we write in each TRB.
    if (numTrbsAva >= numTrbsNeeded + (needZlp ? 1 : 0))
    {
        totalTrbsToUse = numTrbsNeeded;
    }
    else
    {
        // always keep one trb for zlp.
        totalTrbsToUse = numTrbsAva - (needZlp ? 1 : 0);
        fullTd = false;
        NN_USB_LOG_INFO("%s(@%d)TRB Ring Full. Avail: %d Required: %d\n",__FUNCTION__,__LINE__,numTrbsAva,numTrbsNeeded);
        pEndpoint->transferRingFull = true;
    }

    for (i = 0; i < totalTrbsToUse; i++)
    {
        if (BufferLength > TrbMaxBufferSize) // Transfer up to TrbMaxBufferSize per iteration
        {
            bufLenTemp = TrbMaxBufferSize;
        }
        else
        {
            bufLenTemp = BufferLength; // single transfers or the last transfer
        }

        BufferLength -= bufLenTemp; // update the transfer lenght that we are scheduling

        if (UsbEndpointIsHostToDevice(&pEndpoint->usbEndpointDescriptor))
        {
            shortPkt = 1; // all Bulk OUT transfers are sent as "short" packets
        }

        if (BufferLength == 0) // last transfer
        {
            ChainBit = 0;
            intrOnCompl = 1; // Generate an interrupt upon the completion of the last transfer
        }

        if  ((!fullTd) && (j == intrRate))
        {
            intrOnCompl = 1;
            NN_USB_LOG_INFO("%s(@%d)!fullTd intrOnCompl = 1\n",__FUNCTION__,__LINE__);
            j = 0;
        }

        const size_t NumTrbsToStart = trbsUsed - 1;

        // TODO: upper layer provides StreamId?

        // Start TRB
        memset(pEnqPtrTransferTrb, 0, sizeof(Trb));

        TrbWriteDataPtr(pEnqPtrTransferTrb, transferData);
        TrbWriteTransferLen(pEnqPtrTransferTrb, bufLenTemp);
        TrbWriteTdSize(pEnqPtrTransferTrb, NumTrbsToStart);
        TrbWriteCycle(pEnqPtrTransferTrb, pEndpoint->pcs);
        TrbWriteIsp(pEnqPtrTransferTrb, shortPkt);
        TrbWriteChain(pEnqPtrTransferTrb, ChainBit);
        TrbWriteIoc(pEnqPtrTransferTrb, intrOnCompl);
        TrbWriteType(pEnqPtrTransferTrb, trbType);
        TrbWriteStreamId(pEnqPtrTransferTrb, 0);

        transferData += bufLenTemp;

        trbsUsed--;
        pEnqPtrTransferTrb++;
        j++;

        pEnqPtrTransferTrb = CheckEndofSegment(pEndpoint,pEnqPtrTransferTrb);

        // See if TRB buffer wrapped
        if (pEnqPtrTransferTrb == pEndpoint->pFirstTransferTrb)
        {
            if (trbsUsed) // more TRBs, assert chain bit
            {
                TrbWriteChain(pEndpoint->pLinkTrb, 1);
            }
            else
            {
                TrbWriteChain(pEndpoint->pLinkTrb, 0);
            }
        }
    }

    pEndpoint->pEnqPtrTransferTrb = pEnqPtrTransferTrb;
    result = Ringdoorbell(pEndpoint);
    return result;
}


//////////////////////////////////////////////////////////////////////////////
uint32_t DsControllerTegra30::RoomOnRing(uint32_t transferRingSize, Trb *pTransferRing, Trb *pEnqPtr, Trb *dQptr) NN_NOEXCEPT
{
    uint32_t i = 0;

    if (pEnqPtr == dQptr)
    {
        return transferRingSize - 1;
    }

    while (pEnqPtr != dQptr)
    {
        i++;
        pEnqPtr++;

        if (TrbReadType(pEnqPtr) == TrbType_Link)
        {
            pEnqPtr = pTransferRing;
        }

        if (i > transferRingSize)
            break;
    }

    return i - 1;
}


//////////////////////////////////////////////////////////////////////////////
inline int DsControllerTegra30::UsbSsMaxStreams(const UsbEndpointCompanionDescriptor *pCompEpDescriptor) NN_NOEXCEPT
{
    NN_USB_TRACE;
    // Bits 4:0 of bmAttributes if this is a bulk endpoint

    int maxStreams;

    if (!pCompEpDescriptor)
        return 0;

    maxStreams = pCompEpDescriptor->bmAttributes & 0x1f;

    if (!maxStreams)
        return 0;

    maxStreams = 1 << maxStreams;

    return maxStreams;
}


//////////////////////////////////////////////////////////////////////////////
Trb* DsControllerTegra30::CheckEndofSegment(Endpoint *pEndpoint, Trb *pTrb) NN_NOEXCEPT
{
    // check if we are at end of trb segment.  If so, update pcs and enq for next segment
    Trb *pTrbTemp = nullptr;

    if (TrbReadType(pTrb) == TrbType_Link)
    {
        if (TrbReadToggleCycle(pTrb))
        {
            TrbWriteCycle(pTrb, pEndpoint->pcs);
            pEndpoint->pcs ^= 0x1;
        }

        pTrbTemp = pEndpoint->pFirstTransferTrb;
    }
    else
    {
        // not at the end of segment
        pTrbTemp = pTrb;
    }

    return pTrbTemp;
}


//////////////////////////////////////////////////////////////////////////////
// ep utilities
//////////////////////////////////////////////////////////////////////////////
uint8_t DsControllerTegra30::EndpointAttributes(Endpoint *pEndpoint)
{
    return pEndpoint->usbEndpointDescriptor.bmAttributes;
}


//////////////////////////////////////////////////////////////////////////////
uint8_t DsControllerTegra30::EndpointDirection(Endpoint *pEndpoint)
{
    return pEndpoint->usbEndpointDescriptor.bEndpointAddress & UsbEndpointAddressMask::UsbEndpointAddressMask_Dir;
}


//////////////////////////////////////////////////////////////////////////////
uint8_t DsControllerTegra30::EndpointType(Endpoint *pEndpoint)
{
    return EndpointAttributes(pEndpoint) & 3;
}


//////////////////////////////////////////////////////////////////////////////
uint16_t DsControllerTegra30::EndpointMaxPacketSize(Endpoint *pEndpoint)
{
    return pEndpoint->usbEndpointDescriptor.wMaxPacketSize;
}


//////////////////////////////////////////////////////////////////////////////
uint8_t DsControllerTegra30::EndpointInterval(Endpoint *pEndpoint)
{
    return pEndpoint->usbEndpointDescriptor.bInterval;
}


//////////////////////////////////////////////////////////////////////////////
uint8_t DsControllerTegra30::EndpointCompMaxBurst(Endpoint *pEndpoint)
{
    return pEndpoint->usbEndpointCompanionDescriptor.bMaxBurst;
}


//////////////////////////////////////////////////////////////////////////////
uint8_t DsControllerTegra30::EndpointCompAttributes(Endpoint *pEndpoint)
{
    return pEndpoint->usbEndpointCompanionDescriptor.bmAttributes;
}


//////////////////////////////////////////////////////////////////////////////
uint16_t DsControllerTegra30::EndpointCompBytesPerInterval(Endpoint *pEndpoint)
{
    return pEndpoint->usbEndpointCompanionDescriptor.wBytesPerInterval;
}


//////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::EndpointInitializeTransferRing(Endpoint *pEndpoint)
{
    if (pEndpoint->pTransferRingBuffer)
    {
        pEndpoint->pFirstTransferTrb = reinterpret_cast<Trb *>(pEndpoint->pTransferRingBuffer);

        int32_t transferRingSize = 0;

        switch (EndpointType(pEndpoint))
        {
        case UsbEndpointAttributeMask::UsbEndpointAttributeMask_XferTypeControl:

            transferRingSize = UsbXusbDevControlTransferRingSize;
            memset(pEndpoint->pFirstTransferTrb, 0, UsbXusbDevControlTransferRingSize);

            break;

        case UsbEndpointAttributeMask::UsbEndpointAttributeMask_XferTypeBulk:

            transferRingSize = UsbXusbDevBulkTransferRingSize;
            memset(pEndpoint->pFirstTransferTrb, 0, UsbXusbDevMaxRingSize);

            break;

        case UsbEndpointAttributeMask::UsbEndpointAttributeMask_XferTypeInt:

            transferRingSize = UsbXusbDevIntTransferRingSize;
            memset(pEndpoint->pFirstTransferTrb, 0, UsbXusbDevMaxRingSize);

            break;

        default:

            break;
        }

        pEndpoint->pEnqPtrTransferTrb   = pEndpoint->pFirstTransferTrb;
        pEndpoint->pDeqPtrTransferTrb   = pEndpoint->pFirstTransferTrb;
        pEndpoint->pcs                  = true;
        pEndpoint->transferRingFull     = false;

        InitializeEpContext(pEndpoint);

        // Setup transfer TRB
        pEndpoint->pLinkTrb = pEndpoint->pFirstTransferTrb + transferRingSize - 1;

        memset(pEndpoint->pLinkTrb, 0, sizeof(Trb));

        TrbWriteDataPtr(pEndpoint->pLinkTrb, pEndpoint->transferRingIoVa);
        TrbWriteType(pEndpoint->pLinkTrb, TrbType_Link);
        TrbWriteToggleCycle(pEndpoint->pLinkTrb, true);
    }
}


//////////////////////////////////////////////////////////////////////////////
// ep primitives
//////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::EpReload(int epIndex) NN_NOEXCEPT
{
    XusbDeviceBaseRegisters.Write32(XusbDevCtrlEpReloadOffset, NN_USB_DEFINE_SET_BIT(epIndex));
    ReadPoll32(XusbDevCtrlEpReloadOffset, NN_USB_DEFINE_SET_BIT(epIndex), 0);
}


//////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::EpPause(int epIndex) NN_NOEXCEPT
{
    uint32_t epPause = XusbDeviceBaseRegisters.Read32(XusbDevCtrlEpPauseOffset);

    if (epPause & NN_USB_DEFINE_SET_BIT(epIndex))
    {
        return;
    }

    epPause |= NN_USB_DEFINE_SET_BIT(epIndex);
    XusbDeviceBaseRegisters.Write32(XusbDevCtrlEpPauseOffset, epPause);
    ReadPoll32(XusbDevCtrlEpStchgOffset, NN_USB_DEFINE_SET_BIT(epIndex), NN_USB_DEFINE_SET_BIT(epIndex));
    XusbDeviceBaseRegisters.Write32(XusbDevCtrlEpStchgOffset, NN_USB_DEFINE_SET_BIT(epIndex));
}


//////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::EpUnpause(int epIndex) NN_NOEXCEPT
{
    uint32_t epPause = XusbDeviceBaseRegisters.Read32(XusbDevCtrlEpPauseOffset);

    if (!(epPause & NN_USB_DEFINE_SET_BIT(epIndex)))
    {
        return;
    }

    epPause &= ~NN_USB_DEFINE_SET_BIT(epIndex);
    XusbDeviceBaseRegisters.Write32(XusbDevCtrlEpPauseOffset, epPause);
    ReadPoll32(XusbDevCtrlEpStchgOffset, NN_USB_DEFINE_SET_BIT(epIndex), NN_USB_DEFINE_SET_BIT(epIndex));
    XusbDeviceBaseRegisters.Write32(XusbDevCtrlEpStchgOffset, NN_USB_DEFINE_SET_BIT(epIndex));
}

//////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::EpHalt(int epIndex) NN_NOEXCEPT
{
    uint32_t epHalt = XusbDeviceBaseRegisters.Read32(XusbDevCtrlEpHaltOffset);

    if (epHalt & NN_USB_DEFINE_SET_BIT(epIndex))
    {
        return;
    }

    epHalt |= NN_USB_DEFINE_SET_BIT(epIndex);
    XusbDeviceBaseRegisters.Write32(XusbDevCtrlEpHaltOffset, epHalt);
    ReadPoll32(XusbDevCtrlEpStchgOffset, NN_USB_DEFINE_SET_BIT(epIndex), NN_USB_DEFINE_SET_BIT(epIndex));
    XusbDeviceBaseRegisters.Write32(XusbDevCtrlEpStchgOffset, NN_USB_DEFINE_SET_BIT(epIndex));
}


//////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::EpUnhalt(int epIndex) NN_NOEXCEPT
{
    uint32_t epHalt = XusbDeviceBaseRegisters.Read32(XusbDevCtrlEpHaltOffset);

    if (!(epHalt & NN_USB_DEFINE_SET_BIT(epIndex)))
    {
        return;
    }

    epHalt &= ~NN_USB_DEFINE_SET_BIT(epIndex);
    XusbDeviceBaseRegisters.Write32(XusbDevCtrlEpHaltOffset, epHalt);
    ReadPoll32(XusbDevCtrlEpStchgOffset, NN_USB_DEFINE_SET_BIT(epIndex), NN_USB_DEFINE_SET_BIT(epIndex));
    XusbDeviceBaseRegisters.Write32(XusbDevCtrlEpStchgOffset, NN_USB_DEFINE_SET_BIT(epIndex));
}


//////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::EpClearStop(int epIndex) NN_NOEXCEPT
{
    if (XusbDeviceBaseRegisters.Read32(XusbDevCtrlDevEpStoppedOffset) & NN_USB_DEFINE_SET_BIT(epIndex))
    {
        XusbDeviceBaseRegisters.Write32(XusbDevCtrlDevEpStoppedOffset, NN_USB_DEFINE_SET_BIT(epIndex));
    }
}


//////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::EpWaitForStopped(int epIndex) NN_NOEXCEPT
{
    ReadPoll32(XusbDevCtrlDevEpStoppedOffset, NN_USB_DEFINE_SET_BIT(epIndex), NN_USB_DEFINE_SET_BIT(epIndex));
    XusbDeviceBaseRegisters.Write32(XusbDevCtrlDevEpStoppedOffset, NN_USB_DEFINE_SET_BIT(epIndex));
}


//////////////////////////////////////////////////////////////////////////////
void DsControllerTegra30::EpWaitForInactive(int epIndex) NN_NOEXCEPT
{
    ReadPoll32(XusbDevCtrlDevEpThreadActiveOffset, NN_USB_DEFINE_SET_BIT(epIndex), 0);
}


} // end of namespace ds
} // end of namespace usb
} // end of namespace nn
