﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_SystemThreadDefinition.h>
#include <nn/os/os_SdkThreadCommon.h>
#include <nn/hid/system/hid_InputDetection.h>

#include "usb_InputNotifier.h"
#include "usb_Util.h"

namespace nn {
namespace usb {
namespace detail {

Result InputNotifier::Initialize() NN_NOEXCEPT
{
    // create and start input notifier thread
    nn::os::CreateThread(&m_InputNotifierThread,
                         InputNotifierThreadEntry,
                         this,
                         m_InputNotifierThreadStack,
                         sizeof(m_InputNotifierThreadStack),
                         NN_SYSTEM_THREAD_PRIORITY(usb, InputNotifier));
    nn::os::SetThreadNamePointer(&m_InputNotifierThread, NN_SYSTEM_THREAD_NAME(usb, InputNotifier));
    nn::os::StartThread(&m_InputNotifierThread);

    return ResultSuccess();
}

Result InputNotifier::Finalize() NN_NOEXCEPT
{
    // terminate input notifier thread
    m_BreakEvent.Signal();
    nn::os::WaitThread(&m_InputNotifierThread);
    nn::os::DestroyThread(&m_InputNotifierThread);

    return ResultSuccess();
}

void InputNotifier::Notify() NN_NOEXCEPT
{
    m_InputNotifyEvent.Signal();
}

void InputNotifier::InputNotifierThreadLoop() NN_NOEXCEPT
{
    bool running = true;
    while (running)
    {
        switch (nn::os::WaitAny(m_BreakEvent.GetBase(), m_InputNotifyEvent.GetBase()))
        {
        case 0:
            m_BreakEvent.Clear();
            running = false;
            break;

        case 1:
            m_InputNotifyEvent.Clear();
            // Notify input detector, which causes idle time counter reset
            nn::hid::system::NotifyInputDetector(nn::hid::system::InputSourceId::UsbPort::Mask);
            NN_USB_LOG_INFO("Notified hid input detector\n");
            break;

        default:
            NN_UNEXPECTED_DEFAULT;
            break;
        }
    }
}

} // end of namespace detail
} // end of namespace usb
} // end of namespace nn
