﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "usb_ComplexTegra21x.h"
#include "usb_Platform.h"
#include <nn/pcv/pcv.h>

namespace nn {
namespace usb {
namespace detail {

Result UsbComplexTegra21x::Initialize(UsbPlatform *pPlatform) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    UsbController::Config config;

    NN_USB_ABORT_UPON_ERROR(UsbComplex::Initialize(pPlatform));

    // GPIO
    nn::gpio::Initialize();

    // Note that SMMU address space is specific to each device. In other words, no these
    // are not overlapping.
    NN_USB_ABORT_UPON_ERROR(
        m_SmmuXusbHost.Initialize(nn::dd::DeviceName::DeviceName_XusbHost,
                                  0x80000000ULL,
                                  0x70000000)
    );
    NN_USB_ABORT_UPON_ERROR(
        m_SmmuXusbDevice.Initialize(nn::dd::DeviceName::DeviceName_XusbDev,
                                    0x80000000ULL,
                                    0x70000000)
    );

    // registers
    XusbPadctlRegister.Initialize(XusbPadctlBaseAddress, XusbPadctlRegionSize);

    // FUSE
    RegisterBlock fuse;
    fuse.Initialize(FuseBaseAddress, FuseRegionSize);
    m_FuseUsbCalib0    = fuse.Read32(FuseUsbCalib0Offset);
    m_FuseUsbCalibExt0 = fuse.Read32(FuseUsbCalibExt0Offset);
    fuse.Finalize();

    /*
     * Following shared resources are controlled by Eris PCIe:
     *   - 1.05V rail
     *   - PLLe
     *   - PADs for SuperSpeed ports
     * Wait for PCIe to make sure those are properly initialized.
     *
     * PSC cannot help us upon cold boot.
     */
    WaitForPcie();

    // host XHCI UsbController
    memset(&config, 0, sizeof(config));
    config.reg.basePhysicalAddress   = XusbHostBaseAddress;
    config.reg.size                  = XusbHostRegionSize;
    config.pSmmu                     = &m_SmmuXusbHost;
    config.pName                     = "Host-Xusb";
    NN_USB_ABORT_UPON_ERROR(AddController(&m_HostControllerXusb, &config));

    // XUSB Device Controller
    memset(&config, 0, sizeof(config));
    config.reg.basePhysicalAddress   = XusbDevBaseAddress;
    config.reg.size                  = XusbDevRegionSize;
    config.pSmmu                     = &m_SmmuXusbDevice;
    config.pName                     = "Device-Xusb";
    NN_USB_ABORT_UPON_ERROR(AddController(&m_DeviceControllerXusb, &config)); // m_DeviceController30

    return result;
}

Result UsbComplexTegra21x::Finalize() NN_NOEXCEPT
{
    Result result = ResultSuccess();

    // controllers
    NN_USB_ABORT_UPON_ERROR(DelController(&m_DeviceControllerXusb));
    NN_USB_ABORT_UPON_ERROR(DelController(&m_HostControllerXusb));

    // sMMU
    NN_USB_ABORT_UPON_ERROR(m_SmmuXusbHost.Finalize());
    NN_USB_ABORT_UPON_ERROR(m_SmmuXusbDevice.Finalize());

    NN_USB_ABORT_UPON_ERROR(UsbComplex::Finalize());

    XusbPadctlRegister.Finalize();

    // GPIO
    nn::gpio::Finalize();

    return result;
}

Result UsbComplexTegra21x::Enable()
{
    Result result = ResultSuccess();

    /*
     * USB2 PAD MUX: All PADs (BIAS, port 1~4) are assigned to XUSB
     *
     * Note Mariko and TX1 have different reset defaults. Mariko removed
     * SNPS controller so port 1 is assigned to XUSB by default, while
     * on TX1 it's assigned to SNPS.
     *
     * This has to be set after cold boot AND sleep/wake since they both
     * reset it to default.
     */
    XusbPadctlRegister.Write32(XusbPadctlUsb2PadMuxOffset, 0x00040055);

    XusbSsPadctlSetWakeDetect(0xf, false);

    // Perform USB 2.0 pad tracking
    XusbPadctlSetUtmiPhyEnable(true);

    return result;
}

Result UsbComplexTegra21x::Disable()
{
    Result result = ResultSuccess();

    XusbPadctlSetUtmiPhyEnable(false);

    return result;
}

Result UsbComplexTegra21x::CreatePort(uint32_t hsLane, uint32_t ssLane, uint32_t capability)
{
    Result result = ResultSuccess();

    // Lane# must be in valid range
    if (hsLane >= HighSpeedPortCount || ssLane >= SuperSpeedPortCount)
    {
        return ResultInvalidParameter();
    }

    // Cannot be both Host and Device at the same time
    if ((capability & UsbCapability_Host) && (capability & UsbCapability_Device))
    {
        return ResultInvalidParameter();
    }

    if (capability & UsbCapability_Host)
    {
        XusbPadctlHsInit(hsLane, PadMode_Host);

        if (capability & UsbCapability_SuperSpeed)
        {
            XusbPadctlSsInit(ssLane, PadMode_Host);
            XusbPadctlEnableSsPhy((ssLane == 0) ? SsPhyPad_PexP6 : SsPhyPad_PexP5);
        }

        NN_USB_RETURN_UPON_ERROR(m_HostControllerXusb.Enable());
    }
    else if (capability & UsbCapability_Device)
    {
        // Only USB2 P0 can be put into device mode
        if (hsLane != 0)
        {
            return ResultInvalidParameter();
        }

        // Use local override for VBUS status reporting. Required for RxDetect.
        XusbPadctlRegister.SetField32(
            XusbPadctlUsb2VbusId0Offset,
            XusbPadctlUsb2VbusId0VbusOverrideMask
        );

        XusbPadctlHsInit(hsLane, PadMode_Device);

        if (capability & UsbCapability_SuperSpeed)
        {
            XusbPadctlSsInit(ssLane, PadMode_Device);

            // Begin SS device mode workaround NSBG-7670
            if(ssLane == 0)
            {
               XusbPadctlSsInit(1, PadMode_Host);
               XusbPadctlSsInit(2, PadMode_Host);
               XusbPadctlSsInit(3, PadMode_Host);
            }
            if(hsLane == 0)
            {
               XusbPadctlHsInit(1, PadMode_Host);
               XusbPadctlHsInit(2, PadMode_Host);
               XusbPadctlHsInit(3, PadMode_Host);
            }
            // End SS Device mode workaround NSBG-7670

            XusbPadctlEnableSsPhy((ssLane == 0) ? SsPhyPad_PexP6 : SsPhyPad_PexP5);
        }

        NN_USB_RETURN_UPON_ERROR(m_DeviceControllerXusb.Enable());
    }
    else
    {
        return ResultInvalidParameter();
    }

    return result;
}

Result UsbComplexTegra21x::DestroyPort(uint32_t hsLane, uint32_t ssLane, uint32_t capability)
{
    Result result = ResultSuccess();

    // Lane# must be in valid range
    if (hsLane >= HighSpeedPortCount || ssLane >= SuperSpeedPortCount)
    {
        return ResultInvalidParameter();
    }

    // Cannot be both Host and Device at the same time
    if ((capability & UsbCapability_Host) && (capability & UsbCapability_Device))
    {
        return ResultInvalidParameter();
    }

    if (capability & UsbCapability_Host)
    {
        XusbPadctlHsInit(hsLane, PadMode_Disabled);

        if (capability & UsbCapability_SuperSpeed)
        {
            XusbPadctlSsInit(ssLane, PadMode_Disabled);
        }

        NN_USB_RETURN_UPON_ERROR(m_HostControllerXusb.Disable());
    }
    else if (capability & UsbCapability_Device)
    {
        // Only USB2 P0 can be put into device mode
        if (hsLane != 0)
        {
            return ResultInvalidParameter();
        }

        XusbPadctlHsInit(hsLane, PadMode_Disabled);

        if (capability & UsbCapability_SuperSpeed)
        {
            XusbPadctlSsInit(ssLane, PadMode_Disabled);

            // Begin SS device mode workaround NSBG-7670
            if(ssLane == 0)
            {
                XusbPadctlSsInit(1, PadMode_Disabled);
                XusbPadctlSsInit(2, PadMode_Disabled);
                XusbPadctlSsInit(3, PadMode_Disabled);
            }
            if(hsLane == 0)
            {
                XusbPadctlHsInit(1, PadMode_Disabled);
                XusbPadctlHsInit(2, PadMode_Disabled);
                XusbPadctlHsInit(3, PadMode_Disabled);
            }
            // End SS device mode workaround NSBG-7670
        }

        NN_USB_RETURN_UPON_ERROR(m_DeviceControllerXusb.Disable());
    }
    else
    {
        return ResultInvalidParameter();
    }

    return result;
}

void UsbComplexTegra21x::OverrideDriveStrength(uint32_t port, int32_t offset,
                                               uint32_t *pOutStrength) NN_NOEXCEPT
{
    int pad = m_pPlatform->GetConfig().port[port].hsLane;

    uint32_t hsCurrLevelMap[] = {
        NN_USB_GET_FIELD32(FuseUsbCalib0HsCurrLevel0, m_FuseUsbCalib0),
        NN_USB_GET_FIELD32(FuseUsbCalib0HsCurrLevel1, m_FuseUsbCalib0),
        NN_USB_GET_FIELD32(FuseUsbCalib0HsCurrLevel2, m_FuseUsbCalib0),
        NN_USB_GET_FIELD32(FuseUsbCalib0HsCurrLevel3, m_FuseUsbCalib0),
    };

    int32_t hsCurrLevel = (int32_t)hsCurrLevelMap[pad] + offset;
    if (hsCurrLevel < 0)
    {
        hsCurrLevel = 0;
    }
    if (hsCurrLevel > 0x3f)
    {
        hsCurrLevel = 0x3f;
    }

    XusbPadctlRegister.SetField32(
        XusbPadctlUsb2OtgPadCtl0Offsets[pad],
        XusbPadctlUsb2OtgPad0Ctl0HsCurrLevelMask,
        hsCurrLevel << XusbPadctlUsb2OtgPad0Ctl0HsCurrLevelShift
    );

    NN_USB_LOG_WARN("Override HS_CURR_LEVEL[%d] to %d (FUSE_USB_CALIB0=0x%08x, offset=%+d)\n",
                    pad, hsCurrLevel, m_FuseUsbCalib0, offset);

    *pOutStrength = hsCurrLevel;
}

void UsbComplexTegra21x::XusbPadctlToggleVbusId0(bool isSet) NN_NOEXCEPT
{
    if (isSet)
    {
        XusbPadctlRegister.SetField32(XusbPadctlUsb2VbusId0Offset,XusbPadctlUsb2VbusId0IdOverrideMask);
    }
    else
    {
        XusbPadctlRegister.ClearField32(XusbPadctlUsb2VbusId0Offset,XusbPadctlUsb2VbusId0IdOverrideMask);
    }
}

void UsbComplexTegra21x::XusbPadctlSetUtmiPhyEnable(bool isEnabled)
{
    // Enable UTMI pad clock
    // TODO

    if(isEnabled)
    {
        // Enable the tracking clocks
        // CLK_RST_CONTROLLER_CLK_ENB_Y_SET_0[SET_CLK_ENB_USB2_TRK] = 1
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::pcv::SetClockEnabled(nn::pcv::Module_Usb2TrkClk, true)
        );

        // Begin with cold reset defaults
        XusbPadctlRegister.Write32(XusbPadctlUsb2BiasPadCtl0Offset, 0x260e0810);
        XusbPadctlRegister.Write32(XusbPadctlUsb2BiasPadCtl1Offset, 0x04820000);

        // Setup the timing parameters
        XusbPadctlRegister.SET_FIELD32_2(XusbPadctlUsb2BiasPadCtl1,
                                         TrkStartTimer,     0x1E,
                                         TrkDoneResetTimer, 0x0A);

        // Power up the pads and the tracking circuit
        // Ref [NSBG-8267] - HsDisconLevel and TermOffset don't use register default
        XusbPadctlRegister.SET_FIELD32_3(XusbPadctlUsb2BiasPadCtl0,
                                         HsDisconLevel, 7,
                                         TermOffset,    GetTermOffset(),
                                         Pd,            0);

        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(1));

        XusbPadctlRegister.SET_FIELD32_1(XusbPadctlUsb2BiasPadCtl1,
                                         PdTrk, 0);

        // Wait for tracking complete
        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(100));

        // Disable the tracking clocks
        // CLK_RST_CONTROLLER_CLK_ENB_Y_CLR_0[CLR_CLK_ENB_USB2_TRK] = 1
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::pcv::SetClockEnabled(nn::pcv::Module_Usb2TrkClk, false)
        );
    }
    else
    {
        XusbPadctlRegister.SET_FIELD32_1(XusbPadctlUsb2BiasPadCtl0,
                                         Pd,    1);
        XusbPadctlRegister.SET_FIELD32_1(XusbPadctlUsb2BiasPadCtl1,
                                         PdTrk, 1);
    }

    // wait for tracking complete
    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(60));

    // utmi_phy_iddq_override
    // TODO
}

void UsbComplexTegra21x::XusbPadctlEnableSsPhy(SsPhyPad ssPhyPad) NN_NOEXCEPT
{
    // Set mux
    XusbPadctlUphyMisc(ssPhyPad, true);
    if(ssPhyPad==SsPhyPad_SataS0)
    {
        XusbPadctlRegister.SetField32(XusbPadctlUsb3PadMuxOffset,
                                      XusbPadctlUsb3PadMuxSataPadLane0Mask,
                                      NN_USB_MAKE_MASK32(XusbPadctlUsb3PadMuxSataPadLane0, 1));
    }
    else
    {
        XusbPadctlRegister.SetField32(XusbPadctlUsb3PadMuxOffset,
                                      XusbPadctlUsb3PadMuxPciePadLane0Mask << (ssPhyPad * 2),
                                      1 << ((ssPhyPad * 2) + XusbPadctlUsb3PadMuxPciePadLane0Shift));
    }
    XusbPadctlUphyMisc(ssPhyPad, false);

    // Bring enabled lane out of IDDQ
    XusbPadctlRegister.SetBit32(XusbPadctlUsb3PadMuxOffset, XusbPadctlUsb3PadMuxPciePadIddqDisableMask0Shift + ssPhyPad);

    // Release pad mux state latching
    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(1));
    XusbPadctlRegister.ClearBit32(XusbPadctlElpgProgram1Offset,
                                  XusbPadctlElpgProgram1AuxMuxLp0ClampEnShift);

    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(100));
    XusbPadctlRegister.ClearBit32(XusbPadctlElpgProgram1Offset,
                                  XusbPadctlElpgProgram1AuxMuxLp0ClampEnEarlyShift);

    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(100));
    XusbPadctlRegister.ClearBit32(XusbPadctlElpgProgram1Offset,
                                  XusbPadctlElpgProgram1Ssp3ElpgVcoreDownShift);
}

void UsbComplexTegra21x::XusbSsPadctlSetWakeDetect(uint32_t ssPortMask, bool isWakeDetectionEnabled) NN_NOEXCEPT
{
    if(isWakeDetectionEnabled)
    {
        /*
         * This code flow enables wake detection, upon entering deep sleep (LP0)
         */
        XusbPadctlRegister.SetField32(XusbPadctlElpgProgram1Offset,
                                      ((ssPortMask & 1) ? XusbPadctlElpgProgram1Ssp0ElpgClampEnEarlyMask : 0) |
                                      ((ssPortMask & 2) ? XusbPadctlElpgProgram1Ssp1ElpgClampEnEarlyMask : 0) |
                                      ((ssPortMask & 4) ? XusbPadctlElpgProgram1Ssp2ElpgClampEnEarlyMask : 0) |
                                      ((ssPortMask & 8) ? XusbPadctlElpgProgram1Ssp3ElpgClampEnEarlyMask : 0));
        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(100));
        XusbPadctlRegister.SetField32(XusbPadctlElpgProgram1Offset,
                                      ((ssPortMask & 1) ? XusbPadctlElpgProgram1Ssp0ElpgClampEnMask : 0) |
                                      ((ssPortMask & 2) ? XusbPadctlElpgProgram1Ssp1ElpgClampEnMask : 0) |
                                      ((ssPortMask & 4) ? XusbPadctlElpgProgram1Ssp2ElpgClampEnMask : 0) |
                                      ((ssPortMask & 8) ? XusbPadctlElpgProgram1Ssp3ElpgClampEnMask : 0));
        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(250));

        XusbPadctlRegister.SetField32(XusbPadctlElpgProgram1Offset,
                                      ((ssPortMask & 1) ? XusbPadctlElpgProgram1Ssp0ElpgVcoreDownMask : 0) |
                                      ((ssPortMask & 2) ? XusbPadctlElpgProgram1Ssp1ElpgVcoreDownMask : 0) |
                                      ((ssPortMask & 4) ? XusbPadctlElpgProgram1Ssp2ElpgVcoreDownMask : 0) |
                                      ((ssPortMask & 8) ? XusbPadctlElpgProgram1Ssp3ElpgVcoreDownMask : 0));
    }
    else
    {
        /*
         * This code flow disables wake detection, upon exiting deep sleep (LP0)
         */
       XusbPadctlRegister.ClearField32(XusbPadctlElpgProgram1Offset,
                                       ((ssPortMask & 1) ? XusbPadctlElpgProgram1Ssp0ElpgClampEnEarlyMask : 0) |
                                       ((ssPortMask & 2) ? XusbPadctlElpgProgram1Ssp1ElpgClampEnEarlyMask : 0) |
                                       ((ssPortMask & 4) ? XusbPadctlElpgProgram1Ssp2ElpgClampEnEarlyMask : 0) |
                                       ((ssPortMask & 8) ? XusbPadctlElpgProgram1Ssp3ElpgClampEnEarlyMask : 0));
       nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(100));

       XusbPadctlRegister.ClearField32(XusbPadctlElpgProgram1Offset,
                                       ((ssPortMask & 1) ? XusbPadctlElpgProgram1Ssp0ElpgClampEnMask : 0) |
                                       ((ssPortMask & 2) ? XusbPadctlElpgProgram1Ssp1ElpgClampEnMask : 0) |
                                       ((ssPortMask & 4) ? XusbPadctlElpgProgram1Ssp2ElpgClampEnMask : 0) |
                                       ((ssPortMask & 8) ? XusbPadctlElpgProgram1Ssp3ElpgClampEnMask : 0));
       nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(100));

       XusbPadctlRegister.ClearField32(XusbPadctlElpgProgram1Offset,
                                        ((ssPortMask & 1) ? XusbPadctlElpgProgram1Ssp0ElpgVcoreDownMask : 0) |
                                        ((ssPortMask & 2) ? XusbPadctlElpgProgram1Ssp1ElpgVcoreDownMask : 0) |
                                        ((ssPortMask & 4) ? XusbPadctlElpgProgram1Ssp2ElpgVcoreDownMask : 0) |
                                        ((ssPortMask & 8) ? XusbPadctlElpgProgram1Ssp3ElpgVcoreDownMask : 0));
       nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(100));
    }
}

void UsbComplexTegra21x::XusbPadctlUphyMisc(SsPhyPad ssPhyPad, bool isOverride) NN_NOEXCEPT
{
    const size_t miscPadCtl2Regs[] = {XusbPadctlUphyMiscPadP0Ctl2Offset, XusbPadctlUphyMiscPadP1Ctl2Offset, XusbPadctlUphyMiscPadP2Ctl2Offset,
                                      XusbPadctlUphyMiscPadP3Ctl2Offset, XusbPadctlUphyMiscPadP4Ctl2Offset, XusbPadctlUphyMiscPadP5Ctl2Offset,
                                      XusbPadctlUphyMiscPadP6Ctl2Offset, XusbPadctlUphyMiscPadS0Ctl2Offset};

    // WAR: Override pad controls and keep UPHy under IDDQ and
    //   SLEEP 3 state during lane ownership changes and powergating
    if (isOverride)
    {

        XusbPadctlRegister.SetField32(miscPadCtl2Regs[ssPhyPad],
                                      XusbPadctlUphyMiscPadP0Ctl2TxIddqMask     | XusbPadctlUphyMiscPadP0Ctl2RxIddqMask     |
                                      XusbPadctlUphyMiscPadP0Ctl2TxIddqOvrdMask | XusbPadctlUphyMiscPadP0Ctl2RxIddqOvrdMask |
                                      XusbPadctlUphyMiscPadP0Ctl2TxSleepMask    | XusbPadctlUphyMiscPadP0Ctl2RxSleepMask    |
                                      XusbPadctlUphyMiscPadP0Ctl2TxPwrOvrdMask  | XusbPadctlUphyMiscPadP0Ctl2RxPwrOvrdMask);
        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(1));
    }
    else
    {
        XusbPadctlRegister.ClearField32(miscPadCtl2Regs[ssPhyPad],
                                        XusbPadctlUphyMiscPadP0Ctl2TxIddqOvrdMask | XusbPadctlUphyMiscPadP0Ctl2RxIddqOvrdMask |
                                        XusbPadctlUphyMiscPadP0Ctl2TxPwrOvrdMask  | XusbPadctlUphyMiscPadP0Ctl2RxPwrOvrdMask);
    }
}

void UsbComplexTegra21x::XusbPadctlHsInit(int32_t pad, PadMode padMode) NN_NOEXCEPT
{
    XusbPadctlRegister.SetField32(XusbPadctlUsb2PortCapOffset,
                                  XusbPadctlUsb2PortCapPort0CapMask << (4 * pad),
                                  (padMode & 3) << (pad * 4));

    // Don't use Tegra for OC detection
    XusbPadctlRegister.SetField32(XusbPadctlUsb2OcMapOffset,
                                  XusbPadctlUsb2OcMapPort0OcPinMask << (4 * pad),
                                  XusbPadctlUsb2OcMapDetectionDisabled << (pad * 4));
    XusbPadctlRegister.SetField32(XusbPadctlVbusOcMapOffset,
                                  XusbPadctlVbusOcMap0Mask << (5 * pad + 1),
                                  XusbPadctlVbusOcMapDetectionDisabled << (5 * pad + 1));
    //XusbPadctlVbusOcMapEnable0Mask << (5 * pad));

    XusbPadctlRegister.SetField32(
        XusbPadctlUsb2OtgPadCtl0Offsets[pad],
        XusbPadctlUsb2OtgPad0Ctl0HsCurrLevelMask
        | XusbPadctlUsb2OtgPad0Ctl0PdZiMask
        | XusbPadctlUsb2OtgPad0Ctl0Pd2Mask
        | XusbPadctlUsb2OtgPad0Ctl0PdMask,
        NN_USB_MAKE_MASK32(XusbPadctlUsb2OtgPad0Ctl0HsCurrLevel, GetHsCurrLevel(pad))
    );
    XusbPadctlRegister.SetField32(
        XusbPadctlUsb2OtgPadCtl1Offsets[pad],
        XusbPadctlUsb2OtgPad0Ctl1PdDrMask
        | XusbPadctlUsb2OtgPad0Ctl1TermRangeAdjMask
        | XusbPadctlUsb2OtgPad0Ctl1RpdCtrlMask,
        NN_USB_MAKE_MASK32(XusbPadctlUsb2OtgPad0Ctl1TermRangeAdj, GetTermRangeAdj(pad))
        | NN_USB_MAKE_MASK32(XusbPadctlUsb2OtgPad0Ctl1RpdCtrl,    GetRpdCtrl(pad))
    );

    // Turn off battery charge and otg related, set to defaults
    XusbPadctlRegister.Write32(XusbPadctlUsb2BatteryChrgOtgpadCtl0Offsets[pad],  0x00000001);
    XusbPadctlRegister.Write32(XusbPadctlUsb2BatteryChrgOtgpadCtl1Offsets[pad],  0x00000000);

    XusbPadctlRegister.ClearField32(XusbPadctlUsb2BatteryChrgOtgpadCtl1Offsets[pad],
                                    XusbPadctlUsb2BatteryChrgOtgpad0Ctl1VregFix18Mask |
                                    XusbPadctlUsb2BatteryChrgOtgpad0Ctl1VregLevMask);
    if(padMode == PadMode_Device)
    {
        // This setting applies only to the special OTG charge detect capable pad
        if(pad == OtgPad)
        {
            XusbPadctlSetVbusPadProtection(m_LastDetectedChargerType);
        }
    }
    else if(padMode == PadMode_Host)
    {
        XusbPadctlRegister.SetField32(XusbPadctlUsb2BatteryChrgOtgpadCtl1Offsets[pad],
                                      XusbPadctlUsb2BatteryChrgOtgpad0Ctl1VregFix18Mask);
    }
}

size_t UsbComplexTegra21x::XusbPadctlReadBatteryChrgOtgPadCtl0Hs() NN_NOEXCEPT
{
    return XusbPadctlRegister.Read32(XusbPadctlUsb2BatteryChrgOtgpadCtl0Offsets[PadMode_Device]);
}

void UsbComplexTegra21x::XusbPadctlSsInit(int32_t pad, PadMode padMode) NN_NOEXCEPT
{
    size_t ectlPadOffset = pad * XusbPadctlUphyUsb3EctlBlockSize;
    XusbPadctlRegister.SetField32(XusbPadctlSsPortMapOffset,
                                  (XusbPadctlSsPortMap0Mask | XusbPadctlSsPortMap0InternalMask) << (pad * 5),
                                  (padMode != PadMode_Disabled ? pad : 7) << (pad * 5));

    if(padMode != PadMode_Disabled)
    {
        // These ECTL settings were provided by Raptor per NSBG-6166
        XusbPadctlRegister.Write32(XusbPadctlUphyUsb3Ectl1Offset + ectlPadOffset, 0x0002001f);
        XusbPadctlRegister.Write32(XusbPadctlUphyUsb3Ectl2Offset + ectlPadOffset, 0x000000fc);
        XusbPadctlRegister.Write32(XusbPadctlUphyUsb3Ectl3Offset + ectlPadOffset, 0xc0077f1f);
        XusbPadctlRegister.Write32(XusbPadctlUphyUsb3Ectl6Offset + ectlPadOffset, 0xfcf01368);
    }
    else
    {
        // These ECTL settings are default per TRM
        XusbPadctlRegister.Write32(XusbPadctlUphyUsb3Ectl1Offset + ectlPadOffset, 0x0000001f);
        XusbPadctlRegister.Write32(XusbPadctlUphyUsb3Ectl2Offset + ectlPadOffset, 0x0000008f);
        XusbPadctlRegister.Write32(XusbPadctlUphyUsb3Ectl3Offset + ectlPadOffset, 0x00000000);
        XusbPadctlRegister.Write32(XusbPadctlUphyUsb3Ectl6Offset + ectlPadOffset, 0x00000000);
    }
}

void UsbComplexTegra21x::XusbPadctlSetVbusPadProtection(UsbChargerType chargerType) NN_NOEXCEPT
{
    uint8_t vregLevel = XusbPadctlUsb2BatteryChrgOtgpad0Ctl1VregLev500mA;

    switch (chargerType)
    {
    case UsbChargerType_Dcp:
    case UsbChargerType_Cdp:
        vregLevel = XusbPadctlUsb2BatteryChrgOtgpad0Ctl1VregLev1500mA;
        break;
    case UsbChargerType_Sdp:
        vregLevel = XusbPadctlUsb2BatteryChrgOtgpad0Ctl1VregLev500mA;
        break;
    case UsbChargerType_Apple500ma:
        vregLevel = XusbPadctlUsb2BatteryChrgOtgpad0Ctl1VregLev500mA;
        break;
    case UsbChargerType_Apple1000ma:
        vregLevel = XusbPadctlUsb2BatteryChrgOtgpad0Ctl1VregLev900mA;
        break;
    case UsbChargerType_Apple2000ma:
        vregLevel = XusbPadctlUsb2BatteryChrgOtgpad0Ctl1VregLev2000mA;
        break;
    default:
        break;
    }

    XusbPadctlRegister.SetField32(XusbPadctlUsb2BatteryChrgOtgpad0Ctl1Offset,
                                  XusbPadctlUsb2BatteryChrgOtgpad0Ctl1VregLevMask,
                                  vregLevel);
}

uint32_t UsbComplexTegra21x::GetHsCurrLevel(uint32_t pad) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(pad, 0, 3);

    int32_t hsCurrLevel;
    uint32_t hsCurrLevelMap[] = {
        NN_USB_GET_FIELD32(FuseUsbCalib0HsCurrLevel0, m_FuseUsbCalib0),
        NN_USB_GET_FIELD32(FuseUsbCalib0HsCurrLevel1, m_FuseUsbCalib0),
        NN_USB_GET_FIELD32(FuseUsbCalib0HsCurrLevel2, m_FuseUsbCalib0),
        NN_USB_GET_FIELD32(FuseUsbCalib0HsCurrLevel3, m_FuseUsbCalib0),
    };

    switch (GetPlatform()->GetConfig().socName)
    {
    case UsbPlatform::SocName_Tx1:
        // NSBG-8267: We decided to keep using hardcoded value for TX1
        hsCurrLevel = 0x20;
        break;

    case UsbPlatform::SocName_Mariko:
        hsCurrLevel = (int32_t)hsCurrLevelMap[pad] + GetPlatform()->GetConfig().hsCurrLevelOffset;
        if (hsCurrLevel < 0)
        {
            hsCurrLevel = 0;
            NN_USB_LOG_WARN("Port %d: Drive strength saturated (%d / %+d -> 0)\n",
                            pad,
                            hsCurrLevelMap[pad],
                            GetPlatform()->GetConfig().hsCurrLevelOffset);
        }
        if (hsCurrLevel > 0x3f)
        {
            hsCurrLevel = 0x3f;
            NN_USB_LOG_WARN("Port %d: Drive strength saturated (%d / %+d -> 0x3f)\n",
                            pad,
                            hsCurrLevelMap[pad],
                            GetPlatform()->GetConfig().hsCurrLevelOffset);
        }
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

    return hsCurrLevel;
}

uint32_t UsbComplexTegra21x::GetTermRangeAdj(uint32_t pad) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(pad, 0, 3);

    uint32_t termRangeAdj;
    uint32_t termRangeAdjMap[] = {
        NN_USB_GET_FIELD32(FuseUsbCalib0TermRangeAdj0,    m_FuseUsbCalib0),
        NN_USB_GET_FIELD32(FuseUsbCalibExt0TermRangeAdj1, m_FuseUsbCalibExt0),
        NN_USB_GET_FIELD32(FuseUsbCalibExt0TermRangeAdj2, m_FuseUsbCalibExt0),
        NN_USB_GET_FIELD32(FuseUsbCalibExt0TermRangeAdj3, m_FuseUsbCalibExt0),
    };

    switch (GetPlatform()->GetConfig().socName)
    {
    case UsbPlatform::SocName_Tx1:
        // NSBG-8267: We decided to keep using hardcoded value for TX1
        termRangeAdj = 0x08;
        break;


    case UsbPlatform::SocName_Mariko:
        termRangeAdj = termRangeAdjMap[pad];
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

    return termRangeAdj;
}

uint32_t UsbComplexTegra21x::GetRpdCtrl(uint32_t pad) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(pad, 0, 3);

    NN_UNUSED(pad);

    uint32_t rpdCtrl;

    switch (GetPlatform()->GetConfig().socName)
    {
    case UsbPlatform::SocName_Tx1:
        // NSBG-8267: We decided to keep using hardcoded value for TX1
        rpdCtrl = 0;
        break;

    case UsbPlatform::SocName_Mariko:
        rpdCtrl = NN_USB_GET_FIELD32(FuseUsbCalibExt0RpdCtrl, m_FuseUsbCalibExt0);
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

    return rpdCtrl;
}

uint32_t UsbComplexTegra21x::GetTermOffset() const NN_NOEXCEPT
{
    uint32_t termOffset;

    // NSBG-8267: Mariko don't use register default
    switch (GetPlatform()->GetConfig().socName)
    {
    case UsbPlatform::SocName_Tx1:
        termOffset = 0;
        break;

    case UsbPlatform::SocName_Mariko:
        termOffset = 3;
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

    return termOffset;
}

void UsbComplexTegra21x::StartChargerDetection() NN_NOEXCEPT
{
    uint32_t otgpadCtl1;

    m_LastDetectedChargerType = UsbChargerType_Unknown;

    XusbPadctlHsInit(OtgPad, PadMode_Device);

    // Disable HS transmiter
    XusbPadctlRegister.SET_FIELD32_2(XusbPadctlUsb2OtgPad0Ctl0,
                                     Pd2,       1,
                                     Pd2OvrdEn, 1);

    // Enable the charger detecion circuit
    XusbPadctlRegister.SET_FIELD32_1(XusbPadctlUsb2BatteryChrgOtgpad0Ctl0,
                                     PdChg, 0);

    // Set DCD debounce timer
    XusbPadctlRegister.SET_FIELD32_1(XusbPadctlUsb2BatteryChrgTdcdDbncTimer,
                                     TdcdDbnc, 0x0a);

    // Set DP/DN Pull up/down to zero by default
    otgpadCtl1 = XusbPadctlRegister.Read32(XusbPadctlUsb2BatteryChrgOtgpad0Ctl1Offset);
    otgpadCtl1 &= ~(XusbPadctlUsb2BatteryChrgOtgpad0Ctl1UsbopRpdOvrdValMask|
                    XusbPadctlUsb2BatteryChrgOtgpad0Ctl1UsbopRpuOvrdValMask|
                    XusbPadctlUsb2BatteryChrgOtgpad0Ctl1UsbonRpdOvrdValMask|
                    XusbPadctlUsb2BatteryChrgOtgpad0Ctl1UsbonRpuOvrdValMask);
    otgpadCtl1 |= (XusbPadctlUsb2BatteryChrgOtgpad0Ctl1UsbopRpdOvrdMask|
                   XusbPadctlUsb2BatteryChrgOtgpad0Ctl1UsbopRpuOvrdMask|
                   XusbPadctlUsb2BatteryChrgOtgpad0Ctl1UsbonRpdOvrdMask|
                   XusbPadctlUsb2BatteryChrgOtgpad0Ctl1UsbonRpuOvrdMask);
    XusbPadctlRegister.Write32(XusbPadctlUsb2BatteryChrgOtgpad0Ctl1Offset, otgpadCtl1);

    // Disable DP/DN as src/sink
    XusbPadctlRegister.SET_FIELD32_4(XusbPadctlUsb2BatteryChrgOtgpad0Ctl0,
                                     OpSrcEn,  0,
                                     OnSinkEn, 0,
                                     OpSinkEn, 0,
                                     OnSrcEn,  0);

    XusbPadctlRegister.SET_FIELD32_4(XusbPadctlUsb2BatteryChrgOtgpad0Ctl0,
                                     VdcdDetFilterEn, 1,
                                     VdatDetFilterEn, 1,
                                     ZipFilterEn,     1,
                                     ZinFilterEn,     1);
}

void UsbComplexTegra21x::StopChargerDetection(UsbChargerType chargerType) NN_NOEXCEPT
{
    uint32_t otgpadCtl1;

    XusbPadctlRegister.SET_FIELD32_4(XusbPadctlUsb2BatteryChrgOtgpad0Ctl0,
                                     VdcdDetFilterEn, 0,
                                     VdatDetFilterEn, 0,
                                     ZipFilterEn,     0,
                                     ZinFilterEn,     0);

    // Set DP/DN Pull up/down to default
    otgpadCtl1 = XusbPadctlRegister.Read32(XusbPadctlUsb2BatteryChrgOtgpad0Ctl1Offset);
    otgpadCtl1 &= ~(XusbPadctlUsb2BatteryChrgOtgpad0Ctl1UsbopRpdOvrdMask|
                    XusbPadctlUsb2BatteryChrgOtgpad0Ctl1UsbopRpuOvrdMask|
                    XusbPadctlUsb2BatteryChrgOtgpad0Ctl1UsbonRpdOvrdMask|
                    XusbPadctlUsb2BatteryChrgOtgpad0Ctl1UsbonRpuOvrdMask);
    XusbPadctlRegister.Write32(XusbPadctlUsb2BatteryChrgOtgpad0Ctl1Offset, otgpadCtl1);

    // Remove pd2_ovrd_en, leave it under HW control
    XusbPadctlRegister.SET_FIELD32_1(XusbPadctlUsb2OtgPad0Ctl0,
                                     Pd2OvrdEn, 0);

    if (chargerType == UsbChargerType_Dcp)
    {
        /*
         * When attached to a DCP, we shall disable VDM_SRC and enable VDP_SRC
         *
         * FIXME: Perhaps we shouldn't disable charger detection circuit and
         *        PADCTL, but in practice I don't see the difference.
         */
        XusbPadctlRegister.SET_FIELD32_4(XusbPadctlUsb2BatteryChrgOtgpad0Ctl0,
                                         OpSrcEn,  1,
                                         OnSinkEn, 1,
                                         OpSinkEn, 0,
                                         OnSrcEn,  0);
    }
    else
    {
        // Disable DP/DN as src/sink
        XusbPadctlRegister.SET_FIELD32_4(XusbPadctlUsb2BatteryChrgOtgpad0Ctl0,
                                         OpSrcEn,  0,
                                         OnSinkEn, 0,
                                         OpSinkEn, 0,
                                         OnSrcEn,  0);
    }


    // Disable the charger detecion circuit
    XusbPadctlRegister.SET_FIELD32_1(XusbPadctlUsb2BatteryChrgOtgpad0Ctl0,
                                     PdChg, 1);

    XusbPadctlHsInit(OtgPad, PadMode_Disabled);
}

// Ref NSBG-5485
UsbChargerType UsbComplexTegra21x::ChargerDetection() NN_NOEXCEPT
{
    uint32_t       otgpadCtl0;
    uint32_t       otgpadCtl1;
    UsbChargerType chargerType = UsbChargerType_Unknown;

    StartChargerDetection();

    // Non Compliant Charger: Apple Charger
    XusbPadctlRegister.SET_FIELD32_1(XusbPadctlUsb2BatteryChrgOtgpad0Ctl1, DivDetEn, 1);
    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(20));

    otgpadCtl1 = XusbPadctlRegister.Read32(XusbPadctlUsb2BatteryChrgOtgpad0Ctl1Offset);

    XusbPadctlRegister.SET_FIELD32_1(XusbPadctlUsb2BatteryChrgOtgpad0Ctl1, DivDetEn, 0);

    if (NN_USB_CHECK_FIELD32_2(XusbPadctlUsb2BatteryChrgOtgpad0Ctl1, otgpadCtl1,
                               VopDiv2p7Det, 1, VonDiv2p7Det, 1))
    {

        NN_USB_LOG_INFO("Detected Charger A1 (Apple 2000mA)\n");
        chargerType = UsbChargerType_Apple2000ma;
        goto done;
    }
    else if (NN_USB_CHECK_FIELD32_2(XusbPadctlUsb2BatteryChrgOtgpad0Ctl1, otgpadCtl1,
                                    VopDiv2p7Det, 1, VonDiv2p0Det, 1))
    {
        NN_USB_LOG_INFO("Detected Charger A2 (Apple 2000mA)\n");
        chargerType = UsbChargerType_Apple2000ma;
        goto done;
    }
    else if (NN_USB_CHECK_FIELD32_2(XusbPadctlUsb2BatteryChrgOtgpad0Ctl1, otgpadCtl1,
                                    VopDiv2p0Det, 1, VonDiv2p7Det, 1))
    {
        NN_USB_LOG_INFO("Detected Charger A3 (Apple 1000mA)\n");
        chargerType = UsbChargerType_Apple1000ma;
        goto done;
    }
    else if (NN_USB_CHECK_FIELD32_2(XusbPadctlUsb2BatteryChrgOtgpad0Ctl1, otgpadCtl1,
                                    VopDiv2p0Det, 1, VonDiv2p0Det, 1))
    {
        NN_USB_LOG_INFO("Detected Charger A4 (Apple 500mA\n");
        chargerType = UsbChargerType_Apple500ma;
        goto done;
    }

    // Non Compliant Charger: D+/D- tied to high
    otgpadCtl0 = XusbPadctlRegister.Read32(XusbPadctlUsb2BatteryChrgOtgpad0Ctl0Offset);

    if (NN_USB_CHECK_FIELD32_2(XusbPadctlUsb2BatteryChrgOtgpad0Ctl0, otgpadCtl0,
                               Zip, 1, Zin, 1))
    {
        NN_USB_LOG_INFO("Detected Charger B1\n");
        chargerType = UsbChargerType_Unknown;
        goto done;
    }
    else if (NN_USB_CHECK_FIELD32_2(XusbPadctlUsb2BatteryChrgOtgpad0Ctl0, otgpadCtl0,
                                    Zip, 0, Zin, 1))
    {
        NN_USB_LOG_INFO("Detected Charger B2\n");
        chargerType = UsbChargerType_Unknown;
        goto done;
    }
    else if (NN_USB_CHECK_FIELD32_2(XusbPadctlUsb2BatteryChrgOtgpad0Ctl0, otgpadCtl0,
                                    Zip, 1, Zin, 0))
    {
        NN_USB_LOG_INFO("Detected Charger B3\n");
        chargerType = UsbChargerType_Unknown;
        goto done;
    }

    // Non Compliant Charger: D+/D- tied to high
    XusbPadctlRegister.SET_FIELD32_2(XusbPadctlUsb2BatteryChrgOtgpad0Ctl1,
                                     UsbonRpuOvrdVal, 1,
                                     UsbopRpuOvrdVal, 1);
    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(20));

    otgpadCtl0 = XusbPadctlRegister.Read32(XusbPadctlUsb2BatteryChrgOtgpad0Ctl0Offset);

    XusbPadctlRegister.SET_FIELD32_2(XusbPadctlUsb2BatteryChrgOtgpad0Ctl1,
                                     UsbonRpuOvrdVal, 0,
                                     UsbopRpuOvrdVal, 0);

    if (NN_USB_CHECK_FIELD32_2(XusbPadctlUsb2BatteryChrgOtgpad0Ctl0, otgpadCtl0,
                               Zip, 1, Zin, 1))
    {
        NN_USB_LOG_INFO("Detected Charger C1\n");
        chargerType = UsbChargerType_Unknown;
        goto done;
    }
    else if (NN_USB_CHECK_FIELD32_2(XusbPadctlUsb2BatteryChrgOtgpad0Ctl0, otgpadCtl0,
                                    Zip, 0, Zin, 1))
    {
        NN_USB_LOG_INFO("Detected Charger C2\n");
        chargerType = UsbChargerType_Unknown;
        goto done;
    }
    else if (NN_USB_CHECK_FIELD32_2(XusbPadctlUsb2BatteryChrgOtgpad0Ctl0, otgpadCtl0,
                                    Zip, 1, Zin, 0))
    {
        NN_USB_LOG_INFO("Detected Charger C3\n");
        chargerType = UsbChargerType_Unknown;
        goto done;
    }
    else if (NN_USB_CHECK_FIELD32_2(XusbPadctlUsb2BatteryChrgOtgpad0Ctl0, otgpadCtl0,
                                    Zip, 0, Zin, 0))
    {
        chargerType = DetectCompliantCharger();
        goto done;
    }

done:
    StopChargerDetection(chargerType);
    m_LastDetectedChargerType = chargerType;
    XusbPadctlSetVbusPadProtection(m_LastDetectedChargerType);

    return chargerType;
}

/*
 * Ref [TX1 TRM] 23.8.11
 */
UsbChargerType UsbComplexTegra21x::DetectCompliantCharger() NN_NOEXCEPT
{
    uint32_t       otgpadCtl0;

    /*
     * Data Contact Detect
     */

    // Turn on IDP_SRC
    XusbPadctlRegister.SET_FIELD32_1(XusbPadctlUsb2BatteryChrgOtgpad0Ctl0,
                                     OpISrcEn, 1);

    // Turn on D- pull-down resistor
    XusbPadctlRegister.SET_FIELD32_1(XusbPadctlUsb2BatteryChrgOtgpad0Ctl1,
                                     UsbonRpdOvrdVal, 1);

    // Wait for TDCD_DBNC (min 10ms)
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(20));

    // Poll DCD state for TDCD_TIMEOUT (min 300ms, max 900ms)
    for (int i = 0; i < 10; i++)
    {
        otgpadCtl0 = XusbPadctlRegister.Read32(XusbPadctlUsb2BatteryChrgOtgpad0Ctl0Offset);
        if (NN_USB_CHECK_FIELD32_1(XusbPadctlUsb2BatteryChrgOtgpad0Ctl0, otgpadCtl0,
                                   DcdDetected, 1))
        {
            // Data pin contact detected
            break;
        }

        if (i == 9)
        {
            NN_USB_LOG_WARN("DCD timeout!\n");
        }
        else
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        }
    }

    // Turn off IDP_SRC, clear DCD DETECTED
    XusbPadctlRegister.SET_FIELD32_2(XusbPadctlUsb2BatteryChrgOtgpad0Ctl0,
                                     OpISrcEn,    0,
                                     DcdDetected, 1);

    // Turn off D- pull-down resistor
    XusbPadctlRegister.SET_FIELD32_1(XusbPadctlUsb2BatteryChrgOtgpad0Ctl1,
                                     UsbonRpdOvrdVal, 0);

    /*
     * Primary Detection: Source current from D+ to D-
     */
    XusbPadctlRegister.SET_FIELD32_4(XusbPadctlUsb2BatteryChrgOtgpad0Ctl0,
                                     OpSrcEn,  1,
                                     OnSinkEn, 1,
                                     OpSinkEn, 0,
                                     OnSrcEn,  0);

    // Wait for TVDPSRC_ON (min 40ms)
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(40));

    otgpadCtl0 = XusbPadctlRegister.Read32(XusbPadctlUsb2BatteryChrgOtgpad0Ctl0Offset);
    if (NN_USB_CHECK_FIELD32_1(XusbPadctlUsb2BatteryChrgOtgpad0Ctl0, otgpadCtl0,
                               VdatDet, 1))
    {
        /*
         * Secondary Detection: Source current from D- to D+
         */
        XusbPadctlRegister.SET_FIELD32_4(XusbPadctlUsb2BatteryChrgOtgpad0Ctl0,
                                         OpSrcEn,  0,
                                         OnSinkEn, 0,
                                         OpSinkEn, 1,
                                         OnSrcEn,  1);

        // Wait for TVDMSRC_ON (min 40ms)
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(40));

        otgpadCtl0 = XusbPadctlRegister.Read32(XusbPadctlUsb2BatteryChrgOtgpad0Ctl0Offset);
        if (NN_USB_CHECK_FIELD32_1(XusbPadctlUsb2BatteryChrgOtgpad0Ctl0, otgpadCtl0,
                                   VdatDet, 1))
        {
            NN_USB_LOG_INFO("Detected DCP (Dedicated Charging Port)\n");
            return UsbChargerType_Dcp;
            // FIXME: Enable V_DP_SRC or pull D+ to V_DP_UP through R_DP_UP
        }
        else
        {
            NN_USB_LOG_INFO("Detected CDP (Charging Downstream Port)\n");
            return UsbChargerType_Cdp;
        }
    }
    else
    {
        NN_USB_LOG_INFO("Detected SDP (Standard Downstream Port)\n");
        return UsbChargerType_Sdp;
    }
}

void UsbComplexTegra21x::WaitForPcie() NN_NOEXCEPT
{
    Result result = ResultSuccess();
    nn::pcv::ModuleState pcieSubsystemState;

    while (true)
    {
        NN_USB_ABORT_UPON_ERROR(
            nn::pcv::GetState(&pcieSubsystemState, nn::pcv::Module_Pcie)
        );

        if (!pcieSubsystemState.resetAsserted &&
            pcieSubsystemState.powerEnabled   &&
            pcieSubsystemState.clockEnabled    )
        {
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(25));
    }
}

} // end of namespace detail
} // end of namespace usb
} // end of namespace nn
