﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>

#include "updater_Common.h"
#include "updater_PartitionAccessor.h"

namespace nn { namespace updater {

namespace
{
// bct 内の、特定データのオフセット
const uint32_t EksOffset = 0x450;
const uint32_t BootLoaderVersionOffset = 0x2330;

// BootPartition1 にアクセスするためのテーブル
const OffsetSize<BootPartition1Type> s_BootPartition1Definition[] =
{
    { BootPartition1Type::BctNormalMain_BCT0,      0x0, BctSize },
    { BootPartition1Type::BctSafeMain_BCT1,     0x4000, BctSize },
    { BootPartition1Type::BctNormalSub_BCT2,    0x8000, BctSize },
    { BootPartition1Type::BctSafeSub_BCT3,      0xC000, BctSize },
    { BootPartition1Type::BctSave_BCT63,      0x100000 - BctSize, BctSize },
    { BootPartition1Type::Package1NormalMain, 0x100000, 0x40000 },
    { BootPartition1Type::Package1NormalSub,  0x140000, 0x40000 },
    { BootPartition1Type::Eks,                0x180000, EksSize },
};

// BootPartition2 にアクセスするためのテーブル
const OffsetSize<BootPartition2Type> s_BootPartition2Definition[] =
{
    { BootPartition2Type::Package1SafeMain,       0x0, 0x40000 },
    { BootPartition2Type::Package1SafeSub,    0x40000, 0x40000 },
    { BootPartition2Type::Package1RepairMain, 0x80000, 0x40000 },
    { BootPartition2Type::Package1RepairSub,  0xC0000, 0x40000 },
};

// Package2 にアクセスするためのテーブル
const OffsetSize<Package2Type> s_Package2Definition[] =
{
    { Package2Type::Package2,  0x4000, 0x800000 - 0x4000 },
};

// Package2 の BisPartition を引くためのテーブル
fs::BisPartitionId GetBisId(Package2Index index) NN_NOEXCEPT
{
    switch(index)
    {
    case Package2Index::NormalMain:
        return fs::BisPartitionId::BootConfigAndPackage2Part1;
    case Package2Index::NormalSub:
        return fs::BisPartitionId::BootConfigAndPackage2Part2;
    case Package2Index::SafeMain:
        return fs::BisPartitionId::BootConfigAndPackage2Part3;
    case Package2Index::SafeSub:
        return fs::BisPartitionId::BootConfigAndPackage2Part4;
    case Package2Index::RepairMain:
        return fs::BisPartitionId::BootConfigAndPackage2Part5;
    case Package2Index::RepairSub:
        return fs::BisPartitionId::BootConfigAndPackage2Part6;
    default:
        break;
    }
    NN_ABORT("Unknown index: %d\n", index);

    return fs::BisPartitionId::BootConfigAndPackage2Part1;
}

}

BootPartition1Accessor::BootPartition1Accessor() NN_NOEXCEPT:
TAccessor(fs::BisPartitionId::BootPartition1Root, s_BootPartition1Definition, sizeof(s_BootPartition1Definition) / sizeof(s_BootPartition1Definition[0]))
{
}
BootPartition2Accessor::BootPartition2Accessor() NN_NOEXCEPT:
TAccessor(fs::BisPartitionId::BootPartition2Root, s_BootPartition2Definition, sizeof(s_BootPartition2Definition) / sizeof(s_BootPartition2Definition[0]))
{
}

Result BootPartition1Accessor::UpdateSecureInfo(void* bctBuffer, size_t bctSize, const void* eksBuffer, size_t eksSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(bctSize == BctSize);
    NN_ABORT_UNLESS(eksSize == EksSize);

    auto bootLoaderVersion = ReadBootLoaderVersion(bctBuffer, bctSize);
    auto eksIndex = CalculateEksIndex(bootLoaderVersion);

    CopyEks(bctBuffer, eksBuffer, eksIndex);

    NN_RESULT_SUCCESS;
}
Result BootPartition1Accessor::UpdateSecureInfoAuto(void* bctBuffer, size_t bctSize, void* eksBuffer, size_t eksSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(bctSize == BctSize);
    NN_ABORT_UNLESS(eksSize == EksSize);

    size_t eksReadSize;
    NN_RESULT_DO(Read(eksBuffer, &eksReadSize, eksSize, BootPartition1Type::Eks));

    auto bootLoaderVersion = ReadBootLoaderVersion(bctBuffer, bctSize);
    auto eksIndex = CalculateEksIndex(bootLoaderVersion);

    CopyEks(bctBuffer, eksBuffer, eksIndex);

    NN_RESULT_SUCCESS;
}

int32_t BootPartition1Accessor::ReadBootLoaderVersion(void* buffer, size_t size) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(size == BctSize);

    auto version = *reinterpret_cast<int32_t*>(reinterpret_cast<uint8_t*>(buffer) + BootLoaderVersionOffset);

    NN_ABORT_UNLESS_MINMAX(version, 0, 32);

    return version;
}
int32_t BootPartition1Accessor::CalculateEksIndex(int32_t bootLoaderVersion) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(0 <= bootLoaderVersion && bootLoaderVersion <= 32);

    if(bootLoaderVersion == 0)
    {
        return 0;
    }
    else
    {
        return bootLoaderVersion - 1;
    }
}
void BootPartition1Accessor::CopyEks(void* bctBuffer, const void* eksBuffer, int32_t eksIndex) NN_NOEXCEPT
{
    auto sourceEksOffset = EksEntrySize * eksIndex;

    std::memcpy
    (
        reinterpret_cast<uint8_t *>(bctBuffer) + EksOffset,
        reinterpret_cast<const uint8_t *>(eksBuffer) + sourceEksOffset,
        CopyEksSize
    );
}

Package2Accessor::Package2Accessor(Package2Index index) NN_NOEXCEPT:
TAccessor(GetBisId(index), s_Package2Definition, sizeof(s_Package2Definition) / sizeof(s_Package2Definition[0]))
{
}

}}
