﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <cstring>

#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_BitUtil.h>
#include <nn/crypto/crypto_Sha256Generator.h>

#include <nn/fs.h>
#include <nn/fs/fs_SystemData.h>

#include <nn/updater/updater.h>

#include "updater_Common.h"
#include "updater_BisAccessor.h"

namespace nn { namespace updater {

Result VerifyBuffer(void* workBuffer, size_t size) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(size >= sizeof(BctBuffer), ResultOutOfWorkBuffer());
    NN_RESULT_THROW_UNLESS(
        util::is_aligned(reinterpret_cast<uintptr_t>(workBuffer), 4096),
        ResultInvalidAlignmentOfWorkBuffer()
    );
    NN_RESULT_THROW_UNLESS(
        util::is_aligned(size, BisAccessor::GetBlockSize()),
        ResultInvalidAlignmentOfWorkBuffer()
    );

    NN_RESULT_SUCCESS;
}

Result ReadFile(size_t* outSize, char* buffer, size_t bufferSize, const char* path) NN_NOEXCEPT
{
    fs::FileHandle handle;

    NN_RESULT_TRY(fs::OpenFile(&handle, path, fs::OpenMode_Read))
        NN_RESULT_CATCH(fs::ResultPathNotFound) {NN_RESULT_THROW(ResultInvalidBootImagePackage());}
    NN_RESULT_END_TRY
    NN_UTIL_SCOPE_EXIT{ fs::CloseFile(handle); };

    // ブロック単位で書き込むので、変なデータを書き込まないようにクリア
    std::memset(buffer, 0, bufferSize);
    NN_RESULT_DO(fs::ReadFile(outSize, handle, 0, buffer, bufferSize, fs::ReadOption()));

    NN_RESULT_SUCCESS;
}

Result ReadFileAndCalculateHash(size_t* pOutSize, void* pOutHash, size_t hashSize, const char* path, void* workBuffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_UNUSED(hashSize);

    fs::FileHandle handle;

    NN_RESULT_TRY(fs::OpenFile(&handle, path, fs::OpenMode_Read))
        NN_RESULT_CATCH(fs::ResultPathNotFound) {NN_RESULT_THROW(ResultInvalidBootImagePackage());}
    NN_RESULT_END_TRY
    NN_UTIL_SCOPE_EXIT{ fs::CloseFile(handle); };

    size_t readSize = 0;
    crypto::Sha256Generator generator;
    generator.Initialize();
    for(;;)
    {
        size_t size;
        NN_RESULT_DO(fs::ReadFile(&size, handle, readSize, workBuffer, bufferSize, fs::ReadOption()));
        generator.Update(workBuffer, size);
        readSize += size;

        if (size != bufferSize)
        {
            break;
        }
    }
    generator.GetHash(pOutHash, hashSize);
    *pOutSize = readSize;

    NN_RESULT_SUCCESS;
}

}}
