﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/uart/uart_PortName.h>
#include <nn/uart/uart_PortTypes.h>
#include <nn/os/os_InterruptEvent.h>

namespace nn {
namespace uart {
namespace driver {
namespace detail {

// ポート総数（番号は 0 始まりのため、この値未満の非負整数がポート番号として使用できる）
const int PortCountMax = 4;

// ハンドルする割り込み番号の数
const int InterruptCountMax = 4;

const int DmaChannelCountPerPort = 2;
const int DmaChannelCountMax = PortCountMax * DmaChannelCountPerPort;

// 割り込み番号テーブル
const nn::os::InterruptName InterruptNameUartA = 32 + 36;
const nn::os::InterruptName InterruptNameUartB = 32 + 37;
const nn::os::InterruptName InterruptNameUartC = 32 + 46;
const nn::os::InterruptName InterruptNameUartD = 32 + 90;
const nn::os::InterruptName InterruptNameTable[InterruptCountMax] =
{
    InterruptNameUartA,  // UART A
    InterruptNameUartB,  // UART B
    InterruptNameUartC,  // UART C
    InterruptNameUartD   // UART D
};

// ポートと割り込み番号の対応表
const nn::os::InterruptName AssignedInterrupt[PortCountMax] =
{
    InterruptNameUartA,  // Port 0
    InterruptNameUartB,  // Port 1
    InterruptNameUartC,  // Port 2
    InterruptNameUartD   // Port 3
};


// サポートしていないポートとして使用する番号
const int UnsupportedPortIndex = -1;

// 外部の定義と内部ポート番号の組み合わせの構造体の定義
struct PortMapCombination
{
    PortName publicPortName;
    int      portIndex;
};

// 外部の定義と内部のポート番号への変換一覧
const PortMapCombination PortMapCombinationList[] =
{
    // 公開ヘッダーの定義   内部のポート番号
    { PortName_Bluetooth,   3 },
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    { PortName_ExtConU,     1 },
    { PortName_ExtConS,     2 },
#endif
};

// 対応ボーレートリスト
struct BaudRateSupportInfo
{
    int      portIndex;
    BaudRate baudRate;
};
const BaudRateSupportInfo SupportedBaudRateList[] =
{
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    // UART B
    {1, BaudRate_57600},
    {1, BaudRate_115200},
    {1, BaudRate_1M},
    {1, BaudRate_3M},
#endif

    // UART C
    {2, BaudRate_57600},
    {2, BaudRate_115200},
    {2, BaudRate_1M},
    {2, BaudRate_3M},

    // UART D
    {3, BaudRate_57600},
    {3, BaudRate_115200},
    {3, BaudRate_1M},
    {3, BaudRate_3M},
};

// 対応フロー制御モードリスト
struct FlowControlModeSupportInfo
{
    int             portIndex;
    FlowControlMode flowControlMode;
};
const FlowControlModeSupportInfo SupportedFlowControlModeList[] =
{
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    // UART B
    {1, FlowControlMode_None},
    {1, FlowControlMode_Hardware},
#endif

    // UART C
    {2, FlowControlMode_None},
    {2, FlowControlMode_Hardware}, // Jetson TK2 は Expansion Connector 経由で配線していることが前提

    // UART D
    {3, FlowControlMode_None},
    {3, FlowControlMode_Hardware}
};

// 対応ポートイベントリスト
struct PortEventSupportInfo
{
    int           portIndex;
    PortEventType eventType;
};
const PortEventSupportInfo SupportedPortEventList[] =
{
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    // UART B
    {1, PortEventType_SendBufferEmpty},
    {1, PortEventType_SendBufferReady},
    {1, PortEventType_ReceiveBufferReady},
    {1, PortEventType_ReceiveEnd},
#endif

    // UART C
    {2, PortEventType_SendBufferEmpty},
    {2, PortEventType_SendBufferReady},
    {2, PortEventType_ReceiveBufferReady},
    {2, PortEventType_ReceiveEnd},

    // UART D
    {3, PortEventType_SendBufferEmpty},
    {3, PortEventType_SendBufferReady},
    {3, PortEventType_ReceiveBufferReady},
    {3, PortEventType_ReceiveEnd}
};

} // detail
} // driver
} // uart
} // nn
