﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   UART ドライバのレジスタアクセサクラスの実装部(BD-SL-i.MX6 向け)
 * @details UART channel 1 を使用し、以下のセッティングで通信します。
 *            - Baud rate: 115200
 *            - Word size: 8-bit
 *            - Parity: none
 *          参考文献：
 *            - [Ref1] i.MX 6Dual/6Quad Applications Processor Reference Manual, Rev. 1, 04/2013
 */

#include <climits>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>

#include <nn/dd.h>

#include <nn/uart/uart_PortTypes.h>
#include <nn/uart/detail/uart_Log.h>
#include "uart_PortAccessor.h"
#include "uart_CircularBuffer.h"
#include "uart_DdUtil.h"
#include "uart_PortAccessor-hardware.bdsl-imx6.h"
#include "uart_TargetSpec-hardware.bdsl-imx6.h"

namespace nn {
namespace uart {
namespace driver {
namespace detail {

namespace {

/**
 * @brief 参照クロックの分周比に応じて RFDIV(UFCR[7:9]) に設定する値 (3 bit) を返す
 */
const nn::Bit8 GetRfdivRegValue(int clockDivisor)
{
    switch (clockDivisor)
    {
    case 6: return 0; // 000 Divide input clock by 6
    case 5: return 1; // 001 Divide input clock by 5
    case 4: return 2; // 010 Divide input clock by 4
    case 3: return 3; // 011 Divide input clock by 3
    case 2: return 4; // 100 Divide input clock by 2
    case 1: return 5; // 101 Divide input clock by 1
    case 7: return 6; // 110 Divide input clock by 7
    default: // 上記以外は不可
        {
            NN_ABORT("Invalid clock divisor (%d) was specified\n", clockDivisor);
        }
    }
}

/**
 * @brief ターゲットボーレートと参照クロックを元に、UBIR と UBMR にセットする値を決定する
 *        ([Ref1] 64.5 Binary Rate Multiplier (BRM))
 */
void CalcBrmRegisterValues(uint16_t* pOutUbir, uint16_t* pOutUbmr, int baudRate, int refFreq)
{
    NN_SDK_ASSERT(baudRate < INT_MAX / 16); // 値域チェック
    int brmBaudRateClock = 16 * baudRate; // BRMの生成するボーレートクロック

    // (UBMR + 1) / (UBIR + 1) == refFreq / brmBaudRateClock なので、
    // 参照クロックとBRMの生成するボーレートクロックの公約数で割り、それぞれ UBMR と UBIR の値域に収まるなら最良近似
    // 適当な有限の公約数の候補からそれを見つける
    const int commonDivisorTable[] = { 10000, 8192, 3200, 1600, 800 };
    for (const int commonDivisor : commonDivisorTable)
    {
        if (refFreq          % commonDivisor == 0 &&
            brmBaudRateClock % commonDivisor == 0 &&
            refFreq          / commonDivisor <= 0x10000 &&
            brmBaudRateClock / commonDivisor <= 0x10000)
        {
            *pOutUbir = (brmBaudRateClock / commonDivisor) - 1;
            *pOutUbmr = (refFreq / commonDivisor) - 1;
            return;
        }
    }

    // 持ち合わせていない場合は、誤差が出る可能性はあるがアバウトな計算式で出す
    NN_DETAIL_UART_WARN("UART Warning: 使用する BRM 設定では指定されたボーレートから誤差がある可能性があります。\n");
    *pOutUbir = 0xf;
    *pOutUbmr = refFreq / baudRate - 1;
}

}

void PortAccessor::Initialize(int portIdx) NN_NOEXCEPT
{
    SetBaseAddr(portIdx);
}

void PortAccessor::Finalize() NN_NOEXCEPT
{
    m_Reg = nullptr;
}

void PortAccessor::SetBaseAddr(int portIdx) NN_NOEXCEPT
{
    nn::dd::PhysicalAddress   UartPhysicalAddress;
    size_t                    UartAddressSize;

    switch (portIdx)
    {
    case 0:
        {
            UartPhysicalAddress   = 0x002020000ull;
            UartAddressSize       = 0x4000;
        }
        break;
    default:
        NN_ABORT("Unsupported port index (%d).\n", portIdx);
    }

    m_Reg = reinterpret_cast<RegTable*>(GetVirtualAddress(UartPhysicalAddress, UartAddressSize));
    // NN_DETAIL_UART_INFO("uartVirtualAddress = %p\n", uartVirtualAddress);
}

void PortAccessor::EnablePowerAndClock(int baudRate) NN_NOEXCEPT
{
    NN_UNUSED(baudRate);
}

void PortAccessor::DisablePowerAndClock() NN_NOEXCEPT
{
}

/**
 * @brief ポートのレジスタをリセット。BaudRate の設定や FIFO の設定などを行う。
 *        最初は一切の割り込みを有効にしない。
 *        UARTEN ビットはまだ立てない。
 */
void PortAccessor::Setup(int baudRate, FlowControlMode flowControlMode) NN_NOEXCEPT
{
    const int UartModuleClock = 80000000; // 実際のクロック数をランタイムに取得できる外部定義がないので決め打ちとする
    const int UartClockDivisor = 2;

    NN_DETAIL_UART_INFO("UART Info: baudRate=%d\n", baudRate);

    Write32(&m_Reg->ucr1, 0);

    // ソフトリセット ([Ref1] 64.10.2 Software reset)
    Write32(&m_Reg->ucr2, 0); // SRST=0 to trigger reset
    nn::dd::EnsureMemoryAccess();
    while (Read32(&m_Reg->uts) & (1<<0)) {} // Wait for reset completion

    Write32(&m_Reg->ucr3, 0x0704); // Reset value: 0x0700 & RXDMUXSEL=1 ([Ref1] 64.15.6)
    Write32(&m_Reg->ucr4, 0x8000);
    Write32(&m_Reg->uesc, 0x002b); // '+' (0x2b) をエスケープキャラにしているが、
                                     // エスケープシーケンス自体無効なので影響なし
    Write32(&m_Reg->uts, 0);

    // TXTL,RXTL で指定したバイト数だけFIFOが埋まった状態が、Ready/非Readyの閾値となる
    // Tx の場合、FIFO 長 32 のうち 2 個埋まるまでは TRDY が立つ（= 送信 OK）
    // Rx の場合、FIFO 長 32 のうち 1 個埋まっただけで RRDY が立つ (= 受信 OK)
    // TXTL: 2 (default:2, max:32), RFDIV: Divide input clock by 2, RXTL: default
    Write32(&m_Reg->ufcr, (2<<10) | (GetRfdivRegValue(UartClockDivisor)<<7) | (1<<0));

    // Baud rate の設定 ([Ref1] 64.5 Binary Rate Multiplier (BRM))
    uint16_t ubir;
    uint16_t ubmr;
    CalcBrmRegisterValues(&ubir, &ubmr, baudRate, UartModuleClock / UartClockDivisor);

    // UBIR は UBMR より先に書かなくてはならない ([Ref1] 64.15.13)
    Write32(&m_Reg->ubir, ubir);
    nn::dd::EnsureMemoryAccess();
    Write32(&m_Reg->ubmr, ubmr);

    // NN_DETAIL_UART_INFO("UART Info: (ubir+1)/(ubmr+1) = %d/%d\n", static_cast<int>(ubir + 1), static_cast<int>(ubmr + 1));

    // WS, TXEN, RXEN, SRST
    uint32_t ucr2Val = (1<<5) | (1<<2) | (1<<1) | (1<<0);
    switch (flowControlMode)
    {
    case FlowControlMode_None:
        {
            ucr2Val |= (1<<14); // IRTS
        }
        break;
    case FlowControlMode_Hardware: // 標準の Sabre Lite では CTS/RTS の配線がされていないため対応不可
    default:
        NN_ABORT("Unsupported flow control mode (%d)\n", flowControlMode);
    }
    Write32(&m_Reg->ucr2, ucr2Val);

    nn::dd::EnsureMemoryAccess();
}

void PortAccessor::Start() NN_NOEXCEPT
{
    // UARTEN
    SetBitOn32(&m_Reg->ucr1, (1<<0));
    nn::dd::EnsureMemoryAccess();
}

void PortAccessor::Stop() NN_NOEXCEPT
{
    // UARTEN
    SetBitOff32(&m_Reg->ucr1, (1<<0));
    nn::dd::EnsureMemoryAccess();
}

bool PortAccessor::IsTransmitterReady() const NN_NOEXCEPT
{
    // See TRDY bit
    return (Read32(&m_Reg->usr1) & (1<<13)) ? true : false;
}

bool PortAccessor::IsTransmitterReadyInterruptEnabled() const NN_NOEXCEPT
{
    // TRDYEN
    return (Read32(&m_Reg->ucr1) & (1<<13)) ? true : false;
}

void PortAccessor::SetTransmitterReadyInterruptEnabled(bool enable) NN_NOEXCEPT
{
    // TRDYEN
    if (enable)
    {
        SetBitOn32(&m_Reg->ucr1, (1<<13));
    }
    else
    {
        SetBitOff32(&m_Reg->ucr1, (1<<13));
    }
    nn::dd::EnsureMemoryAccess();
}

bool PortAccessor::IsReceiveReady() const NN_NOEXCEPT
{
    // See RRDY bit
    return (Read32(&m_Reg->usr1) & (1<<9)) ? true : false;
}

bool PortAccessor::IsReceiveTimeout() const NN_NOEXCEPT
{
    // TORIAEZU : レジスタが存在するかどうか未確認です。
    // false で返しても影響はないはず(未確定)
    return false;
}

bool PortAccessor::IsReceiveReadyInterruptEnabled() const NN_NOEXCEPT
{
    // RRDYEN
    return (Read32(&m_Reg->ucr1) & (1<<9)) ? true : false;
}

void PortAccessor::SetReceiveInterruptEnabled(bool enable) NN_NOEXCEPT
{
    // RRDYEN
    if (enable)
    {
        SetBitOn32(&m_Reg->ucr1, (1 << 9));
    }
    else
    {
        SetBitOff32(&m_Reg->ucr1, (1 << 9));
    }
    nn::dd::EnsureMemoryAccess();
}

void PortAccessor::SetReceiveReadyInterruptEnabled(bool enable) NN_NOEXCEPT
{
    // RRDYEN
    if (enable)
    {
        SetBitOn32(&m_Reg->ucr1, (1<<9));
    }
    else
    {
        SetBitOff32(&m_Reg->ucr1, (1<<9));
    }
    nn::dd::EnsureMemoryAccess();
}

bool PortAccessor::IsReceiveEnd() const NN_NOEXCEPT
{
    // 非対応
    return false;
}

bool PortAccessor::IsReceiveEndInterruptEnabled() const NN_NOEXCEPT
{
    // 非対応
    return false;
}

void PortAccessor::SetReceiveEndInterruptEnabled(bool enable) NN_NOEXCEPT
{
    // 非対応
    NN_UNUSED(enable);
}

void PortAccessor::ClearReceiveEndInterrupt() NN_NOEXCEPT
{
    // 非対応
}

void PortAccessor::SendByte(char data) NN_NOEXCEPT
{
    while ( !(Read32(&m_Reg->usr1) & (1<<13)) ) {}   // Wait until TRDY is high
    Write32(&m_Reg->utxd, data);
}

PortStatusType PortAccessor::ReceiveByte(char *data) NN_NOEXCEPT
{
    uint32_t val = Read32(&m_Reg->urxd);
    if (val & (1<<14)) // Check error bit
    {
        // Error detected
        // BRK (BREAK キャラクタ受信) はハンドリングしない

#if 0 // パリティは今サポートしていないのでこれは立たない
        if (val & (1<<10)) // PRERR (xxx: not available in 9-bit mode)
        {
            return PortStatusType::ParityError;
        }
        else
#endif
        if (val & (1<<12)) // FRMERR
        {
            return PortStatusType::FrameError;
        }
        else if (val & (1<<13)) // OVRRUN
        {
            *data = val & 0xff; // Received byte seems intact
            return PortStatusType::HardwareOverrun;
        }
        else
        {
            NN_DETAIL_UART_WARN("UART Warning: Rx error reported but no detail bit was set\n");
            // エラーの誤報？想定外だがログだけ出して正常系にレジュームすることにする
        }
    }

    *data = val & 0xff; // Return received data
    return PortStatusType::Success;
}

void PortAccessor::SendDataFromCircularBuffer(PortStatus& outErrorStatus, CircularBuffer& sendBuffer) NN_NOEXCEPT
{
    NN_UNUSED(outErrorStatus);

    if(IsTransmitterReady() && !sendBuffer.IsEmpty())
    {
        const size_t SendChunkSize = 32;
        char sendingData[SendChunkSize];
        size_t sendingSize = sendBuffer.Read(static_cast<char*>(sendingData), SendChunkSize);
        for (size_t i=0; i<sendingSize; i++)
        {
            SendByte(sendingData[i]);
        }
    }
}

void PortAccessor::ReceiveDataToCircularBuffer(PortStatus& outErrorStatus, CircularBuffer& receiveBuffer) NN_NOEXCEPT
{
    const size_t ReceiveChunkSize = 16;
    char receivedData[ReceiveChunkSize];
    size_t receivedSize = 0;

    while(IsReceiveReady())
    {
        auto result = PortStatusType::Success;

        result = ReceiveByte(&receivedData[receivedSize]);
        if (result == PortStatusType::Success)
        {
            receivedSize++;

            if (receivedSize == ReceiveChunkSize || !IsReceiveReady())
            {
                bool wasOverwritten = false;
                size_t written = receiveBuffer.Overwrite(&wasOverwritten,
                                                           static_cast<char*>(receivedData), receivedSize);
                NN_UNUSED(written);
                if (wasOverwritten) // Receive buffer was full, mark error status
                {
                    outErrorStatus.SetStatus(PortStatusType::OutOfBuffer);
                }
                receivedSize = 0;
            }
        }
        else
        {
            // Record last error to return to the next caller (do not overwrite the first error till it is handled)
            outErrorStatus.SetStatus(result);
        }
    }
}

void PortAccessor::ReceiveDataToCircularBufferByPio(PortStatus& outErrorStatus, CircularBuffer& receiveBuffer) NN_NOEXCEPT
{
    // ReceiveDataToCircularBuffer と同じ
    ReceiveDataToCircularBuffer(outErrorStatus, receiveBuffer);
}

bool PortAccessor::MoveRestDataToCircularBuffer(PortStatus& outErrorStatus, CircularBuffer& receiveBuffer) NN_NOEXCEPT
{
    // TORIAEZU : 今は bdsl-imx6 ではソフト的な RTS のコントロールはしない
    NN_UNUSED(outErrorStatus);
    NN_UNUSED(receiveBuffer);
    return false;
}

void PortAccessor::SetRtsToHigh(bool enable) NN_NOEXCEPT
{
    // TORIAEZU : 今は bdsl-imx6 ではソフト的な RTS のコントロールはしない
    NN_UNUSED(enable);
}

bool PortAccessor::IsRestDataExist() NN_NOEXCEPT
{
    // TORIAEZU : 今は bdsl-imx6 ではソフト的な RTS のコントロールはしない
    return false;
}

} // detail
} // driver
} // uart
} // nn
