﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   UART で DMA を使うための DmaAccessor の実装
 * @details DMA 有効時に PortAcsessor のメンバとして使用される
 */

#include <algorithm>
#include <climits>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>

#include <nn/os.h>
#include <nn/os/os_MemoryAttribute.h>
#include <nn/dd.h>

#include <nn/uart/uart_PortTypes.h>
#include <nn/uart/detail/uart_Log.h>
#include "uart_PortAccessor.h"
#include "uart_CircularBuffer.h"
#include "uart_DdUtil.h"
#include "uart_RegisterUart-soc.tegra.h"
#include "uart_Interrupt.h"

#include "uart_DmaAccessor-soc.tegra.h"

namespace nn {
namespace uart {
namespace driver {
namespace detail {

namespace {

// DMA 用のバッファは MemoryPageSize でアラインされている必要がある
const size_t DmaBufferSize  = nn::os::MemoryPageSize;

// 各ポート用の DMA 送受信バッファ
//  XXX: CircularBuffer を流用できるように拡張した方が無駄がない
// TORIAEZU : Session に持たせるとサイズが変わってしまうので、こちらに記載
NN_ALIGNAS(nn::os::MemoryPageSize) char m_DmaSendBuffer[PortCountMax][DmaBufferSize];
NN_ALIGNAS(nn::os::MemoryPageSize) char m_DmaRecvBuffer[PortCountMax][DmaBufferSize];


// DMA ライブラリの初期化カウンタ
int g_DmaInitializeCount = 0;

NN_FORCEINLINE size_t RoundDown(size_t value, size_t align)
{
    return value & ~(align - 1);
}

}

void DmaAccessor::Initialize(int portIdx) NN_NOEXCEPT
{
    m_PortIndex = portIdx;
    if (g_DmaInitializeCount == 0)
    {
        nne::dma::tegra::Initialize();
    }
    g_DmaInitializeCount++;
}

void DmaAccessor::Finalize() NN_NOEXCEPT
{
    if (g_DmaInitializeCount > 0)
    {
        g_DmaInitializeCount--;
        if (g_DmaInitializeCount <= 0)
        {
            nne::dma::tegra::Finalize();
        }
    }
}

void DmaAccessor::Start(const nn::dd::PhysicalAddress uartPhysicalAddress) NN_NOEXCEPT
{
    for (int i = 0; i < DmaChannelIndex_Max; ++i)
    {
        bool  isSend = (i == DmaChannelIndex_Send);
        auto& params = m_DmaParams[i];
        params.buffer = ( isSend ? m_DmaSendBuffer : m_DmaRecvBuffer )[m_PortIndex];
        memset(params.buffer, 0, DmaBufferSize);
        // Make sure to flush the cache to prevent overwriting DMA results
        nn::dd::FlushDataCache(params.buffer, DmaBufferSize);

        const auto& uartRegs = UartRegisterTable[m_PortIndex];
        if (isSend)
        {
            // Send (AHB to APB)
            params.clientBuffer.memoryAddress               = reinterpret_cast<uint64_t>(params.buffer);
            params.clientBuffer.deviceAddress               = uartPhysicalAddress + uartRegs.uartAddress;
            params.clientBuffer.memoryAddressWrapSize       = 0;
            params.clientBuffer.deviceAddressWrapSize       = 4;
        }
        else
        {
            // Receive (APB to AHB)
            params.clientBuffer.deviceAddress               = uartPhysicalAddress + uartRegs.uartAddress;
            params.clientBuffer.memoryAddress               = reinterpret_cast<uint64_t>(params.buffer);
            params.clientBuffer.deviceAddressWrapSize       = 4;
            params.clientBuffer.memoryAddressWrapSize       = 0;
        }
        params.clientBuffer.transferSize = 0;

        auto& interruptNotifier = InterruptNotifier::GetInstance();
        auto* pDmaEvent = interruptNotifier.GetDmaEventType(m_PortIndex, i);
        auto result = nne::dma::tegra::AllocateSession(
                &params.sessionHandle,
                nne::dma::tegra::ModuleId_Uart,
                m_PortIndex,
                reinterpret_cast<uint64_t>(params.buffer),
                DmaBufferSize,
                pDmaEvent);
        // 前回の終了状態によってこの時点でイベントが signal されている可能性があるため、クリアしておく
        nn::os::ClearInterruptEvent(pDmaEvent);
        NN_ABORT_UNLESS_EQUAL(result, nne::dma::tegra::Result_Success);
    }
}

void DmaAccessor::Stop() NN_NOEXCEPT
{
    auto& interruptNotifier = InterruptNotifier::GetInstance();
    for (int i = 0; i < DmaChannelIndex_Max; ++i)
    {
        nne::dma::tegra::AbortTransfer(m_DmaParams[i].sessionHandle);
        interruptNotifier.SetDmaHandler(m_PortIndex, i, nullptr, 0);
        nne::dma::tegra::FreeSession(m_DmaParams[i].sessionHandle);
    }
}

bool DmaAccessor::CopyDataFromDmaBufferToCircularBuffer(PortStatus& outErrorStatus, CircularBuffer& receiveBuffer) NN_NOEXCEPT
{
    auto& dmaParam = m_DmaParams[DmaChannelIndex_Receive];

    size_t receivedSize = 0;

    // バッファに格納
    auto result = nne::dma::tegra::GetTransferredCount( dmaParam.sessionHandle, &receivedSize);

    if (result == nne::dma::tegra::Result_Success && receivedSize > 0)
    {
        nn::dd::InvalidateDataCache( dmaParam.buffer, receivedSize);

        if (receiveBuffer.IsWritableSpaceExist(receivedSize))
        {
            bool wasOverwritten = false;
            size_t written = receiveBuffer.Overwrite(
                &wasOverwritten,
                static_cast<char*>(dmaParam.buffer),
                receivedSize);
            NN_UNUSED(written);
            if (wasOverwritten) // Receive buffer was full, mark error status
            {
                outErrorStatus.SetStatus(PortStatusType::OutOfBuffer);
            }
            m_RestDataSize = 0;
        }
        else
        {
            dmaParam.clientBuffer.transferSize = 0;
            m_RestDataSize = receivedSize;
            return false;
        }
    }
    dmaParam.clientBuffer.transferSize = 0;

    return true;
}

bool DmaAccessor::IsDmaComplete(DmaChannelIndex index) NN_NOEXCEPT
{
    auto& dmaParam = m_DmaParams[index];

    // Check for data that has already been sent
    if (dmaParam.clientBuffer.transferSize > 0)
    {

        // 前回の DMA 送信が完了していない場合は受信関連の割り込みを一旦 disable にして抜ける
        // 完了していれば、セマフォが取れて次に進める
        if (!nn::os::TryAcquireSemaphore(&dmaParam.completeSemaphore))
        {
            return false;
        }

        // セマフォが取れた場合は返却しておく
        nn::os::ReleaseSemaphore(&dmaParam.completeSemaphore);
        return true;
    }
    else
    {
        // transferSize が 0 の時は完了しているものとして true を返す
        return true;
    }
}

void DmaAccessor::SetSendDmaTransferSize(size_t size) NN_NOEXCEPT
{
    auto& dmaParam = m_DmaParams[DmaChannelIndex_Send];
    dmaParam.clientBuffer.transferSize = size;
}

void DmaAccessor::RequestSendDataByDma(CircularBuffer& sendBuffer) NN_NOEXCEPT
{
    auto& dmaParam = m_DmaParams[DmaChannelIndex_Send];
    const size_t BytesToSend = std::min( RoundDown(sendBuffer.GetReadableLength(), DmaBufferAlign), DmaBufferSize);

    // Circular Buffer に値が入っていれば 読み出して if 文の中に入る
    if ((dmaParam.clientBuffer.transferSize = sendBuffer.Read(dmaParam.buffer, BytesToSend)) > 0 )
    {
        //NN_DETAIL_UART_TRACE("BytesToSendByDMA = %d\n", BytesToSend );

        // キャッシュを dma のメモリに書き出す
        nn::dd::FlushDataCache(dmaParam.buffer, dmaParam.clientBuffer.transferSize);

        nne::dma::tegra::Result result;
        do
        {
            // dma による送信の開始
            result = nne::dma::tegra::StartTransfer( dmaParam.sessionHandle,
                                                     &dmaParam.clientBuffer,
                                                     nne::dma::tegra::Direction_MemoryToDevice,
                                                     &dmaParam.completeSemaphore);

            if (result != nne::dma::tegra::Result_Success && result != nne::dma::tegra::Result_Busy)
            {
                NN_DETAIL_UART_ERROR("DMA transfer failed to start (%d)\n", result);
            }

        } while (result == nne::dma::tegra::Result_Busy); // Result_Busy の間はループし続ける
    }
    else
    {
        dmaParam.clientBuffer.transferSize = 0;
    }
}

// TORIAEZU: 関数の名前を変えた方がいいかも
bool DmaAccessor::AbortReceiveDataTransfer(PortStatus& outErrorStatus, CircularBuffer& receiveBuffer) NN_NOEXCEPT
{
    auto& dmaParam = m_DmaParams[DmaChannelIndex_Receive];

    size_t receivedSize = 0;

    nne::dma::tegra::AbortTransfer(dmaParam.sessionHandle);

    // Check for data that has already been sent
    if (dmaParam.clientBuffer.transferSize > 0)
    {
        // DMA で転送できている分のサイズを確認
        auto result = nne::dma::tegra::GetTransferredCount( dmaParam.sessionHandle, &receivedSize);

        // DMA で転送が完了している場合、DMA のバッファから CircularBuffer に移す
        if (result == nne::dma::tegra::Result_Success && receivedSize > 0)
        {
            nn::dd::InvalidateDataCache(dmaParam.buffer, receivedSize);

            if (receiveBuffer.IsWritableSpaceExist(receivedSize))
            {
                bool wasOverwritten = false;
                size_t written = receiveBuffer.Overwrite(
                    &wasOverwritten,
                    static_cast<char*>(dmaParam.buffer),
                    receivedSize);
                //NN_DETAIL_UART_INFO("[---1] Write Data %d bytes\n", receivedSize);
                NN_UNUSED(written);
                if (wasOverwritten) // Receive buffer was full, mark error status
                {
                    outErrorStatus.SetStatus(PortStatusType::OutOfBuffer);
                }
            }
            else
            {
                dmaParam.clientBuffer.transferSize = 0;
                m_RestDataSize = receivedSize;
                return false;
            }
        }
    }
    dmaParam.clientBuffer.transferSize = 0;
    return true;
}

bool DmaAccessor::MoveRestDataToCircularBuffer(PortStatus& outErrorStatus, CircularBuffer& receiveBuffer) NN_NOEXCEPT
{
    auto& dmaParam = m_DmaParams[DmaChannelIndex_Receive];

    if (receiveBuffer.IsWritableSpaceExist(m_RestDataSize))
    {
        bool wasOverwritten = false;
        size_t written = receiveBuffer.Overwrite(
            &wasOverwritten,
            static_cast<char*>(dmaParam.buffer),
            m_RestDataSize);
        NN_UNUSED(written);
        if (wasOverwritten) // Receive buffer was full, mark error status
        {
            outErrorStatus.SetStatus(PortStatusType::OutOfBuffer);
        }
        m_RestDataSize = 0;
        return true;
    }
    else
    {
        return false;
    }
}

void DmaAccessor::SuspendDmaTransfer() NN_NOEXCEPT
{
    // スレッドが走ってないことのチェック
    NN_SDK_ASSERT(!InterruptNotifier::GetInstance().IsThreadRunning());

    // Send / Receive 両方とも AbortTransfer する
    // DMA 用のバッファに入ったデータは無視しておく。(Send / Receive ともに上書きされる)
    nne::dma::tegra::AbortTransfer(m_DmaParams[DmaChannelIndex_Send].sessionHandle);
    nne::dma::tegra::AbortTransfer(m_DmaParams[DmaChannelIndex_Receive].sessionHandle);

    auto& interruptNotifier = InterruptNotifier::GetInstance();
    for (int i = 0; i < DmaChannelIndex_Max; ++i)
    {
        // Suspend 時は MultiWait しているスレッドが終了しているので、そのまま Unlink する。
        interruptNotifier.LinkDmaHandler(m_PortIndex, i, nullptr, 0);
        nne::dma::tegra::FreeSession(m_DmaParams[i].sessionHandle);
    }

}

void DmaAccessor::ResumeDmaTransfer(const nn::dd::PhysicalAddress uartPhysicalAddress) NN_NOEXCEPT
{

    Start(uartPhysicalAddress);
}

void DmaAccessor::RequestReceiveDataByDma() NN_NOEXCEPT
{
    auto& dmaParam = m_DmaParams[DmaChannelIndex_Receive];

    nne::dma::tegra::Result result;

    // 前回の DMA が完了していた場合、新しく DMA を行う
    nn::dd::FlushDataCache(dmaParam.buffer, DmaBufferSize);

    dmaParam.clientBuffer.transferSize = DmaBufferSize;
    result = nne::dma::tegra::StartTransfer(
            dmaParam.sessionHandle,
            &dmaParam.clientBuffer,
            nne::dma::tegra::Direction_DeviceToMemory,
            &dmaParam.completeSemaphore);

    if (result == nne::dma::tegra::Result_Busy)
    {
        NN_DETAIL_UART_ERROR("DMA transfer busy (%d)\n", result);
    }
    else if (result == nne::dma::tegra::Result_Success)
    {
        //NN_DETAIL_UART_INFO("[uart] Start DMA receive\n");
    }
    else
    {
        NN_DETAIL_UART_ERROR("DMA transfer failed to start (%d)\n", result);
        dmaParam.clientBuffer.transferSize = 0;
    }
}

void DmaAccessor::RunInterruptHandler(int channelIndex) NN_NOEXCEPT
{
    nne::dma::tegra::InterruptHandler(m_DmaParams[channelIndex].sessionHandle);
}

} // detail
} // driver
} // uart
} // nn
