﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <climits>
#include <mutex>
#include <cmath>

#include <nn/nn_Common.h>

#include <nn/result/result_HandlingUtility.h>

#include <nn/ts/detail/ts_Log.h>
#include <nn/ts/impl/ts.h>

#include "ts_DeviceControllerImpl.tmp451.h"
#include "ts_ResourceManager.h"
#include "ts_SettingsHolder.h"
#include "ts_TargetSpecTmp451.h"

namespace nn {
namespace ts {
namespace impl {
namespace detail {

namespace {

nnd::tmp451::Temperature ConvertMilliCToTemperature(int milliC)
{
    auto integer = milliC / 1000;
    auto decimal = (milliC % 1000) * (nnd::tmp451::DecimalUnitDigit / 1000);

    // temperature.decimal は TMP451 の仕様から正の値を設定する
    if ( decimal < 0 )
    {
        integer--;
        decimal += nnd::tmp451::DecimalUnitDigit;
    }

    return { integer, decimal };
}

} // namespace

nn::Result Tmp451Controller::InitializeDevice() NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        m_SettingsHolder.LoadSettings();

        nnd::tmp451::Initialize();

        // TODO:エラーハンドリング
        // デバイス動作の設定
        NN_RESULT_DO(nnd::tmp451::SetMeasurementRange(RangeSetting));
        NN_RESULT_DO(nnd::tmp451::SetPin1DetectionSettings(ConvertMilliCToTemperature(m_SettingsHolder.GetRemoteShutdownTemperatureHighLimitMilliC()), Pin1LocalHighLimit));
        NN_RESULT_DO(nnd::tmp451::SetPin2DetectionSettings(Pin2RemoteLowLimit, Pin2RemoteHighLimit, Pin2LocalLowLimit, Pin2LocalHighLimit, nnd::tmp451::DetectionMode::Off));
        NN_RESULT_DO(nnd::tmp451::SetMeasurementRateBy2ToThePowerN(Rate));
        NN_RESULT_DO(nnd::tmp451::SetBusTimeoutFunctionEnabled(BusTimeout));
        NN_RESULT_DO(nnd::tmp451::SetRemoteDigitalFilterControl(RemoteFilter));
        NN_RESULT_DO(nnd::tmp451::SetRemoteTemperatureOffset(ConvertMilliCToTemperature(m_SettingsHolder.GetRemoteTemperatureOffsetMilliC())));
        NN_RESULT_DO(nnd::tmp451::SetRemoteEtaFactorCorrectionValueByN(RemoteEta));
        NN_RESULT_DO(nnd::tmp451::SetThermHysterisis(Hysterisis));
        NN_RESULT_DO(nnd::tmp451::SetConsecutiveAlert(Consecutive));

        // THERM 機能有効化のために常時計測は実行しておく
        NN_RESULT_DO(nnd::tmp451::SetMeasurementMode(nnd::tmp451::MeasurementMode::Continuous));

        m_Initialized = true;
    }
    return nn::ResultSuccess();
}



nn::Result Tmp451Controller::FinalizeDevice() NN_NOEXCEPT
{
    if(IsInitialized())
    {
        NN_RESULT_DO(nnd::tmp451::Finalize());
        m_Initialized = false;
    }
    return nn::ResultSuccess();
}



LimitTemperature Tmp451Controller::GetTemperatureRange(nnd::tmp451::Location location) NN_NOEXCEPT
{
    // TODO: Tmp451 にレンジを数値で取得する関数を nnd::tmp451 に追加する
    switch (location)
    {
    case nnd::tmp451::Location::Remote:
        {
            return RangeTemperature;// FIXME: 暫定値
        }
        break;
    case nnd::tmp451::Location::Local:
        {
            return RangeTemperature;// FIXME: 暫定値
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}



nn::Result Tmp451Controller::GetTemperature(int* pOutTemperatureInteger, nnd::tmp451::Location location) NN_NOEXCEPT
{
    nnd::tmp451::Temperature temp;
    NN_RESULT_DO(nnd::tmp451::ReadTemperature(&temp, location));

    *pOutTemperatureInteger = temp.integer;

    return nn::ResultSuccess();
}

nn::Result Tmp451Controller::GetTemperatureMilliC(int* pOutTemperatureMilliCInteger, nnd::tmp451::Location location) NN_NOEXCEPT
{
    nnd::tmp451::Temperature temp;
    NN_RESULT_DO(nnd::tmp451::ReadTemperature(&temp, location));

    *pOutTemperatureMilliCInteger = temp.integer * 1000 + temp.decimal / 10;

    return nn::ResultSuccess();
}

nn::Result Tmp451Controller::SetMeasurementMode(nnd::tmp451::MeasurementMode measurementMode) NN_NOEXCEPT
{
    NN_DETAIL_TS_TRACE("measurement mode: %d\n", static_cast<int>(measurementMode));
    return nnd::tmp451::SetMeasurementMode(measurementMode);
}



}}}} // nn::ts::impl::detail
