﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "tmipc_node_tics.h"
#include "tmipc_packet.h"
#include "tmipc_node_tics_interface.h"
#include "../thread_tracker.h"
#include "../Version.h"
#include "../performance_monitor/performance_monitor_service.h"
#include <nn/nn_SdkLog.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/os.h>

//==============================================================================
namespace tmipc {
//==============================================================================

Thread      NodeTICS::m_ListenThread;                       // Threads for listen, send and
Thread      NodeTICS::m_SendThread;                         // receive.
Thread      NodeTICS::m_ReceiveThread;                      //
void*       NodeTICS::m_pListenThreadStack = { nullptr };   // The thread stacks for listen,
void*       NodeTICS::m_pSendThreadStack   = { nullptr };   // send and receive.
void*       NodeTICS::m_pReceiveThreadStack;                //
const int   s_StackSize = { 16 * 1024 };
u8          s_ListenStack[s_StackSize] __attribute__((__aligned__(4096)));
u8          s_SendStack[s_StackSize] __attribute__((__aligned__(4096)));
u8          s_ReceiveStack[s_StackSize] __attribute__((__aligned__(4096)));

// ============================================================================

NodeTICS::NodeTICS()
: Node()
{
    DEJA_TRACE( "NodeTICS::NodeTICS", "NodeTICS" );
    m_ListenEvent.Create();
    m_ListenEvent.Reset();
    // Set up the send and receive queues.
    m_SendQueue.Create( 64 );

    // Create the Send/Receive thread stacks.
    m_pListenThreadStack    = s_ListenStack;
    m_pSendThreadStack      = s_SendStack;
    m_pReceiveThreadStack   = s_ReceiveStack;

    Init();

    m_bIsListening   = false;
    m_bStopListening = false;
    m_bIsConnected   = false;
}

// ============================================================================

NodeTICS::~NodeTICS()
{
    StopListening();
    Disconnect();

    m_SendQueue.Destroy();

    m_pSendThreadStack = nullptr;
    m_pReceiveThreadStack = nullptr;

    m_ListenEvent.Destroy();
}

// ============================================================================
// Initialize the TICS interface.
void NodeTICS::Initialize()
{
    TICSInterface::Initialize();
}

// ============================================================================
// Finalize NodeTICS.
void NodeTICS::Finalize()
{
    TICSInterface::Finalize();
}

// ============================================================================
// Listen thread.
s32 NodeTICS::ListenThread( void* lpThis )
{
    DEJA_CONTEXT( "NodeTICS::ListenThread", "Listen( %p )\n", lpThis );
    NodeTICS *pThis = reinterpret_cast<NodeTICS*>(lpThis);
    NN_ABORT_UNLESS( pThis != nullptr, "NodeTICS::ListenThread(...) lpThis == null." );

TMA_POWER_TEST_PRINT( "[%s] !!! - Start.\n", _BestFunctionName_ );
    // Continues to keep attempting to connect, until m_bStopListening is set
    // to false.
    tmipc::Result Result = { tmipc::TMIPC_RESULT_OK };
    while( !pThis->m_bStopListening && (Result == tmipc::TMIPC_RESULT_OK) )
    {
        // Start TICS.
        if( Result == tmipc::TMIPC_RESULT_OK )
        {
            Result = TICSInterface::StartSession( pThis->GetThreadPriority() );
        }

        if( Result == tmipc::TMIPC_RESULT_OK )
        {
            // Start the send/receive threads.
            pThis->StartThreads();
            // Wait for the Listen event to trigger - either through a SendThread
            // shutdown, or a StopListening().
            pThis->m_ListenEvent.Wait( TMIPC_INFINITE );
            // Disconnect.
            pThis->Disconnect();

            // Reset the Listen event.
            pThis->m_ListenEvent.Reset();
        }

        // This needs to happen, regardless if a previous error occurred.
        const tmipc::Result CleanupResult = TICSInterface::CleanupSession();
        if( Result == tmipc::TMIPC_RESULT_OK )
        {
            Result = CleanupResult;
        }
    }
TMA_POWER_TEST_PRINT( "[%s] !!! - Finish.  Result: %d.\n", _BestFunctionName_, Result );

    return 0;
}

// ============================================================================
// Send thread.
s32 NodeTICS::SendThread( void* lpThis )
{
    DEJA_TRACE( "NodeTICS::SendThread", "SendThread( %p )", lpThis );
    tmipc::Result Result = TMIPC_RESULT_OK;
TMA_POWER_TEST_PRINT( "[%s] !!! - Start.\n", _BestFunctionName_ );

    NodeTICS* pThis = reinterpret_cast<NodeTICS*>(lpThis);

    // Loop until the connection "closes".
    bool    NullPacketWasReceived = false;
    Packet* pPacket = nullptr;
    while( !NullPacketWasReceived && (Result == TMIPC_RESULT_OK) )
    {
        // Read a packet from the Message Queue.
        pPacket = (Packet*)pThis->m_SendQueue.Receive();
        NullPacketWasReceived = pPacket == nullptr;

        // Terminate thread if a nullptr packet.
        if( pPacket != nullptr)
        {
            // Send the packet over the Socket.
            if( pThis->IsConnected() )
            {
                Result = TICSInterface::Write( *pPacket );
                if( Result == TMIPC_RESULT_OK )
                {
                    ADD_SENT_PACKET( *pPacket );
                }
                else
                {
                    DEJA_TRACE( "NodeTICS::SendThread", "Failed to Send: %p, %d bytes. Result == %d", pPacket->GetBuffer(), pPacket->GetDataLen(), Result );
                }
            }

            // Free the packet.
            pThis->FreePacket( pPacket );
            pThis->Tick();
        }
    }

    // Send Diconnected to the Work Thread.
    pThis->Disconnected();

    // Delete any pending packets.
    while( pThis->m_SendQueue.GetCount() > 0 )
    {
        Packet* pPacket = (Packet*)pThis->m_SendQueue.Receive();
        if( pPacket )
        {
            pThis->FreePacket( pPacket );
        }
    }

    // ASSERT no more packets.
    ASSERT( pThis->m_SendQueue.GetCount() == 0 );

    DEJA_TRACE( "NodeTICS::SendThread", "SendThread Stopped" );

    // Trigger the Listen to "iterate to the next session".
    pThis->m_ListenEvent.Set();
TMA_POWER_TEST_PRINT( "[%s] !!! - Finish.\n", _BestFunctionName_ );

    // Thread exit.
    return Result;
}

// --------------------------------------------------------------------------
// Receive thread.
s32 NodeTICS::ReceiveThread( void* lpThis )
{
    DEJA_THREAD_LABEL( "NodeTICS::ReceiveThread" );
    DEJA_TRACE( "NodeTICS::ReceiveThread", "ReceiveThread Started" );
    tmipc::Result Result = { TMIPC_RESULT_OK };
TMA_POWER_TEST_PRINT( "[%s] !!! - Start.\n", _BestFunctionName_ );

    NodeTICS* pThis = reinterpret_cast<NodeTICS*>(lpThis);

    // Loop until the connection is shut down or disconnected.
    bool bDone = false;
    while( !bDone && (Result == TMIPC_RESULT_OK) )
    {
        // Allocate a packet.
        Packet* pPacket = pThis->AllocRecvPacket();
        ASSERT( pPacket );

        // Read the packet.
        if( Result == TMIPC_RESULT_OK )
        {
            Result = TICSInterface::Read( *pPacket );

            if( Result == TMIPC_RESULT_OK )
            {
                ADD_RECEIVED_PACKET( *pPacket );
            }
            else
            {
                DEJA_TRACE( "NodeTICS::ReceiveThread", "Failed to read packet: %p", pPacket->GetBuffer() );
            }
        }

        // Dispatch the packet as a Work Item.
        if( Result == TMIPC_RESULT_OK )
        {
            // What type of packet was this?
            const u32 ServiceId = pPacket->GetServiceId();
            const bool bNormalPacket =
            {
                    (ServiceId != ServiceId_NodeTICS_BeaconQuery)
                &&  (ServiceId != ServiceId_NodeTICS_TMSInfo )
            };
            if( bNormalPacket )
            {
                pThis->ProcessReceived( pPacket );
                pPacket = nullptr;
            }
            else
            {
TMA_POWER_TEST_PRINT( "[%s]!!! - Received 'Special Connection Packet'.  ServiceID: 0x%08X.  Start.\n", _BestFunctionName_, ServiceId );
                // This is a special packet.  It needs to be handled differently.
                switch( ServiceId )
                {
                case ServiceId_NodeTICS_BeaconQuery:
                {
                    // "Beacon packet"

                    // Read the TargetManager's version number.
                    if( Result == TMIPC_RESULT_OK )
                    {
                        u32 Version = 0;
                        pPacket->ReadU32( Version );
                    }

                    // Cancel any pending tasks.
                    if( Result == TMIPC_RESULT_OK )
                    {
//                        pThis->CancelAllPendingTasks();
                    }

                    // Send the requested information back to TargetManager.
                    if( Result == TMIPC_RESULT_OK )
                    {
                        // Reset the packet cursor, for the response packet.
                        pPacket->ResetCursor();
                        Result = TICSInterface::SendBeaconPacket( *pPacket );

                        // Assume a connection has been made, until TargetManager
                        // "breaks the connection" (rejection based the beacon
                        // information and/or a disconnect or app termination).
                        pThis->m_bIsConnected = Result == tmipc::TMIPC_RESULT_OK;
                        // Success.
                        if( Result == tmipc::TMIPC_RESULT_OK )
                        {
                            pThis->OnEvent( NodeEvent_Connected );
                        }
                        else
                        {
                            TMIPC_ERROR_LOG( "ReceiveThread SendBeaconPacket failed.\n" );
                        }
                    }
                }
                break;
                case ServiceId_NodeTICS_TMSInfo:
                {
TMA_POWER_TEST_PRINT( "[tmipc_node_tics] Received ServiceId_NodeTICS_TMSInfo.  Packet length: %d.\n", pPacket->GetDataLen() );
                    // Parse the TMSInfo.
                    s32 StringLength = { 0 };
                    const s32 TMSInfoLength = { TMIPC_TMSINFO_PACKET_SIZE - sizeof(StringLength) };
                    char TMSInfo[TMSInfoLength];
                    pPacket->ReadS32( StringLength );
                    if( StringLength <= TMSInfoLength )
                    {
                        pPacket->ReadData( TMSInfo, StringLength );
                        pThis->ProcessTMSInfoString( TMSInfo );
                    }
                }
                break;
                default:
                    // Do nothing.  Just let the packet get Deleted.
                    break;
                }

                // There is no connection, therefore the packet must be
                // manually deallocated.
                pThis->FreePacket( pPacket );
TMA_POWER_TEST_PRINT( "[%s]!!! - Received 'Special Connection Packet'.  ServiceID: 0x%08X.  Finished.\n", _BestFunctionName_, ServiceId );
            }
        }
        else
        {
            bDone = true;
            pThis->FreePacket( pPacket );
        }
    }

    // Send a nullptr to terminate the SendThread.
    pThis->m_SendQueue.Send( nullptr );

    // The send/receive threads will end, which means the connection be gone.
    pThis->m_bIsConnected = false;

    DEJA_TRACE( "NodeTICS::ReceiveThread", "ReceiveThread Stopped" );
TMA_POWER_TEST_PRINT( "[%s] !!! - Finish.\n", _BestFunctionName_ );

    // Thread exit.
    return Result;
}

// ============================================================================
// Starts the Send/Receive threads.
void NodeTICS::StartThreads()
{
    // Cancel any existing blocking calls.
    VERIFY( m_ReceiveThread.Join() == Result::TMIPC_RESULT_OK );
    VERIFY( m_SendThread.Join() == Result::TMIPC_RESULT_OK );

    // Start Send/Receive threads.
    m_SendThread.Start( SendThread, this, m_pSendThreadStack, s_StackSize, NN_SYSTEM_THREAD_PRIORITY(tma, TmaTicsSend), NN_SYSTEM_THREAD_NAME(tma, TmaTicsSend) );
    m_ReceiveThread.Start( ReceiveThread, this, m_pReceiveThreadStack, s_StackSize, NN_SYSTEM_THREAD_PRIORITY(tma, TmaTicsRecv), NN_SYSTEM_THREAD_NAME(tma, TmaTicsRecv) );

    // Register the threads.
    tma::ThreadTracker::RegisterThread( m_SendThread.GetThreadType(), tma::ThreadTracker::ThreadId::TmipcSend );
    tma::ThreadTracker::RegisterThread( m_ReceiveThread.GetThreadType(), tma::ThreadTracker::ThreadId::TmipcReceive );
}

// ============================================================================
// Stops the Send/Receive threads.
void NodeTICS::StopThreads()
{
    // Cancel any existing blocking calls.
    VERIFY( m_ReceiveThread.Join() == Result::TMIPC_RESULT_OK );
    VERIFY( m_SendThread.Join() == Result::TMIPC_RESULT_OK );

    // Unregister the threads.
    tma::ThreadTracker::UnregisterThread( tma::ThreadTracker::ThreadId::TmipcSend );
    tma::ThreadTracker::UnregisterThread( tma::ThreadTracker::ThreadId::TmipcReceive );
}

// ============================================================================
// Listens for connections.
tmipc::Result NodeTICS::Listen()
{
    Result Result = { tmipc::TMIPC_RESULT_OK };
    m_bStopListening = false;

    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        if (!IsListening())
        {
            // Start listening.
            m_bIsListening = true;
            Result = m_ListenThread.Start( ListenThread, this, m_pListenThreadStack, s_StackSize, NN_SYSTEM_THREAD_PRIORITY(tma, TmaTicsListen), NN_SYSTEM_THREAD_NAME(tma, TmaTicsListen) );
        }
    }

    // Register the thread.
    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        tma::ThreadTracker::RegisterThread( m_ListenThread.GetThreadType(), tma::ThreadTracker::ThreadId::TmipcListen );
    }

    return Result;
}

// ============================================================================
// Stops listening for connections.  This does *not* cause a disconnect.
void NodeTICS::StopListening()
{
TMA_POWER_TEST_PRINT( "[%s] !!! - Start.\n", _BestFunctionName_ );
    if( IsListening() )
    {
        m_bStopListening = true;
        TICSInterface::StopSession();
        m_ListenEvent.Set();
        VERIFY( m_ListenThread.Join() == Result::TMIPC_RESULT_OK );
    }
    m_bIsListening = false;

    // Register the thread.
    tma::ThreadTracker::UnregisterThread( tma::ThreadTracker::ThreadId::TmipcListen );

TMA_POWER_TEST_PRINT( "[%s] !!! - Finish.\n", _BestFunctionName_ );
}

// ============================================================================
// Disconnect from the target.
Result NodeTICS::Disconnect()
{
    DEJA_TRACE( "NodeTICS::Disconnect", "Starting thread" );
TMA_POWER_TEST_PRINT( "[%s] !!! - Start.\n", _BestFunctionName_ );

    Result Result = TMIPC_RESULT_OK;
    if( IsConnected() )
    {
        // Trigger the Listen to "iterate to the next session".
        m_ListenEvent.Set();

        // Recv thread should terminate when socket closes.
        VERIFY( m_ReceiveThread.Join() == TMIPC_RESULT_OK );

        // The ReceiveThread sends a nullptr to terminate the Send Thread.
        VERIFY( m_SendThread.Join() == TMIPC_RESULT_OK );
    }

TMA_POWER_TEST_PRINT( "[%s] !!! - Finish.\n", _BestFunctionName_ );

    DEJA_TRACE( "NodeTICS::Disconnect", "Exiting thread" );

    return Result;
}

// ============================================================================
// Is the Node listening for connections?
bool NodeTICS::IsListening() const
{
    return m_bIsListening;
}

// ============================================================================
// Is the Target connected a host app (Win32 - TargetManager, etc.)?
bool NodeTICS::IsConnected()
{
    return m_bIsConnected;
}

// ============================================================================
// Send data to the host app (Win32 - TargetManager, etc.).
// Sends a packet to the remote connection.
Result NodeTICS::Send( Packet* pPacket )
{
    DEJA_TRACE( "NodeTICS::Send", "Send( %p )", pPacket );
    tmipc::Result Result = { tmipc::TMIPC_RESULT_OK };

    // Validate the input parameter.
    if( Result == tmipc::TMIPC_RESULT_OK )
    {
        if( pPacket == nullptr )
        {
            Result = tmipc::TMIPC_RESULT_FAILED;
            TMIPC_ERROR_LOG( "Send failed: null packet.\n" );
        }
    }

    // Make sure a connection has been made.
    if( Result == TMIPC_RESULT_OK )
    {
        ScopedLock lock( m_Lock );

        if( IsConnected() )
        {
            // Add the packet to the Message Queue.
            m_SendQueue.Send( pPacket );
        }
        else
        {
            // No connection, just free the packet.
            FreePacket( pPacket );
            Result = TMIPC_RESULT_DISCONNECTED;

            // Need to Tick the WorkThread if we get here to process the FreePacket.
            Tick();
        }
    }

    return Result;
}

//==============================================================================
} // namespace tmipc
//==============================================================================
