﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/socket.h>

#include "tmipc_node_tcp.h"
#include "tmipc_result.h"
#include "tmipc_packet.h"
#include "tmipc_service.h"
#include "tmipc_task.h"
#include "../DejaInsight.h"
#include "../Version.h"

//==============================================================================
namespace tmipc {
//==============================================================================

s32 WriteSocket( int s, void* pBuffer, s32 nBytes )
{
    ASSERT( s != nn::socket::InvalidSocket );

    // Loop to send all data to the socket.
    s32 bytesSent = 0;
    u8* p = (u8*)pBuffer;
    while( nBytes > 0 )
    {
        s32 Len = (s32)nn::socket::Send( s, (char*)p, nBytes, nn::socket::MsgFlag::Msg_None );
        if( Len == nn::socket::SocketError )
        {
            // Abort on a socket error.
            bytesSent = nn::socket::SocketError;
            //TMIPC_ERROR_LOG( "WriteSocket socket error.\n" );
            break;
        }
        p         += Len;
        nBytes    -= Len;
        bytesSent += Len;
    }

    // Return number of bytes sent
    return( bytesSent );
}

//==============================================================================

s32 ReadSocket( int s, void* pBuffer, s32 nBytes )
{
    ASSERT( s != nn::socket::InvalidSocket );

    // Loop to receive all the data.
    s32 bytesReceived = 0;
    u8* p = (u8*)pBuffer;
    while( nBytes > 0 )
    {
        s32 Len = (s32)nn::socket::Recv( s, (char*)p, nBytes, nn::socket::MsgFlag::Msg_None );
        if( Len == nn::socket::SocketError )
        {
            // Abort on a socket error.
            bytesReceived = nn::socket::SocketError;
            //TMIPC_ERROR_LOG( "ReadSocket socket error.\n" );
            break;
        }
        else if( Len == 0 )
        {
            // This indicates a graceful socket close.
            break;
        }

        p             += Len;
        nBytes        -= Len;
        bytesReceived += Len;
    }

    // Return number of bytes received
    return( bytesReceived );
}

//==============================================================================
#if defined( TMIPC_TARGET_WIN )
NodeAddressTCP::NodeAddressTCP()
{
}

//==============================================================================

NodeAddressTCP::NodeAddressTCP( const char* pIP, s32 port )
{
    Set( pIP, port );
}

//==============================================================================

NodeAddressTCP::~NodeAddressTCP()
{
}

//==============================================================================

bool AddrFromDotString( const char* pString, u32& Addr )
{
    Addr = 0;
    s32 iByte = 0;

    while( (*pString != 0) && (iByte < 4) )
    {
        s32 Value = 0;
        while( isdigit( *pString ) )
        {
            Value *= 10;
            Value += *pString++ - '0';
        }

        if( ((*pString != '.') && (*pString != 0  )) )
        {
            Addr = 0;
            return( false );
        }
        pString++;

#ifdef TARGET_LITTLE_ENDIAN
        s32 Shift = iByte * 8;
#else
        s32 Shift = 24 - (iByte * 8);
#endif
        Addr = Addr | (Value << Shift);
        iByte++;
    }

    return( true );
}

//==============================================================================

void NodeAddressTCP::Set( const char* pIP, s32 port )
{
    // Convert the string based IP to a u32.
#if defined( TMIPC_TARGET_WIN )
    nn::socket::InAddr addr;
    nn::socket::InetPton( nn::socket::Family::Af_Inet, pIP, &addr );
    u32 ip = addr.S_addr;
#elif defined( TMIPC_TARGET_HORIZON )
    u32 ip = 0;
    AddrFromDotString( pIP, ip );
#else
    #error Target Platform Undefined.
#endif

    // Fill the sockaddr.
    memset( &m_SockAddr, 0, sizeof( m_SockAddr ) );
    m_SockAddr.sin_family      = nn::socket::Family::Af_Inet;
    m_SockAddr.sin_port        = nn::socket::InetHtons( (u16)port );
    m_SockAddr.sin_addr.S_addr = ip;
}

//==============================================================================

const nn::socket::SockAddrIn& NodeAddressTCP::SockAddr() const
{
    return( m_SockAddr );
}
#endif // TMIPC_TARGET_WIN

//==============================================================================

NodeTCP::NodeTCP()
:   m_ListenSocket      ( nn::socket::InvalidSocket )
,   m_ConnectedSocket   ( nn::socket::InvalidSocket )
,   m_pSendThreadStack  ( nullptr )
,   m_pRecvThreadStack  ( nullptr )
,   m_pListenThreadStack( nullptr )
{
    m_SendQ.Create( 64 );
    DEJA_TRACE( "NodeTCP::NodeTCP", "NodeTCP constructor" );
    //TMA_TRACE( "NodeTCP::NodeTCP", "NodeTCP constructor" );

#if defined( TMIPC_TARGET_WIN )
    // Initialize Winsock
    WSADATA wsaData;
    s32 result = WSAStartup( MAKEWORD(2,2), &wsaData );
    (void)result;
    ASSERT( result == 0 );
#endif
}

//==============================================================================

NodeTCP::~NodeTCP()
{
    DEJA_TRACE( "NodeTCP::~NodeTCP", "NodeTCP destructor" );

    // Call this *before* calling Disconnect, because a connection may not have
    // finished before tearing down the threads.
    StopListening();

    //TMA_TRACE( "NodeTCP::~NodeTCP", "NodeTCP destructor" );
    if( IsConnected() )
    {
        Disconnect();
    }

    m_SendQ.Destroy();

    if( m_pListenThreadStack )
    {
        tma::s_Deallocate( m_pListenThreadStack, m_ListenThreadStackSize + nn::os::StackRegionAlignment - 1 );
        m_pListenThreadStack = nullptr;
    }

    if( m_pSendThreadStack )
    {
        tma::s_Deallocate( m_pSendThreadStack, m_SendThreadStackSize + nn::os::StackRegionAlignment - 1 );
        m_pSendThreadStack = nullptr;
    }

    if( m_pRecvThreadStack )
    {
        tma::s_Deallocate( m_pRecvThreadStack, m_RecvThreadStackSize + nn::os::StackRegionAlignment - 1 );
        m_pRecvThreadStack = nullptr;
    }

#if defined( TMIPC_TARGET_WIN )
    int result = WSACleanup();
    (void)result;
// Oasis-832: SDK Assertion Failure on tmipc_node_tcp.cpp:250.
// Note: The actual fix was the tma_Thread.h's IsRunning() function.
//    ASSERT( result == 0 );
#endif
}

//==============================================================================

Result NodeTCP::Listen()
{
    DEJA_TRACE( "NodeTCP::Listen", "Listen" );
    //TMA_TRACE( "NodeTCP::Listen", "Listen" );

    // m_ListenPort will get set to the "actual port" after the nn::socket::GetSockName( ... ) returns.
    // This was done [a while ago] to allow for multiple Sim Instances to run at the same time.
    // (This code runs on the Windows machine, not an SDEV or EDEV - at least, not yet...)
    m_ListenPort = 0;
    m_NodeId     = TMIPC_SERVER_NODE_ID;

    // Create the listen socket.
    ASSERT( m_ListenSocket == nn::socket::InvalidSocket );
    m_ListenSocket = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ASSERT( m_ListenSocket != nn::socket::InvalidSocket );
    if( m_ListenSocket != nn::socket::InvalidSocket )
    {
        // Bind the listen socket.
        nn::socket::SockAddrIn Addr;
        memset( &Addr, 0, sizeof(Addr) );
        Addr.sin_family = nn::socket::Family::Af_Inet;
        Addr.sin_port   = nn::socket::InetHtons( (u16)GetListenPort() );
        Addr.sin_addr.S_addr = nn::socket::InAddr_Any;
        int result1 = nn::socket::Bind( m_ListenSocket, (nn::socket::SockAddr*)&Addr, sizeof(Addr) );
        if( result1 == 0 )
        {
            // Get the local address.
            nn::socket::SockAddrIn LocalAddr;
            nn::socket::SockLenT LocalAddrLen = sizeof(LocalAddr);
            int Error = nn::socket::GetSockName( m_ListenSocket, (nn::socket::SockAddr*)&LocalAddr, &LocalAddrLen );
            if( Error != nn::socket::SocketError )
            {
                m_ListenPort = nn::socket::InetNtohs( LocalAddr.sin_port );
            }

            // Start listening.
            int result2 = nn::socket::Listen( m_ListenSocket, 0 );
            if( result2 == 0 )
            {
                // Start the listen thread.
#if defined( TMIPC_TARGET_WIN )
                m_ListenThread.Start( ListenThread, this, THREAD_PRIORITY_NORMAL );
#elif defined( TMIPC_TARGET_HORIZON )
                if( !m_pListenThreadStack )
                {
                    m_ListenThreadStackSize = ((TMIPC_STACK_SIZE + (nn::os::StackRegionAlignment - 1)) & ~(nn::os::StackRegionAlignment - 1));
                    m_pListenThreadStack = tma::s_Allocate( m_ListenThreadStackSize + (nn::os::StackRegionAlignment - 1) );
                    ASSERT( m_pListenThreadStack );
                }
                NN_SDK_ASSERT(m_ThreadPriority == NN_SYSTEM_THREAD_PRIORITY(tma, TcpListen));
                m_ListenThread.Start( ListenThread, this,
                    (m_pListenThreadStack - 1 + nn::os::StackRegionAlignment) & ~(nn::os::StackRegionAlignment - 1),
                    m_ListenThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(tma, TcpListen), NN_SYSTEM_THREAD_NAME(tma, TcpListen) );
#else
    #error Target Platform Undefined.
#endif
                // Success.
                return( TMIPC_RESULT_OK );
            }
        }
    }

    // Failure.
    return( TMIPC_RESULT_FAILED );
}

//==============================================================================

void NodeTCP::StopListening()
{
    DEJA_TRACE( "NodeTCP::StopListening", "StopListening" );
    //TMA_TRACE( "NodeTCP::StopListening", "StopListening" );

    if( m_ListenSocket != nn::socket::InvalidSocket )
    {
        // Copy the m_ListenSocket, so that it can be set to nn::socket::InvalidSocket.
        // The ListenThread will test the m_ListenSocket value when it fails
        // its call to the accept( m_ListenSocket, ...) function.
        // See ListenThread for more details.
        int CopyOfListenSocket = m_ListenSocket;
        m_ListenSocket = nn::socket::InvalidSocket;
        nn::socket::Close( CopyOfListenSocket );
        VERIFY( m_ListenThread.Join() == TMIPC_RESULT_OK );
    }
}

//==============================================================================

s32 NodeTCP::GetListenPort()
{
    return( m_ListenPort );
}

//==============================================================================
#if defined( TMIPC_TARGET_WIN )
s32 NodeTCP::Connect( NodeAddressTCP& TargetAddress )
{
    DEJA_TRACE( "NodeTCP::Connect", "Connect" );
    //TMA_TRACE( "NodeTCP::Connect", "Connect" );

    m_NodeId = TMIPC_CLIENT_NODE_ID;

    // TODO: Make sure we aren't connecting when already connected?
    ASSERT( !IsConnected() );

    // Create the connection socket.
    int s = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ASSERT( s != nn::socket::InvalidSocket );
    if( s != nn::socket::InvalidSocket )
    {
        Thread Thread;
        // Disable Nagle.
        DWORD val = 1;
        int result = nn::socket::SetSockOpt( s, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_NoDelay, (const char*)&val, sizeof(val) );
        (void)result;
        ASSERT( result == 0 );

        // Start a thread here to negotiate connection.
        m_ConnSocket = s;
        m_RemoteAddress = TargetAddress;
        Thread.Start( ConnectThread, this, THREAD_PRIORITY_NORMAL );

        // Wait for TMIPC_CONNECT_TIMEOUT_MS before aborting connection.
        s32 Result = Thread.Join( TMIPC_CONNECT_TIMEOUT_MS );
        if( (Result == TMIPC_RESULT_OK) && (m_ConnResult == TMIPC_RESULT_OK) )
        {
            // Success, Save the connection socket.
            m_ConnectedSocket = s;

            // Start the Send / Recv threads.
            m_SendThread.Start( SendThread, this, THREAD_PRIORITY_NORMAL );
            m_RecvThread.Start( RecvThread, this, THREAD_PRIORITY_NORMAL );
        }
        else
        {
            // Failed. Socket closed in ConnectThread.
            Thread.Join();
        }
    }

    // Save the remote node address.
    m_RemoteAddress = TargetAddress;

    return( m_ConnResult );
}
#endif // TMIPC_TARGET_WIN

//==============================================================================

Result NodeTCP::Disconnect()
{
    DEJA_TRACE( "NodeTCP::Disconnect", "Disconnect" );
    //TMA_TRACE( "NodeTCP::Disconnect", "Disconnect" );

    // TODO: Evaluate this shutdown code...
    if( m_ConnectedSocket != nn::socket::InvalidSocket )
    {
        nn::socket::Shutdown( m_ConnectedSocket, nn::socket::ShutdownMethod::Shut_RdWr );
        nn::socket::Close( m_ConnectedSocket );
        m_ConnectedSocket = nn::socket::InvalidSocket;
    }

    // Recv thread should terminate when socket closes.
    VERIFY( m_RecvThread.Join() == TMIPC_RESULT_OK );

    // The RecvThread sends a nullptr to terminate the Send Thread.
    // Send a nullptr packet to terminate the Send Thread.
    VERIFY( m_SendThread.Join() == TMIPC_RESULT_OK );

    return( TMIPC_RESULT_OK );
}

//==============================================================================

bool NodeTCP::IsConnected()
{
    return( m_ConnectedSocket != nn::socket::InvalidSocket );
}

//==============================================================================

Result NodeTCP::Send( Packet* pPacket )
{
    DEJA_TRACE( "NodeTCP::Send", "Send TaskId = %p", pPacket->GetTaskId() );
    //TMA_TRACE( "NodeTCP", "Send TaskId = %p", pPacket->GetTaskId() );

    // Scope block for the lock below.
    {
        ScopedLock lock( m_Lock );

        if( m_ConnectedSocket != nn::socket::InvalidSocket )
        {
            // Add the packet to the Message Queue.
            m_SendQ.Send( pPacket );

            // All good.
            return( TMIPC_RESULT_OK );
        }
        else
        {
            // No connection, just free the packet.
            FreePacket( pPacket );
        }
    }

    // Need to Tick the WorkThread if we get here to process the FreePacket.
    Tick();

    // We are diconnected.
    return( TMIPC_RESULT_DISCONNECTED );
}

//==============================================================================
#if defined( TMIPC_TARGET_WIN )
s32 NodeTCP::ConnectThread( void* pArg )
{
    DEJA_TRACE( "NodeTCP::ConnectThread", "ConnectThread Started" );
    NodeTCP* pThis = (NodeTCP*)pArg;

    pThis->m_ConnResult = TMIPC_RESULT_CONNECT_FAILED;

    // Connect.
    int result = nn::socket::Connect( pThis->m_ConnSocket, (nn::socket::SockAddr*)&pThis->m_RemoteAddress.SockAddr(), sizeof(nn::socket::SockAddrIn) );
    if( result == 0 )
    {
        // Send Version.
        u32 Version = nn::socket::InetHtonl( TMIPC_VERSION );
        s32 Result = WriteSocket( pThis->m_ConnSocket, &Version, sizeof(Version) );
        if( Result != sizeof(Version) )
        {
            pThis->m_ConnResult = TMIPC_RESULT_DISCONNECTED;
            goto ConnectFail;
        }

        // Recv Agent Version.
        u32 AgentVersion;
        Result = ReadSocket( pThis->m_ConnSocket, &AgentVersion, 4 );
        if( Result != 4 )
        {
            pThis->m_ConnResult = TMIPC_RESULT_DISCONNECTED;
            goto ConnectFail;
        }
        AgentVersion = nn::socket::InetNtohl( AgentVersion );

        // Test for a valid version number to connect to.
        if( AgentVersion != TMIPC_VERSION )
        {
           pThis-> m_ConnResult = TMIPC_RESULT_CONNECT_VERSION_ERROR;
            goto ConnectFail;
        }

        pThis->m_ConnResult = TMIPC_RESULT_OK;
    }
    else
    {
        pThis->m_ConnResult = TMIPC_RESULT_CONNECT_FAILED;
        goto ConnectFail;
    }

    // Success.
    pThis->OnEvent( NodeEvent_Connected );
    return( pThis->m_ConnResult );

ConnectFail:
    // Connect failed.
    nn::socket::Close( pThis->m_ConnSocket );
    pThis->m_ConnSocket = nn::socket::InvalidSocket;
    return( pThis->m_ConnResult );
}
#endif

//==============================================================================

s32 NodeTCP::ListenThread( void* pArg )
{
    DEJA_TRACE( "NodeTCP::ListenThread", "ListenThread Started" );

    NodeTCP* pThis = (NodeTCP*)pArg;

    // Fill in the Beacon Response - this should not change for the duration
    // of this thread.
    char BeaconResponse[256] = { "" };
    const bool bValidBeaconResponse = tma::GenerateBeaconResponse( BeaconResponse, sizeof(BeaconResponse), "TCP", true );
    if( !bValidBeaconResponse )
    {
        // This is very serious.  Not sure how to handle this though...
        DEJA_TRACE( "NodeTCP::ListenThread", "Could not generate beacon response." );
    }
    const s32 BeaconResponseLength = bValidBeaconResponse ? static_cast<s32>(strlen( BeaconResponse ) + 1) : 0;
    const s32 BeaconResponseLengthByteSwapped = nn::socket::InetHtonl( BeaconResponseLength );

    // Listen until the listen socket is closed.
    while( pThis->m_ListenSocket != nn::socket::InvalidSocket )
    {
        // Accept a new connection.
        nn::socket::SockAddrIn Addr;
        nn::socket::SockLenT AddrLen = sizeof(Addr);
        int s = nn::socket::Accept( pThis->m_ListenSocket, (nn::socket::SockAddr*)&Addr, &AddrLen );
        if( s != nn::socket::InvalidSocket )
        {
            DEJA_TRACE( "NodeTCP::ListenThread", "ListenThread Accepted" );

            bool bConnectSuccess = false;

            // Read the remote version number.
            u32 RemoteVersion;
            s32 Result = ReadSocket( s, &RemoteVersion, sizeof(RemoteVersion) );
            if( Result == sizeof(RemoteVersion) )
            {
                bool bCorrectVersion = false;
                RemoteVersion = nn::socket::InetNtohl( RemoteVersion );

                // Send the local version number.
                u32 LocalVersion = nn::socket::InetHtonl( TMIPC_VERSION );
                Result = WriteSocket( s, &LocalVersion, sizeof(LocalVersion) );
                if( Result == 4 )
                {
                    LocalVersion = nn::socket::InetNtohl( LocalVersion);

                    // Test for the same version.
                    bCorrectVersion = RemoteVersion == LocalVersion;
                    if( !bCorrectVersion )
                    {
                        TMIPC_ERROR_LOG( "Version number mismatch detected.\n" );
                    }
                }

                // Send the beacon response.
                if( bCorrectVersion )
                {
                    // Send the beacon response length first.
                    Result = WriteSocket( s, const_cast<s32 *>(&BeaconResponseLengthByteSwapped), sizeof(BeaconResponseLengthByteSwapped) );
                    if( Result == sizeof(BeaconResponseLengthByteSwapped) )
                    {
                        // Send the beacon response string.
                        Result = WriteSocket( s, BeaconResponse, BeaconResponseLength );
                        if( Result == BeaconResponseLength )
                        {
                            bConnectSuccess = true;
                        }
                        else
                        {
                            TMIPC_ERROR_LOG( "Write Beacon response string failed.\n" );
                        }
                    }
                    else
                    {
                        TMIPC_ERROR_LOG( "Write Beacon response string length failed.\n" );
                    }
                }
            }

            // Did the handshake succeed?
            if( bConnectSuccess )
            {
                // Close any current connection.
                pThis->Disconnect();

                // Save this as the connection socket.
                pThis->m_ConnectedSocket = s;




                // Disable Nagle.
                int val = 1;
                int result = nn::socket::SetSockOpt( pThis->m_ConnectedSocket, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_NoDelay, (const char*)&val, sizeof(val) );
                ASSERT( result == 0 );
                NN_UNUSED(result);
                pThis->OnEvent( NodeEvent_Connected );


                // Create the Send / Recv threads.
#if defined( TMIPC_TARGET_WIN )
                pThis->m_SendThread.Start( SendThread, pThis, THREAD_PRIORITY_NORMAL );
                pThis->m_RecvThread.Start( RecvThread, pThis, THREAD_PRIORITY_NORMAL );
#elif defined( TMIPC_TARGET_HORIZON )
                if( !pThis->m_pSendThreadStack )
                {
                    pThis->m_SendThreadStackSize = ((TMIPC_STACK_SIZE + (nn::os::StackRegionAlignment - 1)) & ~(nn::os::StackRegionAlignment - 1));
                    pThis->m_pSendThreadStack = tma::s_Allocate( pThis->m_SendThreadStackSize + (nn::os::StackRegionAlignment - 1) );
                    ASSERT( pThis->m_pSendThreadStack );
                }
                if( !pThis->m_pRecvThreadStack )
                {
                    pThis->m_RecvThreadStackSize = ((TMIPC_STACK_SIZE + (nn::os::StackRegionAlignment - 1)) & ~(nn::os::StackRegionAlignment - 1));
                    pThis->m_pRecvThreadStack = tma::s_Allocate( pThis->m_RecvThreadStackSize + (nn::os::StackRegionAlignment - 1) );
                    ASSERT( pThis->m_pRecvThreadStack );
                }
                pThis->m_SendThread.Start( SendThread, pThis,
                    (pThis->m_pSendThreadStack - 1 + nn::os::StackRegionAlignment) & ~(nn::os::StackRegionAlignment - 1),
                    pThis->m_SendThreadStackSize, pThis->GetThreadPriority(), "TmaTcpSend" );
                pThis->m_RecvThread.Start( RecvThread, pThis,
                    (pThis->m_pRecvThreadStack - 1 + nn::os::StackRegionAlignment) & ~(nn::os::StackRegionAlignment - 1),
                    pThis->m_RecvThreadStackSize, pThis->GetThreadPriority(), "TmaTcpRecv" );
#else
    #error Target Platform Undefined.
#endif
            }
            else
            {
                // Failed to connect.
                nn::socket::Shutdown( s, nn::socket::ShutdownMethod::Shut_RdWr );
                nn::socket::Close( s );
                s = nn::socket::InvalidSocket;
            }
        }
        else
        {
            // If NodeTCP::StopListening() is called, then m_ListenSocket will
            // be destroyed and become nn::socket::InvalidSocket - which will cause
            // accept(...) to fail.  If the m_ListenSocket is not nn::socket::InvalidSocket
            // then assume the "socket allocation" ran out.
            if( pThis->m_ListenSocket != nn::socket::InvalidSocket )
            {
                // The accept returned an error code.  A Deja trace seemes like a
                // good idea.
                ASSERT( (s != nn::socket::InvalidSocket) && "Ran out of sockets!" );
                DEJA_TRACE( "NodeTCP::ListenThread", "Accept returned %d. LastError == %d", s );
                nn::socket::Shutdown( pThis->m_ListenSocket, nn::socket::ShutdownMethod::Shut_RdWr );
                nn::socket::Close( pThis->m_ListenSocket );
                pThis->m_ListenSocket = nn::socket::InvalidSocket;
            }
            break;
        }
    }

    DEJA_TRACE( "NodeTCP::ListenThread", "ListenThread Stopped" );

    // Thread exit.
    return( 0 );
} // NOLINT(readability/fn_size)

//==============================================================================

s32 NodeTCP::SendThread( void* pArg )
{
    DEJA_THREAD_LABEL( "NodeTCP::SendThread" );
    DEJA_TRACE( "NodeTCP::SendThread", "SendThread Started" );

    NodeTCP* pThis = (NodeTCP*)pArg;

    // Loop until the socket closes.
    while( pThis->m_ConnectedSocket != nn::socket::InvalidSocket )
    {
        // Read a packet from the Message Queue.
        Packet* pPacket = (Packet*)pThis->m_SendQ.Receive();

        // Terminate thread if a nullptr packet.
        if( !pPacket )
        {
            break;
        }

        // Send the packet over the Socket.
        if( pThis->m_ConnectedSocket != nn::socket::InvalidSocket )
        {
            u8* pData   = pPacket->GetBuffer();
            s32 SendLen = pPacket->GetSendLen();
            const s32 Result = WriteSocket( pThis->m_ConnectedSocket, pData, SendLen );
            if( Result <= 0 )
            {
                //TMIPC_ERROR_LOG( "SendThread WriteSocket failed.\n" );
                // An error occurred.  Shut the connection down.
                DEJA_TRACE( "NodeTCP::SendThread", "SendThread WriteSocket failed: %d", Result );
                nn::socket::Shutdown( pThis->m_ConnectedSocket, nn::socket::ShutdownMethod::Shut_RdWr );
                nn::socket::Close( pThis->m_ConnectedSocket );
                pThis->m_ConnectedSocket = nn::socket::InvalidSocket;
            }
        }

        // Free the packet.
        pThis->FreePacket( pPacket );
        pThis->Tick();
    }

    // Disconnect the node.
    pThis->Disconnected();

    // Delete any pending packets.
    while( pThis->m_SendQ.GetCount() > 0 )
    {
        Packet* pPacket = (Packet*)pThis->m_SendQ.Receive();
        if( pPacket )
        {
            pThis->FreePacket( pPacket );
        }
    }

    // Comment in this delay to validate no new packets are sent to the dead connection.
    //Thread::SleepMs( 100 );

    // ASSERT no more packets.
    ASSERT( pThis->m_SendQ.GetCount() == 0 );

    DEJA_TRACE( "NodeTCP::SendThread", "SendThread Stopped" );

    // Thread exit.
    return( 0 );
}

//==============================================================================

s32 NodeTCP::RecvThread( void* pArg )
{
    DEJA_THREAD_LABEL( "NodeTCP::RecvThread" );
    DEJA_TRACE( "NodeTCP::RecvThread", "RecvThread Started" );

    NodeTCP* pThis = (NodeTCP*)pArg;

    // Loop until the socket closes.
    while( pThis->m_ConnectedSocket != nn::socket::InvalidSocket )
    {
        // Allocate a packet.
        Packet* pPacket = pThis->AllocRecvPacket();
        ASSERT( pPacket );

        // Read packet from the Socket.
        u8* pBuffer = pPacket->GetBuffer();

        // Read the packet header.
        s32 Result = ReadSocket( pThis->m_ConnectedSocket, pBuffer, sizeof(Packet::Header) );
        if( Result != sizeof(Packet::Header) )
        {
            // If we didn't get a full header then we lost connection.
            //TMIPC_ERROR_LOG( "RecvThread Read header packet failed.\n" );
            DEJA_TRACE( "NodeTCP::RecvThread", "RecvThread ReadSocket failed: %d != sizeof(Packet::Header)", Result );
            pThis->FreePacket( pPacket );
            break;
        }

        s32 DataLen = pPacket->GetDataLen();
        if( DataLen > 0 )
        {
            // Read the packet contents.
            Result = ReadSocket( pThis->m_ConnectedSocket, &pBuffer[sizeof(Packet::Header)], DataLen );
            if( Result != DataLen )
            {
                // If we didn't get all the data then we lost connection.
                TMIPC_ERROR_LOG( "RecvThread Read data packet failed.\n" );
                DEJA_TRACE( "NodeTCP::RecvThread", "RecvThread ReadSocket failed: %d != DataLen", Result );
                pThis->FreePacket( pPacket );
                break;
            }
        }

        //TMA_TRACE( "NodeTCP", "RecvThread - New Packet" );

        // Dispatch the packet.
        pThis->ProcessReceived( pPacket );
        pPacket = nullptr;
    }

    // No longer connected.
    pThis->OnEvent( NodeEvent_Disconnected );
    nn::socket::Shutdown( pThis->m_ConnectedSocket, nn::socket::ShutdownMethod::Shut_RdWr );
    nn::socket::Close( pThis->m_ConnectedSocket );
    pThis->m_ConnectedSocket = nn::socket::InvalidSocket;

    // Send a nullptr to terminate the SendThread.
    pThis->m_SendQ.Send( nullptr );

    DEJA_TRACE( "NodeTCP::RecvThread", "RecvThread Stopped" );

    // Thread exit.
    return( 0 );
}

//==============================================================================
} // namespace tmipc
//==============================================================================
