﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "tmipc_defines.h"
#include "tmipc_thread.h"
#include "tmipc_result.h"
#include <string>

#ifdef TMIPC_TARGET_HORIZON
#include "..\tma\tma_MemMgr.h"
#endif

#include "..\tmagent.h"

//==============================================================================
namespace tmipc {
//==============================================================================

class Packet;
class ServicesManager;

//==============================================================================

enum eNodeEventCode
{
    NodeEvent_Connected,
    NodeEvent_Disconnected,
};

class NodeEvent
{
public:
    explicit NodeEvent( eNodeEventCode Code )
    {
        m_EventCode = Code;
    }

    eNodeEventCode  GetEventCode()
    {
        return m_EventCode;
    }

protected:
    eNodeEventCode  m_EventCode;
};

typedef void (*EventCallback)( void* pUserData, NodeEvent* pEvent );

//==============================================================================

class Node
{
protected:

        Mutex               m_Lock;
        s32                 m_ThreadPriority;
        u32                 m_NodeId;                   // Server / Client identifier

        // Event notification.
        EventCallback       m_pEventCallback;
        void*               m_pEventCallbackData;

        bool                m_Initialized;

private:
        // Needed to allocate/deallocate Receive packets.
        // TODO: Revisit to see if there is a cleaner way to do this.
        ServicesManager*    m_pServicesManager;

        // This flag represents TargetManager's ability to support "preserving state through a Sleep/Wake cycle".
        bool                m_PreserveStateThroughSleepWake;
        // This flag represents this Target has woken up.
        bool                m_WasWokenUp;

protected:

        // Process the TMSInfo string.
        void                ProcessTMSInfoString( const char*       pTMSInfoString );

public:
                            Node                ();
virtual                    ~Node                ();

        void                Init                ();
        void                Kill                ();

        Packet*             AllocSendPacket     ();
        Packet*             AllocRecvPacket     ();
        void                FreePacket          ( Packet*           pPacket );

        // Listens for connections.
virtual Result              Listen              ();
        // Stops listening for connections.  This does *not* cause a disconnect.
virtual void                StopListening       ();

        // Needed to allocate/deallocate Receive packets.
        // TODO: Revisit to see if there is a cleaner way to do this.
        void                SetServicesManager  ( ServicesManager*  pServicesManager );

        void                SetEventCallback    ( EventCallback     pCallback,
                                                  void*             pUserData );

virtual bool                IsConnected         () = 0;
virtual Result              Disconnect          () = 0;

virtual Result              Send                ( Packet*           pPacket ) = 0;
virtual void                SetThreadPriority   ( s32               Priority );
virtual s32                 GetThreadPriority   ();

        // Returns whether or not TMA supports State preservation through sleep wake.
        bool                PreserveStateThroughSleepWake() const   { return m_PreserveStateThroughSleepWake; }
        // Returns whether or not this Target has woken up or not.
        bool                WasWokenUp() const                      { return m_WasWokenUp; }
        void                SetWasWokenUp( bool WasWokenUp )        { m_WasWokenUp = WasWokenUp; }

protected:
        void                ProcessReceived     ( Packet*           pPacket );
        void                Tick                ();
        void                CancelAllPendingTasks();

        void                OnEvent             ( eNodeEventCode    EventType );
        void                Disconnected        ();
};

//==============================================================================
} // namespace tmipc
//==============================================================================
