﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "stdafx.h"

#include "connection/connection_Api.h"
#include "tma/tma_Thread.h"
#include "Version.h"
#include "tm_result.h"
#include "hio/hio_service.h"
#include "hio/hio_directory_service.h"
#include "hid/hid_Service.h"
#include "htcs/htcs_service.h"
#include "env/env_Service.h"
#include "generic_service/generic_service.h"

// The Socket API is safe to include Windows or otherwise.
// It needs to be included before tmipc_node_tcp.h
#include <nn/socket.h>

#include "tmipc/tmipc_services_manager.h"
#include "tmipc/tmipc_node_tcp.h"
#include "tmipc/tmipc_node_htclow.h"
#include "tmipc/tmipc_node_tics.h"
#include "tmipc/tmipc_node_usb.h"

#include <nn/settings/factory/settings_ConfigurationId.h>
#include <nn/settings/factory/settings_SerialNumber.h>
#include <nn/psc.h>
#include <nn/nn_SystemThreadDefinition.h>

//TODO: Ask Craig if it would be ok to have two sections of code, to avoid interspersing of #if !defined( NN_TMA_SIM_INSTANCE ) blocks.
//      Actually, having two separate files might be easier: one for Sim Instance - tma_siminstance.cpp and tma.cpp???


// Build configuration specific header includes.
#if !defined( NN_TMA_SIM_INSTANCE )
#include <nn/svc/svc_MemoryMapSelect.h>
#include <nn/ro/detail/ro_RoModule.h>
#include <nn/rocrt/rocrt.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>
#include <cctype>
#include "benchmark/benchmark_report_service.h"
#include "benchmark/benchmark_service.h"
#include "dbg/dbg_ProcessMgr.h"
#include "dbg/dbg_Service.h"
#include "dbghlp/dbghlp.h"
#include "hio/hio_HioHipcServer.h"
#include "htcs/htcs_HtcsHipcServer.h"
#include "htc/htc_HtcHipcServer.h"
#include "hbx/hbx_api.h"
#include "performance_monitor/performance_monitor_service.h"
#include "pm_monitor.h"
#include "powermanagement/power_management_service.h"
#include "settings/settings_service.h"
#include "thread_tracker.h"
#include "threadfrozen/thread_frozen_service.h"
#include "tio/tio_Service.h"
#include "coredump/coredump_service.h"
#include "diagnostics/diagnostics_service.h"
#include "diagnostics/diagnostics_task.h"
#else
#include "dbg/dbg_SimService.h"
#include "dbg/dbg_SimTask.h"
//#include <nn/htcs/htcs_Library.h>
#endif

#if defined(NN_BUILD_CONFIG_HARDWARE_BDSLIMX6) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_NX) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2)
#define NN_DETAIL_TMA_NX_RELATED_HARDWARE
#endif

//==============================================================================

#if !defined( NN_TMA_SIM_INSTANCE )
namespace {

NN_OS_ALIGNAS_THREAD_STACK char g_HioServerStack[8192];
nn::os::ThreadType g_HioServerThread;
static const int g_HtcsServerThreadCount = 8;
NN_OS_ALIGNAS_THREAD_STACK char g_HtcsServerStack[g_HtcsServerThreadCount][8192];
nn::os::ThreadType g_HtcsServerThread[g_HtcsServerThreadCount];
NN_OS_ALIGNAS_THREAD_STACK char g_HtcServerStack[8192];
nn::os::ThreadType g_HtcServerThread;

void HipcHioServerFunction( void* arg ) NN_NOEXCEPT
{
    NN_UNUSED(arg);
    // サーバループを実行
    // nn::sfsmpl::RequestStopCalculatorServer() が(別スレッドから)呼ばれるまで、ループし続ける
    nn::tma::LoopHioServer();
}

void HipcHtcsServerFunction( void* arg ) NN_NOEXCEPT
{
    NN_UNUSED(arg);
    nn::tma::LoopHtcsServer();
}

void HipcHtcServerFunction( void* arg ) NN_NOEXCEPT
{
    NN_UNUSED(arg);
    nn::tma::LoopHtcServer();
}

void InitializeHipcServers()
{
    nn::Result result;

    //hio
    nn::tma::InitializeHioServer();
    result = nn::os::CreateThread( &g_HioServerThread, &HipcHioServerFunction, nullptr, g_HioServerStack, sizeof(g_HioServerStack), NN_SYSTEM_THREAD_PRIORITY(tma, IpcHio) );
    ASSERT( result.IsSuccess() );
    tma::ThreadTracker::RegisterThread( g_HioServerThread, tma::ThreadTracker::ThreadId::HioServer );
    nn::os::SetThreadName( &g_HioServerThread, NN_SYSTEM_THREAD_NAME(tma, IpcHio) );
    nn::os::StartThread( &g_HioServerThread );

    //htcs
    const s32 HtcsThreadTrackingCount{ 8 };
    static_assert( g_HtcsServerThreadCount == HtcsThreadTrackingCount, "HTCS Thread count discrepancy!" );
    const tma::ThreadTracker::ThreadId kThreadIds[HtcsThreadTrackingCount] =
    {
        tma::ThreadTracker::ThreadId::HtcsServer0,            // tma::HipcHtcsServerFunction[0].
        tma::ThreadTracker::ThreadId::HtcsServer1,            // tma::HipcHtcsServerFunction[1].
        tma::ThreadTracker::ThreadId::HtcsServer2,            // tma::HipcHtcsServerFunction[2].
        tma::ThreadTracker::ThreadId::HtcsServer3,            // tma::HipcHtcsServerFunction[3].
        tma::ThreadTracker::ThreadId::HtcsServer4,            // tma::HipcHtcsServerFunction[4].
        tma::ThreadTracker::ThreadId::HtcsServer5,            // tma::HipcHtcsServerFunction[5].
        tma::ThreadTracker::ThreadId::HtcsServer6,            // tma::HipcHtcsServerFunction[6].
        tma::ThreadTracker::ThreadId::HtcsServer7,            // tma::HipcHtcsServerFunction[7].
    };

    nn::tma::InitializeHtcsServer();
    for( int i = 0; i < g_HtcsServerThreadCount; ++i )
    {
        result = nn::os::CreateThread( &g_HtcsServerThread[i], &HipcHtcsServerFunction, nullptr, g_HtcsServerStack[i], sizeof(g_HtcsServerStack[i]), NN_SYSTEM_THREAD_PRIORITY(tma, IpcHtcs) );
        ASSERT( result.IsSuccess() );
        tma::ThreadTracker::RegisterThread(g_HtcsServerThread[i], kThreadIds[i]);
        nn::os::SetThreadName( &g_HtcsServerThread[i], NN_SYSTEM_THREAD_NAME(tma, IpcHtcs) );
        nn::os::StartThread( &g_HtcsServerThread[i] );
    }

    //htc
    nn::tma::InitializeHtcServer();
    result = nn::os::CreateThread( &g_HtcServerThread, &HipcHtcServerFunction, nullptr, g_HtcServerStack, sizeof(g_HtcServerStack), NN_SYSTEM_THREAD_PRIORITY(tma, IpcHtc) );
    ASSERT( result.IsSuccess() );
    tma::ThreadTracker::RegisterThread( g_HtcServerThread, tma::ThreadTracker::ThreadId::HtcServer );
    nn::os::SetThreadName( &g_HtcServerThread, NN_SYSTEM_THREAD_NAME(tma, IpcHtc) );
    nn::os::StartThread( &g_HtcServerThread );
}

void FinalizeHipcServers()
{
    //hio
    nn::tma::RequestStopHioServer();
    nn::os::WaitThread( &g_HioServerThread );
    nn::os::DestroyThread( &g_HioServerThread );
    nn::tma::FinalizeHioServer();
    tma::ThreadTracker::UnregisterThread( tma::ThreadTracker::ThreadId::HioServer );

    //htcs
    const s32 HtcsThreadTrackingCount{ 8 };
    static_assert( g_HtcsServerThreadCount == HtcsThreadTrackingCount, "HTCS Thread count discrepancy!" );
    const tma::ThreadTracker::ThreadId kThreadIds[HtcsThreadTrackingCount] =
    {
        tma::ThreadTracker::ThreadId::HtcsServer0,            // tma::HipcHtcsServerFunction[0].
        tma::ThreadTracker::ThreadId::HtcsServer1,            // tma::HipcHtcsServerFunction[1].
        tma::ThreadTracker::ThreadId::HtcsServer2,            // tma::HipcHtcsServerFunction[2].
        tma::ThreadTracker::ThreadId::HtcsServer3,            // tma::HipcHtcsServerFunction[3].
        tma::ThreadTracker::ThreadId::HtcsServer4,            // tma::HipcHtcsServerFunction[4].
        tma::ThreadTracker::ThreadId::HtcsServer5,            // tma::HipcHtcsServerFunction[5].
        tma::ThreadTracker::ThreadId::HtcsServer6,            // tma::HipcHtcsServerFunction[6].
        tma::ThreadTracker::ThreadId::HtcsServer7,            // tma::HipcHtcsServerFunction[7].
    };
    nn::tma::RequestStopHtcsServer();
    for( int i = 0; i < g_HtcsServerThreadCount; ++i )
    {
        nn::os::WaitThread( &g_HtcsServerThread[i] );
        nn::os::DestroyThread( &g_HtcsServerThread[i] );
        tma::ThreadTracker::UnregisterThread( kThreadIds[i] );
    }
    nn::tma::FinalizeHtcsServer();

    //htc
    nn::tma::RequestStopHtcServer();
    nn::os::WaitThread( &g_HtcServerThread );
    nn::os::DestroyThread( &g_HtcServerThread );
    nn::tma::FinalizeHtcServer();
    tma::ThreadTracker::UnregisterThread( tma::ThreadTracker::ThreadId::HtcServer );
}

}
#endif

//==============================================================================
namespace tma {
//==============================================================================

void NodeEventCallback( void* pUserData, tmipc::NodeEvent* pEvent );
void PowerManagementCallback( nn::psc::PmState PmState, const nn::psc::PmFlagSet& PmFlagSet );

Mutex                                   g_EventCallbackMutex;

// TCP Transport
tmipc::NodeTCP*                         s_pAgentNodeTCP             = nullptr;

// TICS Transport
tmipc::NodeTICS*                        s_pAgentNodeTICS            = nullptr;

// USB Transport
tmipc::NodeUSB*                         s_pAgentNodeUSB             = nullptr;

// HTCLOW Transport
tmipc::NodeHtclow*                      s_pAgentNodeHtclow          = nullptr;

tmipc::Node*                            s_pAgentNode                = nullptr;  // This is the active Node connection.
tmipc::ServicesManager*                 s_pServicesManager          = nullptr;

tma::env::AgentEnvService*              s_pEnvService               = nullptr;
tma::generic_service::Service*          s_pGenericService           = nullptr;
tma::hio::AgentHostIOService*           s_pHostIOService            = nullptr;
tma::hio::AgentHostDirectoryIOService*  s_pHostDirectoryIOService   = nullptr;
tma::htcs::AgentHTCSService*            s_pHTCSService              = nullptr;
tma::hid::AgentControllerService*       s_pControllerService        = nullptr;

bool                                    g_bNotifyDisconnect         = true;

#if !defined( NN_TMA_SIM_INSTANCE )
#if ENABLE_BENCHMARK_TESTING
tma::benchmark::BenchmarkReportService* s_pBenchmarkReportService   = nullptr;
tma::benchmark::BenchmarkService*       s_pBenchmarkService         = nullptr;
#endif // ENABLE_BENCHMARK_TESTING
#if ENABLE_PERFORMANCE_MONITOR_SERVICE
tma::performance_monitor::Service*      s_pPerformanceMonitorService= nullptr;
#endif // ENABLE_PERFORMANCE_MONITOR_SERVICE
tma::dbg::DebugService*                 s_pDebugService             = nullptr;
dbg::ProcessMgr*                        s_pProcessMgr               = nullptr;
tma::powermanagement::Service*          s_pPowerManagementService   = nullptr;
tma::settings::Service*                 s_pSettingsService          = nullptr;
tma::threadfrozen::Service*             s_pThreadFrozenService      = nullptr;
tma::target_io::Service*                s_pTargetIOService          = nullptr;
tma::coredump::Service*                 s_pCoredumpService          = nullptr;
tma::diagnostics::Service*              s_pDiagnosticsService       = nullptr;
#else
tma::dbg::SimDebugService*              s_pSimDebugService          = nullptr;
#endif
static bool                             s_bInitialized              = false;

extern void* HeapMalloc( size_t Size );
extern void HeapFree( void* pMemory );

//==============================================================================

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
const int TMA_HEAP_SIZE = (10 * 1024 * 1024);
static char s_Heap[TMA_HEAP_SIZE] __attribute__( ( __aligned__( 4096 ) ) );
static nn::lmem::HeapHandle     g_HeapHandle;

void memmgr_init()
{
    g_HeapHandle = nn::lmem::CreateExpHeap(s_Heap, sizeof(s_Heap), nn::lmem::CreationOption_ThreadSafe);
}
#endif

// On NX64, this defaults to nn::lmem::AllocateFromExpHeap.
// On Win32, this defaults to a plain old system call.
#if TRACK_MEMORY
void* DefaultAlloc( size_t size, const char* pFileName, int LineNumber )
#else
void* DefaultAlloc( size_t size )
#endif
{
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    ADD_MEMORY_ALLOCATION_COUNT_INCREMENT();
#endif
    if( size == 0 )
    {
        return NULL;
    }
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    void* p = nn::lmem::AllocateFromExpHeap( g_HeapHandle, size );

#if TRACK_MEMORY
    diagnostics::memory_tracking::Remember( size, p, pFileName, LineNumber );
#endif

#else
    void* p = malloc( size );
#endif
    if( p == nullptr )
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        NN_SDK_LOG( "[tma] DefaultAlloc() returned nullptr when trying to allocate %d bytes (%d bytes available)\n",  size, nn::lmem::GetExpHeapTotalFreeSize( g_HeapHandle ) );
#else
        NN_SDK_LOG( "[tma] DefaultAlloc() returned nullptr when trying to allocate %d bytes\n", size );
#endif
    }
    return p;
}

//==============================================================================

void DefaultDealloc( void* Ptr, size_t Size )
{
    (void)Size;
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    ADD_MEMORY_DEALLOCATION_COUNT_INCREMENT();
    nn::lmem::FreeToExpHeap( g_HeapHandle, Ptr );
#if TRACK_MEMORY
    diagnostics::memory_tracking::Forget( Ptr );
#endif
#else
    free( Ptr );
#endif
}

//==============================================================================

static bool m_StaticInitialized = false;
static uintptr_t m_HeapAddress;

//==============================================================================

#if TRACK_MEMORY
AllocateFunction s_AllocateCall = nullptr;
#else
AllocateFunction s_Allocate = nullptr;
#endif
DeallocateFunction s_Deallocate = nullptr;

//==============================================================================
// Cached Target Settings, to prevent cyclic dependency between fs -> tma -> settings -> fs.
// i.e. SigloNTD-5339: Discuss what to do when the target tries to sleep.
struct TargetSettings
{
    char m_ConfigurationId[128];
    char m_SerialNumber[128];
    bool m_DisableSleep;
    bool m_EnableDisconnectionEmulation;
};
TargetSettings s_TargetSettingsCache =
{
    "Unknown",
    "",
    false
};

//------------------------------------------------------------------------------
// Returns the Target's Configuration ID.  This is guaranteed to not be nullptr or "".
const char* GetTargetConfigurationId()
{
    return s_TargetSettingsCache.m_ConfigurationId;
}

//------------------------------------------------------------------------------
// Returns the Target's Serial Number.  This is guaranteed to not be nullptr, but can be "".
const char* GetTargetSerialNumber()
{
    return s_TargetSettingsCache.m_SerialNumber;
}

//------------------------------------------------------------------------------
// Returns if tma should fall asleep or not
// It is expected that true is set only when being run on CI because the test will fail as TMA-TMS connection breaks if tma sleeps.
const bool GetDisableSleepFlag()
{
    return s_TargetSettingsCache.m_DisableSleep;
}

//------------------------------------------------------------------------------
// Returns if tma should fail to communicate to host or not
const bool GetEnableDisconnectionEmulationFlag()
{
    return s_TargetSettingsCache.m_EnableDisconnectionEmulation;
}

//------------------------------------------------------------------------------
// Returns the total amount of Heap RAM that was allocated to TMA.
const s32 GetHeapRAMTotal()
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    return static_cast<s32>(TMA_HEAP_SIZE);
#else
    return 0;
#endif
}

//------------------------------------------------------------------------------
// Returns the current amount of RAM that was allocated from TMA's HEAP.
const s32 GetHeapRAMAllocated()
{
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    return TMA_HEAP_SIZE - static_cast<s32>(nn::lmem::GetExpHeapTotalFreeSize(tma::g_HeapHandle));
#else
    return 0;
#endif
}

//------------------------------------------------------------------------------
// Record some Target settings.  This needs to happen as infrequent as possible,
// in order to prevent a circular dependency with the file system.
// (i.e. This is related to SigloNTD-5339: Discuss what to do when the target
//  tries to sleep. See struct 'TargetSettings' for details.)
void CacheTargetSettings()
{
    // Configuration ID (i.e. Hardware Type).
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
#if defined(NN_BUILD_CONFIG_ADDRESS_32)
        TMA_STRCPY( s_TargetSettingsCache.m_ConfigurationId, sizeof(s_TargetSettingsCache.m_ConfigurationId), "Win32" );
#endif
#if defined(NN_BUILD_CONFIG_ADDRESS_64)
        TMA_STRCPY( s_TargetSettingsCache.m_ConfigurationId, sizeof(s_TargetSettingsCache.m_ConfigurationId), "Win64" );
#endif
#elif defined(NN_BUILD_CONFIG_HARDWARE_NX)

        nn::settings::factory::ConfigurationId1 ConfigurationId1;
        nn::settings::factory::GetConfigurationId1( &ConfigurationId1 );
        if( strncmp( "", ConfigurationId1.string, sizeof(ConfigurationId1.string)) != 0 )
        {
            TMA_STRCPY( s_TargetSettingsCache.m_ConfigurationId, sizeof(s_TargetSettingsCache.m_ConfigurationId) / sizeof(*s_TargetSettingsCache.m_ConfigurationId), ConfigurationId1.string );
        }
        else
        {
            TMA_STRCPY( s_TargetSettingsCache.m_ConfigurationId, sizeof(s_TargetSettingsCache.m_ConfigurationId) / sizeof(*s_TargetSettingsCache.m_ConfigurationId), "Unknown" );
        }
        nn::settings::fwdbg::GetSettingsItemValue(&s_TargetSettingsCache.m_DisableSleep, sizeof(s_TargetSettingsCache.m_DisableSleep), "systemsleep", "disable_tma_sleep");
        nn::settings::fwdbg::GetSettingsItemValue(&s_TargetSettingsCache.m_EnableDisconnectionEmulation, sizeof(s_TargetSettingsCache.m_EnableDisconnectionEmulation), "htc", "disconnection_emulation");

#elif defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1)
        TMA_STRCPY( s_TargetSettingsCache.m_ConfigurationId, sizeof(s_TargetSettingsCache.m_ConfigurationId) / sizeof(*s_TargetSettingsCache.m_ConfigurationId), "Jetson TK1" );
#elif defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2)
        TMA_STRCPY( s_TargetSettingsCache.m_ConfigurationId, sizeof(s_TargetSettingsCache.m_ConfigurationId) / sizeof(*s_TargetSettingsCache.m_ConfigurationId), "Jetson TK2" );
#else
    #if defined( TMA_USE_WINSOCK )
            TMA_STRCPY( s_TargetSettingsCache.m_ConfigurationId, sizeof(s_TargetSettingsCache.m_ConfigurationId) / sizeof(*s_TargetSettingsCache.m_ConfigurationId), "ARMREF" );
    #else
            TMA_STRCPY( s_TargetSettingsCache.m_ConfigurationId, sizeof(s_TargetSettingsCache.m_ConfigurationId) / sizeof(*s_TargetSettingsCache.m_ConfigurationId), "Unknown" );
    #endif // TMA_USE_WINSOCK
#endif
    }

    // Get the hardware serial number.
    {
        nn::settings::factory::SerialNumber SerialNumber;
        SerialNumber.string[0] = '\0';

#if defined(NN_DETAIL_TMA_NX_RELATED_HARDWARE)
        // The Jetson-TK2 will print warnings about the GetSerialNumber call, and
        // will "become blocking" if a USB connection to the TK2 is reconnected
        // (after having been initially connected).
        const nn::Result NnResult = nn::settings::factory::GetSerialNumber(&SerialNumber);
        if( NnResult.IsFailure() || (SerialNumber.string[0] == '\0') )
        {
            TMA_STRCPY( SerialNumber.string, sizeof(SerialNumber.string) / sizeof(SerialNumber.string[0]), "Corrupted S/N" );
        }

        if( SerialNumber.string[0] != '\0' )
        {
            TMA_STRCPY( s_TargetSettingsCache.m_SerialNumber, sizeof(s_TargetSettingsCache.m_SerialNumber) / sizeof(*s_TargetSettingsCache.m_SerialNumber), SerialNumber.string );
        }
#endif
    }
}

//==============================================================================

void CreateTMAServices()
{
    if( s_pServicesManager == nullptr )
    {
        void* pMem = s_Allocate( sizeof( tmipc::ServicesManager ) );
        s_pServicesManager = new (pMem) tmipc::ServicesManager;
    }
#if !defined( NN_TMA_SIM_INSTANCE )
    if( s_pDebugService == nullptr )
    {
        void* pMem = s_Allocate( sizeof( tma::dbg::DebugService ) );
        s_pDebugService = new (pMem) tma::dbg::DebugService();
    }
    if( s_pPowerManagementService == nullptr )
    {
        void* pMem = s_Allocate( sizeof(*s_pPowerManagementService) );
        s_pPowerManagementService = new (pMem) tma::powermanagement::Service();
    }
    if( s_pSettingsService == nullptr )
    {
        void* pMem = s_Allocate( sizeof(*s_pSettingsService) );
        s_pSettingsService = new (pMem) tma::settings::Service();
    }
    if( s_pThreadFrozenService == nullptr )
    {
        void* pMem = s_Allocate( sizeof(*s_pThreadFrozenService) );
        s_pThreadFrozenService = new (pMem) tma::threadfrozen::Service();
    }
    if( s_pTargetIOService == nullptr )
    {
        void* pMem = s_Allocate( sizeof(*s_pTargetIOService) );
        s_pTargetIOService = new (pMem) tma::target_io::Service();
    }
    if( s_pCoredumpService == nullptr )
    {
        void* pMem = s_Allocate( sizeof(*s_pCoredumpService) );
        s_pCoredumpService = new (pMem) tma::coredump::Service();
    }
    if( s_pDiagnosticsService == nullptr )
    {
        void* pMem = s_Allocate( sizeof(*s_pDiagnosticsService) );
        s_pDiagnosticsService = new (pMem) tma::diagnostics::Service();
    }
#if ENABLE_BENCHMARK_TESTING
    if( s_pBenchmarkReportService == nullptr )
    {
        void* pMem = s_Allocate( sizeof( tma::benchmark::BenchmarkReportService ) );
        s_pBenchmarkReportService = new (pMem) tma::benchmark::BenchmarkReportService();
    }
    if( s_pBenchmarkService == nullptr )
    {
        void* pMem = s_Allocate( sizeof( tma::benchmark::BenchmarkService ) );
        s_pBenchmarkService = new (pMem) tma::benchmark::BenchmarkService();
    }
#endif // ENABLE_BENCHMARK_TESTING
#if ENABLE_PERFORMANCE_MONITOR_SERVICE
    if( s_pPerformanceMonitorService == nullptr )
    {
        void* pMem = s_Allocate( sizeof( tma::performance_monitor::Service ) );
        s_pPerformanceMonitorService = new (pMem) tma::performance_monitor::Service();
    }
#endif // ENABLE_PERFORMANCE_MONITOR_SERVICE
#else
    if( s_pSimDebugService == nullptr )
    {
        void* pMem = s_Allocate( sizeof( tma::dbg::SimDebugService ) );
        s_pSimDebugService = new (pMem) tma::dbg::SimDebugService();
    }
#endif
    if( s_pEnvService == nullptr )
    {
        void* pMem = s_Allocate( sizeof( tma::env::AgentEnvService ) );
        s_pEnvService = new (pMem) tma::env::AgentEnvService();
    }
    if( s_pGenericService == nullptr )
    {
        void* pMem = s_Allocate( sizeof( tma::generic_service::Service ) );
        s_pGenericService = new (pMem) tma::generic_service::Service();
    }
    if( s_pHostIOService == nullptr )
    {
        void* pMem = s_Allocate( sizeof( tma::hio::AgentHostIOService ) );
        s_pHostIOService = new (pMem) tma::hio::AgentHostIOService();
    }
    if( s_pHostDirectoryIOService == nullptr )
    {
        void* pMem = s_Allocate( sizeof( tma::hio::AgentHostDirectoryIOService ) );
        s_pHostDirectoryIOService = new (pMem) tma::hio::AgentHostDirectoryIOService();
    }
    if( s_pHTCSService == nullptr )
    {
        void* pMem = s_Allocate( sizeof( tma::htcs::AgentHTCSService ) );
        s_pHTCSService = new (pMem) tma::htcs::AgentHTCSService();
    }
    if( s_pControllerService == nullptr )
    {
        void* pMem = s_Allocate( sizeof( tma::hid::AgentControllerService ) );
        s_pControllerService = new (pMem) tma::hid::AgentControllerService();
    }
}

//==============================================================================

void DestroyTMAServices()
{
    // Remove (delete) all internal services.
//TODO: This should be considered Delete/DestroyServices for now.
//    s_pServicesManager->RemoveServices();
    s_pServicesManager->Kill();

#if !defined( NN_TMA_SIM_INSTANCE )
    s_pPowerManagementService->~Service();
    s_Deallocate( s_pPowerManagementService, sizeof( *s_pPowerManagementService ) );
    s_pPowerManagementService = nullptr;
    s_pSettingsService->~Service();
    s_Deallocate( s_pSettingsService, sizeof( *s_pSettingsService ) );
    s_pSettingsService = nullptr;
    s_pThreadFrozenService->~Service();
    s_Deallocate( s_pThreadFrozenService, sizeof( *s_pThreadFrozenService ) );
    s_pThreadFrozenService = nullptr;
    s_pTargetIOService->~Service();
    s_Deallocate( s_pTargetIOService, sizeof( *s_pTargetIOService ) );
    s_pTargetIOService = nullptr;
    s_pCoredumpService->~Service();
    s_Deallocate( s_pCoredumpService, sizeof( *s_pCoredumpService ) );
    s_pCoredumpService = nullptr;
    s_pDiagnosticsService->~Service();
    s_Deallocate( s_pDiagnosticsService, sizeof( *s_pDiagnosticsService ) );
    s_pDiagnosticsService = nullptr;
    s_pDebugService->~DebugService();
    s_Deallocate( s_pDebugService, sizeof( tma::dbg::DebugService ) );
    s_pDebugService = nullptr;
#if ENABLE_BENCHMARK_TESTING
    s_pBenchmarkService->~BenchmarkService();
    s_Deallocate( s_pBenchmarkService, sizeof( tma::benchmark::BenchmarkService ) );
    s_pBenchmarkService = nullptr;
    s_pBenchmarkReportService->~BenchmarkReportService();
    s_Deallocate( s_pBenchmarkReportService, sizeof( tma::benchmark::BenchmarkReportService ) );
    s_pBenchmarkReportService = nullptr;
#endif // ENABLE_BENCHMARK_TESTING
#if ENABLE_PERFORMANCE_MONITOR_SERVICE
    s_pPerformanceMonitorService->~Service();
    s_Deallocate( s_pPerformanceMonitorService, sizeof( tma::performance_monitor::Service ) );
    s_pPerformanceMonitorService = nullptr;
#endif // ENABLE_PERFORMANCE_MONITOR_SERVICE
#else
    s_pSimDebugService->~SimDebugService();
    s_Deallocate( s_pSimDebugService, sizeof( tma::dbg::SimDebugService ) );
    s_pSimDebugService = nullptr;
#endif

    s_pEnvService->~AgentEnvService();
    s_Deallocate( s_pEnvService, sizeof( tma::env::AgentEnvService ) );
    s_pEnvService = nullptr;

    s_pGenericService->~Service();
    s_Deallocate( s_pGenericService, sizeof( tma::generic_service::Service ) );
    s_pGenericService = nullptr;

    s_pHostIOService->~AgentHostIOService();
    s_Deallocate( s_pHostIOService, sizeof( tma::hio::AgentHostIOService ) );
    s_pHostIOService = nullptr;

    s_pHostDirectoryIOService->~AgentHostDirectoryIOService();
    s_Deallocate( s_pHostDirectoryIOService, sizeof( tma::hio::AgentHostDirectoryIOService ) );
    s_pHostDirectoryIOService = nullptr;

    s_pHTCSService->~AgentHTCSService();
    s_Deallocate( s_pHTCSService, sizeof( tma::htcs::AgentHTCSService ) );
    s_pHTCSService = nullptr;

    s_pControllerService->~AgentControllerService();
    s_Deallocate( s_pControllerService, sizeof( tma::hid::AgentControllerService ) );
    s_pControllerService = nullptr;

    s_pServicesManager->~ServicesManager();
    s_Deallocate( s_pServicesManager, sizeof( tmipc::ServicesManager ) );
    s_pServicesManager = nullptr;
}

//==============================================================================

void InitServices()
{
    // Set the ServicesManager's thread priority before calling its Init(),
    // because the Init uses the thread priority to create its WorkThread.
    s_pServicesManager->SetThreadPriority( NN_SYSTEM_THREAD_PRIORITY(tma, ServicesMgr) );
    s_pServicesManager->Init();
#if !defined( NN_TMA_SIM_INSTANCE )
    s_pDebugService->Init( s_pProcessMgr );
    s_pServicesManager->AddService( s_pDebugService );
    s_pPowerManagementService->Init();
    s_pServicesManager->AddService( s_pPowerManagementService );
    s_pSettingsService->Init();
    s_pServicesManager->AddService( s_pSettingsService );
    s_pThreadFrozenService->Init();
    s_pServicesManager->AddService( s_pThreadFrozenService );
    s_pTargetIOService->Init();
    s_pServicesManager->AddService( s_pTargetIOService );
    s_pCoredumpService->Init( s_pProcessMgr );
    s_pServicesManager->AddService( s_pCoredumpService );
    s_pDiagnosticsService->Init();
    s_pServicesManager->AddService( s_pDiagnosticsService );
#if ENABLE_BENCHMARK_TESTING
    s_pBenchmarkReportService->Init();
    s_pBenchmarkService->Init();
#endif // ENABLE_BENCHMARK_TESTING
#if ENABLE_PERFORMANCE_MONITOR_SERVICE
    s_pPerformanceMonitorService->Init();
#endif // ENABLE_PERFORMANCE_MONITOR_SERVICE
#else
    s_pSimDebugService->Init();
    s_pServicesManager->AddService( s_pSimDebugService );
#endif
    s_pEnvService->Init();
    s_pGenericService->Init();
    s_pHostIOService->Init();
    s_pHostDirectoryIOService->Init();
    s_pHTCSService->Init();
    s_pControllerService->Init();

    s_pServicesManager->AddService( s_pEnvService );
    s_pServicesManager->AddService( s_pGenericService );
    s_pServicesManager->AddService( s_pHostIOService );
    s_pServicesManager->AddService( s_pHostDirectoryIOService );
    s_pServicesManager->AddService( s_pHTCSService );
    s_pServicesManager->AddService( s_pControllerService );
#if !defined( NN_TMA_SIM_INSTANCE )
#if ENABLE_BENCHMARK_TESTING
    // These service are *not* going to be a common one, so I have added it
    // last to prevent it from adding time to "more frequently used" services.
    s_pServicesManager->AddService( s_pBenchmarkService );
    s_pServicesManager->AddService( s_pBenchmarkReportService );
#endif // ENABLE_BENCHMARK_TESTING
#if ENABLE_PERFORMANCE_MONITOR_SERVICE
    s_pServicesManager->AddService( s_pPerformanceMonitorService );
#endif // ENABLE_PERFORMANCE_MONITOR_SERVICE
#endif
}

//==============================================================================

void KillServices()
{
    s_pServicesManager->Kill();
#if !defined( NN_TMA_SIM_INSTANCE )
    s_pPowerManagementService->Kill();
    s_pSettingsService->Kill();
    s_pThreadFrozenService->Kill();
    s_pTargetIOService->Kill();
    s_pCoredumpService->Kill();
    s_pDiagnosticsService->Kill();
    s_pDebugService->Kill();
#if ENABLE_PERFORMANCE_MONITOR_SERVICE
    s_pPerformanceMonitorService->Kill();
#endif // ENABLE_PERFORMANCE_MONITOR_SERVICE
#if ENABLE_BENCHMARK_TESTING
    s_pBenchmarkService->Kill();
    s_pBenchmarkReportService->Kill();
#endif // ENABLE_BENCHMARK_TESTING
#else
    s_pSimDebugService->Kill();
#endif
    s_pEnvService->Kill();
    s_pGenericService->Kill();
    s_pHostIOService->Kill();
    s_pHostDirectoryIOService->Kill();
    s_pHTCSService->Kill();
    s_pControllerService->Kill();
}

#if !defined( NN_TMA_SIM_INSTANCE )

namespace dbg
{
DebugService& GetDebugService()
{
    return *s_pDebugService;
}
}

namespace powermanagement
{
Service& GetService()
{
    return *s_pPowerManagementService;
}
}

namespace settings
{
Service& GetService()
{
    return *s_pSettingsService;
}
}

namespace threadfrozen
{
Service& GetService()
{
    return *s_pThreadFrozenService;
}
}

namespace target_io
{
Service& GetService()
{
    return *s_pTargetIOService;
}
}

namespace coredump
{
Service& GetService()
{
    return *s_pCoredumpService;
}
}

namespace diagnostics
{
Service& GetService()
{
    return *s_pDiagnosticsService;
}
}

#if ENABLE_BENCHMARK_TESTING
namespace benchmark
{
BenchmarkService& GetBenchmarkService()
{
    return *s_pBenchmarkService;
}
BenchmarkReportService& GetBenchmarkReportService()
{
    return *s_pBenchmarkReportService;
}
}
#endif // ENABLE_BENCHMARK_TESTING

#if ENABLE_PERFORMANCE_MONITOR_SERVICE
namespace performance_monitor
{
    bool IsValid()
    {
        return s_pPerformanceMonitorService != nullptr;
    }
    Service& GetService()
    {
        return *s_pPerformanceMonitorService;
    }
}
#endif // ENABLE_PERFORMANCE_MONITOR_SERVICE

#endif

namespace env
{
AgentEnvService& GetEnvService()
{
    return *s_pEnvService;
}
}

namespace generic_service
{
Service& GetService()
{
    return *s_pGenericService;
}
}

namespace hio
{

AgentHostIOService& GetHostIOService()
{
    return *s_pHostIOService;
}

AgentHostDirectoryIOService& GetHostDirectoryIOService()
{
    return *s_pHostDirectoryIOService;
}
}
namespace htcs
{
AgentHTCSService& GetHTCSService()
{
    return *s_pHTCSService;
}
}
namespace hid
{
AgentControllerService& GetControllerService()
{
    return *s_pControllerService;
}
}

//==============================================================================

void SetActiveNode( tmipc::Node* pActiveNode )
{
    if( s_pAgentNode != pActiveNode )
    {
        // Finalize the previous node's "connection and services".
        if( s_pAgentNode != nullptr )
        {
            KillServices();
            s_pServicesManager->SetNode( nullptr );
            s_pAgentNode->Disconnect();
            s_pAgentNode = nullptr;
        }

        // Set up the new active node's "connection and services".
        if( pActiveNode != nullptr )
        {
            s_pAgentNode = pActiveNode;

            InitServices();
            s_pServicesManager->SetNode( s_pAgentNode );
            s_pAgentNode->SetServicesManager( s_pServicesManager );
        }
    }
}

//==============================================================================

void NodeEventCallback( void* pUserData, tmipc::NodeEvent* pEvent )
{
    ScopedLock MutexLock( g_EventCallbackMutex );

    switch( pEvent->GetEventCode() )
    {
    case tmipc::eNodeEventCode::NodeEvent_Connected:
    {
        if( pUserData != nullptr )
        {
            bool bFoundConnection = false;

            // Update the "Beacon Response" to inform future connections that TMA was not put to sleep.
            tma::SetHasWokenUp( false );

            if( pUserData == s_pAgentNodeTCP )
            {
                DEJA_CONTEXT( "tma::NodeEventCallback", "NodeEvent_Connected TCP\n" );
                SetActiveNode( s_pAgentNodeTCP );
                bFoundConnection = true;
            }

            if( pUserData == s_pAgentNodeTICS )
            {
                DEJA_CONTEXT( "tma::NodeEventCallback", "NodeEvent_Connected Host Bridge\n" );
                SetActiveNode( s_pAgentNodeTICS );
                bFoundConnection = true;
            }

            if( pUserData == s_pAgentNodeUSB )
            {
                if( s_pAgentNodeUSB != nullptr )
                {
                    DEJA_CONTEXT( "tma::NodeEventCallback", "NodeEvent_Connected USB\n" );
                    SetActiveNode( s_pAgentNodeUSB );
                    bFoundConnection = true;
                }
            }

            if( pUserData == s_pAgentNodeHtclow )
            {
                if( s_pAgentNodeHtclow != nullptr )
                {
                    DEJA_CONTEXT( "tma::NodeEventCallback", "NodeEvent_Connected HTCLOW\n" );
                    SetActiveNode( s_pAgentNodeHtclow );
                    bFoundConnection = true;
                }
            }

            // Make sure the connection was set - otherwise: how did it not find a node?
            ASSERT( bFoundConnection );
            if( !bFoundConnection )
            {
                DEJA_CONTEXT( "tma::NodeEventCallback", "Did not switch connection types\n" );
            }
            else
            {
                nn::tma::connection::OnConnected();
            }
        }
    }
    break;
    case tmipc::eNodeEventCode::NodeEvent_Disconnected:
    {
        if( g_bNotifyDisconnect )
        {
            nn::tma::connection::OnDisconnected();
        }
    }
    break;
    default:
        // This is a message that does not require handling.
        break;
    }
}

//==============================================================================
#if !defined( NN_TMA_SIM_INSTANCE )
//==============================================================================

namespace dbg
{

//==============================================================================

tmapi::result CheckProcessHandleForError( nn::svc::Handle ProcessHandle )
{
    if( ProcessHandle.IsValid() == false )
    {
        return tmapi::RESULT_PROCESS_NOT_RUNNING;
    }

    return tmapi::RESULT_OK;
}

//==============================================================================

// Writes a formatted memory dump to NN_SDK_LOG
void MemToLog( void* pMemory, int nTotalBytes, const char* pHeader, int nBytesPerWord, int  nWordsPerLine )
{
    unsigned char       Buf[ 256 ];
    unsigned char*      p       = (unsigned char*)pMemory;
    unsigned char*      pEnd    = p + nTotalBytes;

    if(     ( nTotalBytes   < 1 )
        ||  ( nBytesPerWord < 1 )
        ||  ( nBytesPerWord > 8 )
        ||  ( nWordsPerLine < 1 )
        ||  ( ( nBytesPerWord * nWordsPerLine ) > ( sizeof(Buf) - 1 ) ) )
    {
        NN_SDK_LOG( "MemToLog: Invalid parameters\n" );
        return;
    }

    if( pHeader )
    {
        NN_SDK_LOG( "%s", pHeader );
    }

    while( p < pEnd )
    {
        unsigned char*  pBuf    = Buf;

        for( int w = 0; w < nWordsPerLine; w++ )
        {
            NN_SDK_LOG( " " );
            unsigned long long  u = 0;

            for( int b = 0; b < nBytesPerWord; b++ )
            {
                u <<= 8;

                if( p < pEnd )
                {
                    unsigned char c = *p++;

                    u      |= c;

                    if( ( c < 0x20 ) || ( c > 0x7F ) ) c = 0x2E;

                    *pBuf++ = c;
                }
            }

            for( int b = 0; b < nBytesPerWord; b++ )
            {
                NN_SDK_LOG( "%02X", u & 0x00FF );

                u >>= 8;
            }
        }

        *pBuf = 0;
        NN_SDK_LOG( " - \"%s\"\n", Buf );
    }
}

//==============================================================================
} // namespace dbg
//==============================================================================

enum transport
{
    TRANSPORT_USB,
    TRANSPORT_TCP,
    TRANSPORT_HB,
    TRANSPORT_HTCLOW,
};

bool    g_TransportUSB      = false;
bool    g_TransportTCP      = false;
bool    g_TransportHB       = true;
bool    g_TransportHTCLOW   = false;
s32     g_Transport         = TRANSPORT_HB;
void SetActiveNode( tmipc::Node* pActiveNode );

//==============================================================================

void ReadTransportSettings()
{
#if !defined( NN_DETAIL_TMA_NX_RELATED_HARDWARE )

    g_TransportUSB      = false;
    g_TransportTCP      = false;
    g_TransportHB       = false;
    g_TransportHTCLOW   = true;
    g_Transport         = TRANSPORT_HTCLOW;

#elif defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK2 )

    g_TransportUSB      = true;
    g_TransportTCP      = false;
    g_TransportHB       = false;
    g_TransportHTCLOW   = false;
    g_Transport         = TRANSPORT_USB;

#else // hardware

    // Only allow this code to run if the Hardware is an SDEV.  Currently, the SDEV is the
    // only hardware that can support more than one Transport setting.
    // SDEV: HBX or USB.
    // EDEV: USB.
    const char SDEVToken[]{ "SDEV" };
    if( strncmp( SDEVToken, GetTargetConfigurationId(), sizeof(SDEVToken) - 1 ) == 0 )
    {

        const size_t bufferSize = 16;
        char buffer[bufferSize] = {0};
        if( nn::settings::fwdbg::GetSettingsItemValue( buffer, sizeof(buffer), "bsp0", "tm_transport" ) != 0 )
        {
            buffer[bufferSize - 1] = '\0';
            char* p = buffer;
            while( *p )
            {
                *p = std::tolower( *p );
                p++;
            }

            if( std::strstr( buffer, "usb") != nullptr )
            {
// SIGLONTD-7167: No need to check USB mode settings.
//                // INFO: If usb settings is uhs (USB Host Stack), TM cannot use USB as transport path.
//                char usbValue[bufferSize] = {0};
//                if( nn::settings::fwdbg::GetSettingsItemValue( usbValue, sizeof(usbValue), "bsp0", "usb" ) != 0 )
//                {
//                    usbValue[bufferSize - 1] = '\0';
//                    if( (std::strcmp(usbValue, "UHS") == 0) ||
//                        (std::strcmp(usbValue, "Uhs") == 0) ||
//                        (std::strcmp(usbValue, "uhs") == 0) )
//                    {
//                        g_TransportHB = true;
//                        NN_SDK_LOG( "Conflict found in settings: TargetManager is specified to use USB while usb is specified to use 'uhs' (host stack). usb setting supersedes the other and TargetManager will use HostBridge. To use USB for TargetManager, please do not specify 'uhs' for usb.\n" );
//                    }
//                    else
//                    {
//                        g_TransportUSB = true;
//                        NN_SDK_LOG( "ReadTransportSettings: USB\n" );
//                    }
//                }
//                else
                {
                    g_TransportUSB = true;
                    NN_SDK_LOG( "ReadTransportSettings: USB\n" );
                }
            }

            if( std::strstr( buffer, "tcp") != nullptr )
            {
                g_TransportTCP = true;
                NN_SDK_LOG( "ReadTransportSettings: TCP\n" );
            }

            if( std::strstr( buffer, "hb") != nullptr )
            {
                g_TransportHB = true;
                NN_SDK_LOG( "ReadTransportSettings: HB\n" );
            }
        }
    }
    else
    {
        g_TransportHB       = false;
        g_TransportTCP      = false;
        g_TransportUSB      = true;
        g_TransportHTCLOW   = false;
        g_Transport         = TRANSPORT_USB;
        NN_SDK_LOG( "ReadTransportSettings: NON-SDEV hardware\n" );
    }

    // For now select the single transport that we will use.
    if( g_TransportUSB )
    {
        g_Transport = TRANSPORT_USB;
        NN_SDK_LOG( "ReadTransportSettings: Default USB\n" );
    }
    else if( g_TransportTCP )
    {
        g_Transport = TRANSPORT_TCP;
        NN_SDK_LOG( "ReadTransportSettings: Default TCP\n" );
    }
    else
    {
        g_Transport = TRANSPORT_HB;
        NN_SDK_LOG( "ReadTransportSettings: Default HB\n" );
    }

#endif // hardware
}

//==============================================================================
//bool s_bTMANotifiedTMSAboutSleep = false;
void WakeUp()
{
TMA_POWER_TEST_PRINT( "[%s]!!! Start.\n", _BestFunctionName_ );
    if( s_pServicesManager->IsSleeping() )
    {
        g_bNotifyDisconnect = true;
        s_pAgentNode = nullptr;

TMA_POWER_TEST_PRINT( "[%s]!!! A.\n", _BestFunctionName_ );

        // Update the "Beacon Response" to denote it was woken up.
        tma::SetHasWokenUp( true );

        // Create the tmipc::Node.
        switch( g_Transport )
        {
        case TRANSPORT_TCP:
            NN_ABORT( "TCP Transport not supported" );
            break;

        case TRANSPORT_HB:
#if defined( NN_DETAIL_TMA_NX_RELATED_HARDWARE )
            {
TMA_POWER_TEST_PRINT( "[%s]!!! B.\n", _BestFunctionName_ );
                tmipc::NodeTICS::Initialize();
                void* pMem = s_Allocate( sizeof( tmipc::NodeTICS ) );
                s_pAgentNodeTICS = new (pMem) tmipc::NodeTICS();
                s_pAgentNodeTICS->SetThreadPriority( NN_SYSTEM_THREAD_PRIORITY(tma, AgentNodeTics) );
                s_pAgentNodeTICS->SetEventCallback( NodeEventCallback, s_pAgentNodeTICS );

                // Activate (connect) the TICS node now, so that other code can use the
                // tmagent without hanging or crashing.  Since no nodes are connected the
                // callers may receive error codes (which is preferable to crashing or
                // hanging).
                // This implies there can only be one node active when tma "initializes".
                NN_ABORT_UNLESS( s_pAgentNodeTICS != nullptr, "Allocating NodeTICS failed" );
                s_pAgentNodeTICS->Listen();
                s_pAgentNode = s_pAgentNodeTICS;
                nn::tma::bridge_api::Initialize( s_pAgentNodeTICS );
                nn::tma::bridge_api::SetSleepStatus( false );
TMA_POWER_TEST_PRINT( "[%s]!!! C.\n", _BestFunctionName_ );
            }
#else
            NN_ABORT( "HB Transport not supported" );
#endif
            break;

        case TRANSPORT_HTCLOW:
#if defined( NN_DETAIL_TMA_NX_RELATED_HARDWARE )
            NN_ABORT( "Htclow Transport not supported" );
#else
            {
                tmipc::NodeHtclow::Initialize();
                void* pMem = s_Allocate( sizeof( tmipc::NodeHtclow ) );
                s_pAgentNodeHtclow = new (pMem) tmipc::NodeHtclow();
                s_pAgentNodeHtclow->SetThreadPriority( NN_SYSTEM_THREAD_PRIORITY(tma, AgentNodeHtclow) );
                s_pAgentNodeHtclow->SetEventCallback( NodeEventCallback, s_pAgentNodeHtclow );

                NN_ABORT_UNLESS_NOT_NULL(s_pAgentNodeHtclow);
                s_pAgentNodeHtclow->Listen();
                s_pAgentNode = s_pAgentNodeHtclow;
            }
#endif
            break;
        case TRANSPORT_USB:
        default:
#if defined( NN_DETAIL_TMA_NX_RELATED_HARDWARE )
            {
TMA_POWER_TEST_PRINT( "[%s]!!! D.\n", _BestFunctionName_ );
                // Do the NodeUSB initialization after the nn::sm::Initialize(), otherwise
                // it will fail because "nn::sm::GetServiceHandle(...)" (in
                // usb_DSClientInterface.cpp's Initialize()) is not initialized.
                tmipc::Result Result = tmipc::NodeUSB::Initialize( NN_SYSTEM_THREAD_PRIORITY(tma, AgentNodeUsb) );
TMA_POWER_TEST_PRINT( "[%s]!!! E.\n", _BestFunctionName_ );
                if( Result == tmipc::TMIPC_RESULT_OK )
                {
TMA_POWER_TEST_PRINT( "[%s]!!! F.\n", _BestFunctionName_ );
                    void* pMem = s_Allocate( sizeof( tmipc::NodeUSB ) );
                    s_pAgentNodeUSB = new (pMem) tmipc::NodeUSB();
                    s_pAgentNodeUSB->SetEventCallback( NodeEventCallback, s_pAgentNodeUSB );
                    s_pAgentNodeUSB->SetThreadPriority( NN_SYSTEM_THREAD_PRIORITY(tma, AgentNodeUsb) );
                    s_pAgentNodeUSB->Init();
TMA_POWER_TEST_PRINT( "[%s]!!! G.\n", _BestFunctionName_ );
                }

                // Activate (connect) the USB node now, so that other code can use the
                // tmagent without hanging or crashing.  Since no nodes are connected the
                // callers may receive error codes (which is preferable to crashing or
                // hanging).
                NN_ABORT_UNLESS( s_pAgentNodeUSB != nullptr );
                s_pAgentNode = s_pAgentNodeUSB;
TMA_POWER_TEST_PRINT( "[%s]!!! H.\n", _BestFunctionName_ );
            }
#else
            NN_ABORT( "USB Transport not supported" );
#endif
            break;
        }
TMA_POWER_TEST_PRINT( "[%s]!!! I - Node: %p.\n", _BestFunctionName_, s_pAgentNode);

        // Tell the Services Manager to wake up. This should only be called when
        // coming back from sleep.
        s_pServicesManager->WakeUp( s_pAgentNode );

//        // If TMS was notified that TMA went to sleep, then wait to see if TMS
//        // attempts to automatically recreate the connection.
//TMA_POWER_TEST_PRINT( "[%s]!!! J - s_bTMANotifiedTMSAboutSleep: %s.\n", _BestFunctionName_, s_bTMANotifiedTMSAboutSleep ? "true" : "false");
//        if( s_bTMANotifiedTMSAboutSleep )
//        {
//            const s32 NumberOfSeconds = 10;
//            bool IsConnected{ false };
//            for( s32 Index = 0; (Index < NumberOfSeconds) && !IsConnected; Index++ )
//            {
//TMA_POWER_TEST_PRINT( "[%s]!!! K - Implement 'wait for TMS reconnection' here: %d/%d.\n", _BestFunctionName_, Index + 1, NumberOfSeconds );
//                IsConnected = s_pAgentNode->IsConnected();
//                if( IsConnected )
//                {
//TMA_POWER_TEST_PRINT( "[%s]!!! L - Connected: %d/%d.\n", _BestFunctionName_, Index + 1, NumberOfSeconds );
//                }
//                else
//                {
//                    nn::os::SleepThread( nn::TimeSpan::FromSeconds( 1 ) );
//                }
//            }
//
//TMA_POWER_TEST_PRINT( "[%s]!!! M - Connected? %s.\n", _BestFunctionName_, IsConnected ? "Yes" : "No" );
//        }
//
//        s_bTMANotifiedTMSAboutSleep = false;
    }
TMA_POWER_TEST_PRINT( "[%s]!!! Finish.\n", _BestFunctionName_ );
}

//==============================================================================

void GoToSleep()
{
TMA_POWER_TEST_PRINT( "[%s]!!! - Start.\n", _BestFunctionName_ );
    if( !s_pServicesManager->IsSleeping() )
    {
        g_bNotifyDisconnect = false;
        // Attempt to notify TMS that TMA is going to sleep.
        if( (s_pAgentNode != nullptr) && s_pAgentNode->IsConnected() )
        {
TMA_POWER_TEST_PRINT( "[%s]!!! - A.\n", _BestFunctionName_ );
#if 1
            const tmipc::Result NotifyResult = powermanagement::GetService().NotifyTMSGoingToSleep();
//            s_bTMANotifiedTMSAboutSleep = NotifyResult == tmipc::Result::TMIPC_RESULT_OK;
TMA_POWER_TEST_PRINT( "[%s]!!! - B - Result: %d.\n", _BestFunctionName_, NotifyResult );
#else
TMA_POWER_TEST_PRINT( "[%s]!!! - C - (Skipping powermanagement::GetService().NotifyTMSGoingToSleep() for debugging).\n", _BestFunctionName_ );
#endif
        }

#if defined( NN_DETAIL_TMA_NX_RELATED_HARDWARE )
        if( s_pAgentNode == s_pAgentNodeTICS )
        {
            nn::tma::bridge_api::SetSleepStatus( true );
        }
#endif

        // Let the process manager get ready
        s_pProcessMgr->OnSleep();

        // Tell the services manager to go to sleep.
        s_pServicesManager->GoToSleep();
TMA_POWER_TEST_PRINT( "[%s]!!! - D.\n", _BestFunctionName_ );

        // There will no longer be an active node after this function finishes.
        // This function will also Kill any services that have been initialized.
TMA_POWER_TEST_PRINT( "[%s]!!! - E.\n", _BestFunctionName_ );
        s_pServicesManager->SetNode( nullptr );
TMA_POWER_TEST_PRINT( "[%s]!!! - F %p.\n", _BestFunctionName_, s_pAgentNode);
        // The Host Bridge is actively listening for a connection in a thread,
        // so make sure it stops listening before Disconnecting.  Normally,
        // Disconnecting relies on the listening thread, but in this case the
        // Target is going to sleep so the thread needs to quit.
        if( s_pAgentNode == s_pAgentNodeTICS )
        {
            s_pAgentNodeTICS->StopListening();
        }
        s_pAgentNode->Disconnect();
TMA_POWER_TEST_PRINT( "[%s]!!! - G %p.\n", _BestFunctionName_, s_pAgentNode);
        s_pAgentNode = nullptr;

        // Cancel all pending tasks.  This is really only needed for NodeTICS.
        // NodeUSB does this every time it gets a "Beacon Request", but TICS
        // does not do this.  That being said, this code should be in the
        // ServicesManager.
        s_pServicesManager->CancelAllTasks();

#if defined( NN_DETAIL_TMA_NX_RELATED_HARDWARE )
TMA_POWER_TEST_PRINT( "[%s]!!! - H.\n", _BestFunctionName_ );
        if( s_pAgentNodeTICS )
        {
            nn::tma::bridge_api::Finalize();
TMA_POWER_TEST_PRINT( "[%s]!!! - I.\n", _BestFunctionName_ );
            s_pAgentNodeTICS->Kill();
TMA_POWER_TEST_PRINT( "[%s]!!! - J.\n", _BestFunctionName_ );
            s_pAgentNodeTICS->~NodeTICS();
            s_Deallocate( s_pAgentNodeTICS, sizeof( tmipc::NodeTICS ) );
            s_pAgentNodeTICS = nullptr;
TMA_POWER_TEST_PRINT( "[%s]!!! - K.\n", _BestFunctionName_ );
            tmipc::NodeTICS::Finalize();
TMA_POWER_TEST_PRINT( "[%s]!!! - L.\n", _BestFunctionName_ );
        }
TMA_POWER_TEST_PRINT( "[%s]!!! - M.\n", _BestFunctionName_ );
#endif

#if defined( NN_DETAIL_TMA_NX_RELATED_HARDWARE )
        if( s_pAgentNodeUSB != nullptr )
        {
TMA_POWER_TEST_PRINT( "[%s]!!! - N.\n", _BestFunctionName_ );
////TODO: Note: This happens above, with s_pAgentNode->Disconnect(), so there is no need to call it again. (It will be disconnected by then, so it should not matter, but why take that chance?)
//            s_pAgentNodeUSB->Disconnect();
TMA_POWER_TEST_PRINT( "[%s]!!! - O.\n", _BestFunctionName_ );
            s_pAgentNodeUSB->Kill();
            s_pAgentNodeUSB->~NodeUSB();
            s_Deallocate( s_pAgentNodeUSB, sizeof( tmipc::NodeUSB ) );
            s_pAgentNodeUSB = nullptr;
TMA_POWER_TEST_PRINT( "[%s]!!! - P.\n", _BestFunctionName_ );

            // Finalize the USB interface.
            tmipc::NodeUSB::Finalize();
TMA_POWER_TEST_PRINT( "[%s]!!! - Q.\n", _BestFunctionName_ );
        }
#endif
    }
TMA_POWER_TEST_PRINT( "[%s]!!! - Finish.\n", _BestFunctionName_ );
}

//==============================================================================

void Initialize( AllocateFunction Alloc, DeallocateFunction Dealloc )
{
    if( Alloc != DefaultAlloc )
    {
        NN_SDK_LOG( "\nTMA Process does not support custom allocators.\n" );
        ASSERT( false, "\nTMA Process does not support custom allocators.\n" );
    }

#if TRACK_MEMORY
    s_AllocateCall = Alloc;
#else
    s_Allocate = Alloc;
#endif
    s_Deallocate = Dealloc;

    //=============================================================
    // We'll init memory tracking whether we track memory or not -
    // this way, we can at least call the built-in memory reporting
    // without having to rebuild tmagent.
    //=============================================================
    diagnostics::memory_tracking::Init( g_HeapHandle );

    // Distribute the priority set in Make Initial Program
    int agentPriority = nn::os::GetThreadPriority( nn::os::GetCurrentThread() );

    NN_SDK_ASSERT(agentPriority == NN_SYSTEM_THREAD_PRIORITY(tma, AgentNodeTics));
    NN_SDK_ASSERT(agentPriority == NN_SYSTEM_THREAD_PRIORITY(tma, AgentNodeUsb));
    NN_SDK_ASSERT(agentPriority == NN_SYSTEM_THREAD_PRIORITY(tma, Beacon));
//    NN_SDK_ASSERT(agentPriority == NN_SYSTEM_THREAD_PRIORITY(tma, DbgDispatchTask));
//    NN_SDK_ASSERT(agentPriority == NN_SYSTEM_THREAD_PRIORITY(tma, DbgEvents));
//    NN_SDK_ASSERT(agentPriority == NN_SYSTEM_THREAD_PRIORITY(tma, DbgWorkThread));
    NN_SDK_ASSERT(agentPriority == NN_SYSTEM_THREAD_PRIORITY(tma, HandleAttachOnStart));
    NN_SDK_ASSERT(agentPriority == NN_SYSTEM_THREAD_PRIORITY(tma, HandleLoadExeThread));
    NN_SDK_ASSERT(agentPriority == NN_SYSTEM_THREAD_PRIORITY(tma, PmModuleThread));
    NN_SDK_ASSERT(agentPriority == NN_SYSTEM_THREAD_PRIORITY(tma, ServicesMgr));
    NN_SDK_ASSERT(agentPriority == NN_SYSTEM_THREAD_PRIORITY(tma, TcpListen));

    g_EventCallbackMutex.Create();

    s_pAgentNode = nullptr;

     void* pMem = s_Allocate( sizeof( tma::dbg::ProcessMgr ) );
    s_pProcessMgr = new (pMem) tma::dbg::ProcessMgr();

    // Initialize Services.
    nn::Result result;

    result = nn::sm::Initialize();
    NN_ABORT_UNLESS( result.IsSuccess(), "result=%08x", result );

    dbghlp::Initialize();

    CreateTMAServices();

    s_pProcessMgr->Init();

    // Record some Target settings.  This needs to happen as infrequent as
    // possible, in order to prevent a circular dependency with the file system.
    // (i.e. This is related to SigloNTD-5339: Discuss what to do when the target tries to sleep.
    // See struct 'TargetSettings' for details.)
    CacheTargetSettings();

    ReadTransportSettings();

    switch( g_Transport )
    {
    case TRANSPORT_TCP:
        NN_ABORT( "TCP Transport not supported" );
        break;

    case TRANSPORT_HB:
#if defined( NN_DETAIL_TMA_NX_RELATED_HARDWARE )
    {
        tmipc::NodeTICS::Initialize();
        void* pMem = s_Allocate( sizeof( tmipc::NodeTICS ) );
        s_pAgentNodeTICS = new (pMem) tmipc::NodeTICS();
        s_pAgentNodeTICS->SetThreadPriority( NN_SYSTEM_THREAD_PRIORITY(tma, AgentNodeTics) );
        s_pAgentNodeTICS->SetEventCallback( NodeEventCallback, s_pAgentNodeTICS );

        // Activate (connect) the TICS node now, so that other code can use the
        // tmagent without hanging or crashing.  Since no nodes are connected the
        // callers may receive error codes (which is preferable to crashing or
        // hanging).
        // This implies there can only be one node active when tma "initializes".
        SetActiveNode( s_pAgentNodeTICS );
        s_pAgentNodeTICS->Listen();
        nn::tma::bridge_api::Initialize( s_pAgentNodeTICS );
    }
#else
    NN_ABORT( "HB Transport not supported" );
#endif
    break;

    case TRANSPORT_HTCLOW:
#if defined( NN_DETAIL_TMA_NX_RELATED_HARDWARE )
    NN_ABORT( "HTCLOW Transport not supported" );
#else
    {
        tmipc::NodeHtclow::Initialize();
        void* pMem = s_Allocate(sizeof(tmipc::NodeHtclow));
        s_pAgentNodeHtclow = new (pMem) tmipc::NodeHtclow();
        s_pAgentNodeHtclow->SetThreadPriority(NN_SYSTEM_THREAD_PRIORITY(tma, AgentNodeHtclow));
        s_pAgentNodeHtclow->SetEventCallback(NodeEventCallback, s_pAgentNodeHtclow);

        SetActiveNode( s_pAgentNodeHtclow );
        s_pAgentNodeHtclow->Listen();
    }
#endif
    break;

    case TRANSPORT_USB:
    default:
#if defined( NN_DETAIL_TMA_NX_RELATED_HARDWARE )
    {
        // Do the NodeUSB initialization after the nn::sm::Initialize(), otherwise
        // it will fail because "nn::sm::GetServiceHandle(...)" (in
        // usb_DSClientInterface.cpp's Initialize()) is not initialized.
        tmipc::Result Result = tmipc::NodeUSB::Initialize( NN_SYSTEM_THREAD_PRIORITY(tma, AgentNodeUsb) );
        if( Result == tmipc::TMIPC_RESULT_OK )
        {
            void* pMem = s_Allocate( sizeof( tmipc::NodeUSB ) );
            s_pAgentNodeUSB = new (pMem) tmipc::NodeUSB();
            s_pAgentNodeUSB->SetEventCallback( NodeEventCallback, s_pAgentNodeUSB );
            s_pAgentNodeUSB->SetThreadPriority( NN_SYSTEM_THREAD_PRIORITY(tma, AgentNodeUsb) );
            s_pAgentNodeUSB->Init();
        }

        // Activate (connect) the USB node now, so that other code can use the
        // tmagent without hanging or crashing.  Since no nodes are connected the
        // callers may receive error codes (which is preferable to crashing or
        // hanging).
        if( s_pAgentNodeUSB != nullptr )
        {
            SetActiveNode( s_pAgentNodeUSB );
        }
    }
#else
    NN_ABORT( "USB Transport not supported" );
#endif
    break;
    }

    nn::tma::connection::Initialize();

    // This must be done at the end of Initialize once everything is ready
    // because other processes may start to make request to TMAgent before
    // this call returns.
    InitializeHipcServers();
    s_bInitialized = true;

    // Register with the PowerManagementMonitor for callbacks.
    if (!GetDisableSleepFlag())
    {
        const tmipc::Result Result = tma::PowerManagementMonitor::Initialize( PowerManagementCallback );
        (void)Result;
    }
    NN_SDK_LOG( "Target Manager Agent Version: %s\n", GetTMAVersionNumber() );
}

//==============================================================================

void Initialize()
{
    memmgr_init();

    // Initialize the ThreadTracker and register the main thread.
    {
        tma::ThreadTracker::Initialize();
        const nn::os::ThreadType* pThreadType{ nn::os::GetCurrentThread() };
        if (pThreadType != nullptr)
        {
            tma::ThreadTracker::RegisterThread(*pThreadType, tma::ThreadTracker::ThreadId::Main);
        }
    }

    Initialize(DefaultAlloc, DefaultDealloc);
}

//==============================================================================

void Finalize()
{
    if( s_bInitialized == true )
    {
        s_bInitialized = false;

        FinalizeHipcServers();

        nn::tma::connection::Finalize();

        s_pProcessMgr->Kill();
        s_pProcessMgr->~ProcessMgr();
        s_Deallocate( s_pProcessMgr, sizeof( tma::dbg::ProcessMgr ) );
        s_pProcessMgr = nullptr;

        // There will no longer be an active node after this function finishes.
        // This function will also Kill any services that have been initialized.
        SetActiveNode( nullptr );

        // The Host Bridge is actively listening for a connection in a thread,
        // so make sure it stops listening before Disconnecting.  Normally,
        // Disconnecting relies on the listening thread, but in this case the
        // Target is going to sleep so the thread needs to quit.
        if( s_pAgentNode == s_pAgentNodeTICS )
        {
            s_pAgentNodeTICS->StopListening();
        }
        s_pAgentNode->Disconnect();

#if defined(NN_DETAIL_TMA_NX_RELATED_HARDWARE)
        if( s_pAgentNodeTICS )
        {
            nn::tma::bridge_api::Finalize();
            s_pAgentNodeTICS->Kill();
            s_pAgentNodeTICS->~NodeTICS();
            s_Deallocate( s_pAgentNodeTICS, sizeof( tmipc::NodeTICS ) );
            s_pAgentNodeTICS = nullptr;
            tmipc::NodeTICS::Finalize();
        }
#endif

#if defined(NN_DETAIL_TMA_NX_RELATED_HARDWARE)
        if( s_pAgentNodeUSB != nullptr )
        {
////TODO: Note: This happens above, with s_pAgentNode->Disconnect(), so there is no need to call it again. (It will be disconnected by then, so it should not matter, but why take that chance?)
//            s_pAgentNodeUSB->Disconnect();
            s_pAgentNodeUSB->Kill();
            s_pAgentNodeUSB->~NodeUSB();
            s_Deallocate( s_pAgentNodeUSB, sizeof( tmipc::NodeUSB ) );
            delete s_pAgentNodeUSB;
            s_pAgentNodeUSB = nullptr;

            // Finalize the USB interface.
            tmipc::NodeUSB::Finalize();
        }
#endif

#if !defined(NN_DETAIL_TMA_NX_RELATED_HARDWARE)
        if( s_pAgentNodeHtclow != nullptr )
        {
            s_pAgentNodeHtclow->Kill();
            s_pAgentNodeHtclow->~NodeHtclow();
            s_Deallocate( s_pAgentNodeHtclow, sizeof( tmipc::NodeHtclow ) );
            s_pAgentNodeHtclow = nullptr;
            tmipc::NodeHtclow::Finalize();
        }
#endif

        s_pAgentNode = nullptr;

        DestroyTMAServices();

        diagnostics::memory_tracking::Kill();

        nn::sm::Finalize();

        g_EventCallbackMutex.Destroy();
    }

    // Unregister with the PowerManagementMonitor.
    if (!GetDisableSleepFlag())
    {
        tma::PowerManagementMonitor::Finalize();
    }

    // Unregister the main thread, and finalize the ThreadTracker.
    {
        tma::ThreadTracker::UnregisterThread(tma::ThreadTracker::ThreadId::Main);
        tma::ThreadTracker::Finalize();
    }
}

//==============================================================================
// This callback gets called from pm_monitor.cpp's TmaPmModuleThread function.
void PowerManagementCallback( nn::psc::PmState PmState, const nn::psc::PmFlagSet& PmFlagSet )
{
    DEJA_CONTEXT( "tma::PowerManagementCallback", "PmState: %d; PmFlagSet: 0x%08x", PmState, PmFlagSet );
TMA_POWER_TEST_PRINT( "[%s]!!! PmState: %d - Start; PmFlagSet: 0x%08x.\n", _BestFunctionName_, PmState, PmFlagSet );

    // From comments in psc_PmModule.h:
    // PM modules are expected to handle at least
    // PmState_FullAwake
    // PmState_MinimumAwake
    // PmState_SleepReady
    switch( PmState )
    {
    case nn::psc::PmState_FullAwake:
    {
        WakeUp();
    }
    break;
    case nn::psc::PmState_MinimumAwake:
    {
        DEJA_CONTEXT( "tma::PowerManagementCallback", "PmState: %d - PmState_FullAwake", PmState );
TMA_POWER_TEST_PRINT( "[%s]!!! - PmState: %d; PmFlagSet: 0x%08x - PmState_FullAwake - Start.\n", _BestFunctionName_, PmState, PmFlagSet );
        if( s_pServicesManager->IsSleeping() )
        {
            // Store this before waking up the Services, to prevent *all* chances
            // that a connection occurs before the WakeUp returns.
            const bool bPreserveState{ s_pServicesManager->PreserveStateThroughSleepWake() };
TMA_POWER_TEST_PRINT( "[%s]!!! - A. PreserveState: '%s'.\n", _BestFunctionName_, bPreserveState ? "Yes" : "No" );
            WakeUp();
            if( bPreserveState )
            {
                bool bConnected{ s_pServicesManager->IsConnected() };
                bool bTimedOut{ false };
                const s64 Frequency{ nn::os::GetSystemTickFrequency() };
                const s64 StartTick{ nn::os::GetSystemTick().GetInt64Value() };
                // Wait up to 7 seconds.
                const s64 MaxTick{ StartTick + (Frequency * 7) };
TMA_POWER_TEST_PRINT( "[%s]!!! - B. Frequency: %lld; StartTick: %lld; MaxTick: %lld.\n", _BestFunctionName_, Frequency, StartTick, MaxTick );
                while( !bConnected && !bTimedOut )
                {
                    bConnected = s_pServicesManager->IsConnected();
                    if (!bConnected)
                    {
                        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 1 ) );
                    }
                    bTimedOut = MaxTick <= nn::os::GetSystemTick().GetInt64Value();
                }
                if( bTimedOut )
                {
                    nn::tma::connection::OnDisconnected();
                }
                const s64 EndTick{ nn::os::GetSystemTick().GetInt64Value() };
                const s64 ElapsedTime{ EndTick - StartTick };
                const float fElapsedTime{ static_cast<float>(static_cast<double>(ElapsedTime) / static_cast<double>(Frequency)) };
TMA_POWER_TEST_PRINT( "[%s]!!! - C. bConnected: '%s'; bTimedOut: '%s'; ElapsedTime: %lld (%6.4f seconds).\n", _BestFunctionName_, bConnected ? "Yes" : "No", bTimedOut ? "Yes" : "No", ElapsedTime, fElapsedTime );
            }
        }

        // Let the Services Manager know this variable is done being used.
        s_pServicesManager->ResetPreserveStateThroughSleepWake();
TMA_POWER_TEST_PRINT( "[%s]!!! - PmState: %d; PmFlagSet: 0x%08x - PmState_FullAwake - Finish.\n", _BestFunctionName_, PmState, PmFlagSet );
    }
    break;
    case nn::psc::PmState_SleepReady:
    {
TMA_POWER_TEST_PRINT( "[%s]!!! - PmState: %d; PmFlagSet: 0x%08x - PmState_SleepReady - Start.\n", _BestFunctionName_, PmState, PmFlagSet );
        GoToSleep();
TMA_POWER_TEST_PRINT( "[%s]!!! - PmState: %d; PmFlagSet: 0x%08x - PmState_SleepReady - Finish.\n", _BestFunctionName_, PmState, PmFlagSet );
    }
    break;
    default:
    {
        // no-op, but still return ResultSuccess()
        // for unhandled request types - new types may be
        // added in the future.
    }
    break;
    }
TMA_POWER_TEST_PRINT( "[%s]!!! PmState: %d; PmFlagSet: 0x%08x - Finish.\n", _BestFunctionName_, PmState, PmFlagSet );
}

//==============================================================================
#else // NN_TMA_SIM_INSTANCE
//==============================================================================
static tma::Thread  sAutoConnectThread;
static s32          s_ListenPort;
static bool         s_DoAutoConnect         = true;

void* AutoconnectThread( void* pArgs )
{
    s32 listenPort = *static_cast<s32*>(pArgs);
    WORD wVersionRequested;
    WSADATA wsaData;
    wVersionRequested = MAKEWORD(2, 2);
    int result = WSAStartup( wVersionRequested, &wsaData );
    ASSERT( result == 0 );

    // Get a socket.
    int s = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ASSERT( s != nn::socket::InvalidSocket );

    LARGE_INTEGER Freq;
    LARGE_INTEGER FreqMs;
    LARGE_INTEGER TimeStart;
    LARGE_INTEGER TimeNow;

    QueryPerformanceFrequency( &Freq );
    FreqMs.QuadPart  = Freq.QuadPart / 1000;

    QueryPerformanceCounter( &TimeStart );
    TimeNow = TimeStart;

    nn::socket::SockAddrIn Addr;
    memset( &Addr, 0, sizeof(Addr) );
    Addr.sin_family = nn::socket::Family::Af_Inet;
    Addr.sin_port   = nn::socket::InetHtons( (u16)TMA_AUTOCONNECT_PORT );
    Addr.sin_addr.S_addr = nn::socket::InetHtonl( 0x7F000001 );

    // Use Time as a GUID.
    u32 Buffer[4];
    Buffer[0] = TMA_AUTOCONNECT_VERSION;
    memcpy( &Buffer[1], &TimeStart, 8 );
    Buffer[3] = listenPort;

    // Spin waiting for a TM connection.
    //int64_t TIMEOUT_MS = 20000;
    //while( !s_pAgentNode->IsConnected() && (((TimeNow.QuadPart - TimeStart.QuadPart) / FreqMs.QuadPart) < TIMEOUT_MS) )
    while( s_DoAutoConnect == true && !s_pAgentNode->IsConnected() )
    {
        // Request connection from local target manager.
        result = (int)nn::socket::SendTo( s, (const char*)Buffer, sizeof(Buffer), nn::socket::MsgFlag::Msg_None, (const nn::socket::SockAddr*)&Addr, sizeof(Addr) );
        ASSERT( result >= 0 );

        // Wait 10ms.
        tma::Thread::Sleep( 10 );

        // Read new time.
        QueryPerformanceCounter( &TimeNow );
    }

    //============================================================
    // Clean up our socket.  This fixes SigloNTD-6715, "Should be
    // able to call htcs::Initialize/Finalize multiple times".
    //============================================================
    nn::socket::Close( s );

    return nullptr;
}

//==============================================================================

#ifdef USE_ATEXIT
void AtExit( void );
static bool s_ExitHandlerInstalled  = false;
#endif

//==============================================================================

void Initialize( AllocateFunction Allocate, DeallocateFunction Deallocate )
{
#if TRACK_MEMORY
    s_AllocateCall = Allocate;
#else
    s_Allocate = Allocate;
#endif
    s_Deallocate = Deallocate;
#ifdef USE_ATEXIT
    if( !s_ExitHandlerInstalled )
    {
        s_ExitHandlerInstalled = true;
        atexit( AtExit );
    }
#endif
    // Only Ethernet support for "Win32".
    void* pMem = s_Allocate( sizeof( tmipc::NodeTCP ) );

    g_EventCallbackMutex.Create();

    s_pAgentNodeTCP = new (pMem) tmipc::NodeTCP();
    s_pAgentNodeTCP->SetThreadPriority( NN_SYSTEM_THREAD_PRIORITY(tma, TcpListen) );
    s_pAgentNodeTCP->SetEventCallback( NodeEventCallback, s_pAgentNodeTCP );
    s_pAgentNodeTCP->Init();

    CreateTMAServices();

    // Functions moved from AgentThread TODO: Remove if change is stable - CPC.
    InitServices();
    s_pAgentNode = s_pAgentNodeTCP;
    s_pServicesManager->SetNode( s_pAgentNode );
    s_pAgentNode->SetServicesManager( s_pServicesManager );

    nn::tma::connection::Initialize();

    // Record some Target settings.  This needs to happen as infrequent as
    // possible, in order to prevent a circular dependency with the file system.
    // (i.e. This is related to SigloNTD-5339: Discuss what to do when the target tries to sleep.
    // See function 'CacheTargetSettings' for details.)
    CacheTargetSettings();

    // Start TM Agent.
    s_pAgentNode->Listen();
    s_ListenPort = reinterpret_cast<tmipc::NodeTCP*>(s_pAgentNode)->GetListenPort();

    // Auto connect to Target Manager.
    s_DoAutoConnect = true; // Needs to be set back to true if a person calls Initialize again after Finalize.
    sAutoConnectThread.Start( AutoconnectThread, &s_ListenPort, 16 * 1024, NN_SYSTEM_THREAD_PRIORITY(tma, Autoconnect), NN_SYSTEM_THREAD_NAME(tma, Autoconnect) );

    s_bInitialized = true;
}

//==============================================================================

void Initialize()
{
    Initialize( DefaultAlloc, DefaultDealloc );
}

//==============================================================================

void Log( const char* pMessage, ... )
{
    (void)pMessage;
    //TBD
}

//==============================================================================

void Finalize()
{
    if( s_bInitialized == true )
    {
        s_bInitialized = false;

        if( sAutoConnectThread.IsRunning() )
        {
            s_DoAutoConnect = false;
            sAutoConnectThread.Join();
        }
        else
        {
            nn::tma::connection::OnDisconnected();
        }
        sAutoConnectThread.Destroy();

        nn::tma::connection::Finalize();

        s_pAgentNode->Kill();
        KillServices();
        DestroyTMAServices();

        s_pAgentNode->~Node();
        s_Deallocate( s_pAgentNode, sizeof( tmipc::Node ) );
        s_pAgentNode = nullptr;
        // At this point, s_pAgentNode == s_pAgentNodeTCP.
        s_pAgentNodeTCP = nullptr;

        g_EventCallbackMutex.Destroy();
    }
}

//==============================================================================

#ifdef USE_ATEXIT
void AtExit()
{
    Finalize();
}
#endif

//==============================================================================
#endif // NN_TMA_SIM_INSTANCE
//==============================================================================


//==============================================================================
} // namespace tma
//==============================================================================
