﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdlib>
#include <stdint.h>
#include <cstring>

#ifdef NX
#include <nn/fs/fs_Directory.h>
#endif

//==============================================================================

namespace nn { namespace tma { namespace target_io {

enum
{
#ifdef NX
    MAX_TARGET_IO_PATH = nn::fs::EntryNameLengthMax
#else
    MAX_TARGET_IO_PATH = 768 //Sigh
#endif
};

//==============================================================================

enum AccessMode : uint32_t
{
    TARGET_IO_ACCESS_READ       = 0x80000000,
    TARGET_IO_ACCESS_WRITE      = 0x40000000,
    TARGET_IO_ACCESS_EXECUTE    = 0x20000000,
    TARGET_IO_ACCESS_ALL        = 0x10000000,
};

//==============================================================================

enum CreationDisposition : int32_t
{
    // Always create a new file.
    // If the specified file exists and is writable, the function overwrites the file, the function succeeds, and last-error code is set to ERROR_ALREADY_EXISTS (183).
    // If the specified file does not exist and is a valid path, a new file is created, the function succeeds, and the last-error code is set to zero.
    TARGET_IO_CREATE_ALWAYS = 2,

    // Creates a new file, only if it does not already exist.
    // If the specified file exists, the function fails and the last-error code is set to ERROR_FILE_EXISTS (80).
    // If the specified file does not exist and is a valid path to a writable location, a new file is created.
    TARGET_IO_CREATE_NEW = 1,

    // Opens a file, always.
    // If the specified file exists, the function succeeds and the last-error code is set to ERROR_ALREADY_EXISTS (183).
    // If the specified file does not exist and is a valid path to a writable location, the function creates a file and the last-error code is set to zero.
    TARGET_IO_OPEN_ALWAYS   = 4,

    // Opens a file or device, only if it exists.
    // If the specified file or device does not exist, the function fails and the last-error code is set to ERROR_FILE_NOT_FOUND (2).
    TARGET_IO_OPEN_EXISTING = 3,

    // Opens a file and truncates it so that its size is zero bytes, only if it exists.
    // If the specified file does not exist, the function fails and the last-error code is set to ERROR_FILE_NOT_FOUND (2).
    // The calling process must open the file with the GENERIC_WRITE bit set as part of the dwDesiredAccess parameter.
    TARGET_IO_TRUNCATE_EXISTING = 5,
};

//==============================================================================

enum FileAttributes : int32_t
{
    // A file or directory that is an archive file or directory. Applications typically use this attribute to mark files for backup or removal.
    TARGET_IO_FILE_ATTRIBUTE_ARCHIVE                = 32,

    // A file or directory that is compressed. For a file, all of the data in the file is compressed. For a directory, compression is the default for newly created files and subdirectories.
    TARGET_IO_FILE_ATTRIBUTE_COMPRESSED             = 2048,

    // This value is reserved for system use.
    TARGET_IO_FILE_ATTRIBUTE_DEVICE                 = 64,

    // The handle that identifies a directory.
    TARGET_IO_FILE_ATTRIBUTE_DIRECTORY              = 16,

    // A file or directory that is encrypted. For a file, all data streams in the file are encrypted. For a directory, encryption is the default for newly created files and subdirectories.
    TARGET_IO_FILE_ATTRIBUTE_ENCRYPTED              = 16384,

    // The file or directory is hidden. It is not included in an ordinary directory listing.
    TARGET_IO_FILE_ATTRIBUTE_HIDDEN                 = 2,

    // The directory or user data stream is configured with integrity (only supported on ReFS volumes). It is not included in an ordinary directory listing.
    // The integrity setting persists with the file if it's renamed. If a file is copied the destination file will have integrity set if either the source file or
    // destination directory have integrity set.
    // Windows Server 2008 R2, Windows 7, Windows Server 2008, Windows Vista, Windows Server 2003 and Windows XP:  This flag is not supported until Windows Server 2012.
    TARGET_IO_FILE_ATTRIBUTE_INTEGRITY_STREAM       = 32768,

    // A file that does not have other attributes set. This attribute is valid only when used alone.
    TARGET_IO_FILE_ATTRIBUTE_NORMAL                 =  128,

    // The file or directory is not to be indexed by the content indexing service.
    TARGET_IO_FILE_ATTRIBUTE_NOT_CONTENT_INDEXED    = 8192,

    // The user data stream not to be read by the background data integrity scanner (AKA scrubber).
    // When set on a directory it only provides inheritance. This flag is only supported on Storage Spaces and ReFS volumes.
    // It is not included in an ordinary directory listing.
    // Windows Server 2008 R2, Windows 7, Windows Server 2008, Windows Vista, Windows Server 2003 and Windows XP:  This flag is not supported until Windows 8 and Windows Server 2012.
    TARGET_IO_FILE_ATTRIBUTE_NO_SCRUB_DATA          = 131072,

    // The data of a file is not available immediately. This attribute indicates that the file data is physically moved to offline storage.
    // This attribute is used by Remote Storage, which is the hierarchical storage management software. Applications should not arbitrarily change this attribute.
    TARGET_IO_FILE_ATTRIBUTE_OFFLINE                = 4096,

    // A file that is read-only. Applications can read the file, but cannot write to it or delete it.  This attribute is not honored on directories.
    TARGET_IO_FILE_ATTRIBUTE_READONLY               = 1,

    // A file or directory that has an associated reparse point, or a file that is a symbolic link.
    TARGET_IO_FILE_ATTRIBUTE_REPARSE_POINT          = 1024,

    // A file that is a sparse file.
    TARGET_IO_FILE_ATTRIBUTE_SPARSE_FILE            = 512,

    // A file or directory that the operating system uses a part of, or uses exclusively.
    TARGET_IO_FILE_ATTRIBUTE_SYSTEM                 = 4,

    // A file that is being used for temporary storage.
    TARGET_IO_FILE_ATTRIBUTE_TEMPORARY              = 256,

    // This value is reserved for system use.
    TARGET_IO_FILE_ATTRIBUTE_VIRTUAL                = 65536,
};

//==============================================================================

enum FileFlags : uint32_t
{
    // The file is being opened or created for a backup or restore operation. The system ensures that the calling process overrides file security checks when the process has SE_BACKUP_NAME and SE_RESTORE_NAME privileges.
    // You must set this flag to obtain a handle to a directory. A directory handle can be passed to some functions instead of a file handle.
    TARGET_IO_FILE_FLAG_BACKUP_SEMANTICS = 0x02000000,

    // The file is to be deleted immediately after all of its handles are closed, which includes the specified handle and any other open or duplicated handles.
    // If there are existing open handles to a file, the call fails unless they were all opened with the FILE_SHARE_DELETE share mode.
    // Subsequent open requests for the file fail, unless the FILE_SHARE_DELETE share mode is specified.
    TARGET_IO_FILE_FLAG_DELETE_ON_CLOSE = 0x04000000, //

    // The file or device is being opened with no system caching for data reads and writes. This flag does not affect hard disk caching or memory mapped files.
    // There are strict requirements for successfully working with files opened with CreateFile using the FILE_FLAG_NO_BUFFERING flag.
    TARGET_IO_FILE_FLAG_NO_BUFFERING = 0x20000000,

    // The file data is requested, but it should continue to be located in remote storage. It should not be transported back to local storage. This flag is for use by remote storage systems.
    TARGET_IO_FILE_FLAG_OPEN_NO_RECALL = 0x00100000,

    // Normal reparse point processing will not occur; CreateFile will attempt to open the reparse point. When a file is opened, a file handle is returned,
    // whether or not the filter that controls the reparse point is operational.
    // This flag cannot be used with the CREATE_ALWAYS flag.
    // If the file is not a reparse point, then this flag is ignored.
    TARGET_IO_FILE_FLAG_OPEN_REPARSE_POINT = 0x00200000,

    // The file or device is being opened or created for asynchronous I/O.
    // When subsequent I/O operations are completed on this handle, the event specified in the OVERLAPPED structure will be set to the signaled state.
    // If this flag is specified, the file can be used for simultaneous read and write operations.
    // If this flag is not specified, then I/O operations are serialized, even if the calls to the read and write functions specify an OVERLAPPED structure.
    TARGET_IO_FILE_FLAG_OVERLAPPED      = 0x40000000,

    // Access will occur according to POSIX rules. This includes allowing multiple files with names, differing only in case,
    // for file systems that support that naming. Use care when using this option, because files created with this flag may not be accessible by
    // applications that are written for MS-DOS or 16-bit Windows.
    TARGET_IO_FILE_FLAG_POSIX_SEMANTICS = 0x0100000,

    // Access is intended to be random. The system can use this as a hint to optimize file caching.
    // This flag has no effect if the file system does not support cached I/O and FILE_FLAG_NO_BUFFERING.
    TARGET_IO_FILE_FLAG_RANDOM_ACCESS   = 0x10000000,

    // The file or device is being opened with session awareness. If this flag is not specified, then per-session devices
    // (such as a device using RemoteFX USB Redirection) cannot be opened by processes running in session 0.
    // This flag has no effect for callers not in session 0. This flag is supported only on server editions of Windows.
    TARGET_IO_FILE_FLAG_SESSION_AWARE   = 0x00800000,

    // Access is intended to be sequential from beginning to end. The system can use this as a hint to optimize file caching.
    // This flag should not be used if read-behind (that is, reverse scans) will be used.
    // This flag has no effect if the file system does not support cached I/O and FILE_FLAG_NO_BUFFERING.
    TARGET_IO_FILE_FLAG_SEQUENTIAL_SCAN = 0x08000000,

    // Write operations will not go through any intermediate cache, they will go directly to disk.
    TARGET_IO_FILE_FLAG_WRITE_THROUGH   = 0x80000000,
};

//==============================================================================

typedef struct _FILE_INFORMATION
{
    _FILE_INFORMATION()
    {
        memset( this, 0, sizeof(_FILE_INFORMATION) );
        m_FileAttributes = TARGET_IO_FILE_ATTRIBUTE_NORMAL;
    }

    FileAttributes  m_FileAttributes;
    int32_t         m_VolumeSerialNumber;
    uint64_t        m_CreationTime;
    uint64_t        m_LastAccessTime;
    uint64_t        m_LastWriteTime;
    uint64_t        m_FileSize;
} FILE_INFORMATION;

//==============================================================================

typedef struct _DIRECTORY_ENTRY
{
    _DIRECTORY_ENTRY()
    {
        memset( this, 0, sizeof(struct _DIRECTORY_ENTRY));
        m_FileAttributes = TARGET_IO_FILE_ATTRIBUTE_NORMAL;
    }

    FileAttributes  m_FileAttributes;
    int64_t         m_CreationTime;
    int64_t         m_LastAccessTime;
    int64_t         m_LastWriteTime;
    int64_t         m_FileSize;
    char            m_FileName[ MAX_TARGET_IO_PATH ];
} DIRECTORY_ENTRY;

//==============================================================================

typedef uint64_t    TIO_HANDLE;
typedef uint64_t    TIO_SEARCH_HANDLE;

//==============================================================================

}}} // namespace
//==============================================================================

