﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "pm_monitor.h"

#if !defined(NN_TMA_SIM_INSTANCE)
#include "thread_tracker.h"
#include "tmipc/tmipc_thread.h"
#include <nn/os.h>
#include <nn/nn_SystemThreadDefinition.h>

//==============================================================================
namespace tma {
//==============================================================================

// This namespace is the entry point for handling Power Management (PM) messages.
//==============================================================================
namespace PowerManagementMonitor {
//==============================================================================

// Local data only.
namespace {
static  NN_OS_ALIGNAS_THREAD_STACK u8   s_ThreadStack[TMIPC_STACK_SIZE];
static  tmipc::Thread                   *s_pThread          = nullptr;
static  nn::os::SystemEvent             *s_pQuitEvent       = nullptr;
static  EventCallback                   s_pfEventCallback   = nullptr;
}

// --------------------------------------------------------------------------
// The monitoring thread.
s32 TmaPmModuleThread( void* lpNullptr )
{
    (void)lpNullptr;
    DEJA_TRACE( "tma::PowerManagementMonitor::TmaPmModuleThread", "TmaPmModuleThread - Start" );
TMA_POWER_TEST_PRINT( "[%s] !!! Start.\n", _BestFunctionName_ );

    // Set up the Power Service Coordinator (PSC) module.
    nn::Result          Result;
    nn::psc::PmModule   PMModule;
    // Dependent modules.
//TODO: I'm 100% guessing on this, but "it seems to be working on my machine".
    nn::psc::PmModuleId Dependencies[] =
    {
        nn::psc::PmModuleId_Pcie,
        nn::psc::PmModuleId_Usb
    };
    Result = PMModule.Initialize
        (
            nn::psc::PmModuleId_TmHostIo,   // In this context, Tm == tma (Target Manager Agent).
            Dependencies,
            sizeof(Dependencies) / sizeof(*Dependencies),
            nn::os::EventClearMode_ManualClear
        );
    NN_ABORT_UNLESS_RESULT_SUCCESS(Result);

    if( Result.IsSuccess() )
    {
        // Required for multi wait events.

        // Set up the multi wait holder, so we can detect either a PM or "quit" event.
        nn::os::MultiWaitType       MultiWait;
        nn::os::InitializeMultiWait( &MultiWait );

        // Add the "quit event" to the MultiWait event.
        nn::os::MultiWaitHolderType MultiWaitHolderTypeQuitEvent;
        nn::os::InitializeMultiWaitHolder( &MultiWaitHolderTypeQuitEvent, s_pQuitEvent->GetBase() );
        nn::os::LinkMultiWaitHolder( &MultiWait, &MultiWaitHolderTypeQuitEvent );

        // Add the "PM Module event" to the MultiWait event.
        nn::os::MultiWaitHolderType MultiWaitHolderTypePmEvent;
        nn::os::InitializeMultiWaitHolder( &MultiWaitHolderTypePmEvent, (PMModule.GetEventPointer())->GetBase());
        nn::os::LinkMultiWaitHolder( &MultiWait, &MultiWaitHolderTypePmEvent );

        // Information queried from a signalled PM event.
        nn::psc::PmFlagSet   Flags;
        nn::psc::PmState     State;

        // Start the Pm Module loop.
        bool bQuit = { false };
        while( !bQuit )
        {
            // Wait for either: PM Module event, or a quit event.
            nn::os::MultiWaitHolderType* pHolder = nn::os::WaitAny( &MultiWait );

            // Was this a PM Module event?
            if( pHolder == &MultiWaitHolderTypePmEvent )
            {
TMA_POWER_TEST_PRINT( "[%s] !!! A: Clear the event.\n", _BestFunctionName_ );
                PMModule.GetEventPointer()->Clear();

                // Retrieve the state.
                Result = PMModule.GetRequest( &State, &Flags );
TMA_POWER_TEST_PRINT( "[%s] !!! B: State: %d, Flags: 0x%08x.\n", _BestFunctionName_, State, Flags );

                // Make the callback.
                s_pfEventCallback( State, Flags );
TMA_POWER_TEST_PRINT( "[%s] !!! C: State: %d, Flags: 0x%08x.\n", _BestFunctionName_, State, Flags );

                // Acknowledge the callback.  This is mandatory for *all* events.
                Result = PMModule.Acknowledge( State, Result );
TMA_POWER_TEST_PRINT( "[%s] !!! D: State: %d, Flags: 0x%08x.\n", _BestFunctionName_, State, Flags );
                NN_ABORT_UNLESS_RESULT_SUCCESS( Result );
TMA_POWER_TEST_PRINT( "[%s] !!! E: State: %d, Flags: 0x%08x.\n", _BestFunctionName_, State, Flags );
            }
            else
            {
                // This could happen for some reason, although tma is expected
                // to be present all the time...
                DEJA_TRACE( "tma::PowerManagementMonitor::TmaPmModuleThread", "TMA signalled a quit." );
                bQuit = true;
TMA_POWER_TEST_PRINT( "[%s] !!! TMA Signalled a quit.\n", _BestFunctionName_ );
            }
        }

        // Done with the thread.

        // Finalize resources allocated in this thread.
        PMModule.Finalize();
        // Finalize the MultiWait resources.
        nn::os::UnlinkMultiWaitHolder( &MultiWaitHolderTypeQuitEvent );
        nn::os::UnlinkMultiWaitHolder( &MultiWaitHolderTypePmEvent );
        nn::os::FinalizeMultiWait( &MultiWait );
    }

TMA_POWER_TEST_PRINT( "[%s] !!! Finish.\n", _BestFunctionName_ );
    DEJA_TRACE( "tma::PowerManagementMonitor::TmaPmModuleThread", "TmaPmModuleThread - Finish" );
    return 0;
}

// --------------------------------------------------------------------------
// Starts the Power Management monitoring thread.  Messages from PSC (PM)
// will be sent to the given callback.
//  In: pfEventCallback - The callback to send nn::psc::PmState messages
//                        to, when received from PSC.
// Returns: TMIPC_RESULT_OK upon success; otherwise the error that occurred.
tmipc::Result Initialize( EventCallback pfEventCallback )
{
    DEJA_TRACE( "tma::PowerManagementMonitor::Initialize", "Initialize - Start" );
TMA_POWER_TEST_PRINT( "[%s] !!! Start.\n", _BestFunctionName_ );
    tmipc::Result Result = { tmipc::Result::TMIPC_RESULT_OK };

    // Validate the parameter.
    if( Result == tmipc::Result::TMIPC_RESULT_OK )
    {
        if( pfEventCallback == nullptr )
        {
            Result = tmipc::Result::TMIPC_RESULT_INVALID_PARAMETER;
        }
    }

    // Make sure remnants of a previous session have been cleaned up.
    if( Result == tmipc::Result::TMIPC_RESULT_OK )
    {
        if  (
                (s_pfEventCallback != nullptr)
            ||  (s_pQuitEvent != nullptr)
            )
        {
//TODO: This is technically a monitoring service...  But I can see this being confusing as a return type.
//      Maybe a better result is needed - "failed" might be too generic...
            Result = tmipc::Result::TMIPC_RESULT_DUPLICATE_SERVICE;
        }
    }

    // Create the Quit system event.
    if( Result == tmipc::Result::TMIPC_RESULT_OK )
    {
        void* pMem = s_Allocate( sizeof( nn::os::SystemEvent ) );
        s_pQuitEvent = new (pMem) nn::os::SystemEvent( nn::os::EventClearMode_AutoClear, false );
    }

    // Set the callback.
    if( Result == tmipc::Result::TMIPC_RESULT_OK )
    {
        s_pfEventCallback = pfEventCallback;
    }

    // Allocate and start the thread.
    if( Result == tmipc::Result::TMIPC_RESULT_OK )
    {
        NN_SDK_ASSERT(nn::os::GetThreadPriority( nn::os::GetCurrentThread() ) == NN_SYSTEM_THREAD_PRIORITY(tma, PmModuleThread), "Thread priority invalid\n");
        void* pMem = s_Allocate( sizeof( tmipc::Thread ) );
        s_pThread = new (pMem) tmipc::Thread;
        s_pThread->Start( TmaPmModuleThread, nullptr, s_ThreadStack, sizeof(s_ThreadStack) / sizeof(*s_ThreadStack), NN_SYSTEM_THREAD_PRIORITY(tma, PmModuleThread), NN_SYSTEM_THREAD_NAME(tma, PmModuleThread) );
        tma::ThreadTracker::RegisterThread( s_pThread->GetThreadType(), tma::ThreadTracker::ThreadId::PowerManagement );
    }

    DEJA_TRACE( "tma::PowerManagementMonitor::Initialize", "Initialize - Finish.  Result = %d", Result );
TMA_POWER_TEST_PRINT( "[%s] !!! Finish.  Result == %d.\n", _BestFunctionName_, Result );
    return Result;
}

// --------------------------------------------------------------------------
// Stops listening for PmState messages.  Cleans up and deallocates
// resources.
void Finalize()
{
TMA_POWER_TEST_PRINT( "[%s] !!! Start.\n", _BestFunctionName_ );

    // Signal the Quit event.
    if( s_pQuitEvent != nullptr )
    {
        s_pQuitEvent->Signal();
    }

    // TMA is being shut down.  This could be from the TmaPmModuleThread, but
    // if it was, it would be "after the event was processed", which means the
    // thread is already quitting.  Otherwise, this tricks the thread into
    // processing the event as if the PM gave set it.
    if( s_pThread != nullptr )
    {
        s_pThread->Join( TMIPC_INFINITE );
    }
    s_pThread->~Thread();
    s_Deallocate( s_pThread, sizeof( tmipc::Thread ) );
    s_pThread = nullptr;
    tma::ThreadTracker::UnregisterThread( tma::ThreadTracker::ThreadId::PowerManagement );

    // Deallocate the event.
    s_pQuitEvent->~SystemEvent();
    s_Deallocate( s_pQuitEvent, sizeof( nn::os::SystemEvent ) );
    s_pQuitEvent = nullptr;

    // Null out the event callback function.
    s_pfEventCallback = nullptr;
TMA_POWER_TEST_PRINT( "[%s] !!! Finish.\n", _BestFunctionName_ );
}

//==============================================================================
} // namespace PowerManagementMonitor
//==============================================================================

//==============================================================================
} // namespace tma
//==============================================================================

#endif // NN_TMA_SIM_INSTANCE
