﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "../tmipc/tmipc_defines.h"
#if ENABLE_PERFORMANCE_MONITOR_SERVICE

#include "../thread_tracker.h"
#include "../tmipc/tmipc_thread.h"
#include "../tmipc/tmipc_service.h"

//==============================================================================
namespace tma { namespace performance_monitor {
//==============================================================================

class Service : public tmipc::Service
{
private:
//    struct ServiceCounts
//    {
//        s32 m_Benchmark;                // 0xba2c1174
//        s32 m_BenchmarkReport;          // 0x378d2db4
//        s32 m_Channel;                  // 0x41d1ae3f
//        s32 m_Controller;               // 0x0955e925
//        s32 m_CoreDump;                 // 0x50bf64e6
//        s32 m_Debug;                    // 0xddde6636
//        s32 m_Diagnostics;              // 0xce69fb16
//        s32 m_DirectoryIO;              // 0xb04489f2
//        s32 m_Env;                      // 0xc191dac9
//        s32 m_HostIO;                   // 0x5fae4d7e
//        s32 m_HTCS;                     // 0xb644d830
//        s32 m_Launch;                   // 0x3ac1167b
//        s32 m_Log;                      // 0x373e20c5
//        s32 m_PowerManagement;          // 0xcc11bcfd
//        s32 m_PerformanceMonitor;       // 0xd684da51
//        s32 m_Settings;                 // 0x8a4f6b62
//        s32 m_TargetIO;                 // 0x8e69bdd7
//        s32 m_ThreadFrozen;             // 0x5898b956
//        s32 m_NodeTICSBeaconQuery;      // 0x30eb2f3a
//        s32 m_NodeTICSTMSInfo;          // 0xf8da3cd6
//        s32 m_NodeUSBBeaconQuery;       // 0x3a8ddd94
//        s32 m_NodeUSBTMSInfo;           // 0x4d4a6c24
//        s32 m_NodeUSBConnectHandshake;  // 0xcd5ed1b8
//    };
//
    struct Info
    {
        s32 m_MemoryAllocated;
        s32 m_MemoryTotal;
        s32 m_MemoryAllocationCount;
        s32 m_MemoryDeallocationCount;
        s64 m_StartTime;
        s64 m_EndTime;
        s64 m_CPUFrequencyTicks;
        s64 m_ThreadTimeTicks[static_cast<s32>(tma::ThreadTracker::ThreadId::Count)];
        s32 m_HostIOReadByteCount;
        s32 m_HostIOReadCount;
        s32 m_HostIOWriteByteCount;
        s32 m_HostIOWriteCount;
        s32 m_HTCSReceivedByteCount;
        s32 m_HTCSReceivedCount;
        s32 m_HTCSSentByteCount;
        s32 m_HTCSSentCount;
        s32 m_ReceivedByteCount;
        s32 m_ReceivedPacketCount;
        s32 m_ReceivedPacketQueueAllocated;
        s32 m_ReceivedPacketQueueTotal;
        s32 m_SentByteCount;
        s32 m_SentPacketCount;
        s32 m_SentPacketQueueAllocated;
        s32 m_SentPacketQueueTotal;
        s32 m_TaskCountCancelled;
        s32 m_TaskCountCompleted;
        s32 m_TaskCountCurrent;
        s32 m_TaskCountTotal;
        s32 m_WorkQueueCurrent;
        s32 m_WorkQueueSubmitCount;
        s32 m_WorkQueueSize;
        s32 m_WorkQueueMaxConcurrentCount;
    };

    tmipc::Mutex    m_Mutex;
    Info            m_CurrentInfo;
    Info            m_SnapShotInfo;

    // These stats *never* change.
    s32             m_HeapMemoryTotal;
    s32             m_ReceivePacketQueueTotal;
    s32             m_SendPacketQueueTotal;
    s32             m_TaskCountTotal;

virtual tmipc::Task*    OnNewTask( tmipc::Packet* pPacket );

public:
    Service   ();
    ~Service  ();

    void    Init        ();
    void    Kill        ();

    // Records the accumulated data.
    // This needs to be called before querying the performance monitor information, or
    // else the data will not be updated.
    void    RecordSnapShot();

    // -------------------------------------------------
    // DO NOT USE THESE.  USE THE #define VERSIONS BELOW.
    // The #define version will handle the #if defined(ENABLE_PERFORMANCE_MONITOR_SERVICE) for you.
    // -------------------------------------------------

    // Accumulate data.
    void    AddMemoryAllocationCountIncrement       ();                                 // Increments m_MemoryAllocationCount by 1.
    void    AddMemoryDeallocationCountIncrement     ();                                 // Increments m_MemoryDeallocationCount by 1.
    void    AddReceivedByteIncrement                ( s32 Increment );                  // Adds Increment to m_ReceivedByteCount.
    void    AddReceivedPacket                       ( const tmipc::Packet& Packet );    // Adds 1 to m_ReceivedPacketCount and updates m_ReceivedByteCount by: Packet::GetDataLen() + sizeof(Packet::Header).
    void    AddReceivedPacketCountIncrement         ( s32 Increment = 1 );              // Adds Increment to m_ReceivedPacketCount.
    void    AddReceivedPacketQueueIncrement         ( s32 Increment = 1 );              // Adds Increment to m_ReceivedPacketQueueAllocated.
    void    AddSentByteIncrement                    ( s32 Increment );                  // Adds Increment to m_SentByteCount.
    void    AddSentPacket                           ( const tmipc::Packet& Packet );    // Adds 1 to m_SentPacketCount and updates m_SentByteCount by: Packet::GetSendLen().
    void    AddSentPacketCountIncrement             ( s32 Increment = 1 );              // Adds Increment to m_SentPacketCount.
    void    AddTaskCountCancelledIncrement          ( s32 Increment = 1 );              // Adds Increment to m_TaskCountCancelled.
    void    AddTaskCountCompletedIncrement          ( s32 Increment = 1 );              // Adds Increment to m_TaskCountCompleted.
    void    AddWorkQueueSubmitCountIncrement        ( s32 CurrentCount );               // Increments m_WorkQueueSubmitCount by 1.  CurrentSize: The current number of Tasks in the Work Queue.


    // Query accumulated data.  Note: These values only change after each call to RecordSnapShot().
    s64     GetStartTime() const                        { return m_SnapShotInfo.m_StartTime;                    }
    s64     GetEndTime() const                          { return m_SnapShotInfo.m_EndTime;                      }
    u64     GetCPUFrequencyTicks() const                { return m_SnapShotInfo.m_CPUFrequencyTicks;            }
    u64     GetThreadTicksHio() const                   { return m_SnapShotInfo.m_ThreadTimeTicks[static_cast<s32>(tma::ThreadTracker::ThreadId::HioServer)];          }
    u64     GetThreadTicksHtc() const                   { return m_SnapShotInfo.m_ThreadTimeTicks[static_cast<s32>(tma::ThreadTracker::ThreadId::HtcServer)];          }
    u64     GetThreadTicksHtcs0() const                 { return m_SnapShotInfo.m_ThreadTimeTicks[static_cast<s32>(tma::ThreadTracker::ThreadId::HtcsServer0)];        }
    u64     GetThreadTicksHtcs1() const                 { return m_SnapShotInfo.m_ThreadTimeTicks[static_cast<s32>(tma::ThreadTracker::ThreadId::HtcsServer1)];        }
    u64     GetThreadTicksHtcs2() const                 { return m_SnapShotInfo.m_ThreadTimeTicks[static_cast<s32>(tma::ThreadTracker::ThreadId::HtcsServer2)];        }
    u64     GetThreadTicksHtcs3() const                 { return m_SnapShotInfo.m_ThreadTimeTicks[static_cast<s32>(tma::ThreadTracker::ThreadId::HtcsServer3)];        }
    u64     GetThreadTicksHtcs4() const                 { return m_SnapShotInfo.m_ThreadTimeTicks[static_cast<s32>(tma::ThreadTracker::ThreadId::HtcsServer4)];        }
    u64     GetThreadTicksHtcs5() const                 { return m_SnapShotInfo.m_ThreadTimeTicks[static_cast<s32>(tma::ThreadTracker::ThreadId::HtcsServer5)];        }
    u64     GetThreadTicksHtcs6() const                 { return m_SnapShotInfo.m_ThreadTimeTicks[static_cast<s32>(tma::ThreadTracker::ThreadId::HtcsServer6)];        }
    u64     GetThreadTicksHtcs7() const                 { return m_SnapShotInfo.m_ThreadTimeTicks[static_cast<s32>(tma::ThreadTracker::ThreadId::HtcsServer7)];        }
    u64     GetThreadTicksMain() const                  { return m_SnapShotInfo.m_ThreadTimeTicks[static_cast<s32>(tma::ThreadTracker::ThreadId::Main)];               }
    u64     GetThreadTicksPowerManagement() const       { return m_SnapShotInfo.m_ThreadTimeTicks[static_cast<s32>(tma::ThreadTracker::ThreadId::PowerManagement)];    }
    u64     GetThreadTicksTmipcListen() const           { return m_SnapShotInfo.m_ThreadTimeTicks[static_cast<s32>(tma::ThreadTracker::ThreadId::TmipcListen)];        }
    u64     GetThreadTicksTmipcReceive() const          { return m_SnapShotInfo.m_ThreadTimeTicks[static_cast<s32>(tma::ThreadTracker::ThreadId::TmipcReceive)];       }
    u64     GetThreadTicksTmipcSend() const             { return m_SnapShotInfo.m_ThreadTimeTicks[static_cast<s32>(tma::ThreadTracker::ThreadId::TmipcSend)];          }
    u64     GetThreadTicksTmipcSession() const          { return m_SnapShotInfo.m_ThreadTimeTicks[static_cast<s32>(tma::ThreadTracker::ThreadId::TmipcSession)];       }
    u64     GetThreadTicksTmipcServicesManager() const  { return m_SnapShotInfo.m_ThreadTimeTicks[static_cast<s32>(tma::ThreadTracker::ThreadId::ServicesManager)];    }
    s32     GetHostIOReadByteCount() const              { return m_SnapShotInfo.m_HostIOReadByteCount;          }
    s32     GetHostIOReadCount() const                  { return m_SnapShotInfo.m_HostIOReadCount;              }
    s32     GetHostIOWriteByteCount() const             { return m_SnapShotInfo.m_HostIOWriteByteCount;         }
    s32     GetHostIOWriteCount() const                 { return m_SnapShotInfo.m_HostIOWriteCount;             }

    s32     GetHTCSReceivedByteCount() const            { return m_SnapShotInfo.m_HTCSReceivedByteCount;        }
    s32     GetHTCSReceivedCount() const                { return m_SnapShotInfo.m_HTCSReceivedCount;            }
    s32     GetHTCSSentByteCount() const                { return m_SnapShotInfo.m_HTCSSentByteCount;            }
    s32     GetHTCSSentCount() const                    { return m_SnapShotInfo.m_HTCSSentCount;                }

    s32     GetReceivedByteCount() const                { return m_SnapShotInfo.m_ReceivedByteCount;            }
    s32     GetReceivedPacketCount() const              { return m_SnapShotInfo.m_ReceivedPacketCount;          }
    s32     GetReceivedPacketQueueAllocated() const     { return m_SnapShotInfo.m_ReceivedPacketQueueAllocated; }
    s32     GetReceivedPacketQueueTotal() const         { return m_SnapShotInfo.m_ReceivedPacketQueueTotal;     }
    s32     GetMemoryAllocated() const                  { return m_SnapShotInfo.m_MemoryAllocated;              }
    s32     GetMemoryTotal() const                      { return m_SnapShotInfo.m_MemoryTotal;                  }
    s32     GetMemoryAllocationCount() const            { return m_SnapShotInfo.m_MemoryAllocationCount;        }
    s32     GetMemoryDeallocationCount() const          { return m_SnapShotInfo.m_MemoryDeallocationCount;      }
    s32     GetSentByteCount() const                    { return m_SnapShotInfo.m_SentByteCount;                }
    s32     GetSentPacketCount() const                  { return m_SnapShotInfo.m_SentPacketCount;              }
    s32     GetSentPacketQueueAllocated() const         { return m_SnapShotInfo.m_SentPacketQueueAllocated;     }
    s32     GetSentPacketQueueTotal() const             { return m_SnapShotInfo.m_SentPacketQueueTotal;         }
    s32     GetTaskCountCancelled() const               { return m_SnapShotInfo.m_TaskCountCancelled;           }
    s32     GetTaskCountCompleted() const               { return m_SnapShotInfo.m_TaskCountCompleted;           }
    s32     GetTaskCountCurrent() const                 { return m_SnapShotInfo.m_TaskCountCurrent;             }
    s32     GetTaskCountTotal() const                   { return m_SnapShotInfo.m_TaskCountTotal;               }
    s32     GetWorkQueueCurrent() const                 { return m_SnapShotInfo.m_WorkQueueCurrent;             }
    s32     GetWorkQueueSubmitCount() const             { return m_SnapShotInfo.m_WorkQueueSubmitCount;         }
    s32     GetWorkQueueSize() const                    { return m_SnapShotInfo.m_WorkQueueSize;                }
    s32     GetWorkQueueMaxConcurrentCount() const      { return m_SnapShotInfo.m_WorkQueueMaxConcurrentCount;  }

    // -------------------------------------------------
};

bool     IsValid( void );
Service& GetService(void);

#define ADD_MEMORY_ALLOCATION_COUNT_INCREMENT( )            { if( tma::performance_monitor::IsValid() ) { tma::performance_monitor::GetService().AddMemoryAllocationCountIncrement( ); } }
#define ADD_MEMORY_DEALLOCATION_COUNT_INCREMENT( )          { if( tma::performance_monitor::IsValid() ) { tma::performance_monitor::GetService().AddMemoryDeallocationCountIncrement( ); } }
#define ADD_RECEIVED_BYTE_INCREMENT( Increment )            tma::performance_monitor::GetService().AddReceivedByteIncrement( (Increment) )
#define ADD_RECEIVED_PACKET_COUNT_INCREMENT( Increment )    tma::performance_monitor::GetService().AddReceivedPacketCountIncrement( (Increment) )
#define ADD_SENT_BYTE_INCREMENT( Increment )                tma::performance_monitor::GetService().AddSentByteIncrement( (Increment) )
#define ADD_SENT_PACKET_COUNT_INCREMENT( Increment )        tma::performance_monitor::GetService().AddSentPacketIncrement( (Increment) )
#define ADD_TASK_COUNT_CANCELLED_INCREMENT( Increment )     tma::performance_monitor::GetService().AddTaskCountCancelledIncrement( (Increment) )
#define ADD_TASK_COUNT_COMPLETED_INCREMENT( Increment )     tma::performance_monitor::GetService().AddTaskCountCompletedIncrement( (Increment) )
#define ADD_RECEIVED_PACKET( Packet )                       tma::performance_monitor::GetService().AddReceivedPacket( Packet )
#define ADD_SENT_PACKET( Packet )                           tma::performance_monitor::GetService().AddSentPacket( Packet )
#define ADD_WORK_QUEUE_SUBMIT_INCREMENT( ConcurrentCount )  tma::performance_monitor::GetService().AddWorkQueueSubmitCountIncrement( (ConcurrentCount) )

//==============================================================================
}}
//==============================================================================

#else // ENABLE_PERFORMANCE_MONITOR_SERVICE

#define ADD_MEMORY_ALLOCATION_COUNT_INCREMENT( )
#define ADD_MEMORY_DEALLOCATION_COUNT_INCREMENT( )
#define ADD_RECEIVED_BYTE_INCREMENT( Increment )
#define ADD_RECEIVED_PACKET_COUNT_INCREMENT( Increment )
#define ADD_SENT_BYTE_INCREMENT( Increment )
#define ADD_SENT_PACKET_COUNT_INCREMENT( Increment )
#define ADD_TASK_COUNT_CANCELLED_INCREMENT( Increment )
#define ADD_TASK_COUNT_COMPLETED_INCREMENT( Increment )
#define ADD_RECEIVED_PACKET( Packet )
#define ADD_SENT_PACKET( Packet )
#define ADD_WORK_QUEUE_SUBMIT_INCREMENT( ConcurrentCount )

#endif // ENABLE_PERFORMANCE_MONITOR_SERVICE
