﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "performance_monitor_service.h"
#if ENABLE_PERFORMANCE_MONITOR_SERVICE
#include "performance_monitor_task.h"
#include "../thread_tracker.h"
#include "../tmipc/tmipc_service_ids.h"
#include <nn/svc/svc_Types.common.h>

// --------------------------------------------------------------------------
namespace tma { namespace performance_monitor {
// --------------------------------------------------------------------------

Service::Service()
{
//NN_SDK_LOG( "[performance_monitor::Service::Service::Service( %p )]\n", this );
    m_ServiceId = HashString( "PerformanceMonitor::Service" );
//NN_SDK_LOG( "[performance_monitor::Service::Service::Service( %p )] - HashString('Service') == 0x%08x\n", this, GetId() );

    // Create the mutex.
    m_Mutex.Create();
    memset( &m_CurrentInfo, 0, sizeof(m_CurrentInfo) );
    memset( &m_SnapShotInfo, 0, sizeof(m_SnapShotInfo) );
    // Initialize the timing.
    m_CurrentInfo.m_StartTime   = nn::os::GetSystemTick().GetInt64Value();
    m_CurrentInfo.m_EndTime     = m_CurrentInfo.m_StartTime + 1;
    m_SnapShotInfo.m_StartTime  = m_CurrentInfo.m_StartTime;
    m_SnapShotInfo.m_EndTime    = m_CurrentInfo.m_StartTime + 1;

    // Initialize "stats" that *never* change.
    m_HeapMemoryTotal           = tma::GetHeapRAMTotal();
    m_ReceivePacketQueueTotal   = tmipc::TMIPC_NUM_RECV_PACKETS;
    m_SendPacketQueueTotal      = tmipc::TMIPC_NUM_SEND_PACKETS;
    m_TaskCountTotal            = tmipc::TMIPC_WORK_QUEUE_DEPTH;
}

// --------------------------------------------------------------------------

Service::~Service()
{
//NN_SDK_LOG( "[performance_monitor::Service::~Service( %p )]\n", this );
    // Destroy the mutex.
    m_Mutex.Destroy();
}

// --------------------------------------------------------------------------

void Service::Init()
{
//NN_SDK_LOG( "[performance_monitor::Service::Init( %p )]\n", this );
    Create();
}

// --------------------------------------------------------------------------

void Service::Kill()
{
//NN_SDK_LOG( "[performance_monitor::Service::Kill( %p )]\n", this );
    Destroy();
}

// --------------------------------------------------------------------------

tmipc::Task* Service::OnNewTask( tmipc::Packet* pPacket )
{
//NN_SDK_LOG( "[performance_monitor::Service::OnNewTask( 0x%p )] - Packet: 0x%p\n", this, pPacket );
    void* pMem = s_Allocate( sizeof( Task ) );
    Task* pTask = new (pMem) Task;
    pTask->SetServicesManager( m_pServicesManager );
    pTask->SetServiceId( m_ServiceId );
    pTask->SetTaskId( pPacket->GetTaskId() );
    pTask->OnInitiate( pPacket );
    return( pTask );
}

// --------------------------------------------------------------------------
// Records the accumulated data.
// This needs to be called before querying the performance_monitor information, or
// else the data will not be updated.
void Service::RecordSnapShot()
{
//NN_SDK_LOG( "[performance_monitor::Service::RecordSnapShot( 0x%p )].\n", this );
    const tmipc::ScopedLock kLock{ m_Mutex };
    memcpy( &m_SnapShotInfo, &m_CurrentInfo, sizeof(m_SnapShotInfo) );
    m_SnapShotInfo.m_MemoryTotal                = tma::GetHeapRAMTotal();
    m_SnapShotInfo.m_MemoryAllocated            = tma::GetHeapRAMAllocated();
    m_pServicesManager->GetCurrentTaskCount     ( m_SnapShotInfo.m_TaskCountCurrent );
    m_SnapShotInfo.m_TaskCountTotal             = m_SnapShotInfo.m_TaskCountCancelled + m_SnapShotInfo.m_TaskCountCompleted + m_SnapShotInfo.m_TaskCountCurrent;
    m_SnapShotInfo.m_ReceivedPacketQueueTotal   = m_ReceivePacketQueueTotal;
    m_SnapShotInfo.m_SentPacketQueueTotal       = m_SendPacketQueueTotal;
    m_pServicesManager->GetAllocatedPacketCounts( m_SnapShotInfo.m_SentPacketQueueAllocated, m_SnapShotInfo.m_ReceivedPacketQueueAllocated );
    m_pServicesManager->GetCurrentWorkQueueCount( m_SnapShotInfo.m_WorkQueueCurrent );
    m_SnapShotInfo.m_WorkQueueSize              = tmipc::TMIPC_WORK_QUEUE_DEPTH;

    m_SnapShotInfo.m_EndTime                    = nn::os::GetSystemTick().GetInt64Value();
    m_SnapShotInfo.m_CPUFrequencyTicks          = static_cast<u64>(nn::os::GetSystemTickFrequency());

    // Thread time.
    // Note: Start with index 1.  Using Index 0 will be treated as the "current thread", i.e. the Service Manager thred.
    //       (I am not certain why this happens, because the "if test in the loop" is checking for a handle = 0, which means someone is setting [0] to something not-0.)
//TODO: I need to know how/why this is occurring.
    //       The thread CPU function resets the "start time" per call, so calling it twice (i.e. for "current thread" *and* nn.tma.ServicesMgr)
    //       would require the
    //       While the value will work it will modify the time read
    //       because
//    for( s32 Index = 0; Index < static_cast<s32>(tma::ThreadTracker::ThreadId::Count); Index++ )
for( s32 Index = 1; Index < static_cast<s32>(tma::ThreadTracker::ThreadId::Count); Index++ )
    {
        s64 ThreadTime{ 0 };
        const nn::os::ThreadType& ThreadType{ tma::ThreadTracker::GetThreadType(static_cast<tma::ThreadTracker::ThreadId>(Index)) };
        if( ThreadType._handle != 0 )
        {
            nn::Bit64 Info;
            nn::Bit64 CPUCoreNumber{ static_cast<nn::Bit64>(-1ll) };  // -1 == All CPU cores.
            nn::svc::Handle Handle{ ThreadType._handle };
            if( Handle.IsValid() )
            {
                nn::Result Result{ nn::svc::GetInfo( &Info, nn::svc::InfoType_ThreadTickCount, Handle, CPUCoreNumber ) };
                if( Result.IsSuccess() )
                {
                    ThreadTime = static_cast<s64>(Info);
//{
//const nn::os::ThreadId kThreadId{ nn::os::GetThreadId( &ThreadType ) };
//const char *pThreadName{ nn::os::GetThreadNamePointer( &ThreadType ) };
//const char *pSafeThreadName{ (pThreadName != nullptr) ? pThreadName : "<nullptr>" };
//const uint32_t kHandle{ Handle.GetPrintableBits() };
//NN_SDK_LOG( "Successfully retrieved CPU information['%d']: Handle: %u.  Id: %llu.  Name: '%s'.  Info: %lld.  ThreadTime: %lld.\n", Index, Handle, kThreadId, pSafeThreadName, Info, ThreadTime );
//}
                }
//else
//{
//    bool bInvalidHandle{ Result.CanAccept( nn::svc::ResultInvalidHandle() ) };
//    bool bInvalidCombination{ Result.CanAccept( nn::svc::ResultInvalidCombination() ) };
//    NN_SDK_LOG( "Failed: nn::svc::GetInfo(..., '%u', '%d').  Valid handle: '%s'.  Err1: '%s'.  Err2: '%s'. Result: %d.\n", Handle.GetPrintableBits(), CPUCoreNumber, Handle.IsValid() ? "Yes" : "No", bInvalidHandle ? "Yes" : "No", bInvalidCombination ? "Yes" : "No", Result.GetInnerValueForDebug() );
//}
            }
else
{
    ThreadTime = -1;
}
        }
        m_SnapShotInfo.m_ThreadTimeTicks[Index] = static_cast<u64>(ThreadTime);
    }

    // Reset some of the stats for the next iteration...
    // NOTE: 1) The *only* stats that should be reset to 0 here are:
    //          a) Stats that only increment               (example: do *not* have a decrement pairing, such as m_TaskCountTotal).
    //          b) Stats that are polled per snapshot      (example: m_MemoryTotal).
    //       2) Stats that should *never* be reset to 0 are:
    //          a) m_Start/m_End times.
    //          b) [Currently nothing else so far.  Be aware of this though.]
    m_CurrentInfo.m_MemoryAllocated                 = 0;
    m_CurrentInfo.m_MemoryTotal                     = 0;
    m_CurrentInfo.m_MemoryAllocationCount           = 0;
    m_CurrentInfo.m_MemoryDeallocationCount         = 0;
    m_CurrentInfo.m_StartTime                       = m_SnapShotInfo.m_EndTime;         // These values need to be updated to "now".
    m_CurrentInfo.m_EndTime                         = m_CurrentInfo.m_StartTime + 1;    //
    m_CurrentInfo.m_CPUFrequencyTicks               = 0;
    for( s32 Index = 0; Index < (sizeof(m_CurrentInfo.m_ThreadTimeTicks) / sizeof(m_CurrentInfo.m_ThreadTimeTicks[0])); Index++ )
    {
        m_CurrentInfo.m_ThreadTimeTicks[Index]      = 0;
    }
    m_CurrentInfo.m_HostIOReadByteCount             = 0;
    m_CurrentInfo.m_HostIOReadCount                 = 0;
    m_CurrentInfo.m_HostIOWriteByteCount            = 0;
    m_CurrentInfo.m_HostIOWriteCount                = 0;
    m_CurrentInfo.m_HTCSReceivedByteCount           = 0;
    m_CurrentInfo.m_HTCSReceivedCount               = 0;
    m_CurrentInfo.m_HTCSSentByteCount               = 0;
    m_CurrentInfo.m_HTCSSentCount                   = 0;
    m_CurrentInfo.m_ReceivedByteCount               = 0;
    m_CurrentInfo.m_ReceivedPacketCount             = 0;
    m_CurrentInfo.m_ReceivedPacketQueueAllocated    = 0;
    m_CurrentInfo.m_ReceivedPacketQueueTotal        = 0;
    m_CurrentInfo.m_SentByteCount                   = 0;
    m_CurrentInfo.m_SentPacketCount                 = 0;
    m_CurrentInfo.m_SentPacketQueueAllocated        = 0;
    m_CurrentInfo.m_SentPacketQueueTotal            = 0;
    m_CurrentInfo.m_TaskCountCancelled              = 0;
    m_CurrentInfo.m_TaskCountCompleted              = 0;
    m_CurrentInfo.m_TaskCountCurrent                = 0;
    m_CurrentInfo.m_TaskCountTotal                  = 0;
    m_CurrentInfo.m_WorkQueueCurrent                = 0;
    m_CurrentInfo.m_WorkQueueSubmitCount            = 0;
    m_CurrentInfo.m_WorkQueueSize                   = 0;
    m_CurrentInfo.m_WorkQueueMaxConcurrentCount     = 0;
}

// --------------------------------------------------------------------------

void Service::AddMemoryAllocationCountIncrement()
{
    const tmipc::ScopedLock kLock{ m_Mutex };
    m_CurrentInfo.m_MemoryAllocationCount++;
}

// --------------------------------------------------------------------------

void Service::AddMemoryDeallocationCountIncrement()
{
    const tmipc::ScopedLock kLock{ m_Mutex };
    m_CurrentInfo.m_MemoryDeallocationCount++;
}

// --------------------------------------------------------------------------

void Service::AddReceivedByteIncrement( s32 Increment )
{
    const tmipc::ScopedLock kLock{ m_Mutex };
    m_CurrentInfo.m_ReceivedByteCount += Increment;
}

// --------------------------------------------------------------------------

void Service::AddReceivedPacketCountIncrement( s32 Increment )
{
    const tmipc::ScopedLock kLock{ m_Mutex };
    m_CurrentInfo.m_ReceivedPacketCount += Increment;
}

// --------------------------------------------------------------------------

void Service::AddSentByteIncrement( s32 Increment )
{
    const tmipc::ScopedLock kLock{ m_Mutex };
    m_CurrentInfo.m_SentByteCount += Increment;
}

// --------------------------------------------------------------------------

void Service::AddSentPacketCountIncrement( s32 Increment )
{
    const tmipc::ScopedLock kLock{ m_Mutex };
    m_CurrentInfo.m_SentPacketCount += Increment;
}

// --------------------------------------------------------------------------

void Service::AddTaskCountCancelledIncrement( s32 Increment )
{
    const tmipc::ScopedLock kLock{ m_Mutex };
    m_CurrentInfo.m_TaskCountCancelled += Increment;
}

// --------------------------------------------------------------------------

void Service::AddTaskCountCompletedIncrement( s32 Increment )
{
    const tmipc::ScopedLock kLock{ m_Mutex };
    m_CurrentInfo.m_TaskCountCompleted += Increment;
}

// --------------------------------------------------------------------------

void Service::AddWorkQueueSubmitCountIncrement( s32 CurrentCount )
{
    const tmipc::ScopedLock kLock{ m_Mutex };
    m_CurrentInfo.m_WorkQueueSubmitCount++;
//NN_SDK_LOG( "Service::AddWorkQueueSubmitCountIncrement( %d ).\n", CurrentCount );
    if( m_CurrentInfo.m_WorkQueueMaxConcurrentCount < CurrentCount )
    {
        m_CurrentInfo.m_WorkQueueMaxConcurrentCount = CurrentCount;
    }
}

// --------------------------------------------------------------------------
// Slightly optimized versions, to reduce the number of times the m_Mutex needs to be locked.

// --------------------------------------------------------------------------

void Service::AddReceivedPacket( const tmipc::Packet& Packet )
{
    const tmipc::ScopedLock kLock{ m_Mutex };
    m_CurrentInfo.m_ReceivedByteCount  += static_cast<s32>(sizeof(tmipc::Packet::Header)) + static_cast<s32>(Packet.GetDataLen());
    m_CurrentInfo.m_ReceivedPacketCount++;

    // Extract additional information from the Packet.
    const u32 kServiceId{ Packet.GetServiceId() };
    switch( kServiceId )
    {
//    case tmipc::ServiceId_BenchmarkService:
//        break;
//    case tmipc::ServiceId_BenchmarkReportService:
//        break;
//    case tmipc::ServiceId_ChannelService:
//        break;
//    case tmipc::ServiceId_ControllerService:
//        break;
//    case tmipc::ServiceId_CoreDumpService:
//        break;
//    case tmipc::ServiceId_DebugService:
//        break;
//    case tmipc::ServiceId_EnvService:
//        break;
//    case tmipc::ServiceId_HostDirectoryIOService:
//        break;
    case tmipc::ServiceId_HostIOService:
        m_CurrentInfo.m_HostIOReadByteCount += static_cast<s32>(sizeof(tmipc::Packet::Header)) + static_cast<s32>(Packet.GetDataLen());
        m_CurrentInfo.m_HostIOReadCount++;
        break;
    case tmipc::ServiceId_HTCSService:
        m_CurrentInfo.m_HTCSReceivedByteCount += static_cast<s32>(sizeof(tmipc::Packet::Header)) + static_cast<s32>(Packet.GetDataLen());
        m_CurrentInfo.m_HTCSReceivedCount++;
        break;
//    case tmipc::ServiceId_LaunchService:
//        break;
//    case tmipc::ServiceId_LogService:
//        break;
//    case tmipc::ServiceId_PowerManagementService:
//        break;
//    case tmipc::ServiceId_PerformanceMonitorService:
//        break;
//    case tmipc::ServiceId_SettingsService:
//        break;
//    case tmipc::ServiceId_TargetIOService:
//        break;
//    case tmipc::ServiceId_ThreadFrozenService:
//        break;
//    case tmipc::ServiceId_NodeTICS_BeaconQuery:
//        break;
//    case tmipc::ServiceId_NodeTICS_TMSInfo:
//        break;
//    case tmipc::ServiceId_NodeUSB_BeaconQuery:
//        break;
//    case tmipc::ServiceId_NodeUSB_ConnectHandshake:
//        break;
//    case tmipc::ServiceId_NodeUSB_Disconnect:
//        break;
//    case tmipc::ServiceId_NodeUSB_TMSInfo:
//        break;
    default:
        break;
    }
}

// --------------------------------------------------------------------------

void Service::AddSentPacket( const tmipc::Packet& Packet )
{
    const tmipc::ScopedLock kLock{ m_Mutex };
    m_CurrentInfo.m_SentByteCount  += Packet.GetSendLen();
    m_CurrentInfo.m_SentPacketCount++;

    // Extract additional information from the Packet.
    const u32 kServiceId{ Packet.GetServiceId() };
    switch( kServiceId )
    {
//    case tmipc::ServiceId_BenchmarkService:
//        break;
//    case tmipc::ServiceId_BenchmarkReportService:
//        break;
//    case tmipc::ServiceId_ChannelService:
//        break;
//    case tmipc::ServiceId_ControllerService:
//        break;
//    case tmipc::ServiceId_CoreDumpService:
//        break;
//    case tmipc::ServiceId_DebugService:
//        break;
//    case tmipc::ServiceId_EnvService:
//        break;
//    case tmipc::ServiceId_HostDirectoryIOService:
//        break;
    case tmipc::ServiceId_HostIOService:
        m_CurrentInfo.m_HostIOWriteByteCount += static_cast<s32>(sizeof(tmipc::Packet::Header)) + static_cast<s32>(Packet.GetDataLen());
        m_CurrentInfo.m_HostIOWriteCount++;
        break;
    case tmipc::ServiceId_HTCSService:
        m_CurrentInfo.m_HTCSSentByteCount += static_cast<s32>(sizeof(tmipc::Packet::Header)) + static_cast<s32>(Packet.GetDataLen());
        m_CurrentInfo.m_HTCSSentCount++;
        break;
//    case tmipc::ServiceId_LaunchService:
//        break;
//    case tmipc::ServiceId_LogService:
//        break;
//    case tmipc::ServiceId_PowerManagementService:
//        break;
//    case tmipc::ServiceId_PerformanceMonitorService:
//        break;
//    case tmipc::ServiceId_SettingsService:
//        break;
//    case tmipc::ServiceId_TargetIOService:
//        break;
//    case tmipc::ServiceId_ThreadFrozenService:
//        break;
//    case tmipc::ServiceId_NodeTICS_BeaconQuery:
//        break;
//    case tmipc::ServiceId_NodeTICS_TMSInfo:
//        break;
//    case tmipc::ServiceId_NodeUSB_BeaconQuery:
//        break;
//    case tmipc::ServiceId_NodeUSB_ConnectHandshake:
//        break;
//    case tmipc::ServiceId_NodeUSB_Disconnect:
//        break;
//    case tmipc::ServiceId_NodeUSB_TMSInfo:
//        break;
    default:
        break;
    }
}

// --------------------------------------------------------------------------

}} // namespace tma::profiling.
// --------------------------------------------------------------------------

#endif // ENABLE_PERFORMANCE_MONITOR_SERVICE
