﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "..\tmipc\tmipc_packet.h"
#include "..\tmipc\tmipc_task.h"
#include <nn/htcs/htcs_Types.h>
#include <nn/sf/sf_NativeHandle.h>
#include "..\tma\tma_Mutex.h"
#include <nn/os/os_TransferMemory.h>

//==============================================================================
namespace tma { namespace htcs {
//==============================================================================

nn::htcs::SocketError ConvertToHtcsSocketError( int WinsockErrorCode );

class HTCSTask : public tmipc::Task
{
public:
    explicit        HTCSTask    ( int               Socket );
                   ~HTCSTask    ();

    //================================================================
    // Result API

    int             GetSocket           ( void );
    int32_t         GetResult           ( void );
    int32_t         GetErrorCode        ( void );

virtual void        OnInitiate          ( tmipc::Packet*    pPacket );
virtual void        OnRecvPacket        ( tmipc::Packet*    pPacket );
virtual void        OnSendPacket        ( tmipc::Packet*    pPacket );

protected:
    int             m_Socket;
    int32_t         m_Result;
    int32_t         m_ErrorCode;

    //==============================================================================
    // Static functionality

    static struct task_system_event*    s_pEvents;
    static tma::Mutex                   s_EventMutex; // Mutex to protect event list.

    static nn::os::SystemEventType* AllocEvent  ( void );
    static void                     FreeEvent   ( nn::os::SystemEventType* pEvent );

public:
    static void                     OnServiceKilled ( );

};

//==============================================================================

class HTCSSignalingTask : public HTCSTask
{
public:
    explicit        HTCSSignalingTask   ( int               Socket );
                   ~HTCSSignalingTask   ();

protected:
    nn::os::SystemEventType m_SignalEvent;
};

//==============================================================================

class HTCSSocketTask : public HTCSTask
{
public:

    explicit        HTCSSocketTask      ( int               Socket );
                   ~HTCSSocketTask      ();

virtual void        OnRecvPacket        ( tmipc::Packet*    pPacket );

    void            Socket              ( void );
};

//==============================================================================

class HTCSCloseTask : public HTCSTask
{
public:
    explicit        HTCSCloseTask       ( int               Socket );
                   ~HTCSCloseTask       ();

virtual void        OnRecvPacket        ( tmipc::Packet*    pPacket );

    void            Close               ( void );
};

//==============================================================================

class HTCSBindTask : public HTCSTask
{
public:
    explicit        HTCSBindTask        ( int               Socket );
                   ~HTCSBindTask        ();

virtual void        OnRecvPacket        ( tmipc::Packet*    pPacket );

    void            Bind                ( const nn::htcs::SockAddrHtcs* pAddr );
};

//==============================================================================

class HTCSListenTask : public HTCSTask
{
public:
    explicit        HTCSListenTask      ( int               Socket );
                   ~HTCSListenTask      ();

virtual void        OnRecvPacket        ( tmipc::Packet*    pPacket );

    void            Listen              ( int               backlogCount );
};

//==============================================================================

class HTCSShutdownTask : public HTCSTask
{
public:
    explicit        HTCSShutdownTask    ( int               Socket );
                   ~HTCSShutdownTask    ();

virtual void        OnRecvPacket        ( tmipc::Packet*    pPacket );

    void            Shutdown            ( int               how );
};

//==============================================================================

class HTCSFcntlTask : public HTCSTask
{
public:
    explicit        HTCSFcntlTask       ( int               Socket );
                   ~HTCSFcntlTask       ();

virtual void        OnRecvPacket        ( tmipc::Packet*    pPacket );

    void            Fcntl               ( int               command,
                                          int               value );
};

//==============================================================================

class HTCSConnectTask : public HTCSTask
{
public:
    explicit        HTCSConnectTask     ( int                 Socket );
                   ~HTCSConnectTask     ();

virtual void        OnRecvPacket        ( tmipc::Packet*    pPacket );

    void            Connect             ( const nn::htcs::SockAddrHtcs* pAddr );

};

//==============================================================================

class HTCSAcceptTask : public HTCSSignalingTask
{
public:
    explicit        HTCSAcceptTask      ( int                 Socket );
                   ~HTCSAcceptTask      ();

virtual void        OnRecvPacket        ( tmipc::Packet*    pPacket );

    void            Start               ( nn::os::NativeHandle*     pWaitHandle );
    int             GetResults          ( nn::htcs::SockAddrHtcs*   pAddr,
                                          int*                      pError );

private:
    nn::htcs::SockAddrHtcs  m_AcceptedAddr;
};

//==============================================================================

class HTCSRecvTask : public HTCSSignalingTask
{
public:
    explicit        HTCSRecvTask        ( int               Socket );
                   ~HTCSRecvTask        ();

virtual void        OnRecvPacket        ( tmipc::Packet*    pPacket );

    void            Start               ( int64_t               bufferByteSize,
                                          int                   flags,
                                          nn::os::NativeHandle* pWaitHandle );

    int             GetResults          ( void**            pBuffer,
                                          int64_t*          pAmountRead,
                                          int*              pError );

private:
    void*                   m_Buffer;
    int64_t                 m_AmountRead;
};

//==============================================================================

class HTCSRecvLargeTask : public HTCSSignalingTask
{
public:
    explicit        HTCSRecvLargeTask   ( int               Socket );
                   ~HTCSRecvLargeTask   ();

virtual void        OnRecvPacket        ( tmipc::Packet*    pPacket );

    void            Start               ( int32_t           unalignedStartSize,
                                          int32_t           unalignedEndSize,
                                          int64_t           alignedSendSize,
                                          nn::os::NativeHandle alignedMemoryHandle,
                                          int               flags,
                                          nn::os::NativeHandle* pWaitHandle );

    int             GetResults          ( void**            pBuffer,
                                          int32_t*          pUnalignedStartSize,
                                          int32_t*          pUnalignedEndSize,
                                          int64_t*          pAmountRead,
                                          int*              pError );

private:
    nn::os::TransferMemoryType  m_Dest;
    bool    m_MemoryMapped;
    int     m_Flags;
    int64_t m_AmountRead;
    int64_t m_AlignedLeftToRecv;
    char*   m_pAlignedBuffer;
    int32_t m_PreLeftToRead;
    int32_t m_PostLeftToRead;
    int32_t m_UnalignedStartSize;
    int32_t m_UnalignedEndSize;
    int32_t m_UnalignedRead;
    int32_t m_UnalignedDataSize;
    char*   m_pUnalignedData;
};

//==============================================================================

class HTCSSendTask : public HTCSSignalingTask
{
public:
    explicit        HTCSSendTask        ( int               Socket );
                   ~HTCSSendTask        ();

virtual void        OnRecvPacket        ( tmipc::Packet*        pPacket );

    void            Start               ( const void*           pBuffer,
                                          int32_t               bufferByteSize,
                                          int                   flags,
                                          nn::os::NativeHandle* pWaitHandle );

virtual int         GetResults          ( int*                  pError );

private:
};

//==============================================================================

class HTCSSendLargeTask : public HTCSSendTask
{
public:
    explicit        HTCSSendLargeTask   ( int                   Socket );
                   ~HTCSSendLargeTask   ();

virtual void        OnSendPacket        ( tmipc::Packet*        pPacket );

    void            Start               ( const void*           pUnalignedStart,
                                          int32_t               unalignedStartSize,
                                          const void*           pUnalignedEnd,
                                          int32_t               unalignedEndSize,
                                          nn::os::NativeHandle  alignedDataHandle,
                                          int64_t               alignedSendSize,
                                          int                   flags,
                                          nn::os::NativeHandle* pWaitHandle );
virtual int         GetResults          ( int*                  pError );

private:
    bool            PopulatePacket      ( tmipc::Packet*        pPacket,
                                          int32_t               thisBufferSize = 0,
                                          const void*           pSrc = NULL );

    nn::os::TransferMemoryType  m_Src;
    int     m_Flags;
    int32_t m_NumberOfPacketsSent;
    int64_t m_AlignedLeftToWrite;
    char*   m_pAlignedBuffer;
    int32_t m_PostWritten;
    int32_t m_PostLeftToWrite;
    int32_t m_UnalignedDataSize;
    char*   m_pUnalignedData;
    bool    m_MemoryMapped;
};

//==============================================================================
// Deprecated, backwards-compatible tasks.
class DeprecatedAcceptTask : public HTCSTask
{
public:
    explicit        DeprecatedAcceptTask( int                 Socket );
                   ~DeprecatedAcceptTask();

virtual void        OnRecvPacket        ( tmipc::Packet*    pPacket );

    void            Accept              ( nn::htcs::SockAddrHtcs*    pAddr );

private:
    nn::htcs::SockAddrHtcs*  m_WriteAddr;
};

//==============================================================================

class DeprecatedRecvTask : public HTCSTask
{
public:
    explicit        DeprecatedRecvTask  ( int               Socket );
                   ~DeprecatedRecvTask  ();

virtual void        OnRecvPacket        ( tmipc::Packet*    pPacket );

    void            Recv                ( void*             pBuffer,
                                          size_t            bufferByteSize,
                                          int               flags );
private:
    void*           m_WriteAddr;
};

//==============================================================================

class DeprecatedSendTask : public HTCSTask
{
public:
    explicit        DeprecatedSendTask  ( int               Socket );
                   ~DeprecatedSendTask  ();

virtual void        OnRecvPacket        ( tmipc::Packet*    pPacket );
virtual void        OnSendPacket        ( tmipc::Packet*    pPacket );

    void            Send                ( const void*               pBuffer,
                                          size_t                    bufferByteSize,
                                          int                       flags );

private:
    void            PopulatePacket      ( tmipc::Packet*    pPacket );

    //Internal-use variables.
    int             m_Flags;
    char*           m_pReadFromBuffer;
    size_t          m_AmountLeftToWrite;
    int32_t         m_NumberOfPacketsSent;
};

//==============================================================================
}}
//==============================================================================
