﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// このファイルはユーザ定義

#include "htcs_api.h"
#include "htcs_HtcsSocketImpl.h"
#include <nn/htcs/htcs_Types.h>
#include <nn/tma/tma_Result.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_ObjectFactory.h>

namespace nn { namespace tma {

namespace {

struct SocketImplAllocatorTag;
typedef nn::sf::ExpHeapStaticAllocator<4096 * 8, SocketImplAllocatorTag> SocketImplAllocator;

// SocketImplAllocator を静的コンストラクタで初期化するためのヘルパー
class SocketImplAllocatorInitializer
{
public:
    SocketImplAllocatorInitializer() NN_NOEXCEPT
    {
        SocketImplAllocator::Initialize(nn::lmem::CreationOption_ThreadSafe);
    }
} g_SocketImplAllocatorInitializer;

}

HtcsSocketImpl::HtcsSocketImpl(HtcsManagerImpl* parent, int32_t descriptor) NN_NOEXCEPT
    : m_Parent(parent, true)
{
    m_Descriptor = descriptor;
    m_Parent->Register(this);
}

HtcsSocketImpl::~HtcsSocketImpl() NN_NOEXCEPT
{
    m_Parent->Unregister(this);
    int32_t errorCode;
    htcs::Close( m_Descriptor, errorCode );
}

nn::Result HtcsSocketImpl::Close(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<std::int32_t> pOutIntResult) NN_NOEXCEPT
{
    auto descriptor = m_Descriptor.exchange(-1);
    if( descriptor < 0 )
    {
        pOutErrorCode.Set( nn::htcs::HTCS_EBADF );
        NN_RESULT_SUCCESS;
    }
    int ErrorCode = 0;
    int32_t ret = htcs::Close( descriptor, ErrorCode );
    pOutIntResult.Set(ret);
    if( ret < 0 )
    {
        pOutErrorCode.Set( ErrorCode );
    }
    else
    {
        pOutErrorCode.Set(0);
    }
    NN_RESULT_SUCCESS;
}

nn::Result HtcsSocketImpl::Connect(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<std::int32_t> pOutIntResult, const nn::htcs::SockAddrHtcs& address) NN_NOEXCEPT
{
    int ErrorCode = 0;
    int32_t ret = htcs::Connect( m_Descriptor, &address, ErrorCode );
    pOutIntResult.Set(ret);
    if( ret < 0 )
    {
        pOutErrorCode.Set( ErrorCode );
    }
    else
    {
        pOutErrorCode.Set(0);
    }
    NN_RESULT_SUCCESS;
}

nn::Result HtcsSocketImpl::Bind(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<std::int32_t> pOutIntResult, const nn::htcs::SockAddrHtcs& address) NN_NOEXCEPT
{
    int ErrorCode = 0;
    int32_t ret = htcs::Bind( m_Descriptor, &address, ErrorCode );
    pOutIntResult.Set(ret);
    if( ret < 0 )
    {
        pOutErrorCode.Set( ErrorCode );
    }
    else
    {
        pOutErrorCode.Set(0);
    }
    NN_RESULT_SUCCESS;
}

nn::Result HtcsSocketImpl::Listen(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<std::int32_t> pOutIntResult, std::int32_t backlogCount) NN_NOEXCEPT
{
    int ErrorCode = 0;
    int32_t ret = htcs::Listen( m_Descriptor, backlogCount, ErrorCode );
    pOutIntResult.Set(ret);
    if( ret < 0 )
    {
        pOutErrorCode.Set( ErrorCode );
    }
    else
    {
        pOutErrorCode.Set(0);
    }
    NN_RESULT_SUCCESS;
}

nn::Result HtcsSocketImpl::Accept(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<nn::sf::SharedPointer<nn::tma::ISocket>> pOutSocket, nn::sf::Out<nn::htcs::SockAddrHtcs> pOutAddress) NN_NOEXCEPT
{
    int ErrorCode = 0;
    int32_t ret = htcs::Accept( m_Descriptor, pOutAddress.GetPointer(), ErrorCode );
    if( ret < 0 )
    {
        pOutErrorCode.Set( ErrorCode );
    }
    else
    {
        pOutErrorCode.Set(0);
        auto p = nn::sf::ObjectFactory<SocketImplAllocator::Policy>::CreateSharedEmplaced<
            ISocket,
            HtcsSocketImpl
        >(m_Parent.Get(), ret);

        pOutSocket.Set(std::move(p));
    }

    NN_RESULT_SUCCESS;
}

nn::Result HtcsSocketImpl::Recv(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<std::int64_t> pOutReceivedSize, const nn::sf::OutBuffer& buffer, std::int32_t flags) NN_NOEXCEPT
{
    int ErrorCode = 0;
    int64_t ret = htcs::Recv( m_Descriptor, buffer.GetPointerUnsafe(), buffer.GetSize(), flags, ErrorCode );
    pOutReceivedSize.Set(ret);
    if( ret < 0 )
    {
        pOutErrorCode.Set( ErrorCode );
    }
    else
    {
        pOutErrorCode.Set(0);
    }
    NN_RESULT_SUCCESS;
}

nn::Result HtcsSocketImpl::Send(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<std::int64_t> pOutSentSize, const nn::sf::InBuffer& buffer, std::int32_t flags) NN_NOEXCEPT
{
    int ErrorCode = 0;
    int64_t ret = htcs::Send( m_Descriptor, buffer.GetPointerUnsafe(), buffer.GetSize(), flags, ErrorCode );
    pOutSentSize.Set(ret);
    if( ret < 0 )
    {
        pOutErrorCode.Set( ErrorCode );
    }
    else
    {
        pOutErrorCode.Set(0);
    }
    NN_RESULT_SUCCESS;
}

nn::Result HtcsSocketImpl::Shutdown(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<std::int32_t> pOutIntResult, std::int32_t how) NN_NOEXCEPT
{
    int ErrorCode = 0;
    int32_t ret = htcs::Shutdown( m_Descriptor, how, ErrorCode );
    pOutIntResult.Set(ret);
    if( ret < 0 )
    {
        pOutErrorCode.Set( ErrorCode );
    }
    else
    {
        pOutErrorCode.Set(0);
    }
    NN_RESULT_SUCCESS;
}

nn::Result HtcsSocketImpl::Fcntl(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<std::int32_t> pOutIntResult, std::int32_t command, std::int32_t value) NN_NOEXCEPT
{
    int ErrorCode = 0;
    int32_t ret = htcs::Fcntl( m_Descriptor, command, value, ErrorCode );
    pOutIntResult.Set(ret);
    if( ret < 0 )
    {
        pOutErrorCode.Set( ErrorCode );
    }
    else
    {
        pOutErrorCode.Set(0);
    }
    NN_RESULT_SUCCESS;
}

void HtcsSocketImpl::CancelWait() NN_NOEXCEPT
{
    int32_t errorCode;
    htcs::Close( m_Descriptor, errorCode );
}

nn::Result HtcsSocketImpl::AcceptStart( nn::sf::Out<std::uint32_t> pTaskId, nn::sf::Out<::nn::sf::NativeHandle> pWaitHandle ) NN_NOEXCEPT
{
    nn::os::NativeHandle WaitHandle;
    uint32_t TaskId;
    int32_t ret = htcs::AcceptStart( m_Descriptor, &WaitHandle, &TaskId );
    if( ret >= 0 )
    {
        pWaitHandle.Set(nn::sf::NativeHandle(WaitHandle, true));
        pTaskId.Set(TaskId);
        NN_RESULT_SUCCESS;
    }

    return nn::tma::ResultUnknown();
}

nn::Result HtcsSocketImpl::AcceptResults( nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<nn::sf::SharedPointer<nn::tma::ISocket>> pOutSocket, nn::sf::Out<nn::htcs::SockAddrHtcs> pOutAddress, uint32_t TaskId ) NN_NOEXCEPT
{
    int ErrorCode = 0;
    int32_t ret = htcs::AcceptResults( TaskId, pOutAddress.GetPointer(), &ErrorCode );
    if( ret < 0 )
    {
        pOutErrorCode.Set( ErrorCode );
    }
    else
    {
        pOutErrorCode.Set(0);
        auto p = nn::sf::ObjectFactory<SocketImplAllocator::Policy>::CreateSharedEmplaced<ISocket, HtcsSocketImpl>(m_Parent.Get(), ret);
        pOutSocket.Set(std::move(p));
    }

    NN_RESULT_SUCCESS;
}

nn::Result HtcsSocketImpl::RecvStart( nn::sf::Out<std::uint32_t> pTaskId, nn::sf::Out<::nn::sf::NativeHandle> pWaitHandle, int64_t Size, int32_t flags ) NN_NOEXCEPT
{
    nn::os::NativeHandle WaitHandle;
    uint32_t TaskId;
    int32_t ret = htcs::RecvStart( m_Descriptor, Size, flags, &WaitHandle, &TaskId );
    if( ret >= 0 )
    {
        pWaitHandle.Set(nn::sf::NativeHandle(WaitHandle, true));
        pTaskId.Set(TaskId);
        NN_RESULT_SUCCESS;
    }

    return nn::tma::ResultUnknown();
}

nn::Result HtcsSocketImpl::RecvLargeStart( nn::sf::Out<std::uint32_t> pTaskId, nn::sf::Out<::nn::sf::NativeHandle> pWaitHandle, int32_t unalignedStartSize, int32_t unalignedEndSize,
        int64_t alignedSize, nn::sf::NativeHandle alignedMemoryHandle, int32_t flags )
{
    nn::os::NativeHandle WaitHandle;
    uint32_t TaskId;
    int32_t ret = htcs::RecvLargeStart( m_Descriptor, unalignedStartSize, unalignedEndSize, alignedSize, alignedMemoryHandle.GetOsHandle(), flags, &WaitHandle, &TaskId );

    alignedMemoryHandle.Detach();

    if( ret >= 0 )
    {
        pWaitHandle.Set(nn::sf::NativeHandle(WaitHandle, true));
        pTaskId.Set(TaskId);
        NN_RESULT_SUCCESS;
    }

    return nn::tma::ResultUnknown();
}

nn::Result HtcsSocketImpl::RecvResults(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<std::int64_t> pOutReceivedSize, const nn::sf::OutBuffer& buffer, std::uint32_t TaskId ) NN_NOEXCEPT
{
    int ErrorCode = 0;
    int64_t ReceivedSize = 0;
    nn::htcs::ssize_t Ret = htcs::RecvResults( TaskId, buffer.GetPointerUnsafe(), buffer.GetSize(), &ReceivedSize, &ErrorCode );
    pOutReceivedSize.Set( Ret );
    if( Ret < 0 )
    {
        pOutErrorCode.Set( ErrorCode );
    }
    else
    {
        pOutErrorCode.Set( 0 );
    }

    NN_RESULT_SUCCESS;
}

nn::Result HtcsSocketImpl::SendLargeStart( nn::sf::Out<std::uint32_t> pTaskId, nn::sf::Out<::nn::sf::NativeHandle> pWaitHandle,
            const nn::sf::InBuffer& unalignedStart,  const nn::sf::InBuffer& unalignedEnd, nn::sf::NativeHandle alignedMemoryHandle, int64_t alignedSize, int32_t flags  ) NN_NOEXCEPT
{
    nn::os::NativeHandle WaitHandle;
    uint32_t TaskId;
    int32_t ret = htcs::SendLargeStart( m_Descriptor, unalignedStart.GetPointerUnsafe(), (int32_t)unalignedStart.GetSize(),
        unalignedEnd.GetPointerUnsafe(), (int32_t)unalignedEnd.GetSize(),
        alignedMemoryHandle.GetOsHandle(), alignedSize, flags, &WaitHandle, &TaskId );

    alignedMemoryHandle.Detach();

    if( ret >= 0 )
    {
        pWaitHandle.Set(nn::sf::NativeHandle(WaitHandle, true));
        pTaskId.Set(TaskId);
        NN_RESULT_SUCCESS;
    }

    return nn::tma::ResultUnknown();
}

nn::Result HtcsSocketImpl::SendStart( nn::sf::Out<std::uint32_t> pTaskId, nn::sf::Out<::nn::sf::NativeHandle> pWaitHandle, const nn::sf::InBuffer& buffer, std::int32_t flags) NN_NOEXCEPT
{
    nn::os::NativeHandle WaitHandle;
    uint32_t TaskId;
    int32_t ret = htcs::SendStart( m_Descriptor, buffer.GetPointerUnsafe(), buffer.GetSize(), flags, &WaitHandle, &TaskId );
    if( ret >= 0 )
    {
        pWaitHandle.Set(nn::sf::NativeHandle(WaitHandle, true));
        pTaskId.Set(TaskId);
        NN_RESULT_SUCCESS;
    }

    return nn::tma::ResultUnknown();
}

nn::Result HtcsSocketImpl::SendResults(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<std::int64_t> pOutSentSize, std::uint32_t TaskId) NN_NOEXCEPT
{
    int ErrorCode = 0;
    nn::htcs::ssize_t Ret = htcs::SendResults( TaskId, &ErrorCode );
    pOutSentSize.Set( (int64_t)Ret );
    if( Ret < 0 )
    {
        pOutErrorCode.Set( ErrorCode );
    }
    else
    {
        pOutErrorCode.Set( 0 );
    }

    NN_RESULT_SUCCESS;
}

}}
