﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/tma/sf/tma_IHtcsManager.sfdl.h>
#include <nn/os/os_Mutex.h>
#include <nn/util/util_IntrusiveList.h>
#include <vector>

namespace nn { namespace tma {

class HtcsSocketImpl;

class HtcsManagerImpl
    : public sf::ISharedObject
{
public:
    nn::Result Socket(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<std::int32_t> pOutSocket) NN_NOEXCEPT;
    nn::Result Close(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<std::int32_t> pOutIntResult, std::int32_t descriptor) NN_NOEXCEPT;
    nn::Result Connect(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<std::int32_t> pOutIntResult, std::int32_t descriptor, const nn::htcs::SockAddrHtcs& address) NN_NOEXCEPT;
    nn::Result Bind(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<std::int32_t> pOutIntResult, std::int32_t descriptor, const nn::htcs::SockAddrHtcs& address) NN_NOEXCEPT;
    nn::Result Listen(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<std::int32_t> pOutIntResult, std::int32_t descriptor, std::int32_t backlogCount) NN_NOEXCEPT;
    nn::Result Accept(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<std::int32_t> pOutIntResult, nn::sf::Out<nn::htcs::SockAddrHtcs> pOutAddress, std::int32_t descriptor) NN_NOEXCEPT;
    nn::Result Recv(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<std::int64_t> pOutReceivedSize, const nn::sf::OutBuffer& buffer, std::int32_t descriptor, std::int32_t flags) NN_NOEXCEPT;
    nn::Result Send(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<std::int64_t> pOutSentSize, std::int32_t descriptor, const nn::sf::InBuffer& buffer, std::int32_t flags) NN_NOEXCEPT;
    nn::Result Shutdown(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<std::int32_t> pOutIntResult, std::int32_t descriptor, std::int32_t how) NN_NOEXCEPT;
    nn::Result Fcntl(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<std::int32_t> pOutIntResult, std::int32_t descriptor, std::int32_t command, std::int32_t value) NN_NOEXCEPT;
    nn::Result GetPeerNameAny(nn::sf::Out<nn::htcs::HtcsPeerName> pOutValue) NN_NOEXCEPT;
    nn::Result GetDefaultHostName(nn::sf::Out<nn::htcs::HtcsPeerName> pOutValue) NN_NOEXCEPT;
    nn::Result CreateSocketOld(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<nn::sf::SharedPointer<nn::tma::ISocket>> pOutSocket) NN_NOEXCEPT;
    nn::Result CreateSocket(nn::sf::Out<std::int32_t> pOutErrorCode, nn::sf::Out<nn::sf::SharedPointer<nn::tma::ISocket>> pOutSocket, bool enableDisconnectionEmulation) NN_NOEXCEPT;
    nn::Result RegisterProcessId(Bit64 processId) NN_NOEXCEPT;
    nn::Result MonitorManager(Bit64 processId) NN_NOEXCEPT;
    ~HtcsManagerImpl() NN_NOEXCEPT;

    void Register(HtcsSocketImpl* p) NN_NOEXCEPT
    {
        m_SocketList.Register(p);
    }

    void Unregister(HtcsSocketImpl* p) NN_NOEXCEPT
    {
        m_SocketList.Unregister(p);
    }

private:
    sf::SharedPointer<HtcsManagerImpl> m_MonitorTarget;

    class SocketList
    {
    private:
        os::Mutex m_Mutex;
        util::IntrusiveList<HtcsSocketImpl, util::IntrusiveListBaseNodeTraits<HtcsSocketImpl, HtcsManagerImpl>> m_Sockets;
    public:
        SocketList() NN_NOEXCEPT;
        void Register(HtcsSocketImpl* p) NN_NOEXCEPT;
        void Unregister(HtcsSocketImpl* p) NN_NOEXCEPT;
        void CancelWait() NN_NOEXCEPT;
    };

    SocketList m_SocketList;

    class SocketPrimitiveList
    {
    private:
        os::Mutex m_Mutex;
        std::vector<std::int32_t> m_Sockets;
    public:
        SocketPrimitiveList() NN_NOEXCEPT;
        void Save(std::int32_t p) NN_NOEXCEPT;
        void Remove(std::int32_t p) NN_NOEXCEPT;
        void Clear() NN_NOEXCEPT;
    };

    SocketPrimitiveList m_SocketPrimitiveList;

};

nn::sf::SharedPointer<IHtcsManager> CreateHtcsManagerImpl() NN_NOEXCEPT;

}}
