﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "htcs_HtcsHipcServer.h"
#include <nn/tma/detail/htcs_ServiceName.h>

#include <nn/nn_Common.h>
#include <nn/sf/sf_HipcServer.h>
#include <type_traits>
#include <new>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include "htcs_HtcsManager.h"

namespace nn { namespace tma {

namespace {

const int SessionCountMax = 64;

struct MyServerManagerOption
{
    static const size_t PointerTransferBufferSize = 128;
    static const int    ObjectInSubDomainCountMax = 100;
    static const int    SubDomainCountMax = 16;
};

// HipcSimpleAllInOneServerManager は HIPC のサーバのポートとセッションを一元管理する。
class MyServerManager
    : public nn::sf::HipcSimpleAllInOneServerManager<SessionCountMax, 1, MyServerManagerOption>
{
private:
    virtual nn::Result OnNeedsToAccept(int portIndex, PortForAllInOne* pPort) NN_NOEXCEPT NN_OVERRIDE final
    {
        NN_UNUSED(portIndex);
        return this->AcceptImpl(pPort, CreateHtcsManager());
    }
};

// 繰り返しのサーバの起動と終了が可能となるように placement new で初期化を行う。
// 繰り返しの起動と終了が必要ない場合には MyServerManager は直接配置してもよい。
std::aligned_storage<sizeof(MyServerManager), NN_ALIGNOF(MyServerManager)>::type g_MyServerManagerStorage;
MyServerManager* g_pMyServerManager;

}

void InitializeHtcsServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!g_pMyServerManager);

    // MyServerManager オブジェクトのコンストラクト
    g_pMyServerManager = new (&g_MyServerManagerStorage) MyServerManager;

    // サービス名の登録とポートの初期化
    // portIndex は OnNeedsToAccept の第一引数となるが、今回は使用しないため、適当に 0 を指定している。
    // sessionCountMax は、現時点では十分に大きな値を指定しておけばよい。
    auto portIndex = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_pMyServerManager->InitializePort(0, SessionCountMax, detail::HtcsServiceName));

    // サーバマネージャの開始
    // ただし、実際のサーバ動作は、LoopAuto 関数等を呼び出すことで行う必要がある。
    g_pMyServerManager->Start();
}

void LoopHtcsServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pMyServerManager);

    // サーバの処理ループ実行を実行する。
    // - ポートへの接続要求
    // - セッションへの処理要求
    // RequestStop が呼ばれるまで返らない。
    g_pMyServerManager->LoopAuto();
}

void RequestStopHtcsServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pMyServerManager);

    // 処理ループの停止リクエストを送り、LoopAuto 関数を返す
    g_pMyServerManager->RequestStop();
}

void FinalizeHtcsServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pMyServerManager);

    // MyServerManager のデストラクト
    // 登録したサービスなどはここで登録が解除される。
    g_pMyServerManager->~MyServerManager();
    g_pMyServerManager = nullptr;
}

}}
