﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "..\tmagent.h"
#include "hid_Service.h"
#include "hid_structures.h"

//==============================================================================
namespace tma { namespace hid {
//==============================================================================

AgentControllerService::AgentControllerService()
:   m_iStartChar    ( 0 )
,   m_iEndChar      ( 0 )
{
    DEJA_TRACE( "AgentControllerService::AgentControllerService", "AgentControllerService" );
    m_ServiceId = HashString( "ControllerService" );

    memset( m_CharBuffer, 0, sizeof( s32 ) * MAX_CHARACTERS_TO_KEEP );
    memset( m_ControllerState, 0, sizeof(m_ControllerState) );

#ifdef CTRL_PERFORMANCE
    Counter = 0;

    SecStart = nn::os::GetSystemTick();
#endif // CTRL_PERFORMANCE
}

//==============================================================================

AgentControllerService::~AgentControllerService()
{
    DEJA_TRACE( "AgentControllerService::~AgentControllerService", "~AgentControllerService" );
}

//==============================================================================

void AgentControllerService::Init()
{
    DEJA_TRACE( "AgentControllerService::Init", "Init" );
    Create();
}

//==============================================================================

void AgentControllerService::Kill()
{
    DEJA_TRACE( "AgentControllerService::Kill", "Kill" );
    Destroy();
}

//==============================================================================

tmipc::Task* AgentControllerService::OnNewTask( tmipc::Packet* pPacket )
{
#ifdef CTRL_PERFORMANCE
    Counter++;

    if( Counter >= 60 )
    {
        Counter = 0;

        nn::os::Tick Now = nn::os::GetSystemTick();
        u64 Dif = (Now - SecStart).GetInt64Value();
        auto timeSpan = nn::TimeSpan::FromNanoSeconds( Dif );
        SecStart = Now;

        NN_SDK_LOG( "Took %lld msecs to recieve 60 controller tasks.\n", timeSpan.GetMilliSeconds() );
    }
#endif // CTRL_PERFORMANCE

    void* pMem = s_Allocate( sizeof( ControllerReadTask ) );
    ControllerReadTask* pTask = new (pMem) ControllerReadTask();

    pTask->SetServicesManager( m_pServicesManager );
    pTask->SetServiceId( pPacket->GetServiceId() );
    pTask->SetTaskId( pPacket->GetTaskId() );
    pTask->OnInitiate( pPacket );

    return(pTask);
}

//==============================================================================

bool AgentControllerService::GetControllerState( s32 iController, void* pStateBuffer )
{
    bool bValid = false;

    if( (0 <= iController) && (iController < MAX_CONTROLLERS) )
    {
        bValid = true;

        memcpy( pStateBuffer, &m_ControllerState[iController], sizeof(controller_state) );
    }
    else
    {
        memset( pStateBuffer, 0, sizeof(controller_state) );
    }

    return( bValid );
}

//==============================================================================

void AgentControllerService::UpdateControllerState( controller_data* pBuffer )
{
    if( pBuffer == NULL )
        return;
    //tma::hid::controller_type Type = tma::hid::GetControllerType( pBuffer );

    switch( pBuffer->m_Type )
    {
        case CONTROLLER_TYPE_WII_CLASSIC_PRO:
        {
            wii_classic_pro_data* pData = (wii_classic_pro_data*)pBuffer;

            // Only update the controller state if the controller is connected and
            // is a valid controller index.
            if( pData->m_bConnected
                && (pData->m_Size == sizeof(wii_classic_pro_data))
                && ((0 <= pData->m_Port) && (pData->m_Port < MAX_CONTROLLERS)) )
            {
                controller_state& State = m_ControllerState[pData->m_Port];

                State.m_Buttons = pData->m_ButtonStates;
                memcpy( &State.m_Axis, pData->m_AxisStates, sizeof(pData->m_AxisStates) );
            }
        }
        default:
            break;
    }
}

//==============================================================================

void AgentControllerService::UpdateKeyboardInput( s32 Character )
{
    // Only keep MAX_CHARACTERS_TO_KEEP - 1 in the array
    // ignore characters if it would overwrite a non-consumed character.

    u32 iEnd = (m_iEndChar + 1) % MAX_CHARACTERS_TO_KEEP;
    if( iEnd != m_iStartChar )
    {
        m_CharBuffer[m_iEndChar] = Character;
        m_iEndChar = iEnd;
    }
    else
    {
        NN_SDK_LOG( "Buffer overflow, ignoring %c [0x%08x]\n", Character, Character );
        DEJA_TRACE( "AgentControllerService::UpdateKeyboardInput", "Buffer overflow, ignoring %c [0x%08x]\n", Character, Character );
    }
}

//==============================================================================

void AgentControllerService::GetKeyboardInput( s32& Character )
{
    if( m_iStartChar != m_iEndChar )
    {
        Character = m_CharBuffer[m_iStartChar];
        m_iStartChar = (m_iStartChar + 1) % MAX_CHARACTERS_TO_KEEP;
    }
    else
    {
        Character = -1;
    }
}

//==============================================================================

u32 AgentControllerService::GetControllerBufferSzie()
{
    return ( sizeof(controller_state) );
}

//==============================================================================
}
}
//==============================================================================
