﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "dbghlp.h"
#include <nn/ldr/ldr_ProcessManagerApi.h>
#include <nn/ncm/ncm_ProgramId.h>
#include <cstring>
#include <algorithm>
#include <nn/dmnt/dmnt_Api.h>

//==============================================================================
namespace tma { namespace dbghlp {
//==============================================================================

#define USE_DMNT
//#define ENABLE_PROFILE
//#define ENABLE_CALL_TRACE

enum times
{
    TIME_GETDEBUGTHREADPARAM,
    TIME_READDEBUGPROCESSMEMORY,
    TIME_WRITEDEBUGPROCESSMEMORY,
    TIME_GETPROCESSMODULEINFO,
    TIME_GETPROCESSID,
    TIME_GETPROCESSIDFROMPROGRAMID,
    TIME_WAITSYNCHRONIZATION,
    TIME_GETDEBUGEVENT,
    TIME_CONTINUEDEBUGEVENT,
    TIME_GETDEBUGTHREADCONTEXT,
    TIME_SETDEBUGTHREADCONTEXT,
    TIME_TERMINATEDEBUGPROCESS,
    TIME_CLOSEHANDLE,
    TIME_GETTHREADLIST,
    TIME_LOADIMAGE,
    TIME_BREAKDEBUGPROCESS,
    TIME_INITIALIZETHREADINFO,

    TIME_SIZEOF
};

#ifdef ENABLE_PROFILE

const char* s_StatNames[TIME_SIZEOF] =
{
    "GetDebugThreadParam.....",
    "ReadDebugProcessMemory..",
    "WriteDebugProcessMemory.",
    "GetProcessModuleInfo....",
    "GetProcessId............",
    "WaitSynchronization.....",
    "GetDebugEvent...........",
    "ContinueDebugEvent......",
    "GetDebugThreadContext...",
    "SetDebugThreadContext...",
    "TerminateDebugProcess...",
    "CloseHandle.............",
    "GetThreadList...........",
    "LoadImage...............",
    "BreakDebugProcess.......",
    "InitializeThreadInfo....",
};

struct stat
{
    uint32_t        m_nHits;
    uint32_t        m_tTotal;
    int64_t         m_tStart;
};

stat s_Stat[TIME_SIZEOF] = {0};

class ctx
{
    int         m_id;

public:
    explicit ctx( int id )
    {
        m_id = id;
        ++s_Stat[id].m_nHits;
        s_Stat[id].m_tStart = nn::os::GetSystemTick().GetInt64Value();
    }

    ~ctx()
    {
        int64_t t = nn::os::GetSystemTick().GetInt64Value() - s_Stat[m_id].m_tStart;
        s_Stat[m_id].m_tTotal += t;
    }
};

#define PROFILE( _a_ ) ctx c(_a_)

void PrintStats()
{
    int64_t f = nn::os::GetSystemTickFrequency() / 10000;

    NN_SDK_LOG( "Freq: %d\n", (int)f );

    for( int i=0; i<TIME_SIZEOF; i++ )
    {
        stat& s = s_Stat[i];

        int t = (int)(s.m_tTotal / f);
        NN_SDK_LOG( "%s %4d %8d\n", s_StatNames[i], s.m_nHits, t );
        s.m_nHits  = 0;
        s.m_tTotal = 0;
    }
}

#else // ENABLE_PROFILE

#define PROFILE(_a_)

#endif // ENABLE_PROFILE

#ifdef ENABLE_CALL_TRACE
#define CALL_TRACE( _a_ ) NN_SDK_LOG( _a_ );
#else
#define CALL_TRACE( _a_ )
#endif

//==============================================================================

bool                        g_LocationResolverInitialized = false;
nn::lr::LocationResolver    g_LocationResolver;

//==============================================================================

nn::Result Initialize() NN_NOEXCEPT
{
    CALL_TRACE( "dbghlp::Initialize\n" );

#ifndef USE_DMNT
    nn::Result result = nn::dbg::Initialize();
    NN_ABORT_UNLESS(result.IsSuccess(), "result=%08x", result);
#endif

    return nn::ResultSuccess();
}

//==============================================================================

nn::Result Finalize() NN_NOEXCEPT
{
    CALL_TRACE( "dbghlp::Finalize\n" );

    if( g_LocationResolverInitialized )
    {
        nn::lr::Finalize();
        nn::pm::FinalizeForDebugMonitor();
    }

#ifndef USE_DMNT
    nn::dbg::Finalize();
#endif

    return nn::ResultSuccess();
}

//==============================================================================

nn::Result QueryDebugProcessMemory( nn::svc::MemoryInfo* pBlockInfo, nn::svc::PageInfo* pPageInfo, nn::svc::Handle process, uintptr_t addr ) NN_NOEXCEPT
{
    CALL_TRACE( "dbghlp::QueryDebugProcessMemory\n" );

#ifdef USE_DMNT
    nn::Result result = nn::dmnt::QueryDebugProcessMemory( pBlockInfo, pPageInfo, process, addr );
#else
    nn::Result result = nn::dbg::QueryDebugProcessMemory( pBlockInfo, pPageInfo, process, addr );
#endif

    return( result );
}

//==============================================================================

nn::Result GetDebugThreadParam( nn::Bit64* pOut1, nn::Bit32* pOut2, nn::svc::Handle debug, nn::Bit64 threadId, nn::svc::DebugThreadParam select) NN_NOEXCEPT
{
    CALL_TRACE( "dbghlp::GetDebugThreadParam\n" );
    PROFILE(TIME_GETDEBUGTHREADPARAM);

#ifdef USE_DMNT
    nn::Result result = nn::dmnt::GetDebugThreadParam( pOut1, pOut2, debug, threadId, select );
#else
    nn::Result result = nn::dbg::GetDebugThreadParam( pOut1, pOut2, debug, threadId, select );
#endif

    return( result );
}

//==============================================================================

nn::Result ReadDebugProcessMemory( uintptr_t buf, nn::svc::Handle debug, uintptr_t addr, size_t size ) NN_NOEXCEPT
{
    CALL_TRACE( "dbghlp::ReadDebugProcessMemory\n" );
    PROFILE(TIME_READDEBUGPROCESSMEMORY);

#ifdef USE_DMNT
    nn::Result result = nn::dmnt::ReadDebugProcessMemory( buf, debug, addr, size );
#else
    nn::Result result = nn::dbg::ReadDebugProcessMemory( buf, debug, addr, size );
#endif

    return( result );
}

//==============================================================================

nn::Result WriteDebugProcessMemory(nn::svc::Handle debug, uintptr_t buf, uintptr_t addr, size_t size) NN_NOEXCEPT
{
    CALL_TRACE( "dbghlp::WriteDebugProcessMemory\n" );

    PROFILE(TIME_WRITEDEBUGPROCESSMEMORY);

#ifdef USE_DMNT
    nn::Result result = nn::dmnt::WriteDebugProcessMemory( debug, buf, addr, size );
#else
    nn::Result result = nn::dbg::WriteDebugProcessMemory( debug, buf, addr, size );
#endif

    return( result );
}

//==============================================================================

nn::Result GetProcessModuleInfo(int* pOutCount, nn::dbg::ModuleInfo* pOutModules, int num, nn::os::ProcessId pid) NN_NOEXCEPT
{
    CALL_TRACE( "dbghlp::GetProcessModuleInfo\n" );

    PROFILE(TIME_GETPROCESSMODULEINFO);

#ifdef USE_DMNT
    nn::Result result = nn::dmnt::GetProcessModuleInfo( pOutCount, pOutModules, num, pid );
#else
    nn::Result result = nn::dbg::GetProcessModuleInfo( pOutCount, pOutModules, num, pid );
#endif

    return( result );
}

//=============================================================================================================

nn::Result GetProcessList(int32_t* pNumProcesses, nn::Bit64 pProcessIds[], int32_t arraySize) NN_NOEXCEPT
{
    CALL_TRACE( "dbghlp::GetProcessList\n" );

    PROFILE(TIME_GETPROCESSMODULEINFO);

#ifdef USE_DMNT
    nn::Result result = nn::dmnt::GetProcessList( pNumProcesses, pProcessIds, arraySize);
#else
    nn::Result result = nn::dbg::GetProcessList( pNumProcesses, pProcessIds, arraySize);
#endif

    return( result );
}

//=============================================================================================================

nn::Result GetProcessHandle(nn::svc::Handle* pOut, nn::Bit64 processId) NN_NOEXCEPT
{
    CALL_TRACE( "dbghlp::GetProcessHandle\n" );

    PROFILE(TIME_GETPROCESSMODULEINFO);

#ifdef USE_DMNT
    nn::Result result = nn::dmnt::GetProcessHandle( pOut, processId );
#else
    nn::Result result = nn::dbg::DebugActiveProcess( pOut, processId );
#endif

    return( result );
}

//==============================================================================

nn::Result GetProcessId(nn::Bit64* pOut, nn::svc::Handle process) NN_NOEXCEPT
{
    CALL_TRACE( "dbghlp::GetProcessId\n" );

    PROFILE(TIME_GETPROCESSID);

#ifdef USE_DMNT
    nn::Result result = nn::dmnt::GetProcessId( pOut, process );
#else
    nn::Result result = nn::dbg::GetProcessId( pOut, process );
#endif

    return( result );
}

//==============================================================================
/*
nn::Result GetProcessIdFromProgramId( nn::Bit64* pOut, nn::Bit64 ProgramId ) NN_NOEXCEPT
{
    CALL_TRACE( "dbghlp::GetProcessId\n" );

    PROFILE(TIME_GETPROCESSIDFROMPROGRAMID);

    nn::Result result = nn::dmnt::GetProcessIdFromProgramId( pOut, ProgramId );

    return( result );
}
*/
//==============================================================================

nn::Result WaitSynchronization(int32_t* pOut, const nn::svc::Handle handles[], int32_t numHandles, int64_t ns) NN_NOEXCEPT
{
    CALL_TRACE( "dbghlp::WaitSynchronization\n" );

    PROFILE(TIME_WAITSYNCHRONIZATION);

#ifdef USE_DMNT
    nn::Result result = nn::dmnt::WaitSynchronization( *handles, ns );
#else
    nn::Result result = nn::dbg::WaitSynchronization( pOut, handles, numHandles, ns );
#endif

    return( result );
}

//==============================================================================

nn::Result GetDebugEvent(nn::svc::DebugEventInfo* pInfo, nn::svc::Handle debug) NN_NOEXCEPT
{
    CALL_TRACE( "dbghlp::GetDebugEvent\n" );

    PROFILE(TIME_GETDEBUGEVENT);

#ifdef USE_DMNT
    nn::Result result = nn::dmnt::GetDebugEvent( pInfo, debug );
#else
    nn::Result result = nn::dbg::GetDebugEvent( pInfo, debug );
#endif

    return( result );
}

//==============================================================================

nn::Result ContinueDebugEvent(nn::svc::Handle debug, nn::Bit32 flags, nn::Bit64 threadIds[], nn::Bit32 size) NN_NOEXCEPT
{
    return nn::dmnt::ContinueDebugEvent( debug, flags, threadIds, size );
}

//==============================================================================

nn::Result GetDebugThreadContext(nn::svc::ThreadContext* pContext, nn::svc::Handle debug, nn::Bit64 threadId, nn::Bit32 controlFlags) NN_NOEXCEPT
{
    CALL_TRACE( "dbghlp::GetDebugThreadContext\n" );

    PROFILE(TIME_GETDEBUGTHREADCONTEXT);

#ifdef USE_DMNT
    nn::Result result = nn::dmnt::GetDebugThreadContext( pContext, debug, threadId, controlFlags );
#else
    nn::Result result = nn::dbg::GetDebugThreadContext( pContext, debug, threadId, controlFlags );
#endif

    return( result );
}

//==============================================================================

nn::Result GetAllDebugThreadInfo( nn::svc::Handle debug, void* pThreadData, nn::Bit32 ElementSize, nn::osdbg::ThreadInfo ThreadInfo[], nn::Bit32 ArraySize ) NN_NOEXCEPT
{
    CALL_TRACE( "dbghlp::GetAllDebugThreadInfo\n" );

#ifdef USE_DMNT
    nn::Result result = nn::dmnt::GetAllDebugThreadInfo( debug, pThreadData, ElementSize, ThreadInfo, ArraySize );
#else
    #error Not Implemented
#endif

    return( result );
}

//==============================================================================

nn::Result SetDebugThreadContext(nn::svc::Handle debug, nn::Bit64 threadId, const nn::svc::ThreadContext& context, nn::Bit32 controlFlags) NN_NOEXCEPT
{
    CALL_TRACE( "dbghlp::SetDebugThreadContext\n" );

    PROFILE(TIME_SETDEBUGTHREADCONTEXT);

#ifdef USE_DMNT
    nn::Result result = nn::dmnt::SetDebugThreadContext( debug, threadId, context, controlFlags );
#else
    nn::Result result = nn::dbg::SetDebugThreadContext( debug, threadId, context, controlFlags );
#endif

    return( result );
}

//==============================================================================

nn::Result TerminateDebugProcess(nn::svc::Handle debug) NN_NOEXCEPT
{
    CALL_TRACE( "dbghlp::TerminateDebugProcess\n" );

    nn::Result result;

    {
        PROFILE(TIME_TERMINATEDEBUGPROCESS);

#ifdef USE_DMNT
        result = nn::dmnt::TerminateDebugProcess( debug );
#else
        result = nn::dbg::TerminateDebugProcess( debug );
#endif
    }

#ifdef ENABLE_PROFILE
    PrintStats();
#endif

    return( result );
}

//==============================================================================

nn::Result CloseHandle(nn::svc::Handle handle) NN_NOEXCEPT
{
    CALL_TRACE( "dbghlp::CloseHandle\n" );

    PROFILE(TIME_CLOSEHANDLE);

#ifdef USE_DMNT
    nn::Result result = nn::dmnt::CloseHandle( handle );
#else
    nn::Result result = nn::dbg::CloseHandle( handle );
#endif

    return( result );
}

//==============================================================================

nn::Result GetThreadList(int32_t* pNumThreads, nn::Bit64 pThreadIds[], int32_t arraySize, nn::svc::Handle domain) NN_NOEXCEPT
{
    CALL_TRACE( "dbghlp::GetThreadList\n" );

    PROFILE(TIME_GETTHREADLIST);

#ifdef USE_DMNT
    nn::Result result = nn::dmnt::GetThreadList( pNumThreads, pThreadIds, arraySize, domain );
#else
    nn::Result result = nn::dbg::GetThreadList( pNumThreads, pThreadIds, arraySize, domain );
#endif

    return( result );
}

//==============================================================================

nn::Result LoadImage( nn::svc::Handle* pHandle, const char* pFileName, const char* pArgs ) NN_NOEXCEPT
{
    CALL_TRACE( "dbghlp::LoadImage\n" );

    PROFILE(TIME_LOADIMAGE);

#ifdef USE_DMNT
    nn::Result result = nn::dmnt::LoadImage( pHandle, pFileName, pArgs );
#else
    if( !g_LocationResolverInitialized )
    {
        nn::Result result;

        result = nn::pm::InitializeForDebugMonitor();
        NN_ABORT_UNLESS(result.IsSuccess(), "result=%08x", result);

        nn::lr::Initialize();
        NN_ABORT_UNLESS(nn::lr::OpenLocationResolver(&g_LocationResolver, nn::ncm::StorageId::Host).IsSuccess());

        g_LocationResolverInitialized = true;
    }

    // TODO: This programId came from DebugMonitor. What should it really be?
    static nn::ncm::ProgramId dummyProgramId = { 0xF0F0000000000000ull };

    nn::lr::Path path;
    std::memset( path.string, 0, sizeof(path.string) );

    const char HostMountName[] = "@Host:/";
    const char Fs0HostArchiveName[] = "host:";

    if( std::strncmp(pFileName, Fs0HostArchiveName, std::strlen(Fs0HostArchiveName)) != 0 )
    {
        std::memcpy( path.string, HostMountName, std::strlen(HostMountName) );
    }
    std::memcpy( path.string + std::strlen(path.string), pFileName, std::min(strlen(pFileName), sizeof(path.string) - std::strlen(path.string) - 1) );

    g_LocationResolver.RedirectProgramPath( dummyProgramId, path );

    // Setup the arguments.
    size_t argumentSize = strlen(pArgs);

    // Start the process.
    nn::svc::Handle hProcess;
    nn::Result result = nn::dbg::DebugNewProcess( pHandle, dummyProgramId, pArgs, argumentSize );
#endif

    return( result );
}

//==============================================================================

nn::Result AttachByProgramId( nn::svc::Handle* pHandle, nn::Bit64* pPID, nn::Bit64 ProgramId ) NN_NOEXCEPT
{
    CALL_TRACE( "dbghlp::AttachByProgramId\n" );

    nn::Result result = nn::dmnt::AttachByProgramId( pHandle, pPID, ProgramId );

    return( result );
}

//==============================================================================

nn::Result AttachOnLaunch( nn::svc::Handle* pHandle, nn::Bit64* pPID, nn::Bit64 ProgramId ) NN_NOEXCEPT
{
    CALL_TRACE( "dbghlp::WaitForLaunch\n" );

    nn::Result result = nn::dmnt::AttachOnLaunch( pHandle, pPID, ProgramId );

    return( result );
}

//==============================================================================

nn::Result BreakDebugProcess(nn::svc::Handle debug) NN_NOEXCEPT
{
    CALL_TRACE( "dbghlp::BreakDebugProcess\n" );

    PROFILE(TIME_BREAKDEBUGPROCESS);

#ifdef USE_DMNT
    nn::Result result = nn::dmnt::BreakDebugProcess( debug );
#else
    nn::Result result = nn::dbg::BreakDebugProcess( debug );
#endif

    return( result );
}

//==============================================================================

nn::Result InitializeThreadInfo( nn::osdbg::ThreadInfo* pThreadInfo, nn::svc::Handle debugHandle, const nn::svc::DebugInfoCreateProcess* pDebugInfoCreateProcess, const nn::svc::DebugInfoCreateThread* pDebugInfoCreateThread ) NN_NOEXCEPT
{
    CALL_TRACE( "dbghlp::InitializeThreadInfo\n" );

    PROFILE(TIME_INITIALIZETHREADINFO);

#ifdef USE_DMNT
    nn::Result result = nn::dmnt::InitializeThreadInfo( pThreadInfo, debugHandle, pDebugInfoCreateProcess, pDebugInfoCreateThread );
#else
    nn::Result result = nn::osdbg::InitializeThreadInfo( pThreadInfo, debugHandle, pDebugInfoCreateProcess, pDebugInfoCreateThread );
#endif

    return( result );
}

//==============================================================================

nn::Result SetHardwareBreakPoint( nn::svc::HardwareBreakPointRegisterName regNo,  nn::Bit64 control, nn::Bit64 value ) NN_NOEXCEPT
{
    CALL_TRACE( "dbghlp::SetHardwareBreakPoint\n" );

#ifdef USE_DMNT
    nn::Result result = nn::dmnt::SetHardwareBreakPoint( regNo, control, value );
#else
    nn::Result result = nn::dbg::SetHardwareBreakPoint( regNo, control, value );
#endif

    return( result );
}

//==============================================================================

nn::Result GetProcessMemoryDetails( int32_t* pNumOfMemoryBlocks, nn::svc::MemoryInfo pBlockInfo[], int32_t arraySize, nn::svc::Handle debugHandle ) NN_NOEXCEPT
{
    CALL_TRACE( "dbghlp::GetProcessMemoryDetails\n" );

    nn::Result result = nn::dmnt::GetProcessMemoryDetails( pNumOfMemoryBlocks, pBlockInfo, arraySize, debugHandle );

    return( result );
}

//==============================================================================

nn::Result GetJitDebugProcessList( int32_t* pNumProcesses, nn::os::ProcessId pProcessIds[], int32_t arraySize ) NN_NOEXCEPT
{
    CALL_TRACE( "dbghlp::GetJitDebugProcessList\n" );

    nn::Result result = nn::dmnt::GetJitDebugProcessList( pNumProcesses, pProcessIds, arraySize );

    return( result );
}

//==============================================================================

nn::Result CreateCoreDump( char *FileName, coredump::coredump_compiled_data* pDetails, nn::svc::Handle debugHandle ) NN_NOEXCEPT
{
    CALL_TRACE( "dbghlp::CreateCoreDump\n" );

    nn::Result result = nn::dmnt::CreateCoreDump( FileName, pDetails, debugHandle );

    return( result );
}

//==============================================================================

nn::Result InitiateCoreDump( char *FileName, coredump::coredump_compiled_data* pDetails, nn::svc::Handle debugHandle, u64* pSessionId ) NN_NOEXCEPT
{
    CALL_TRACE( "dbghlp::InitiateCoreDump\n" );

    nn::Result result = nn::dmnt::InitiateCoreDump( FileName, pDetails, debugHandle, pSessionId );

    return( result );
}

//==============================================================================

nn::Result ContinueCoreDump( u64 SessionId, s32* pPercentDone ) NN_NOEXCEPT
{
    CALL_TRACE( "dbghlp::ContinueCoreDump\n" );

    nn::Result result = nn::dmnt::ContinueCoreDump( SessionId, pPercentDone );

    return( result );
}

//==============================================================================

nn::Result AddTTYToCoreDump( char* pData, u32 SizeOfData , u64 SessionId ) NN_NOEXCEPT
{
    CALL_TRACE( "dbghlp::AddTTYToCoreDump\n" );

    nn::Result result = nn::dmnt::AddTTYToCoreDump( pData, SizeOfData, SessionId );

    return( result );
}

//==============================================================================

nn::Result AddImageToCoreDump( char* pData, u32 SizeOfData , u64 SessionId ) NN_NOEXCEPT
{
    CALL_TRACE( "dbghlp::AddImageToCoreDump\n" );

    nn::Result result = nn::dmnt::AddImageToCoreDump( pData, SizeOfData, SessionId );

    return( result );
}

//==============================================================================

nn::Result CloseCoreDump( u64 SessionId ) NN_NOEXCEPT
{
    CALL_TRACE( "dbghlp::CloseCoreDump\n" );

    nn::Result result = nn::dmnt::CloseCoreDump( SessionId );

    return( result );
}

//==============================================================================
}} // namespace
//==============================================================================
