﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// WARNING: THERE ARE TWO COPIES OF THIS FILE WHICH MUST STAY IN SYNC

#pragma once

#include "..\tmagent.h"

//==============================================================================
namespace tma { namespace dbg {
//==============================================================================

enum
{
    TMA_MAX_MEMORY_WRITE_SIZE   = 1024,     // This is the biggest size we support via RPC, and thus from the debug engine.
    TMA_MAX_DUMP_URL_SIZE       = 1024
};

//==============================================================================

struct ReadMemCommandArgs
{
    ReadMemCommandArgs( u64 Address, u32 DataSize )
    {
        m_Address   = Address;
        m_DataSize  = DataSize;
    }

    ReadMemCommandArgs()
    {
        m_DataSize = 0;
    }

    //Just simplistic validity checking
    bool HasValidData()
    {
        return( m_DataSize > 0 );
    };

    u64     m_Address;
    u32     m_DataSize;
};

//==============================================================================

struct WriteMemoryCommandArgs
{
    explicit WriteMemoryCommandArgs()
    {
        memset(m_Data, 0, sizeof(m_Data));
        m_SizeOfData = 0;
        m_WriteToAddress = 0;
    }

    //Just simplistic validity checking ONLY.
    bool HasValidData()
    {
        return ( m_SizeOfData > 0 );
    };

    char    m_Data[TMA_MAX_MEMORY_WRITE_SIZE];
    u32     m_SizeOfData;
    u64     m_WriteToAddress;
};

//==============================================================================

struct ReadRegDefsCommandArgs
{
    explicit ReadRegDefsCommandArgs( u32 DataSize )
    {
        m_DataSize  = DataSize;
    }

    ReadRegDefsCommandArgs()
    {
        m_DataSize = 0;
    }

    //Just simplistic validity checking
    bool HasValidData()
    {
        return( true );
    };

    u32     m_DataSize;
};

//==============================================================================

struct ReadRegDataCommandArgs
{
    ReadRegDataCommandArgs( s32 ThreadId, u32 DataSize )
    {
        m_ThreadId  = ThreadId;
        m_DataSize  = DataSize;
    }

    ReadRegDataCommandArgs()
    {
        m_DataSize  = 0;
    }

    //Just simplistic validity checking
    bool HasValidData()
    {
        return( m_DataSize > 0 );
    };

    s32     m_ThreadId;
    u32     m_DataSize;
};

//==============================================================================

struct WriteRegisterCommandArgs
{
    WriteRegisterCommandArgs( s32 ThreadId, s32 RegisterId, u64 Value )
    {
        m_ThreadId      = ThreadId;
        m_RegisterId    = RegisterId;
        m_Value         = Value;
    }

    WriteRegisterCommandArgs()
    {
        m_ThreadId  = -1;
    }

    //Just simplistic validity checking ONLY.
    bool HasValidData()
    {
        return ( (m_ThreadId >= 0) && ( (m_RegisterId >= 0) || (m_RegisterId == CF_REG_IP) ) );
    };

    s32  m_ThreadId;
    s32  m_RegisterId;
    u64  m_Value;
};

//==============================================================================

enum
{
    NEARBY_STEP_RANGES  = 2,
    MAX_STEP_RANGES     = (1 + NEARBY_STEP_RANGES + NEARBY_STEP_RANGES + 2)
};

struct InstrRange
{
    u64 m_AddrLo;
    u64 m_AddrHi;
};

struct per_thread_step_info
{
        InstrRange              m_InstrRange[ MAX_STEP_RANGES ];
        u32                     m_Instr;        // Original instruction passed by Step command
        s32                     m_StepKind;
};

struct StepDebugCommandArgs
{
    StepDebugCommandArgs( s32 ThreadId, s32 StepKind, u64 IP, u32 Instr )
    {
        m_ThreadId      = ThreadId;
        m_StepKind      = StepKind;
        m_IP            = IP;
        m_Instruction   = Instr;

        for( int i = 0; i < MAX_STEP_RANGES; i++ )
        {
            m_InstrRange[i].m_AddrLo    = 0;
            m_InstrRange[i].m_AddrHi    = 0;
        }
    }

    StepDebugCommandArgs()
    {
        m_ThreadId  = -1;
    }

    //Just simplistic validity checking ONLY.
    bool HasValidData()
    {
        return ( (m_ThreadId >= 0) && (m_InstrRange[0].m_AddrLo != 0) && (m_StepKind <= 3) ) ;
    };

    s32         m_ThreadId;
    s32         m_StepKind;
    InstrRange  m_InstrRange[ MAX_STEP_RANGES ];
    u64         m_IP;
    u32         m_Instruction; // Original instruction at current IP
};

//==============================================================================

struct WriteDumpCommandArgs
{
    WriteDumpCommandArgs()
    {
        memset( this, 0, sizeof(WriteDumpCommandArgs) );
    }

    //Just simplistic validity checking ONLY.
    bool HasValidData()
    {
        return ( true );
    };

    char    m_OutputName[ TMA_MAX_DUMP_URL_SIZE ];
};

//==============================================================================

enum step_state : u16
{
    SS_NONE     = 0,            // Not stepping
    SS_BREAK    = 1,            // Stop stepping and report to TargetManager
    SS_CONT     = 5,            // F5       = Continue running without stepping
    SS_OUT      = 6,            // Shft-F11 = Continue until return address
    SS_OVER     = 10,           // F10      = Stepping over calls until outside of statement bounds
    SS_INTO     = 11,           // F11      = Stepping until outside of statement bounds
};

enum bp_state : u16
{
    BP_NONE         = 0x0000,   // No breakpoint allocated at this address
    BP_STEP         = 0x0001,   // Step     breakpoint               (Data = ThreadId)
    BP_TEMP         = 0x0002,   // Temp     breakpoint (blocks Step) (Data = ThreadId)
    BP_NORMAL       = 0x0004,   // Normal   breakpoint (blocks Temp and Step)

    BP_MODIFIED     = 0x0008,   // Modified since last GetBreakpoints request

    BP_DATA_READ    = 0x0010,   // Data breakpoint before Read
    BP_DATA_WRITE   = 0x0020,   // Data breakpoint before Write
    BP_DATA_BOTH    = 0x0030,   // Data breakpoint before Read or Write
    BP_DATA_CHANGED = 0x0024,   // Break after data changed

    BP_COUNT_EQ     = 0x0040,   // HitCount == PassCount
    BP_COUNT_GE     = 0x0080,   // HitCount >= PassCount
    BP_COUNT_MOD    = 0x00C0,   // HitCount  % PassCount == 0

    BP_COND_TRUE    = 0x0100,   // Break when Condition == true     (Data = Condition*)
    BP_COND_CHG     = 0x0200,   // Break when Condition changes     (Data = Condition*)
};

// WARNING: THERE ARE MULTIPLE COPIES OF THIS STRUCTURE WHICH MUST STAY IN SYNC

struct break_point
{
    u64         m_Address;      // Address of breakpoint
    u64         m_Data;         // Data based on m_State
    u32         m_Instruction;  // Original instruction or size of data range
    bp_state    m_State;        // Breakpoint type and flags
    step_state  m_StepState;    // Stepping state when breakpoint created
    u32         m_HitCount;     // Number of times breakpoint has been hit
    u32         m_PassCount;    // Pass count associated with hit counts

    break_point( u64 Address = 0, bp_state State = BP_NONE, step_state StepState = SS_NONE, u64 Data = 0 )
    {
        m_Address       = Address;
        m_Data          = Data;
        m_State         = State;
        m_StepState     = StepState;
        m_Instruction   = 0;
        m_HitCount      = 0;
        m_PassCount     = 0;
    }
};

//==============================================================================
}} // namespace
//==============================================================================
