﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/timesrv/detail/core/timesrv_StandardUserSystemClockCore.h>

#include <nn/timesrv/detail/core/timesrv_SteadyClockCore.h>
#include <nn/nn_StaticAssert.h>
#include <mutex>

#if NN_DETAIL_TIME_CONFIG_EXTERNAL_STEADY_CLOCK_MODEL == NN_DETAIL_TIME_CONFIG_EXTERNAL_STEADY_CLOCK_MODEL_STD_CHRONO
#include <chrono>
#include <algorithm>
#endif

namespace nn
{
namespace timesrv
{
namespace detail
{
namespace core
{

StandardUserSystemClockCore::StandardUserSystemClockCore(
    SystemClockCore* pLocalClock, SystemClockCore* pNetworkClock) NN_NOEXCEPT:
    SystemClockCore(&pLocalClock->GetSteadyClockCore()),
    m_IsAutomaticCorrectionEnabled(false),
    m_pStandardLocalSystemClockCore(pLocalClock),
    m_pStandardNetworkSystemClockCore(pNetworkClock),
    m_AutomaticCorrectionUpdatedTime{0, nn::util::InvalidUuid}
{
    NN_SDK_ASSERT_NOT_NULL(pLocalClock);
    NN_SDK_ASSERT_NOT_NULL(pNetworkClock);
}

StandardUserSystemClockCore::~StandardUserSystemClockCore() NN_NOEXCEPT
{
}

Result StandardUserSystemClockCore::GetSystemClockContextImpl(nn::time::SystemClockContext* pSystemClockContext) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pSystemClockContext);

    if(m_IsAutomaticCorrectionEnabled)
    {
        NN_RESULT_DO(SyncNetworkClockCurrentTime());
    }

    NN_RESULT_DO(m_pStandardLocalSystemClockCore->GetSystemClockContext(pSystemClockContext));
    NN_RESULT_SUCCESS;
}

Result StandardUserSystemClockCore::SetSystemClockContextImpl(const nn::time::SystemClockContext& systemClockContext) NN_NOEXCEPT
{
    NN_UNUSED(systemClockContext);
    NN_RESULT_THROW(nn::time::ResultNotImplemented());
}

Result StandardUserSystemClockCore::SyncNetworkClockCurrentTime() const NN_NOEXCEPT
{
    if(m_pStandardNetworkSystemClockCore->IsAvailable())
    {
        // SetCurrentTime ではタイミングがずれるのでコンテキストを同期する
        nn::time::SystemClockContext context;
        NN_RESULT_DO(m_pStandardNetworkSystemClockCore->GetSystemClockContext(&context));
        NN_RESULT_DO(m_pStandardLocalSystemClockCore->SetSystemClockContext(context));
    }
    NN_RESULT_SUCCESS;
}

bool StandardUserSystemClockCore::IsAutomaticCorrectionEnabled() const NN_NOEXCEPT
{
    return m_IsAutomaticCorrectionEnabled;
}

Result StandardUserSystemClockCore::SetAutomaticCorrectionEnabled(bool value) NN_NOEXCEPT
{
    if(m_IsAutomaticCorrectionEnabled != value)
    {
        // true->false, false->true の両方の補正フラグ切り替えで、
        // ネットワーク時計が補正済ならば、
        // ローカル時計をネットワーク時計と同じコンテキストにする.
        NN_RESULT_DO(SyncNetworkClockCurrentTime());
    }

    m_IsAutomaticCorrectionEnabled = value;

    NN_RESULT_SUCCESS;
}

nn::time::SteadyClockTimePoint StandardUserSystemClockCore::GetAutomaticCorrectionUpdatedTime() const NN_NOEXCEPT
{
    return m_AutomaticCorrectionUpdatedTime;
}

void StandardUserSystemClockCore::SetAutomaticCorrectionUpdatedTime(const nn::time::SteadyClockTimePoint& value) NN_NOEXCEPT
{
    m_AutomaticCorrectionUpdatedTime = value;
}

}
}
}
}

