﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_TimeSpan.h>

#include <nn/os/os_SystemEvent.h>
#include <nn/tc/tc_Types.h>
#include <nn/tcap/tcap_OperationModeTypes.h>

namespace nn { namespace tcap { namespace server {

class ContinuousHighSkinTemperatureHandler final
{
    NN_DISALLOW_COPY(ContinuousHighSkinTemperatureHandler);
    NN_DISALLOW_MOVE(ContinuousHighSkinTemperatureHandler);

private:
    bool DetectValidInput() NN_NOEXCEPT;
    bool IsSkinTemperatureUnderThreshold() NN_NOEXCEPT;
    void SubmitErrorReport(nn::Result result) NN_NOEXCEPT;

public:
    explicit ContinuousHighSkinTemperatureHandler(int milliSec) NN_NOEXCEPT
        : m_ElapsedTime(0)
        , m_ThresholdTime(nn::TimeSpan::FromMilliSeconds(milliSec))
        , m_ThresholdTemperatureMilliC(48000)
        , m_SystemEvent()
        , m_LatestTimestamp(0)
        , m_OperationMode(OperationMode_Handheld)
    {
        nn::os::CreateSystemEvent(&m_SystemEvent, nn::os::EventClearMode_ManualClear, true);
    }

    ~ContinuousHighSkinTemperatureHandler() NN_NOEXCEPT
    {
        nn::os::DestroySystemEvent(&m_SystemEvent);
    }

    void Initialize(int milliSeconds) NN_NOEXCEPT;
    void Update(nn::TimeSpan elapsedTime) NN_NOEXCEPT;
    void ClearElapsedTime() NN_NOEXCEPT;
    nn::os::SystemEventType* GetSystemEvent() NN_NOEXCEPT;
    void SetOperationMode(OperationMode operationMode) NN_NOEXCEPT;

    nn::TimeSpan m_ElapsedTime;
    nn::TimeSpan m_ThresholdTime;
    nn::tc::TemperatureMilliC m_ThresholdTemperatureMilliC;
    nn::os::SystemEventType m_SystemEvent;
    int64_t m_LatestTimestamp;
    OperationMode m_OperationMode;
};

}}} // namespace nn::tcap::server
