﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/fan/fan.h>
#include <nn/tc/tc_Types.h>

#include "tc_AdvancedThermalHandler.h"
#include "tc_SettingsHolder.h"
#include "tc_ThermalHandler.h"

namespace nn { namespace tc { namespace impl { namespace detail {

int ThermalHandler::GetNumberOfTemperatureLevel(TemperaturePolicy policy) NN_NOEXCEPT
{
    switch ( policy )
    {
    case TemperaturePolicy::Handheld:
        return m_pSettingsHolder->GetNumberOfTemperatureLevelHandheld();
    case TemperaturePolicy::Console:
        return m_pSettingsHolder->GetNumberOfTemperatureLevelConsole();
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

const TemperatureLevel& ThermalHandler::GetTemperatureLevel(TemperaturePolicy policy, int index) NN_NOEXCEPT
{
    switch ( policy )
    {
    case TemperaturePolicy::Handheld:
        return m_pSettingsHolder->GetTemperatureLevelHandheld(index);
    case TemperaturePolicy::Console:
        return m_pSettingsHolder->GetTemperatureLevelConsole(index);
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

nn::fan::RotationSpeedLevel ThermalHandler::GetRotationSpeedLevelFromTable(TemperatureMilliC temperatureMilliC) NN_NOEXCEPT
{
    TemperaturePolicy policy = TemperaturePolicy::Handheld;

    if ( m_OperatingMode == nn::tc::OperatingMode_Handheld || m_pSettingsHolder->GetRateSelect() == RateSelect::TableHandheld )
    {
        policy = TemperaturePolicy::Handheld;
    }
    else if  ( m_OperatingMode == nn::tc::OperatingMode_Console )
    {
        policy = TemperaturePolicy::Console;
    }

    auto tableSize = GetNumberOfTemperatureLevel(policy);

    // 前回からポリシーが変更されている場合は開始位置をテーブルの中で最も高い温度域にします。
    if ( policy != m_PreviousPolicy )
    {
        m_PreviousPolicy = policy;
        m_TemperatureLevelIndex = tableSize - 1;
    }

    // テーブル上で適切な温度域を遷移させます。
    for ( int count = 0; count < tableSize; count++ )
    {
        if ( temperatureMilliC < GetTemperatureLevel(policy, m_TemperatureLevelIndex).minTemperature )
        {
            if ( !(m_TemperatureLevelIndex > 0) )
            {
                break;
            }
            m_TemperatureLevelIndex--;
        }
        else if ( temperatureMilliC > GetTemperatureLevel(policy, m_TemperatureLevelIndex).maxTemperature )
        {
            if ( !(m_TemperatureLevelIndex < (tableSize - 1)) )
            {
                break;
            }
            m_TemperatureLevelIndex++;
        }
        else
        {
            break;
        }
    }

    auto level = GetTemperatureLevel(policy, m_TemperatureLevelIndex);

    // 傾きが無い場合は minSpeedLevel を回転レートとします。
    if ( level.maxSpeedLevel == level.minSpeedLevel )
    {
        return level.minSpeedLevel;
    }
    else
    {
        return ((level.maxSpeedLevel - level.minSpeedLevel)
            * (static_cast<int>(temperatureMilliC - level.minTemperature)))
            / (static_cast<int>(level.maxTemperature - level.minTemperature)) + level.minSpeedLevel;
    }
}

void ThermalHandler::Initialize(SettingsHolder* pSettingsHolder) NN_NOEXCEPT
{
    m_pSettingsHolder = pSettingsHolder;

    nn::fan::Initialize();
    nn::fan::OpenController(&m_Controller, nn::fan::FanName_Cpu);

    InitializeAdvancedPolicy();
}

void ThermalHandler::Finalize() NN_NOEXCEPT
{
    FinalizeAdvancedPolicy();

    nn::fan::CloseController(&m_Controller);
    nn::fan::Finalize();
}

void ThermalHandler::SetTemperature(TemperatureMilliC skinTemperature, TemperatureMilliC pcbTemperature, TemperatureMilliC socTemperature) NN_NOEXCEPT
{
    if ( m_FanControlEnabled )
    {
        switch ( m_PowerMode )
        {
        case PowerMode_FullAwake:
            m_SpeedLevelFromTable = GetRotationSpeedLevelFromTable(skinTemperature);
            m_SpeedLevelFromPrev = GetRotationSpeedLevelFromAdvancedPolicy(skinTemperature, pcbTemperature, socTemperature);
            break;
        case PowerMode_MinimumAwake:
        case PowerMode_SleepReady:
            m_SpeedLevelFromTable = 0;
            m_SpeedLevelFromPrev = 0;
            break;
        default:
            m_SpeedLevelFromTable = 0;
            m_SpeedLevelFromPrev = 0;
            break;
        }

        switch ( m_pSettingsHolder->GetRateSelect() )
        {
        case RateSelect::Table:
        case RateSelect::TableHandheld:
            nn::fan::SetRotationSpeedLevel(&m_Controller, m_SpeedLevelFromTable);
            break;
        case RateSelect::Prev:
            nn::fan::SetRotationSpeedLevel(&m_Controller, m_SpeedLevelFromPrev);
            break;
        case RateSelect::Both:
            nn::fan::SetRotationSpeedLevel(&m_Controller, std::max(m_SpeedLevelFromTable, m_SpeedLevelFromPrev));
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

void ThermalHandler::SetOperatingMode(OperatingMode operatingMode) NN_NOEXCEPT
{
    m_OperatingMode = operatingMode;

    SetOperatingModeToAdvancedPolicy(operatingMode);
}

void ThermalHandler::SetPowerMode(PowerMode powerMode) NN_NOEXCEPT
{
    m_PowerMode = powerMode;

    SetPowerModeToAdvancedPolicy(powerMode);
}

nn::fan::RotationSpeedLevel ThermalHandler::GetRotationSpeedLevel(RateSelect rateSelect) NN_NOEXCEPT
{
    nn::fan::RotationSpeedLevel speedLevel = 0;

    switch ( rateSelect )
    {
    case RateSelect::Table:
    case RateSelect::TableHandheld:
        speedLevel = m_SpeedLevelFromTable;
        break;
    case RateSelect::Prev:
        speedLevel = m_SpeedLevelFromPrev;
        break;
    case RateSelect::Both:
        speedLevel = std::max(m_SpeedLevelFromTable, m_SpeedLevelFromPrev);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    return speedLevel;
}

nn::fan::RotationSpeedLevel ThermalHandler::GetFanRotationSpeedLevel() NN_NOEXCEPT
{
    return nn::fan::GetRotationSpeedLevel(&m_Controller);
}

nn::fan::RotationSpeedLevel ThermalHandler::GetFanActualRotationSpeedLevel() NN_NOEXCEPT
{
    return nn::fan::GetActualRotationSpeedLevel(&m_Controller);
}

void ThermalHandler::SetFanControlEnabled(bool enabled) NN_NOEXCEPT
{
    m_FanControlEnabled = enabled;
}

bool ThermalHandler::GetFanControlEnabled() NN_NOEXCEPT
{
    return m_FanControlEnabled;
}

}}}} // namespace nn::tc::impl::detail
