﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_TimeSpan.h>

#include <nn/os.h>
#include <nn/tc/tc_Types.h>
#include <nn/tc/impl/tc_PowerModeTypes.h>

#include "tc_LocationAccessor.h"
#include "tc_SettingHandler.h"
#include "tc_SettingsHolder.h"
#include "tc_ThermalHandler.h"

namespace nn { namespace tc { namespace impl { namespace detail {

class ThermalCounter
{
public:
    ThermalCounter(TemperatureMilliC temperature, nn::TimeSpan timeSpan) NN_NOEXCEPT
    : m_ThermalThreshold(temperature),
      m_TimeThreshold(timeSpan),
      m_StartTime(0)
    {
    }

    bool Update(TemperatureMilliC temperature) NN_NOEXCEPT;
    void Clear() NN_NOEXCEPT;

private:
    TemperatureMilliC m_ThermalThreshold;
    nn::TimeSpan m_TimeThreshold;
    nn::TimeSpan m_StartTime;
};

class IirFilter
{
public:
    IirFilter() NN_NOEXCEPT
    : m_Gain(DefaultGain),
      m_TemperatureMilliC(DefaultTemperatureMilliC)
    {
    }

    TemperatureMilliC Update(TemperatureMilliC rawTemperatureMilliC)
    {
        m_TemperatureMilliC = static_cast<TemperatureMilliC>((rawTemperatureMilliC - m_TemperatureMilliC) * m_Gain + m_TemperatureMilliC);
        return m_TemperatureMilliC;
    }

    void SetGain(double gain) NN_NOEXCEPT
    {
        m_Gain = gain;
    }

private:
    const double DefaultGain = 0.37;
    const TemperatureMilliC DefaultTemperatureMilliC = 35000;

    double m_Gain;
    TemperatureMilliC m_TemperatureMilliC;
};

class ThermalCoordinator
{
    NN_DISALLOW_COPY(ThermalCoordinator);
    NN_DISALLOW_MOVE(ThermalCoordinator);

public:
    ThermalCoordinator() NN_NOEXCEPT;

    void Initialize() NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;
    void Update() NN_NOEXCEPT;

    nn::Result SetOperatingMode(OperatingMode operatingMode) NN_NOEXCEPT;
    nn::Result GetEventPtr(nn::os::SystemEventType** pOutEventPtr, EventTarget target) NN_NOEXCEPT;
    nn::Result IsActionAllowed(bool* pOutPossible, Action action) NN_NOEXCEPT;
    nn::Result SetVirtualTemperatureEnabled(Location location, bool enabled) NN_NOEXCEPT;
    nn::Result SetVirtualTemperature(Location location, TemperatureMilliC temperature) NN_NOEXCEPT;
    nn::Result SetFanControlEnabled(bool enabled) NN_NOEXCEPT;
    nn::Result GetFanControlEnabled(bool* pOutEnabled) NN_NOEXCEPT;
    void SetPowerMode(PowerMode powerMode) NN_NOEXCEPT;
    int GetSkinTemperatureMilliC() NN_NOEXCEPT;

private:
    void DumpThresholds() NN_NOEXCEPT;
    void EnterSleep(const char* pFormat, ...) NN_NOEXCEPT;
    void UpdateErrorReportContext(TemperatureMilliC socTemperature, TemperatureMilliC pcbTemperature) NN_NOEXCEPT;
    void SubmitErrorReport(Result result, TemperatureMilliC socTemperature, TemperatureMilliC pcbTemperature) NN_NOEXCEPT;

private:
    LocationAccessor m_LocationPcb; //!< 温度センサー IC の内部温度関する処理を担当するオブジェクト。
    LocationAccessor m_LocationSoc; //!< SoC の内部温度に関する処理を担当するオブジェクト。
    IirFilter m_IirFilterPcb;
    IirFilter m_IirFilterSoc;
    ThermalHandler m_ThermalHandler; //!< 温度をファンなどを用いて制御する処理を担当するオブジェクト。
    SettingHandler m_SettingHandler; //!< 温度に応じて設定を更新するモジュールに通知を行うオブジェクト。
    OperatingMode m_OperatingMode;
    nn::os::SystemEventType m_AbnormalTemperatureEvent;
    int m_FailureCount;
    ThermalCounter m_CounterHandheldMiddle;
    ThermalCounter m_CounterHandheldHigh;
    TskinSelect m_TskinSelect;
    bool m_HasHighTemperatureBeenDetected;
    SettingsHolder m_SettingsHolder;
    TemperatureMilliC m_SkinTemperatureMilliC;
};

ThermalCoordinator& GetThermalCoordinator() NN_NOEXCEPT;

}}}} // namespace nn::tc::impl::detail
