﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   ソフトウェアキーボード呼び出しに関する公開ヘッダ
 */

#include <nn/swkbd/swkbd_InlineKeyboardApi.h>

#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>

#include <new>

#include <nn/os/os_MemoryHeapCommon.h>

#include <nn/util/util_BitUtil.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_CharacterEncodingResult.h>
#include <nn/util/util_StringUtil.h>

#include <nn/swkbd/swkbd_Result.h>

#include <nn/swkbd/detail/swkbd_InlineKeyboardApiImpl.h>
#include <algorithm>

namespace nn { namespace swkbd {


// 文字列変換を行う内部クラス
class Converter
{
public:
    //------------------------------------------------------------------------------
    // UTF-8の文字列をUTF-16に変換します。
    void ConvertUtf8ToUtf16( void* dst, int dstLength, const char* src, int srcLength ) NN_NOEXCEPT
    {
        nn::util::CharacterEncodingResult encoding_result;
        encoding_result = nn::util::ConvertStringUtf8ToUtf16Native( (uint16_t*)dst, dstLength, src, srcLength );
        if( encoding_result != nn::util::CharacterEncodingResult_Success )
        {
            if( encoding_result == nn::util::CharacterEncodingResult_InsufficientLength )
            {
                NN_ABORT( "Insufficient Length." );
            }
            else if( encoding_result == nn::util::CharacterEncodingResult_InvalidFormat )
            {
                NN_ABORT( "Invalid Format." );
            }
        }
    }


    //------------------------------------------------------------------------------
    //! @brief UTF-8の文字列をUTF-16に変換します。終端文字まで変換されます。
    //! @param[out] dst 変換先の配列です。終端文字が付加されます。
    //! @param[in] dstLength 変換先の配列の長さです。
    //! @param[in] src 変換元の配列です。
    void ConvertUtf8ToUtf16( void* dst, int dstLength, const char* src ) NN_NOEXCEPT
    {
        nn::util::CharacterEncodingResult encoding_result;
        encoding_result = nn::util::ConvertStringUtf8ToUtf16Native( (uint16_t*)dst, dstLength, src );
        if( encoding_result != nn::util::CharacterEncodingResult_Success )
        {
            if( encoding_result == nn::util::CharacterEncodingResult_InsufficientLength )
            {
                NN_ABORT( "Insufficient Length." );
            }
            else if( encoding_result == nn::util::CharacterEncodingResult_InvalidFormat )
            {
                NN_ABORT( "Invalid Format." );
            }
        }
    }


    //------------------------------------------------------------------------------
    // UTF-8の文字列をUTF-16に変換するために必要な変換先の長さを返します
    int GetLengthOfConvertedStringUtf8ToUtf16( const char* pSrc ) NN_NOEXCEPT
    {
        int src_length = 0;
        while( NN_STATIC_CONDITION( true ) )
        {
            if( pSrc[ src_length ] == 0 )
            {
                break;
            }
            ++src_length;
        }

        int str_length = 0;

        nn::util::CharacterEncodingResult encoding_result = nn::util::GetLengthOfConvertedStringUtf8ToUtf16Native(
            &str_length, pSrc, src_length );

        if( encoding_result != nn::util::CharacterEncodingResult_Success )
        {
            if( encoding_result == nn::util::CharacterEncodingResult_InvalidFormat )
            {
                NN_ABORT( "Invalid Format." );
            }
        }

        return str_length;
    }
};



//---------------------------------------------------------------------------
//
void
InputText::Set( const char16_t* pStr )
{
    // 未指定の場合は初期化する
    if( pStr == nullptr )
    {
        // 先頭にヌル終端をつけて終了する
        text[ 0 ] = 0;
        // 関連して文字列の長さ、カーソル位置も初期化する
        textLength = 0;
        _reserve = 0;
        return;
    }

    // 文字列の長さを取得する
    int str_length = nn::util::Strnlen( pStr, TextMaxLength + 1 );

    // 最大値より大きな文字数を指定していたら、エラーを返す
    NN_SDK_ASSERT( ( str_length > 0 && str_length < TextMaxLength + 1 ), "[swkbd] string size is out of range.\n" );

    // 文字数超え対策
    if( str_length >= TextMaxLength + 1 )
    {
        str_length = TextMaxLength;
    }

    // バッファのコピー
    std::memcpy( text, pStr, str_length * sizeof( char16_t ) );

    // 末尾にヌル終端をつける
    text[ str_length ] = 0;
}


//---------------------------------------------------------------------------
//
void
InputText::SetUtf8( const char* pStr )
{
    // 未指定の場合は初期化する
    if( pStr == nullptr )
    {
        // 先頭にヌル終端をつけて終了する
        text[ 0 ] = 0;
        // 関連して文字列の長さ、カーソル位置も初期化する
        textLength = 0;
        _reserve = 0;
        return;
    }

    // 入力文字列の長さチェック
    Converter converter;
    NN_SDK_ASSERT( ((converter.GetLengthOfConvertedStringUtf8ToUtf16(pStr) > 0                ) &&
                    (converter.GetLengthOfConvertedStringUtf8ToUtf16(pStr) < TextMaxLength + 1)),
                    "[swkbd] string size is out of range.\n" );

    // 文字列をバッファに追加
    converter.ConvertUtf8ToUtf16( text, NN_ARRAY_SIZE(text), pStr );
}


//---------------------------------------------------------------------------
//
size_t
UserWordInfo::GetRequiredWorkBufferSize( int32_t userWordNum )
{
    // 最大値より大きな単語数を指定していたら、エラーを返す
    NN_SDK_ASSERT( ( userWordNum >= 0 && userWordNum <= UserWordMax ),
        "[swkbd] user word size is out of range.\n" );

    if( userWordNum >= 0 && userWordNum <= UserWordMax )
    {
        size_t required_size = 0;
        // コマンド + (単語総数) + ユーザ単語数が、要求するサイズ
        required_size += sizeof( RequestCommand );
        required_size += sizeof( int32_t ); // userWordNum
        required_size += sizeof( UserWord ) * userWordNum;

        // nn::os::MemoryPageSize でアライメント
        required_size = nn::util::align_up( required_size, nn::os::MemoryPageSize );

        return required_size;
    }
    else
    {
        // 範囲外の指定だった場合のケア
        return 0;
    }
}


//---------------------------------------------------------------------------
//
InlineKeyboard::InlineKeyboard() NN_NOEXCEPT
: m_pImpl( nullptr )
, m_pWorkBuf( nullptr )
{
}

//---------------------------------------------------------------------------
//
InlineKeyboard::~InlineKeyboard() NN_NOEXCEPT
{
}


size_t
InlineKeyboard::GetRequiredWorkBufferSize() NN_NOEXCEPT
{
    return sizeof( detail::InlineKeyboardImpl );
}


//---------------------------------------------------------------------------
//
bool
InlineKeyboard::Initialize( void* pWorkBuf ) NN_NOEXCEPT
{
    // バッファ未指定であれば、何もしない
    NN_SDK_REQUIRES( pWorkBuf != nullptr );
    if( pWorkBuf == nullptr )
    {
        return false;
    }

    m_pWorkBuf = pWorkBuf;

    // バッファの確保
    m_pImpl = new( m_pWorkBuf ) detail::InlineKeyboardImpl();

    // 起動パラメータの設定
    InitializeArg arg;
    arg.isOverlayLayer = false; // アプリ側に描画してもらう
    // windowOriginMode は使わなくなったが、プロトコル互換性のために残す

    return m_pImpl->Initialize( arg );
}

//---------------------------------------------------------------------------
//
void
InlineKeyboard::Finalize() NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return;
    }

    m_pImpl->Finalize();

    // m_pImpl インスタンスを削除
    m_pImpl->~InlineKeyboardImpl();
}


//---------------------------------------------------------------------------
//
bool
InlineKeyboard::Launch() NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return false;
    }

    return m_pImpl->Launch();
}


//---------------------------------------------------------------------------
//
void
InlineKeyboard::SetVolume( float volume ) NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return;
    }

    m_pImpl->SetVolume( volume );
}


//---------------------------------------------------------------------------
//
void
InlineKeyboard::Appear( const AppearArg& arg ) NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return;
    }

    m_pImpl->Appear( arg );
}

//---------------------------------------------------------------------------
//
void
InlineKeyboard::Disappear() NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return;
    }

    m_pImpl->Disappear();
}

//---------------------------------------------------------------------------
//
void
InlineKeyboard::SetInputText( const InputText& info ) NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return;
    }

    m_pImpl->SetInputText( info );
}

//---------------------------------------------------------------------------
//
void
InlineKeyboard::SetCursorPos( int32_t cursorPos ) NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return;
    }


    m_pImpl->SetCursorPos( cursorPos );
}

//---------------------------------------------------------------------------
//
bool
InlineKeyboard::SetUserWordInfo( const UserWordInfo& info ) NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return false;
    }

    return m_pImpl->SetUserWordInfo( info );
}

//---------------------------------------------------------------------------
//
bool
InlineKeyboard::UnsetUserWordInfo() NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return false;
    }

    return m_pImpl->UnsetUserWordInfo();
}

//---------------------------------------------------------------------------
//
void
InlineKeyboard::SetUtf8Mode( bool isUseUtf8 ) NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return;
    }

    m_pImpl->SetUtf8Mode( isUseUtf8 );
}

//---------------------------------------------------------------------------
//
bool
InlineKeyboard::SetCustomizeDic( const void* pBuffer, size_t size, const CustomizeDicInfo& info ) NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return false;
    }

    return m_pImpl->SetCustomizeDic( pBuffer, size, info );
}

//---------------------------------------------------------------------------
//
bool
InlineKeyboard::UnsetCustomizeDic() NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return false;
    }

    return m_pImpl->UnsetCustomizeDic();
}

//---------------------------------------------------------------------------
//
void
InlineKeyboard::SetKeytopAlpha( float alpha ) NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return;
    }

    m_pImpl->SetKeytopBgAlpha( alpha );
}

//---------------------------------------------------------------------------
//
void
InlineKeyboard::SetFooterAlpha( float alpha ) NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return;
    }

    m_pImpl->SetFooterBgAlpha( alpha );
}

//---------------------------------------------------------------------------
//
void
InlineKeyboard::SetKeytopScale( float scale ) NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return;
    }

    m_pImpl->SetKeytopScale( scale, scale );
    // SWHUID-10402 での指示に従う。
    float balloonScale = std::min( scale + 0.15f, 1.f );
    m_pImpl->SetBalloonScale( balloonScale );
}

//---------------------------------------------------------------------------
//
void
InlineKeyboard::SetKeytopTranslate( float translateX, float translateY ) NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return;
    }

    m_pImpl->SetKeytopTranslate( translateX, translateY );
}

//---------------------------------------------------------------------------
void
InlineKeyboard::SetWindowMode( WindowMode mode ) NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return;
    }

    bool footerScalable = true;
    bool showLowerFrame = false;

    switch ( mode )
    {
    case WindowMode_Single:
        // デフォルト値のまま
        break;
    case WindowMode_Floating:
        footerScalable = false;
        showLowerFrame = true;
        break;
    case WindowMode_AdjacentFloating:
        footerScalable = false;
        showLowerFrame = false;
        break;
    default:
        NN_SDK_ASSERT(false, "Invalid Parameter");
    }

    m_pImpl->SetKeytopAsFloating( showLowerFrame );
    m_pImpl->SetFooterScalable( footerScalable );
}

//---------------------------------------------------------------------------
//
void
InlineKeyboard::SetTouchEnabled( bool enabled ) NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return;
    }

    m_pImpl->SetDisableTouch( ! enabled );
}

//---------------------------------------------------------------------------
//
void
InlineKeyboard::SetUsbKeyboardEnabled( bool enabled ) NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return;
    }

    m_pImpl->SetDisableUSBKeyboard( !enabled );
}

//---------------------------------------------------------------------------
//
State
InlineKeyboard::Calc() NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return State::State_None;
    }

    m_pImpl->NormalizeParams();
    return m_pImpl->Calc();
}


//---------------------------------------------------------------------------
//
int
InlineKeyboard::GetMaxHeight() NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return 0;
    }

    return m_pImpl->GetMaxHeight();
}


//---------------------------------------------------------------------------
//
bool
InlineKeyboard::IsUsedTouchPointByKeyboard( int32_t x, int32_t y ) NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return false;
    }

    return m_pImpl->IsUsedTouchPointByKeyboard( x, y );
}


//---------------------------------------------------------------------------
//
void
InlineKeyboard::SetFinishedInitializeCallback( FinishedInitializeCallback pCallback ) NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return;
    }

    m_pImpl->SetFinishedInitializeCallback( pCallback );
}


//---------------------------------------------------------------------------
//
void
InlineKeyboard::SetFinishedKeyboardCallback( FinishedKeyboardCallback pCallback ) NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return;
    }

    m_pImpl->SetFinishedKeyboardCallback( pCallback );
}


//---------------------------------------------------------------------------
//
void
InlineKeyboard::SetChangedStringCallback( ChangedStringCallback pCallback ) NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return;
    }

    m_pImpl->SetChangedStringCallback( pCallback );
}


//---------------------------------------------------------------------------
//
void
InlineKeyboard::SetChangedStringCallbackUtf8( ChangedStringCallbackUtf8 pCallback ) NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return;
    }

    m_pImpl->SetChangedStringCallbackUtf8( pCallback );
}


//---------------------------------------------------------------------------
//
void
InlineKeyboard::SetMovedCursorCallback( MovedCursorCallback pCallback ) NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return;
    }

    m_pImpl->SetMovedCursorCallback( pCallback );
}


//---------------------------------------------------------------------------
//
void
InlineKeyboard::SetMovedCursorCallbackUtf8( MovedCursorCallbackUtf8 pCallback ) NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return;
    }

    m_pImpl->SetMovedCursorCallbackUtf8( pCallback );
}


//---------------------------------------------------------------------------
//
void
InlineKeyboard::SetDecidedEnterCallback( DecidedEnterCallback pCallback ) NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return;
    }

    m_pImpl->SetDecidedEnterCallback( pCallback );
}


//---------------------------------------------------------------------------
//
void
InlineKeyboard::SetDecidedEnterCallbackUtf8( DecidedEnterCallbackUtf8 pCallback ) NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return;
    }

    m_pImpl->SetDecidedEnterCallbackUtf8( pCallback );
}


//---------------------------------------------------------------------------
//
void
InlineKeyboard::SetDecidedCancelCallback( DecidedCancelCallback pCallback ) NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return;
    }

    m_pImpl->SetDecidedCancelCallback( pCallback );
}


//---------------------------------------------------------------------------
//
void
InlineKeyboard::SetReleasedUserWordInfoCallback( ReleasedUserWordInfoCallback pCallback ) NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return;
    }

    m_pImpl->SetReleasedUserWordInfoCallback( pCallback );
}


//---------------------------------------------------------------------------
//
void
InlineKeyboard::GetWindowSize( int* pWidth, int* pHeight ) const NN_NOEXCEPT
{
    // 未初期化時でも呼んでも良い API なため、static な API 経由で行っている

    detail::InlineKeyboardImpl::GetWindowSize( pWidth, pHeight );
}


//---------------------------------------------------------------------------
int
InlineKeyboard::GetTouchRectangles( Rect* primary, Rect* secondary ) const NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return 0;
    }

    return m_pImpl->GetTouchRectangles( primary, secondary );
}

//---------------------------------------------------------------------------
//
void
InlineKeyboard::GetImageMemoryRequirement( size_t* pOutRequiredSize,
    size_t* pOutRequiredAlignment ) NN_NOEXCEPT
{
    // 未初期化時でも呼んでも良い API なため、static な API 経由で行っている

    detail::InlineKeyboardImpl::GetImageMemoryRequirement(
        pOutRequiredSize, pOutRequiredAlignment );
}


//---------------------------------------------------------------------------
//
bool
InlineKeyboard::GetImage( void* pBuffer, size_t bufferSize ) NN_NOEXCEPT
{
    // m_pImpl インスタンスが存在しないならなにもしない
    NN_SDK_REQUIRES( m_pImpl != nullptr );
    if( m_pImpl == nullptr )
    {
        return false;
    }

    return m_pImpl->GetImage( pBuffer, bufferSize );
}


}} // namespace nn::swkbd

