﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <new>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/ssl/detail/ssl_Build.h>
#include <nn/ssl/ssl_Types.h>
#include <nn/ssl/ssl_Api.h>
#include <nn/ssl/ssl_Context.h>
#include <nn/ssl/ssl_Connection.h>
#include <nn/ssl/ssl_Result.h>

#include <nnc/nn_Result.h>
#include <nnc/result/result_CConverter.h>
#include <nnc/ssl/ssl_Types.h>
#include <nnc/ssl/ssl_Context.h>
#include <nnc/ssl/ssl_Connection.h>


#define RETURN_UPON_INVALID_CLIENT_POINTER(pClientContext)                                         \
do{                                                                                                \
    NN_SDK_REQUIRES_NOT_NULL(pClientContext);                                                      \
    if(pClientContext == nullptr)                                                                  \
    {                                                                                              \
        return nn::result::ConvertToC(nn::ssl::ResultInvalidPointer());                            \
    }                                                                                              \
}while(NN_STATIC_CONDITION(false))

#ifdef __cplusplus
extern "C"
{
#endif // __cplusplus

// -------------------------------------------------------------------------------------------------
// API
// -------------------------------------------------------------------------------------------------
nnResult nnsslInitialize()
{
    nn::Result result = nn::ssl::Initialize();
    return nn::result::ConvertToC(result);
}

nnResult nnsslInitializeWithConcurrencyLimit(uint32_t concurLimit)
{
    nn::Result result = nn::ssl::Initialize(concurLimit);
    return nn::result::ConvertToC(result);
}

nnResult nnsslFinalize()
{
    nn::Result result = nn::ssl::Finalize();
    return nn::result::ConvertToC(result);
}

// -------------------------------------------------------------------------------------------------
// nn::ssl::Context
// -------------------------------------------------------------------------------------------------
nnResult nnsslContextCreate(
    nnsslSslContextType* pSslContext,
    nnsslContextSslVersion version)
{
    RETURN_UPON_INVALID_CLIENT_POINTER(pSslContext);

    nn::Result result = nn::ResultSuccess();
    do{
    // Allocate object and run constructor of nn::ssl::Context
    nn::ssl::Context* pObject = new (pSslContext) nn::ssl::Context();
    if(pObject == nullptr)
    {
        result = nn::ssl::ResultInsufficientMemory();
        break;
    }

    result = pObject->Create(static_cast<nn::ssl::Context::SslVersion>(version));
    }while(NN_STATIC_CONDITION(false));
    return nn::result::ConvertToC(result);
}

nnResult nnsslContextDestroy(nnsslSslContextType* pSslContext)
{
    RETURN_UPON_INVALID_CLIENT_POINTER(pSslContext);

    nn::Result result = nn::ResultSuccess();
    do{
    nn::ssl::Context* pContext = reinterpret_cast<nn::ssl::Context*>(pSslContext);
    result = pContext->Destroy();
    if(result.IsFailure())
    {
        break;
    }

    // Manually call destructor here because the object of nn::ssl::Context were allocated
    // in nnsslContextCreate manually hence its destructor need to be called explicitly
    pContext->~Context();

    }while(NN_STATIC_CONDITION(false));
    return nn::result::ConvertToC(result);
}

nnResult nnsslContextGetContextId(
    nnsslSslContextType* pInSslContext,
    nnsslSslContextId* pOutContextId)
{
    RETURN_UPON_INVALID_CLIENT_POINTER(pInSslContext);
    RETURN_UPON_INVALID_CLIENT_POINTER(pOutContextId);

    nn::Result result = nn::ResultSuccess();
    do{
    nn::ssl::Context* pContext = reinterpret_cast<nn::ssl::Context*>(pInSslContext);
    result = pContext->GetContextId(reinterpret_cast<nn::ssl::SslContextId*>(pOutContextId));
    }while(NN_STATIC_CONDITION(false));
    return nn::result::ConvertToC(result);
}


// -------------------------------------------------------------------------------------------------
// nn::ssl::Connection
// -------------------------------------------------------------------------------------------------
nnResult nnsslConnectionCreate(
    nnsslSslConnectionType* pSslConnection,
    nnsslSslContextType* pInClientContext)
{
    RETURN_UPON_INVALID_CLIENT_POINTER(pSslConnection);
    RETURN_UPON_INVALID_CLIENT_POINTER(pInClientContext);

    nn::Result result = nn::ResultSuccess();
    do{
    // Allocate object and run constructor of nn::ssl::Connection
    nn::ssl::Connection* pObject = new (pSslConnection) nn::ssl::Connection();
    if(pObject == nullptr)
    {
        result = nn::ssl::ResultInsufficientMemory();
        break;
    }

    result = pObject->Create(reinterpret_cast<nn::ssl::Context*>(pInClientContext));
    }while(NN_STATIC_CONDITION(false));
    return nn::result::ConvertToC(result);
}

nnResult nnsslConnectionDestroy(nnsslSslConnectionType* pSslConnection)
{
    RETURN_UPON_INVALID_CLIENT_POINTER(pSslConnection);

    nn::Result result = nn::ResultSuccess();
    do{
    nn::ssl::Connection* pConnection = reinterpret_cast<nn::ssl::Connection*>(pSslConnection);
    result = pConnection->Destroy();
    if(result.IsFailure())
    {
        break;
    }

    // Manually call destructor here because the object of nn::ssl::Connection were allocated
    // in nnsslConnectionCreate manually hence its destructor need to be called explicitly
    pConnection->~Connection();
    }while(NN_STATIC_CONDITION(false));
    return nn::result::ConvertToC(result);
}


nnResult nnsslConnectionSetSocketDescriptor(
    nnsslSslConnectionType* pSslConnection,
    int socketDescriptor)
{
    RETURN_UPON_INVALID_CLIENT_POINTER(pSslConnection);

    nn::Result result = nn::ResultSuccess();
    do{
    nn::ssl::Connection* pConnection = reinterpret_cast<nn::ssl::Connection*>(pSslConnection);
    result = pConnection->SetSocketDescriptor(socketDescriptor);
    }while(NN_STATIC_CONDITION(false));
    return nn::result::ConvertToC(result);
}

nnResult nnsslConnectionSetHostName(
    nnsslSslConnectionType* pSslConnection,
    const char* pInHostName,
    uint32_t hostNameLength)
{
    RETURN_UPON_INVALID_CLIENT_POINTER(pSslConnection);

    nn::Result result = nn::ResultSuccess();
    do{
    nn::ssl::Connection* pConnection = reinterpret_cast<nn::ssl::Connection*>(pSslConnection);
    result = pConnection->SetHostName(pInHostName, hostNameLength);
    }while(NN_STATIC_CONDITION(false));
    return nn::result::ConvertToC(result);
}

nnResult nnsslConnectionSetVerifyOption(
    nnsslSslConnectionType* pSslConnection,
    nnsslConnectionVerifyOption optionValue)
{
    RETURN_UPON_INVALID_CLIENT_POINTER(pSslConnection);

    nn::Result result = nn::ResultSuccess();
    do{
    nn::ssl::Connection* pConnection = reinterpret_cast<nn::ssl::Connection*>(pSslConnection);
    result = pConnection->SetVerifyOption(static_cast<nn::ssl::Connection::VerifyOption>(optionValue));
    }while(NN_STATIC_CONDITION(false));
    return nn::result::ConvertToC(result);
}

nnResult nnsslConnectionSetServerCertBuffer(
    nnsslSslConnectionType* pSslConnection,
    char* pInBuffer,
    uint32_t bufferLength)
{
    RETURN_UPON_INVALID_CLIENT_POINTER(pSslConnection);

    nn::Result result = nn::ResultSuccess();
    do{
    nn::ssl::Connection* pConnection = reinterpret_cast<nn::ssl::Connection*>(pSslConnection);
    result = pConnection->SetServerCertBuffer(pInBuffer, bufferLength);
    }while(NN_STATIC_CONDITION(false));
    return nn::result::ConvertToC(result);
}

nnResult nnsslConnectionSetIoMode(
    nnsslSslConnectionType* pSslConnection,
    nnsslConnectionIoMode mode)
{
    RETURN_UPON_INVALID_CLIENT_POINTER(pSslConnection);

    nn::Result result = nn::ResultSuccess();
    do{
    nn::ssl::Connection* pConnection = reinterpret_cast<nn::ssl::Connection*>(pSslConnection);
    result = pConnection->SetIoMode(static_cast<nn::ssl::Connection::IoMode>(mode));
    }while(NN_STATIC_CONDITION(false));
    return nn::result::ConvertToC(result);
}

nnResult nnsslConnectionSetSessionCacheMode(
    nnsslSslConnectionType* pSslConnection,
    nnsslConnectionSessionCacheMode mode)
{
    RETURN_UPON_INVALID_CLIENT_POINTER(pSslConnection);

    nn::Result result = nn::ResultSuccess();
    do{
    nn::ssl::Connection* pConnection = reinterpret_cast<nn::ssl::Connection*>(pSslConnection);
    result = pConnection->SetSessionCacheMode(static_cast<nn::ssl::Connection::SessionCacheMode>(mode));
    }while(NN_STATIC_CONDITION(false));
    return nn::result::ConvertToC(result);
}

nnResult nnsslConnectionGetSocketDescriptor(
    nnsslSslConnectionType* pSslConnection,
    int* pOutSocketDescriptor)
{
    RETURN_UPON_INVALID_CLIENT_POINTER(pSslConnection);

    nn::Result result = nn::ResultSuccess();
    do{
    nn::ssl::Connection* pConnection = reinterpret_cast<nn::ssl::Connection*>(pSslConnection);
    result = pConnection->GetSocketDescriptor(pOutSocketDescriptor);
    }while(NN_STATIC_CONDITION(false));
    return nn::result::ConvertToC(result);
}

nnResult nnsslConnectionGetHostName(
    nnsslSslConnectionType* pSslConnection,
    char* pOutHostName,
    uint32_t* pOutHostNameLength,
    uint32_t hostNameBufferLength)
{
    RETURN_UPON_INVALID_CLIENT_POINTER(pSslConnection);

    nn::Result result = nn::ResultSuccess();
    do{
    nn::ssl::Connection* pConnection = reinterpret_cast<nn::ssl::Connection*>(pSslConnection);
    result = pConnection->GetHostName(pOutHostName, pOutHostNameLength, hostNameBufferLength);
    }while(NN_STATIC_CONDITION(false));
    return nn::result::ConvertToC(result);

}

nnResult nnsslConnectionGetVerifyOption(
    nnsslSslConnectionType* pSslConnection,
    nnsslConnectionVerifyOption* pOutOptionValue)
{
    RETURN_UPON_INVALID_CLIENT_POINTER(pSslConnection);

    nn::Result result = nn::ResultSuccess();
    do{
    nn::ssl::Connection* pConnection = reinterpret_cast<nn::ssl::Connection*>(pSslConnection);
    result = pConnection->GetVerifyOption(reinterpret_cast<nn::ssl::Connection::VerifyOption*>(pOutOptionValue));
    }while(NN_STATIC_CONDITION(false));
    return nn::result::ConvertToC(result);
}

nnResult nnsslConnectionGetIoMode(
    nnsslSslConnectionType* pSslConnection,
    nnsslConnectionIoMode* pOutMode)
{
    RETURN_UPON_INVALID_CLIENT_POINTER(pSslConnection);

    nn::Result result = nn::ResultSuccess();
    do{
    nn::ssl::Connection* pConnection = reinterpret_cast<nn::ssl::Connection*>(pSslConnection);
    result = pConnection->GetIoMode(reinterpret_cast<nn::ssl::Connection::IoMode*>(pOutMode));
    }while(NN_STATIC_CONDITION(false));
    return nn::result::ConvertToC(result);
}

nnResult nnsslConnectionDoHandshake(nnsslSslConnectionType* pSslConnection)
{
    RETURN_UPON_INVALID_CLIENT_POINTER(pSslConnection);

    nn::Result result = nn::ResultSuccess();
    do{
    nn::ssl::Connection* pConnection = reinterpret_cast<nn::ssl::Connection*>(pSslConnection);
    result = pConnection->DoHandshake();
    }while(NN_STATIC_CONDITION(false));
    return nn::result::ConvertToC(result);
}

nnResult nnsslConnectionDoHandshakeWithCertBuffer(
    nnsslSslConnectionType* pSslConnection,
    uint32_t* pOutServerCertSize,
    uint32_t* pOutServerCertCount)
{
    RETURN_UPON_INVALID_CLIENT_POINTER(pSslConnection);

    nn::Result result = nn::ResultSuccess();
    do{
    nn::ssl::Connection* pConnection = reinterpret_cast<nn::ssl::Connection*>(pSslConnection);
    result = pConnection->DoHandshake(pOutServerCertSize, pOutServerCertCount);
    }while(NN_STATIC_CONDITION(false));
    return nn::result::ConvertToC(result);
}

nnResult nnsslConnectionDoHandshakeWithBuffer(
    nnsslSslConnectionType* pSslConnection,
    uint32_t* pOutCertSize,
    uint32_t* pOutServerCertCount,
    char* pOutBuffer,
    uint32_t bufferLen)
{
    RETURN_UPON_INVALID_CLIENT_POINTER(pSslConnection);

    nn::Result result = nn::ResultSuccess();
    do {
        nn::ssl::Connection *pConnection = reinterpret_cast<nn::ssl::Connection *>(pSslConnection);
        result = pConnection->DoHandshake(pOutCertSize, pOutServerCertCount, pOutBuffer, bufferLen);
    } while (NN_STATIC_CONDITION(false));

    return nn::result::ConvertToC(result);
}

nnResult nnsslConnectionGetServerCertDetail(
    nnsslSslConnectionType* pSslConnection,
    nnsslConnectionServerCertDetailType *pOutCertDetail,
    const char *pInBuffer,
    uint32_t index)
{
    RETURN_UPON_INVALID_CLIENT_POINTER(pSslConnection);
    RETURN_UPON_INVALID_CLIENT_POINTER(pOutCertDetail);
    RETURN_UPON_INVALID_CLIENT_POINTER(pInBuffer);

    nn::Result result = nn::ResultSuccess();

    do {
        nn::ssl::Connection::ServerCertDetail *pDetail =
            reinterpret_cast<nn::ssl::Connection::ServerCertDetail *>(pOutCertDetail);
        nn::ssl::Connection *pConnection = reinterpret_cast<nn::ssl::Connection *>(pSslConnection);
        result = pConnection->GetServerCertDetail(pDetail, pInBuffer, index);
    } while (NN_STATIC_CONDITION(false));

    return nn::result::ConvertToC(result);
}

nnResult nnsslConnectionRead(
    nnsslSslConnectionType* pSslConnection,
    char* pOutBuffer,
    int* pOutReadSizeCourier,
    uint32_t bufferLength)
{
    RETURN_UPON_INVALID_CLIENT_POINTER(pSslConnection);

    nn::Result result = nn::ResultSuccess();
    do{
    nn::ssl::Connection* pConnection = reinterpret_cast<nn::ssl::Connection*>(pSslConnection);
    result = pConnection->Read(pOutBuffer, pOutReadSizeCourier, bufferLength);
    }while(NN_STATIC_CONDITION(false));
    return nn::result::ConvertToC(result);
}

nnResult nnsslConnectionWrite(
    nnsslSslConnectionType* pSslConnection,
    const char* pInBuffer,
    int* pOutWrittenSizeCourier,
    uint32_t bufferLength)
{
    RETURN_UPON_INVALID_CLIENT_POINTER(pSslConnection);

    nn::Result result = nn::ResultSuccess();
    do{
    nn::ssl::Connection* pConnection = reinterpret_cast<nn::ssl::Connection*>(pSslConnection);
    result = pConnection->Write(pInBuffer, pOutWrittenSizeCourier, bufferLength);
    }while(NN_STATIC_CONDITION(false));
    return nn::result::ConvertToC(result);
}

nnResult nnsslConnectionPending(nnsslSslConnectionType* pSslConnection, int* pOutPendingSizeCourier)
{
    RETURN_UPON_INVALID_CLIENT_POINTER(pSslConnection);

    nn::Result result = nn::ResultSuccess();
    do{
    nn::ssl::Connection* pConnection = reinterpret_cast<nn::ssl::Connection*>(pSslConnection);
    result = pConnection->Pending(pOutPendingSizeCourier);
    }while(NN_STATIC_CONDITION(false));
    return nn::result::ConvertToC(result);
}

nnResult nnsslConnectionPeek(
    nnsslSslConnectionType* pSslConnection,
    char* pOutBuffer,
    int* pOutReadSizeCourier,
    uint32_t bufferLength)
{
    RETURN_UPON_INVALID_CLIENT_POINTER(pSslConnection);

    nn::Result result = nn::ResultSuccess();
    do{
    nn::ssl::Connection* pConnection = reinterpret_cast<nn::ssl::Connection*>(pSslConnection);
    result = pConnection->Peek(pOutBuffer, pOutReadSizeCourier, bufferLength);
    }while(NN_STATIC_CONDITION(false));
    return nn::result::ConvertToC(result);
}

nnResult nnsslConnectionPoll(
    nnsslSslConnectionType* pSslConnection,
    nnsslConnectionPollEvent* pOutEvent,
    nnsslConnectionPollEvent* pInEvent,
    uint32_t msecTimeout)
{
    RETURN_UPON_INVALID_CLIENT_POINTER(pSslConnection);

    nn::Result result = nn::ResultSuccess();
    do{
    nn::ssl::Connection* pConnection = reinterpret_cast<nn::ssl::Connection*>(pSslConnection);
    result = pConnection->Poll(
        reinterpret_cast<nn::ssl::Connection::PollEvent*>(pOutEvent),
        reinterpret_cast<nn::ssl::Connection::PollEvent*>(pInEvent),
        msecTimeout);
    }while(NN_STATIC_CONDITION(false));
    return nn::result::ConvertToC(result);
}

nnResult nnsslConnectionGetVerifyCertError(nnsslSslConnectionType* pSslConnection, nnResult* pOutVerifyError)
{
    RETURN_UPON_INVALID_CLIENT_POINTER(pSslConnection);

    nn::Result result = nn::ResultSuccess();
    do{
    nn::ssl::Connection* pConnection = reinterpret_cast<nn::ssl::Connection*>(pSslConnection);
    result = pConnection->GetVerifyCertError(reinterpret_cast<nn::Result*>(pOutVerifyError));
    }while(NN_STATIC_CONDITION(false));
    return nn::result::ConvertToC(result);
}

nnResult nnsslConnectionGetContextId(nnsslSslConnectionType* pSslConnection, nnsslSslContextId* pOutContextId)
{
    RETURN_UPON_INVALID_CLIENT_POINTER(pSslConnection);

    nn::Result result = nn::ResultSuccess();
    do{
    nn::ssl::Connection* pConnection = reinterpret_cast<nn::ssl::Connection*>(pSslConnection);
    result = pConnection->GetContextId(reinterpret_cast<nn::ssl::SslContextId*>(pOutContextId));
    }while(NN_STATIC_CONDITION(false));
    return nn::result::ConvertToC(result);
}

nnResult nnsslConnectionGetConnectionId(nnsslSslConnectionType* pSslConnection, nnsslSslConnectionId* pOutConnectionId)
{
    RETURN_UPON_INVALID_CLIENT_POINTER(pSslConnection);

    nn::Result result = nn::ResultSuccess();
    do{
    nn::ssl::Connection* pConnection = reinterpret_cast<nn::ssl::Connection*>(pSslConnection);
    result = pConnection->GetConnectionId(reinterpret_cast<nn::ssl::SslConnectionId*>(pOutConnectionId));
    }while(NN_STATIC_CONDITION(false));
    return nn::result::ConvertToC(result);
}

nnResult nnsslConnectionSetOption(
    nnsslSslConnectionType* pSslConnection,
    nnsslConnectionOptionType optionType,
    bool enable)
{
    RETURN_UPON_INVALID_CLIENT_POINTER(pSslConnection);

    nn::Result result = nn::ResultSuccess();
    do{
    nn::ssl::Connection* pConnection = reinterpret_cast<nn::ssl::Connection*>(pSslConnection);
    result = pConnection->SetOption(
        static_cast<nn::ssl::Connection::OptionType>(optionType),
        enable);
    }while(NN_STATIC_CONDITION(false));
    return nn::result::ConvertToC(result);
}

nnResult nnsslConnectionGetOption(
    nnsslSslConnectionType* pSslConnection,
    bool* pOutIsEnabled,
    nnsslConnectionOptionType optionType)
{
    RETURN_UPON_INVALID_CLIENT_POINTER(pSslConnection);

    nn::Result result = nn::ResultSuccess();
    do{
    nn::ssl::Connection* pConnection = reinterpret_cast<nn::ssl::Connection*>(pSslConnection);
    result = pConnection->GetOption(
        pOutIsEnabled,
        static_cast<nn::ssl::Connection::OptionType>(optionType));
    }while(NN_STATIC_CONDITION(false));
    return nn::result::ConvertToC(result);
}


#ifdef __cplusplus
}
#endif // __cplusplus
