﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/ssl/detail/ssl_Build.h>


namespace nn { namespace ssl { namespace detail {

class NssCore
{
public:
    enum InitStatus
    {
        InitStatus_InitFail       = -2,
        InitStatus_Idle           = -1,
        InitStatus_NotInitialized = 0,
        InitStatus_Initializing   = 1,
        InitStatus_InitDone       = 2,
    };

    typedef InitStatus (*DeferredInitCb)(void *pInArg);
    typedef void (*BgTaskCb)(void *pArg);
    typedef uint64_t BgTask;

    static const uint32_t               g_Ssl3SessionCacheTimeout = 3600; // an hour
    static const uint32_t               g_SidCacheEntryCount      = 128;
    static const uint32_t               g_DeferredInitThreadStackSize = 0x40000;
    static const uint32_t               g_BgTaskThreadStackSize = 0x40000;
    static const int                    g_MaxBgTasks = static_cast<int>(sizeof(uint64_t) * 8);
    static const int64_t                g_NssBgTaskWaitTimeoutMs = 500;

private:
    struct BgTaskInfo
    {
        BgTaskCb                pCb;
        void                    *pArg;
    };

    struct DeferredInitInfo
    {
        char                    *pId;
        InitStatus              status;
        DeferredInitCb          pCb;
        void                    *pArg;
    };

    static void                         *g_pNssDeferredInitThread;
    static nn::os::Semaphore            g_NssDeferredInitStart;
    static void                         *g_NssDeferredInitInfoList;    //  PRCList *, but left as void * to avoid header inclusion mess
    static InitStatus                   g_NssDeferredInitStatus;
    static nn::os::Mutex                g_NssDeferredInitLock;
    static nn::os::ConditionVariable    g_NssDeferredInitCv;
    static bool                         g_NssInitDone;

    static void                         *g_pNssBgTaskThread;
    static bool                         g_BgTaskShutdown;
    static nn::os::Mutex                g_NssBgTaskLock;
    static nn::os::ConditionVariable    g_NssBgTaskCv;
    static uint64_t                     g_NssBgTasks;
    static uint64_t                     g_ActiveBgTasks;
    static BgTaskInfo                   g_AllBgTaskInfo[];

    static InitStatus GetInitStatusLocked(const char *pInId);
    static void DeferredInitThreadCb(void *pArg);
    static void BgTaskThreadCb(void *pArg);
    static nn::Result QueueBgTask(BgTask task);

public:
    static nn::Result Initialize();
    static nn::Result Finalize();
    static void StartDeferredInit();

    static InitStatus GetInitStatus(bool wait);
    static InitStatus GetInitStatus(const char *pInId, bool wait);
    static nn::Result AddDeferredInit(const char     *pInId,
                                      DeferredInitCb pInCb,
                                      void           *pInArg);
    static nn::Result AllocBgTask(BgTask *pOutTask, BgTaskCb pInCb, void *pInArg);
    static nn::Result FreeBgTask(BgTask task);
    static void RunBgTask(BgTask task);
    static void ScheduleBgTask(BgTask task);
};

} } }
