﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdlib>
#include <cstring>

#include <nn/fs.h>

#include <nn/ssl/detail/ssl_Build.h>
#include <nn/ssl/detail/ssl_Common.h>

#include "ssl_NssCommon.h"
#include "ssl_Util.h"
#include "ssl_NssConfigurator.h"

namespace nn { namespace ssl { namespace detail {

// -------------------------------------------------------------------------------------------------
// Pkcs12ModuleConfigurator
// -------------------------------------------------------------------------------------------------
//  Taken from pk12util.c
SECStatus NssConfigurator::Initializer::Pkcs12ModuleConfigurator::SwapUnicodeBytes(SECItem *pOutUnicodeItem)
{
    uint32_t      i;
    unsigned char ascii;

    if ((pOutUnicodeItem == NULL) || (pOutUnicodeItem->len % 2))
    {
        return SECFailure;
    }

    for (i = 0; i < pOutUnicodeItem->len; i += 2)
    {
        ascii = pOutUnicodeItem->data[i];
        pOutUnicodeItem->data[i] = pOutUnicodeItem->data[i + 1];
        pOutUnicodeItem->data[i + 1] = ascii;
    }

    return SECSuccess;
}


PRBool NssConfigurator::Initializer::Pkcs12ModuleConfigurator::ConvertUnicodeToAscii(
    PRBool        toUnicode,
    unsigned char *inBuf,
    unsigned int  inBufLen,
    unsigned char *outBuf,
    unsigned int  maxOutBufLen,
    unsigned int  *outBufLen,
    PRBool        swapBytes)
{
    SECItem  it;
    SECItem* dup = NULL;
    PRBool   ret;

    NN_DETAIL_SSL_DBG_PRINT("[ConvertUnicodeToAscii] converting %d bytes: %s\n",
        inBufLen, inBuf);

    it.data = inBuf;
    it.len  = inBufLen;
    dup     = SECITEM_DupItem(&it);

    // If converting Unicode to ASCII, swap bytes before conversion as neccessary
    if (!toUnicode && swapBytes)
    {
        NN_DETAIL_SSL_DBG_PRINT("[ConvertUnicodeToAscii] swap needed\n");
        if (SwapUnicodeBytes(dup) != SECSuccess)
        {
            SECITEM_ZfreeItem(dup, PR_TRUE);
            return PR_FALSE;
        }
    }

    // Perform the conversion
    NN_DETAIL_SSL_DBG_PRINT("[ConvertUnicodeToAscii] do conversion to UTF8\n");
    ret = PORT_UCS2_UTF8Conversion(
        toUnicode,
        dup->data,
        dup->len,
        outBuf,
        maxOutBufLen,
        outBufLen);
    if (dup)
    {
        SECITEM_ZfreeItem(dup, PR_TRUE);
    }

    NN_DETAIL_SSL_DBG_PRINT("[ConvertUnicodeToAscii] returning %d\n", ret);
    return ret;
}

nn::Result NssConfigurator::Initializer::Pkcs12ModuleConfigurator::Initialize()
{
    //  Enable PKCS12 ciphers used for decrypting stores
    SEC_PKCS12EnableCipher(PKCS12_RC4_40, 1);
    SEC_PKCS12EnableCipher(PKCS12_RC4_128, 1);
    SEC_PKCS12EnableCipher(PKCS12_RC2_CBC_40, 1);
    SEC_PKCS12EnableCipher(PKCS12_RC2_CBC_128, 1);
    SEC_PKCS12EnableCipher(PKCS12_DES_56, 1);
    SEC_PKCS12EnableCipher(PKCS12_DES_EDE3_168, 1);
    SEC_PKCS12SetPreferredCipher(PKCS12_DES_EDE3_168, 1);

    //  Setup unicode converstion callback
    NN_DETAIL_SSL_DBG_PRINT("[Pkcs12ModuleConfigurator::Initialize] hooking UCS to ASCII\n");
    PORT_SetUCS2_ASCIIConversionFunction(ConvertUnicodeToAscii);

    return nn::ResultSuccess();
}


nn::Result NssConfigurator::Initializer::SetDefaultCipherPolicy(CipherPolicyRegion region)
{
    do
    {
        if(region == CipherPolicyRegion_US)
        {
            NN_DETAIL_SSL_BREAK_ON_NSS_FAILURE(NSS_SetDomesticPolicy());
        }
        else if(region == CipherPolicyRegion_FR)
        {
            NN_DETAIL_SSL_BREAK_ON_NSS_FAILURE(NSS_SetExportPolicy());
        }
        else if(region == CipherPolicyRegion_Export)
        {
            NN_DETAIL_SSL_BREAK_ON_NSS_FAILURE(NSS_SetFrancePolicy());
        }
        else
        {
            break;
        }

        // SIGLO-61732: Disabling RC4 ciphers (use of RC4 is prohibited in RFC7465)
        NN_DETAIL_SSL_BREAK_ON_NSS_FAILURE(SSL_CipherPolicySet(TLS_RSA_WITH_RC4_128_SHA, SSL_NOT_ALLOWED));
        NN_DETAIL_SSL_BREAK_ON_NSS_FAILURE(SSL_CipherPolicySet(TLS_RSA_WITH_RC4_128_MD5, SSL_NOT_ALLOWED));

        return ResultSuccess();
    } while(NN_STATIC_CONDITION(false));

    return ResultInternalNssFailedToSetCipherPolicy();
}

nn::Result NssConfigurator::Initializer::SetDefaultCipherPreference()
{
    // Reference
    // https://developer.mozilla.org/en-US/docs/Mozilla/Projects/NSS/SSL_functions/sslfnc.html#1068466
    do
    {
        //NN_DETAIL_SSL_BREAK_ON_NSS_FAILURE(SSL_CipherPrefSetDefault(TLS_DHE_RSA_WITH_AES_256_CBC_SHA, PR_TRUE));
        return ResultSuccess();
    } while(NN_STATIC_CONDITION(false));
}

nn::Result NssConfigurator::Initializer::SetDefaultOptions()
{
    // Configures Default NSS by SSL_OptionSetDefault which will change the default values for
    // all subsequently opened sockets
    //
    // [Options]
    // SSL_SECURITY: Enables use of security protocol. Factory setting is on.
    //   If you turn this option off, the session will not be an SSL session and will not have
    //    certificate-based authentication, tamper detection, or encryption.
    // SSL_REQUEST_CERTIFICATE: Server option
    // SSL_REQUIRE_CERTIFICATE: Server option
    // SSL_HANDSHAKE_AS_CLIENT: Controls the behavior of PR_Accept
    //   If this option is off, the PR_Accept configures the SSL socket to handshake as a server.
    //   If it is on, then PR_Accept configures the SSL socket to handshake as a client, even
    //   though it accepted the connection as a TCP server
    // SSL_HANDSHAKE_AS_SERVER: Controls the behavior of PR_Connect. If this option is off, then
    //    PR_Connect configures the SSL socket to handshake as a client. If it is on, then
    //    PR_Connect configures the SSL socket to handshake as a server, even though it connected
    //    as a TCP client.
    // SSL_ENABLE_FDX: Tells the SSL library whether the application will have two threads,
    //    one reading and one writing, or just one thread doing reads and writes alternately.
    //    The factory setting for this option (which is the default, unless the application changes
    //    the default) is off (PR_FALSE), which means that the application will not do simultaneous
    //    reads and writes. An application that wishes to do sumultaneous reads and writes should
    //    set this to PR_TRUE.
    // SSL_ENABLE_SSL3: Enables the application to communicate with SSL v3. Factory setting is on.
    //    If you turn this option off, an attempt to establish a connection with a peer that only
    //    understands SSL v3 will fail.
    // SSL_ENABLE_SSL2: Enables the application to communicate with SSL v2. Factory setting is on.
    //    If you turn this option off, an attempt to establish a connection with a peer that only
    //    understands SSL v2 will fail.
    // SSL_ENABLE_TLS: A peer of the SSL_ENABLE_SSL2 and SSL_ENABLE_SSL3 options.
    //    The IETF standard Transport Layer Security (TLS) protocol, RFC 2246, is a modified
    //    version of SSL3. It uses the SSL version number 3.1, appearing to be a "minor" revision
    //    of SSL 3.0. NSS 2.8 supports TLS in addition to SSL2 and SSL3. You can think of it as
    //    "SSL_ENABLE_SSL3.1". See the description below for more information about this option.
    // SSL_V2_COMPATIBLE_HELLO: Tells the SSL library whether or not to send SSL3 client hello
    //    messages in SSL2-compatible format. If set to PR_TRUE, it will; otherwise, it will not.
    //    Factory setting is on (PR_TRUE).
    // SSL_NO_CACHE: Disallows use of the session cache. Factory setting is off. If you turn this
    //     option on, this socket will be unable to resume a session begun by another socket.
    //     When this socket's session is finished, no other socket will be able to resume the session
    //     begun by this socket.
    // SSL_ROLLBACK_DETECTION: Disables detection of a rollback attack. Factory setting is on.
    //     You must turn this option off to interoperate with TLS clients ( such as certain versions of
    //     Microsoft Internet Explorer) that do not conform to the TLS specification regarding rollback
    //     attacks. Important: turning this option off means that your code will not comply with the
    //     TLS 3.1 and SSL 3.0 specifications regarding rollback attack and will therefore be vulnerable
    //     to this form of attack.
    do
    {
        NN_DETAIL_SSL_BREAK_ON_NSS_FAILURE(SSL_OptionSetDefault(SSL_SECURITY, PR_TRUE));
        NN_DETAIL_SSL_BREAK_ON_NSS_FAILURE(SSL_OptionSetDefault(SSL_HANDSHAKE_AS_CLIENT, PR_TRUE));
        NN_DETAIL_SSL_BREAK_ON_NSS_FAILURE(SSL_OptionSetDefault(SSL_HANDSHAKE_AS_SERVER, PR_FALSE));
        NN_DETAIL_SSL_BREAK_ON_NSS_FAILURE(SSL_OptionSetDefault(SSL_ENABLE_FDX, PR_TRUE));
        NN_DETAIL_SSL_BREAK_ON_NSS_FAILURE(SSL_OptionSetDefault(SSL_ENABLE_SSL3, PR_FALSE));
        NN_DETAIL_SSL_BREAK_ON_NSS_FAILURE(SSL_OptionSetDefault(SSL_ENABLE_SSL2, PR_FALSE));
        NN_DETAIL_SSL_BREAK_ON_NSS_FAILURE(SSL_OptionSetDefault(SSL_NO_CACHE, PR_FALSE) );
        NN_DETAIL_SSL_BREAK_ON_NSS_FAILURE(SSL_OptionSetDefault(SSL_ROLLBACK_DETECTION, PR_TRUE));
        return ResultSuccess();
    } while(NN_STATIC_CONDITION(false));

    return ResultInternalNssFailedToSetDefaultSslOption();
};

}}}
