﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/os.h>

#include "ssl_BdfReader.h"
#include "ssl_StaticDataReader.h"
#include "ssl_SystemDataReader.h"
#include "ssl_Util.h"

namespace nn { namespace ssl { namespace detail {

//////////////////////////////////////////////////////////////////////////////
//  BdfReader - The reader factory and interface for getting built-in data.
//////////////////////////////////////////////////////////////////////////////
nn::Result BdfReaderFactory::Initialize() NN_NOEXCEPT
{
    return SystemDataReader::Initialize();
}


nn::Result BdfReaderFactory::Finalize() NN_NOEXCEPT
{
    return SystemDataReader::Finalize();
}


BdfReader *BdfReaderFactory::Create(BdfReader::BdfId id) NN_NOEXCEPT
{
    BdfReader                   *pNewReader = nullptr;
    nn::Result                  result = ResultSuccess();

    do
    {
        switch (id)
        {
            case BdfReader::BdfId_TrustedCerts:
                pNewReader = new SystemDataReader(id);
                if (pNewReader == nullptr)
                {
                    NN_DETAIL_SSL_DBG_PRINT("[BdfReaderFactory::Create] failed to create new instance of SystemDataReader for id %8.8X\n",
                                            id);
                    break;
                }

                break;

            case BdfReader::BdfId_Crl:
                pNewReader = new SystemDataReader(id);
                if (pNewReader == nullptr)
                {
                    NN_DETAIL_SSL_DBG_PRINT("[BdfReaderFactory::Create] failed to create new instance of SystemDataReader for id %8.8X\n",
                                            id);
                    break;
                }

                break;

            case nn::ssl::detail::BdfReader::BdfId_TestCrl:
            {
                if(Util::IsInitializeForTestEnabled())
                {
                    pNewReader = new SystemDataReader(id);
                    if (pNewReader == nullptr)
                    {
                        NN_DETAIL_SSL_DBG_PRINT("[BdfReaderFactory::Create] failed to create new instance of SystemDataReader for id %8.8X\n",
                                                id);
                        break;
                    }
                }
                else
                {
                    NN_DETAIL_SSL_DBG_PRINT("[BdfReaderFactory::Create] Please enable initialize_for_test for ssl.test\n", id);
                }

                break;
            }

            default:
                NN_DETAIL_SSL_DBG_PRINT("[BdfReaderFactory::Create] invalid id: %8.8X\n", id);
                break;
        }

        if (pNewReader != nullptr)
        {
            result = pNewReader->Open();
            if (result.IsFailure())
            {
                NN_DETAIL_SSL_DBG_PRINT("[BdfReaderFactory::Create] failed to open BdfReader for id %8.8X: %d-%d\n",
                                        id,
                                        result.GetModule(),
                                        result.GetDescription());
                delete pNewReader;
                pNewReader = nullptr;
                break;
            }
        }
    } while (NN_STATIC_CONDITION(false));

    return pNewReader;
}

} } }    //  nn::ssl::detail
