﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "ssl_BdfReader.h"
#include "ssl_Util.h"

namespace nn { namespace ssl { namespace detail {


BdfReader::BdfReader()
{
    //  Clear the header to ensure it is in an empty, uninitialized state
    memset(&m_hdr, 0, sizeof(m_hdr));
}


nn::Result BdfReader::GetDataCount(uint32_t *pOutCount) NN_NOEXCEPT
{
    nn::Result                  result = ResultSuccess();
    size_t                      bytesRead;

    do
    {
        if (m_hdr.magicNum != BuiltinDataInfo::g_BdfMagicNum)
        {
            result = Read(&bytesRead,
                          0,
                          reinterpret_cast<uint8_t *>(&m_hdr),
                          sizeof(m_hdr));
            if (result.IsFailure())
            {
                NN_DETAIL_SSL_DBG_PRINT("[BdfReader::GetDataCount] error reading hdr: %d-%d\n",
                                        result.GetModule(),
                                        result.GetDescription());
                break;
            }

            if (m_hdr.magicNum != BuiltinDataInfo::g_BdfMagicNum)
            {
                NN_DETAIL_SSL_DBG_PRINT("[BdfReader::GetDataCount] invalid hdr in data, magic number is 8.8X\n",
                                        m_hdr.magicNum);
                result = ResultErrorLower();
                break;
            }
        }

        *pOutCount = m_hdr.dataCount;
    } while (NN_STATIC_CONDITION(false));

    return result;
}


nn::Result BdfReader::GetMetaDataEntry(BdfMetaDataEntry  *pOutMetaDataEntry,
                                       uint32_t          dataIndex) NN_NOEXCEPT
{
    nn::Result                  result = nn::ResultSuccess();
    uint32_t                    dataCount = m_hdr.dataCount;
    int64_t                     baseOffset;
    size_t                      bytesRead;

    //  The base offset here is from the start of the BDF since that is where
    //  meta data begins.  The offsets in the meta data are taken from
    //  AFTER this point, so they are added to this to get the correct offset
    //  within the file.
    baseOffset = sizeof(BdfMetaHdr);

    do
    {
        //  If we haven't already read the header, do it now
        if (m_hdr.magicNum != BuiltinDataInfo::g_BdfMagicNum)
        {
            result = GetDataCount(&dataCount);
            if (result.IsFailure())
            {
                NN_DETAIL_SSL_DBG_PRINT("[BdfReader::GetMetaDataEntry] failed to get hdr data: %d-%d\n",
                                        result.GetModule(),
                                        result.GetDescription());
                break;
            }
        }

        //  Make sure the requested index is within the bounds
        if (dataIndex >= m_hdr.dataCount)
        {
            NN_DETAIL_SSL_DBG_PRINT("[BdfReader::GetMetaDataEntry] invalid index %u (count %u)\n",
                                    dataIndex,
                                    m_hdr.dataCount);
            result = ResultErrorLower();
            break;
        }

        //  Read out the cert meta data for this cert (index)
        result = Read(&bytesRead,
                      baseOffset + (dataIndex * sizeof(*pOutMetaDataEntry)),
                      reinterpret_cast<uint8_t *>(pOutMetaDataEntry),
                      sizeof(*pOutMetaDataEntry));
        if (!result.IsSuccess())
        {
            NN_DETAIL_SSL_DBG_PRINT("[BdfReader::GetMetaDataEntry] failed to read data %u: %d-%d\n",
                                    dataIndex,
                                    result.GetModule(),
                                    result.GetDescription());
            break;
        }
    } while (NN_STATIC_CONDITION(false));

    return result;
}


nn::Result BdfReader::GetData(uint8_t           *pOutBuf,
                              uint32_t          bufLen,
                              BdfMetaDataEntry  *pMetaDataEntry) NN_NOEXCEPT
{
    nn::Result                  result = nn::ResultSuccess();
    size_t                      bytesRead;

    do
    {
        if (bufLen < pMetaDataEntry->dataSize)
        {
            NN_DETAIL_SSL_DBG_PRINT("[BdfReader::GetMetaDataEntry] buf too small, size %u, need %u\n",
                                    bufLen,
                                    pMetaDataEntry->dataSize);
            result = ResultInsufficientMemory();
            break;
        }

        result = Read(&bytesRead,
                      sizeof(BdfMetaHdr) + pMetaDataEntry->dataOffset,
                      pOutBuf,
                      static_cast<size_t>(pMetaDataEntry->dataSize));
        if (result.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[BdfReader::GetMetaDataEntry] ERROR unable to read data id 0x%8.8X, at offset %u, size %u: %d-%d\n",
                                    pMetaDataEntry->dataId,
                                    pMetaDataEntry->dataOffset,
                                    pMetaDataEntry->dataSize,
                                    result.GetModule(),
                                    result.GetDescription());
            break;
        }
    } while (NN_STATIC_CONDITION(false));

    return result;
}


} } }    //  nn::ssl::detail
