﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>

#include "spsm_InternalTimer.h"

namespace nn { namespace spsm { namespace observer {

namespace {
    nn::os::EventType   g_StartTimerEvent;
    nn::os::EventType   g_CancelTimerEvent;
    nn::TimeSpan        g_Duration;
}

//-----------------------------------------------------------------------------
//  初期化
//
void InternalTimer::Initialize(nn::spsm::server::PowerStateMessageQueue* pPowerStateMessageQueue) NN_NOEXCEPT
{
    m_pPowerStateMessageQueue = pPowerStateMessageQueue;

    os::InitializeEvent(&g_StartTimerEvent, false, os::EventClearMode_AutoClear);
    os::InitializeEvent(&g_CancelTimerEvent, false, os::EventClearMode_AutoClear);
    os::InitializeMultiWaitHolder(&m_TimerEventHolder, m_TimerEvent.GetBase());
    os::InitializeMultiWaitHolder(&m_StartTimerEventHolder, &g_StartTimerEvent);
    os::InitializeMultiWaitHolder(&m_CancelTimerEventHolder, &g_CancelTimerEvent);
    m_TimerEventHolder.userData = reinterpret_cast<uintptr_t>(this);
    m_StartTimerEventHolder.userData = reinterpret_cast<uintptr_t>(this);
    m_CancelTimerEventHolder.userData = reinterpret_cast<uintptr_t>(this);
}


//-----------------------------------------------------------------------------
//  デストラクタ
//
void InternalTimer::Finalize() NN_NOEXCEPT
{
    CancelTimer();
    os::FinalizeMultiWaitHolder(&m_TimerEventHolder);
    os::FinalizeMultiWaitHolder(&m_StartTimerEventHolder);
    os::FinalizeMultiWaitHolder(&m_CancelTimerEventHolder);
    os::FinalizeEvent(&g_StartTimerEvent);
    os::FinalizeEvent(&g_CancelTimerEvent);
}

//-----------------------------------------------------------------------------
//  TimerEvent ハンドラ関数（多重待ち解除時に呼ばれる）
//
void InternalTimer::ProcessTimerEvent() NN_NOEXCEPT
{
    if (!m_TimerEvent.TryWait())
    {
        return;
    }
    NotifyInternalTimerExpired();
    CancelTimer();
}

void InternalTimer::ProcessStartTimerEvent() NN_NOEXCEPT
{
    if (!os::TryWaitEvent(&g_StartTimerEvent))
    {
        return;
    }
    StartTimer();
}

void InternalTimer::ProcessCancelTimerEvent() NN_NOEXCEPT
{
    if (!os::TryWaitEvent(&g_CancelTimerEvent))
    {
        return;
    }
    CancelTimer();
}

void InternalTimer::StartTimer() NN_NOEXCEPT
{
    if ( !m_State )
    {
        m_TimerEvent.Clear();
        m_TimerEvent.StartOneShot(g_Duration);
        m_State = true;
    }
}

void InternalTimer::CancelTimer() NN_NOEXCEPT
{
    if ( m_State )
    {
        m_TimerEvent.Stop();
        m_TimerEvent.Clear();
        m_State = false;
    }
}

//-----------------------------------------------------------------------------

void StartInternalTimer(nn::TimeSpan duration) NN_NOEXCEPT
{
    g_Duration = duration;
    os::SignalEvent(&g_StartTimerEvent);
}

void CancelInternalTimer() NN_NOEXCEPT
{
    os::SignalEvent(&g_CancelTimerEvent);
}

}}} // namespace nn::spsm::observer

