﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/hid/system/hid_HomeButton.h>
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#include <nn/btm/btm_Api.h>
#endif
#include <mutex>

#include "spsm_HomeButton.h"

namespace nn { namespace spsm { namespace observer {

//-----------------------------------------------------------------------------
//  初期化
//  TORIAEZU:
//      初期化時は HOME ボタンが押されていないと仮定する。
//      仮に押されていたとしても長押しとして検知しない。
//
void HomeButton::Initialize(nn::spsm::server::PowerStateMessageQueue* pPowerStateMessageQueue) NN_NOEXCEPT
{
    m_pPowerStateMessageQueue = pPowerStateMessageQueue;

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    hid::system::BindHomeButtonEvent(m_HomeButtonEvent.GetBase(), os::EventClearMode_AutoClear);
    m_HomeButtonEvent.Clear();

    btm::RegisterSystemEventForAwakeReq(m_WakeupEvent.GetBase());
    m_WakeupEvent.Clear();
#endif

    m_TimeShortPressing = GetTimeSpanShortPressing();
    m_TimeLongPressing  = GetTimeSpanLongPressing();

    os::InitializeMultiWaitHolder(&m_TimerEventHolder,      m_TimerEvent.GetBase());
    os::InitializeMultiWaitHolder(&m_HomeButtonEventHolder, m_HomeButtonEvent.GetBase());
    os::InitializeMultiWaitHolder(&m_WakeupEventHolder,     m_WakeupEvent.GetBase());

    m_TimerEventHolder.userData      = reinterpret_cast<uintptr_t>(this);
    m_HomeButtonEventHolder.userData = reinterpret_cast<uintptr_t>(this);
    m_WakeupEventHolder.userData     = reinterpret_cast<uintptr_t>(this);
}


//-----------------------------------------------------------------------------
//  デストラクタ
//
void HomeButton::Finalize() NN_NOEXCEPT
{
    os::FinalizeMultiWaitHolder(&m_TimerEventHolder);
    os::FinalizeMultiWaitHolder(&m_HomeButtonEventHolder);
    os::FinalizeMultiWaitHolder(&m_WakeupEventHolder);
}


//-----------------------------------------------------------------------------
//  現在のボタン押下を無効化（効力は１回だけ）
//
void HomeButton::InvalidateCurrentButtonPressing(TimeSpan extensionTime) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    this->m_State = false;
    m_TimerEvent.Stop();
    m_TimerEvent.Clear();
    m_HomeButtonEvent.Clear();

    m_EndTickToInvalidate = os::GetSystemTick() + os::ConvertToTick( extensionTime );
}


//-----------------------------------------------------------------------------
//  TimerEvent ハンドラ関数（多重待ち解除時に呼ばれる）
//
void HomeButton::TimerEventHandler() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    if (!m_TimerEvent.TryWait())
    {
        return;
    }

    if (m_State)
    {
        auto interval = (os::GetSystemTick() - m_StartTick).ToTimeSpan();

        // m_TimeLongPressing 通過時点で長押しを検知
        if (interval >= m_TimeLongPressing)
        {
            NotifyLongPressingHomeButton();
        }
    }
}


//-----------------------------------------------------------------------------
//  HomeButtonEvent ハンドラ関数（多重待ち解除時に呼ばれる）
//
void HomeButton::HomeButtonEventHandler() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    if (!m_HomeButtonEvent.TryWait())
    {
        return;
    }

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    hid::system::HomeButtonState    homeButtonState;
    int count = hid::system::GetHomeButtonStates(&homeButtonState, 1);
    NN_SDK_ASSERT(count == 1);
    NN_UNUSED(count);

    auto currentTick = os::GetSystemTick();
    if ( homeButtonState.buttons.Test<hid::system::HomeButton::Active>() )
    {
        if (currentTick >= m_EndTickToInvalidate)
        {
            // 押下エッジ検出なら時間計測開始
            m_StartTick = currentTick;
            m_State     = true;
            m_TimerEvent.Clear();
            m_TimerEvent.StartOneShot( m_TimeLongPressing );
        }

        NotifyStartPressingHomeButton();
    }
    else
    {
        // 開放エッジ検出なら時間計測停止
        m_TimerEvent.Stop();

        // ボタンが押されていたら、その時間によってメッセージを通知
        if (m_State)
        {
            auto interval = (currentTick - m_StartTick).ToTimeSpan();

            // ここでは 30msec～m_TimeLongPressing 区間の時のみ短押しのメッセージを通知
            if (m_TimeShortPressing <= interval &&
                                       interval < m_TimeLongPressing)
            {
                NotifyShortPressingHomeButton();
            }
        }
        m_State = false;
    }
#endif
}

void HomeButton::WakeupEventHandler() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    if (!m_WakeupEvent.TryWait())
    {
        return;
    }

    NotifyFullAwakeRequestByController();
}

void HomeButton::ChangeLongPressingTime(TimeSpan timeLongPressing) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    if (timeLongPressing < m_TimeShortPressing)
    {
        timeLongPressing = m_TimeShortPressing;
    }

    if (m_TimeLongPressing == timeLongPressing)
    {
        return;
    }

    this->m_TimeLongPressing = timeLongPressing;
    if (m_State)
    {
        // 押し下げ中だったら、タイマーを停止
        m_TimerEvent.Stop();
        m_TimerEvent.Clear();
        auto elapsed = (os::GetSystemTick() - m_StartTick).ToTimeSpan();

        if (elapsed >= m_TimeLongPressing)
        {
            // すでに m_TimeLongPressing 経過していたら、長押し検知
            NotifyLongPressingHomeButton();
            return;
        }
        else
        {
            // そうでなければ、残り時間で再度タイマーをセット
            m_TimerEvent.StartOneShot(m_TimeLongPressing - elapsed);
        }
    }
}

//-----------------------------------------------------------------------------

}}} // namespace nn::spsm::observer

