﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/spsm/server/spsm_Server.h>

#include "spsm_EventHandlerManager.h"

namespace nn { namespace spsm { namespace observer {

EventHandlerManager::EventHandlerManager() NN_NOEXCEPT
{
}

//-----------------------------------------------------------------------------
//  イベントハンドラ用の自動ループ関数
//  - POWER ボタンの短押し／中押し／長押し検知
//  - 自動スリープ (APD) の発動検知
//  - 温度異常検知
//  - システム電圧低下／電池残量僅少検知
//  - CEC システムスタンバイ受信
//  - HOME ボタンの短押し／長押し検知
//  - バックグラウンドタスクの完了
//  - 給電状態の変化
//  - SD カード/ゲームカードの挿抜
//

void EventHandlerManager::EventHandlerLoopAutoImpl(nn::os::SystemEventType* pCecEventType) NN_NOEXCEPT
{
    IEventHandler* const EventHandlerImplList[] =
    {
        &m_PowerButton,
        &m_AutoSleepRequest,
        &m_ThermalCoordinator,
        &m_ThermalCoordinatorAdvancedPolicy,
        &m_LowBattery,
        &m_CecSystemStandby,
        &m_BackgroundTask,
        &m_HomeButton,
        &m_InternalTimer,
        &m_StorageDetection,
        &m_DebugPowerStateMessageSender,
    };

    auto pPowerStateMessageQueue = nn::spsm::server::GetPowerStateMessageQueue();
    for ( auto& handler : EventHandlerImplList )
    {
        handler->Initialize(pPowerStateMessageQueue);
    }

    // FIXME: nn::cec のインタフェースの制約により、同一プロセス内で複数のイベントを登録することができず、外部から供給されなければならない
    m_CecSystemStandby.SetCecEvent(pCecEventType);

    // 前処理
    os::MultiWaitType multi;
    os::InitializeMultiWait(&multi);

#if defined(NN_BUILD_CONFIG_OS_WIN)
    // スレッド終了用イベントのリンク
    os::InitializeEvent(&m_StopEvent, false, os::EventClearMode_AutoClear);
    os::MultiWaitHolderType stopEventHolder;
    os::InitializeMultiWaitHolder(&stopEventHolder, &m_StopEvent);
    stopEventHolder.userData = reinterpret_cast<uintptr_t>(&m_StopEvent);
    os::LinkMultiWaitHolder(&multi, &stopEventHolder);
#endif

    // ステートマシンへのオブザーバの登録
    nn::spsm::server::StateChangeObserver stateChangeObserver(
        // OnStateEntry
        [&](PowerState enteredState)
        {
            for ( auto& handler : EventHandlerImplList )
            {
                handler->OnPowerStateEntry(enteredState);
            }
        },
        // OnStateExit
        [&](PowerState exitedState)
        {
            for ( auto& handler : EventHandlerImplList )
            {
                handler->OnPowerStateExit(exitedState);
            }
        }
    );
    nn::spsm::server::AddStateChangeObserver(&stateChangeObserver);

    // 関連ホルダのリンク
    for ( auto& handler : EventHandlerImplList )
    {
        handler->LinkMultiWaitHolders(&multi);
    }

    // ここからループ処理
    for (;;)
    {
        auto* holder = os::WaitAny(&multi);

#if defined(NN_BUILD_CONFIG_OS_WIN)
        if ( holder->userData == reinterpret_cast<uintptr_t>(&m_StopEvent) )
        {
            break;
        }
#endif

        for ( auto& handler : EventHandlerImplList )
        {
            if ( holder->userData == reinterpret_cast<uintptr_t>(handler) )
            {
                if ( handler->HandleEventIfHolderOwner(holder) )
                {
                    break;
                }
            }
        }
    }

#if defined(NN_BUILD_CONFIG_OS_WIN)
    // NN_DETAIL_SPSM_INFO("Quitting event handler loop\n");

    // 後処理
    nn::spsm::server::RemoveStateChangeObserver(&stateChangeObserver);

    for ( auto& handler : EventHandlerImplList )
    {
        handler->UnlinkMultiWaitHolders(&multi);
    }

    os::UnlinkMultiWaitHolder(&stopEventHolder);
    os::FinalizeMultiWaitHolder(&stopEventHolder);
    os::FinalizeEvent(&m_StopEvent);
    os::FinalizeMultiWait(&multi);

    for ( auto& handler : EventHandlerImplList )
    {
        handler->Finalize();
    }
#endif
}

#if defined(NN_BUILD_CONFIG_OS_WIN)
// ユニットテストでプログラムを正常終了するために必要（ないと MultiWaitHolder オブジェクトが Unlink されないまま終了し、os でアサートが出る）
void EventHandlerManager::StopEventHandlerLoopAutoImpl() NN_NOEXCEPT
{
    os::SignalEvent(&m_StopEvent);
}
#endif

//-----------------------------------------------------------------------------


}}} // namespace nn::spsm::observer

