﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_SystemSaveData.h>

#include "settings_SystemDataFile-os.win.h"

namespace nn { namespace settings { namespace detail {

//!< システムセーブデータを扱うクラスです。
class SystemSaveData final
{
private:
    //!< インメモリモードか否か
    bool m_IsInMemoryMode;

    //!< メモリのアドレス
    ::nn::Bit8* m_MemoryAddress;

    //!< メモリのサイズ
    size_t m_MemorySize;

    //!< システムデータファイル
    SystemDataFile m_SystemDataFile;

public:
    SystemSaveData() NN_NOEXCEPT;

    explicit SystemSaveData(const SystemDataFile& file) NN_NOEXCEPT;

    //!< システムセーブデータの識別子を設定します。
    void SetSystemSaveDataId(::nn::fs::SystemSaveDataId value) NN_NOEXCEPT;

    //!< システムセーブデータの合計サイズを設定します。
    void SetTotalSize(int64_t value) NN_NOEXCEPT;

    //!< システムセーブデータのジャーナルサイズを設定します。
    void SetJournalSize(int64_t value) NN_NOEXCEPT;

    //!< システムセーブデータのフラグ集合を設定します。
    void SetFlags(uint32_t value) NN_NOEXCEPT;

    //!< システムセーブデータのマウント名を設定します。
    void SetMountName(const char* name) NN_NOEXCEPT;

    //!< システムセーブデータをインメモリモードを有効にします。（デバッグ用）
    ::nn::Result EnableInMemoryModeForDebug(
        void* buffer, size_t size) NN_NOEXCEPT;

    //!< システムセーブデータをマウントします。
    ::nn::Result Mount(bool creates) NN_NOEXCEPT;

    //!< システムセーブデータの更新をコミットします。
    ::nn::Result Commit(bool synchronizes) NN_NOEXCEPT;

    //!< システムセーブデータを作成します。
    ::nn::Result Create(int64_t size) NN_NOEXCEPT;

    //!< システムセーブデータを読み込み権限で開きます。
    ::nn::Result OpenToRead() NN_NOEXCEPT;

    //!< システムセーブデータを書き込み権限で開きます。
    ::nn::Result OpenToWrite() NN_NOEXCEPT;

    //!< システムセーブデータを閉じます。
    void Close() NN_NOEXCEPT;

    //!< システムセーブデータから読み込みます。
    ::nn::Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT;

    //!< システムセーブデータへ書き込みます。
    ::nn::Result Write(
        int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT;

    //!< システムセーブデータへの書き込みをフラッシュします。
    ::nn::Result Flush() NN_NOEXCEPT;
};

}}} // namespace nn::settings::detail
