﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_StaticAssert.h>

namespace nn { namespace settings { namespace detail {

//!< 指定された文字を文字列としてバッファに書き出します。
char* WriteString(char* buffer, int size, char value) NN_NOEXCEPT;

//!< 指定された文字列を文字列としてバッファに書き出します。
char* WriteString(char* buffer, int size, const char* value) NN_NOEXCEPT;

//!< 可変型の文字列を表す構造体です。
template<int Capacity>
struct StringBuilder final
{
    NN_STATIC_ASSERT(0 < Capacity);

    //!< 文字列を格納するメモリ領域
    char buffer[Capacity];

    //!< 文字列の末尾を指すポインタ
    char* tail;

    //!< 文字列のポインタを返します。
    const char* Get() const NN_NOEXCEPT
    {
        return this->buffer;
    }

    //!< 文字列の長さを返します。
    int GetLength() const NN_NOEXCEPT
    {
        return static_cast<int>(this->tail - this->buffer);
    }

    //!< 文字列をクリアします。
    StringBuilder& Clear() NN_NOEXCEPT
    {
        this->tail = &(this->buffer[0]);

        *(this->tail) = '\0';

        return *this;
    }

    //!< 指定された文字を文字列に追加します。
    StringBuilder& Append(char value) NN_NOEXCEPT
    {
        const auto size = static_cast<int>(buffer + Capacity - tail);

        tail = WriteString(tail, size, value);

        return *this;
    }

    //!< 指定された文字列を文字列に追加します。
    StringBuilder& Append(const char* value) NN_NOEXCEPT
    {
        const auto size = static_cast<int>(buffer + Capacity - tail);

        tail = WriteString(tail, size, value);

        return *this;
    }
};

}}} // namespace nn::settings::detail
